/*
 * File:		 fmt_3ds.c
 * 
 * Description:  loads 3ds files; interfaces with lib3ds
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */

/*
 * Portions of this file were ripped from:
 *	- player.c from the lib3ds examples directory
 *		 player.c is copyright J.E. Hoffmann <je-h@gmx.net>
 */

#ifdef HAVE_LIB3DS

#include <lib3ds/file.h>
#include <lib3ds/camera.h>
#include <lib3ds/mesh.h>
#include <lib3ds/node.h>
#include <lib3ds/material.h>
#include <lib3ds/matrix.h>
#include <lib3ds/vector.h>
#include <lib3ds/light.h>

#include <config.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <gtk/gtk.h>

#include "gui.h"

#include "transform.h"
#include "globals.h"
#include "mesh.h"
#include "model.h"
#include "group.h"
#include "polygon.h"
#include "vertex.h"
#include "texture.h"
#include "bottombar.h"
#include "tools.h"

#include "model_load_save.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


gint f3ds_load_doc(struct fu_file *file, Model *doc);
gint f3ds_save_doc(struct fu_file *file, Model *doc);
void f3ds_destroy_priv_data(Model *doc);


struct doc_file_fmt fmt_f3ds = {
	"3d Studio Format (3ds)",
	"3ds",
	f3ds_load_doc,
	NULL,
	NULL
};


/* FILE-SCOPE VARS **********************************************/
static Lib3dsFile *lib3ds_file=0;
static int f3ds_current_frame = 0;
static float f3ds_scale_factor = 1.0;
static Model * model;


/* PROTOTYPES ***************************************************/
static int f3ds_init( Model *doc );
static GNode *f3ds_load_node( Lib3dsNode *node );
static void f3ds_load( Model *doc );

gint f3ds_prompt_user( void ) ;



/* Load the document from fu_file and set the root member of doc */
gint f3ds_load_doc( struct fu_file *file, Model *doc ) {
	int ret;

	/* Note - model_new has already created an empty mesh and groupnode.  
	We don't need either, so we'll delete them here. */
	if( doc->root )
		group_delete( doc->root );
	
	ret = f3ds_init( doc );
	if( !ret )
		return FALSE;

	ret = f3ds_prompt_user();
	if( !ret )
		return FALSE;

	/* set current frame */
	lib3ds_file_eval( lib3ds_file, f3ds_current_frame );

	f3ds_load( doc );
	
	/* scale the model using f3ds_scale_factor */
	tools_vertices_scale_about_origin( doc, doc->verts, f3ds_scale_factor, TRUE );

	return TRUE;
}


/* Save the document to fu_file */
gint f3ds_save_doc( struct fu_file *file, Model *doc ) {

	fu_truncate(file, 0);

	return TRUE;
}


/* Destroy the contents of doc->fmt_data */
void f3ds_destroy_priv_data(Model *doc) {}


/************************************************************************
* 
* LOADING STUFF
* 
*************************************************************************/


/*!
 *
 */
static int
f3ds_init( Model *doc ) {

	/* I *think* it's ok to open the same file several times w/in the same
	   process, but only for reading... we're going to have trouble in the 
	   save routines...
	 */
	lib3ds_file = lib3ds_file_load( doc->fname );
	if (!lib3ds_file) {
		puts("***ERROR*** Loading 3DS file failed.");
		return FALSE;
	}

	model = doc;

	return TRUE;
}



static Mesh *
f3ds_load_mesh( Lib3dsMesh *lib3ds_mesh ) {

	Mesh *kl3d_mesh = NULL;

	ASSERT( lib3ds_mesh );
	if( !lib3ds_mesh ) {
		return NULL;
	}

printf( "%s - doing node %s\n", __FUNCTION__, lib3ds_mesh->name );

	kl3d_mesh = mesh_new( model );
	mesh_set_name( kl3d_mesh, lib3ds_mesh->name );

	{
		unsigned i;

		/* fixme - ignoring xform matrices and normals...
				Lib3dsVector *normalL=malloc(3*sizeof(Lib3dsVector)*mesh->faces);
		 
				{
				  Lib3dsMatrix M;
				  lib3ds_matrix_copy(M, mesh->matrix);
				  lib3ds_matrix_inv(M);
				  glMultMatrixf(&M[0][0]);
				}
				lib3ds_mesh_calculate_normals(mesh, normalL);
		*/

		/* for each vertex in the mesh */
		for( i = 0; i < lib3ds_mesh->points; ++i ) {
			Lib3dsPoint *p = &lib3ds_mesh->pointL[i];
			Vertex *new_vert = vertex_new();

			new_vert->v[0] = p->pos[0];
			new_vert->v[1] = p->pos[1];
			new_vert->v[2] = p->pos[2];

			model_vertex_add( model, new_vert );
			mesh_vertex_add( kl3d_mesh, new_vert );
		}

		/* for each poly in the mesh */
		for( i = 0; i < lib3ds_mesh->faces; ++i ) {
			int j;
			Lib3dsFace *f = &lib3ds_mesh->faceL[i];
			Poly *new_poly = poly_new();

			/* for each vertex in this face... */
			for( j = 0; j < 3; ++j ) {
				poly_add_vertex( new_poly,
								 g_slist_nth_data( kl3d_mesh->vertices, f->points[j] ));
			}

			mesh_polygon_add( kl3d_mesh, new_poly );
		}

		/*
		free(normalL);
		*/
	}

	return kl3d_mesh;
}




/*!
 *
 */
static GNode *
f3ds_load_node( Lib3dsNode *node ) {
	GNode *result = NULL;
	Mesh *kl3d_mesh = NULL;
	Lib3dsMesh *lib3ds_mesh = NULL;

	ASSERT( lib3ds_file );


	if( node->type != LIB3DS_OBJECT_NODE ) {
		return NULL;
	}
	if( strcmp(node->name,"$$$DUMMY")==0 ) {
		return NULL;
	}

	lib3ds_mesh = lib3ds_file_mesh_by_name(lib3ds_file, node->name);
	kl3d_mesh = f3ds_load_mesh( lib3ds_mesh );
	
	if( kl3d_mesh == NULL )
		return NULL;

	result = g_node_new( kl3d_mesh );

	/* now descend farther down the tree, and handle the children as well */
	{
		Lib3dsNode *p;
		GNode *n;
		for( p = node->childs; p != 0; p = p->next ) {
			n = f3ds_load_node( p );
			if( n ) {
				g_node_append( result, n );
			} else {
				printf( "%s - got a null node...\n", __FUNCTION__ );
			}
		}
	}

	/* return our GNode* */
	return result;
}




/*!
 *
 */
static void
f3ds_load( Model *doc ) {
	if( !lib3ds_file ) {
		return;
	}
	if( !doc ) {
		return;
	}

	doc->root = group_new( doc );

/*printf( "%s: file %s contains pointers: %x nodes, %x meshes\n", __FUNCTION__, doc->fname, lib3ds_file->nodes, lib3ds_file->meshes );*/

	/* add children to the root node */
	if( lib3ds_file->nodes ) {
		Lib3dsNode *p;
		GNode *n;
		for( p = lib3ds_file->nodes; p != 0; p = p->next ) {
			n = f3ds_load_node( p );
			if( n ) {
				g_node_append( doc->root, n );
			} else {
				printf( "%s - got a null node...\n", __FUNCTION__ );
			}
		}
	} else {
	
		Lib3dsMesh *l3dsm;
		Mesh *kl3dm;
		for( l3dsm = lib3ds_file->meshes; l3dsm != 0; l3dsm = l3dsm->next ) {
			kl3dm = f3ds_load_mesh( l3dsm );
			if( kl3dm ) {
				g_node_append( doc->root, g_node_new( kl3dm ) );
			} else {
				printf( "%s - got a null mesh...\n", __FUNCTION__ );
			}
		}
	}

}


/* GUI STUFF ***************************************************************/

GtkWidget *f3ds_scale_entry = NULL;
GtkWidget *f3ds_frame_spinner = NULL;

gint f3ds_prompt_user( void ) {

	GtkWidget *dialog;
	GtkWidget *hbox, *widget;
	GtkAdjustment *adj;
	gchar buf[32];
	gint dlg_response;
	int result = TRUE;

	dialog = gtk_dialog_new_with_buttons ("3ds Options",
										  GTK_WINDOW( TopLevelWindow ),
										  GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
										  GTK_STOCK_OK,
										  GTK_RESPONSE_OK,
										  GTK_STOCK_CANCEL,
										  GTK_RESPONSE_CANCEL,
										  NULL);
	
	/* Scaling - frame */
	widget = gtk_frame_new("Scaling");
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), widget, TRUE, TRUE, 0);
	
	/* Scaling - hbox */
	hbox = gtk_hbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(widget), hbox);
	
	/* Scaling - entry box */
	widget = gtk_label_new("Scaling Factor:");
	gtk_box_pack_start(GTK_BOX(hbox), widget, FALSE, FALSE, 0);
	f3ds_scale_entry = gtk_entry_new();
	sprintf(buf, "%g", f3ds_scale_factor);
	gtk_entry_set_text(GTK_ENTRY(f3ds_scale_entry), buf);
	gtk_box_pack_start(GTK_BOX(hbox), f3ds_scale_entry, TRUE, TRUE, 0);


	/* Select Frame - frame */
	widget = gtk_frame_new("Select Frame");
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), widget, TRUE, TRUE, 0);
	
	/* Select Frame - hbox */
	hbox = gtk_hbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(widget), hbox);
	
	/* Select Frame - entry box */
	widget = gtk_label_new("Frame to load:");
	gtk_box_pack_start(GTK_BOX(hbox), widget, FALSE, FALSE, 0);
	adj = (GtkAdjustment *)gtk_adjustment_new( 
			0., 0., (gdouble)(lib3ds_file->frames) - 1., 
			1., 1., 1. );
	f3ds_frame_spinner = gtk_spin_button_new( adj, 1.0, 0 );
	gtk_box_pack_start(GTK_BOX(hbox), f3ds_frame_spinner, TRUE, TRUE, 0);

	/* set up the OK button as the default widget */
	gtk_dialog_set_default_response( GTK_DIALOG(dialog), GTK_RESPONSE_OK );
	g_signal_connect( G_OBJECT( f3ds_scale_entry ), "activate",
					G_CALLBACK(gui_dialog_ok_cb), dialog );
	g_signal_connect( G_OBJECT( f3ds_frame_spinner ), "activate",
					G_CALLBACK(gui_dialog_ok_cb), dialog );
	
	/* shows the dialog, and since it's modal, prevents rest of 
	   kludge3d from doing stuff */
	gtk_widget_show_all (dialog);
	
	/* runs a gtkmain until the user clicks on something */
	dlg_response = gtk_dialog_run( GTK_DIALOG (dialog) );
	
	switch( dlg_response ) {
	case GTK_RESPONSE_OK:
		f3ds_scale_factor =
			atof(gtk_entry_get_text(GTK_ENTRY(f3ds_scale_entry)));
		if( f3ds_scale_factor < 0.0 )
			f3ds_scale_factor = -f3ds_scale_factor;
		f3ds_current_frame = gtk_spin_button_get_value_as_int( 
			GTK_SPIN_BUTTON(f3ds_frame_spinner) );
		break;
	default:
		f3ds_scale_factor = -1.;
		f3ds_current_frame = 0;
		result = FALSE;
		break;
	}
	
	gtk_widget_destroy( dialog );

	f3ds_scale_entry = NULL;

printf( "f3ds_scale_factor is %f\n", f3ds_scale_factor );
printf( "f3ds_current_frame is %i\n", f3ds_current_frame );

/* FIXME - possible leak - will the gtkadjustment be properly dereferenced 
when the spinner is destroyed, or do *we* need to do that? */

	return result;
}



#endif
