/* 
 * File:		 fmt_ac3d.c
 * 
 * Description:  handles loading and saving of model data in AC3D format
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 
/*
 * Portions of this file were ripped from:
 *	- ac3d.c from the "orbit" space combat game.
 *    "orbit" took the code from the ac_to_gl program.
 * 
 */



#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>

#include "transform.h"
#include "globals.h"
#include "mesh.h"
#include "model.h"
#include "group.h"
#include "polygon.h"
#include "vertex.h"
#include "texture.h"
#include "bottombar.h"
#include "vector.h"

#include "model_load_save.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


/* This number is the latest version of the ac3d file format that this module 
can load/save.  */
#define AC3D_FORMAT_VERSION 0xb


gint ac3d_load_doc(struct fu_file *file, Model *doc);
gint ac3d_save_doc(struct fu_file *file, Model *doc);
void ac3d_destroy_priv_data(Model *doc);

struct doc_file_fmt fmt_ac3d = {
  "AC3D Format (ac)",
  "ac",
  ac3d_load_doc,
  ac3d_save_doc,
  NULL			/* <-- replace with NULL if not used */
};

/* FILE-SCOPE VARS ******************************************************/

Model *ac3dModel = NULL;


/* STRUCTS **************************************************************/

struct Tag {
	char *token;
	int ( *func ) ( char *s, GNode * node );
};


/* PROTOTYPES ***********************************************************/

void ac3d_write_materials( struct fu_file *file, Model *model );
void ac3d_write_mesh( Mesh *m, FILE *fp ) ;
void ac3d_write_child( GNode *n, FILE *fp );

GNode * ac3d_load ( FILE *fp );

int Search ( struct Tag *tags, char *s, GNode * );
int DoMaterial ( char *s, GNode * );
int DoObject ( char *s, GNode * );
int DoACName ( char *s, GNode * );
int DoData ( char *s, GNode * );
int DoTexture ( char *s, GNode * );
int DoTexrep ( char *s, GNode * );
int DoTexoff ( char *s, GNode * );
int DoRot ( char *s, GNode * );
int DoLoc ( char *s, GNode * );
int DoUrl ( char *s, GNode * );
int DoNumvert ( char *s, GNode * );
int DoNumsurf ( char *s, GNode * );
int DoSurf ( char *s, GNode * );
int DoMat ( char *s, GNode * );
int DoRefs ( char *s, GNode * );
int DoKids ( char *s, GNode * );

int DoObjWorld ( char *s, GNode * );
int DoObjPoly ( char *s, GNode * );
int DoObjGroup ( char *s, GNode * );

void SkipSpaces ( char **s );
void SkipQuotes ( char **s );
void RemoveQuotes ( char *s );


/* FUNCS ****************************************************************/


/* Load the document from fu_file and set the root member of doc */
gint ac3d_load_doc(struct fu_file *file, Model *model)
{

	if( model == NULL ) return FALSE;
	
	/* Note - model_new has already created an empty mesh and groupnode.  
	We don't need either, so we'll delete them here. */
	if( model->root )
		group_delete( model->root );
	
	ac3dModel = model;
	
	model->root = ac3d_load( file->fp );
	
	if( model->root == NULL )
		return FALSE;
	
	return TRUE;
}


/* Save the document to fu_file */
gint ac3d_save_doc(struct fu_file *file, Model *model)
{
	GNode *n;
	
	if( model == NULL ) return FALSE;
	
	n = model->root;

	fu_truncate(file, 0);	/* it's a good idea to truncate first */
				/* unless you have a more complex format */

	/* header */
	fu_printf( file, "AC3D%x\n", AC3D_FORMAT_VERSION );
	ac3d_write_materials( file, model );

	/* world object (root node) */
	fu_printf( file, "OBJECT world\n" );
	fu_printf( file, "kids %i\n", g_node_n_children( n ) );

	ac3d_write_child( n->children, file->fp );
	
	return TRUE;
}


/* Destroy the contents of model->fmt_data */
void ac3d_destroy_priv_data(Model *model)
{
	/* nothing to do */
}




/***************************************************************************
*
* OUTPUT FUNCS
*
***************************************************************************/

void ac3d_write_materials( struct fu_file *file, Model *model ) {
	fu_printf( file, "MATERIAL \"\" rgb 1 1 1  amb 0.2 0.2 0.2  emis 0 0 0  spec 0.5 0.5 0.5  shi 10  trans 0\n" );
}


void ac3d_write_mesh( Mesh *m, FILE *fp ) {

	GSList * c;
	Poly *p;
	Vertex *v;
	int num_vertices, num_polygons;


	num_vertices = g_slist_length( m->vertices );
	num_polygons = g_slist_length( m->polygons );
	
	fprintf( fp, "OBJECT poly\n" );

	if( m->name != NULL ) 
		fprintf( fp, "name \"%s\"\n", m->name );

	if( m->texture != NULL ) {
		fprintf( fp, "texture \"%s\"\n", m->texture->filename );
		fprintf( fp, "texrep %f %f\n", m->texrep_x, m->texrep_y );
	}
	
#if 0
	if( m->location != NULL ) {
		v = m->location;
#ifdef NO_SWAP_YZ
		fprintf( fp, "loc %f %f %f\n", v->v[0], v->v[1], v->v[2] );
#else
		fprintf( fp, "loc %f %f %f\n", v->v[0], v->v[2], -1.0 * v->v[1] );
#endif
	}
#endif

	fprintf( fp, "numvert %i\n", num_vertices );

	for ( c = m->vertices; c != NULL; c = c->next ) {
		v = ( Vertex * ) c->data;
#ifdef NO_SWAP_YZ
		fprintf( fp, "%f %f %f\n", v->v[ 0 ], v->v[ 1 ], v->v[ 2 ] );
#else
		fprintf( fp, "%f %f %f\n", v->v[ 0 ], v->v[ 2 ], -1.0 * v->v[ 1 ] );
#endif
	}

	fprintf( fp, "numsurf %i\n", num_polygons );

	for ( c = m->polygons; c != NULL; c = c->next ) {
		int i;
		p = ( Poly * ) c->data;

		fprintf( fp, "SURF 0x0\n" );
		fprintf( fp, "mat 0\n" );
		fprintf( fp, "refs %i\n", p->num_verts );


		for ( i = 0; i < p->num_verts; i++ ) {

			fprintf( fp, "%d %f %f\n",
					 g_slist_index( m->vertices, p->verts[ i ] ),
					 p->tc[ i ]->x,
					 p->tc[ i ]->y );
		}

	}
}


void ac3d_write_child( GNode *n, FILE *fp ) {

	for( ; n != NULL; n = n->next ) {
		int kids = g_node_n_children( n );
		ac3d_write_mesh( (Mesh*)(n->data), fp );
		fprintf( fp, "kids %i\n", kids );
		if( kids > 0 )
			ac3d_write_child( n->children, fp );
	}

}



/***************************************************************************
*
*  INPUT FUNCS
*
***************************************************************************/


int acsmooth;			/* Smooth surface? */
int acsurfpass;			/* Pass through surface */
#define PASS_NORMAL (0)		/* Get normals */
#define PASS_RENDER (1)		/* Render */


int num_materials = 0;
int num_textures = 0;
int ntex = 0;

int last_flags = -1;
int last_mat;
int last_num_kids = -1;
int current_flags = -1;
int texture_enabled = 0;
int need_texture = 0;

#define PARSE_CONT   0
#define PARSE_POP	1
/*FIXME - should have a PARSE_ERROR so that we can gracefully back out 
of all these funcs */

FILE *acfp;

struct Tag top_tags [] = {
	{"MATERIAL", DoMaterial },
	{"OBJECT" , DoObject },
	{NULL, NULL }
};



GNode * ac3d_load ( FILE *fp ) {

	char buffer[ 1024 ], *s;
	int firsttime;
	GNode *n;

	printf( "LoadAC3D: Loading model \n" );

	n = group_new( ac3dModel );

	acfp = fp;
	if ( acfp == NULL ) {
		printf( "LoadAC3D: Can't open , giving up!\n" );
		return ( NULL );
	}

	firsttime = 1;

	num_materials = 0;
	num_textures = 0;
	last_flags = -1 ;
	last_num_kids = -1 ;
	current_flags = -1 ;
	texture_enabled = 0;
	need_texture = 0;
	last_mat = -1;

	while ( fgets ( buffer, 1024, acfp ) != NULL ) {

		s = buffer ;

		/* Skip leading whitespace */
		SkipSpaces ( &s );

		/* Skip blank lines and comments */
		if ( *s < ' ' && *s != '\t' ) continue;
		if ( *s == '#' || *s == ';' ) continue;

		if ( firsttime ) {
			int version;
			firsttime = 0;

			if ( strncmp ( s, "AC3D", 4 ) != 0 ) {
				printf( "LoadAC3d: is not an AC3D format file\n" );
				return ( NULL );
			}
			sscanf( s, "AC3D%x", &version );
#ifdef VERBOSE
			printf( "LoadAC3d: file is in format version %x\n", version );
#endif
			if( version > AC3D_FORMAT_VERSION ) {
				printf( "LoadAC3d: file is in AC3D file format version %i, and "
						"this program can\nonly read files up to version %i\n", 
						version, AC3D_FORMAT_VERSION );
				return ( NULL );
			}
		} else {
			Search( top_tags, s, n );
		}
	}

	return n;
}


void SkipSpaces ( char **s ) {
	while ( **s == ' ' || **s == '\t' ) ( *s ) ++;
}

void SkipQuotes ( char **s ) {
	char * t;

	SkipSpaces ( s );

	if ( **s == '\"' ) {
		( *s ) ++;
		t = *s;
		while ( *t != '\0' && *t != '\"' ) t++;

		if ( *t != '\"' )
			printf( "SkipQuotes: Mismatched double-quote in '%s'\n", *s );

		*t = '\0';
	} else {
		printf( "SkipQuotes: Expected double-quote in '%s'\n", *s );
	}
}

void RemoveQuotes ( char *s ) {
	/*
	 *  Remove quotes from string
	 */

	char * t;
	int i, j, len;

	len = strlen ( s );
	t = ( char * ) malloc ( len + 1 );
	strcpy ( t, s );
	j = 0;

	for ( i = 0; i < len; i++ ) {
		if ( t[ i ] != '"' ) {
			s[ j++ ] = t[ i ];
		}
	}
	s[ j ] = 0;

	free ( t );
}

int Search ( struct Tag *tags, char *s, GNode *node ) {
	int i;

	SkipSpaces ( &s );

/*printf( "Search has been handed: %s", s ); */

	for ( i = 0; tags[ i ].token != NULL; i++ ) {
		if ( !strncasecmp ( tags[ i ].token, s, strlen ( tags[ i ].token ) ) ) {
			s += strlen ( tags[ i ].token );
			SkipSpaces ( &s );
			return ( *( tags[ i ].func ) ) ( s, node );
		}
	}

	printf( "Search: Unrecognised token '%s'\n", s );

	return PARSE_POP;
}


/*****************************************
 *
 * OBJECT-LEVEL FUNCTIONS AND STRUCTS 
 *
 *****************************************/


struct Tag object_tags[] = {
	{"name" , DoACName },
	{"data" , DoData},
	{"texture", DoTexture},
	{"texrep" , DoTexrep},
	{"texoff" , DoTexoff},
	{"rot" , DoRot},
	{"loc" , DoLoc},
	{"url" , DoUrl},
	{"numvert", DoNumvert},
	{"numsurf", DoNumsurf},
	{"kids" , DoKids},
	{NULL, NULL }
};

struct Tag surf_tag[] = {
	{"SURF", DoSurf },
	{NULL, NULL}
};

struct Tag surface_tags[] = {
	{"mat" , DoMat },
	{"refs", DoRefs},
	{NULL, NULL}
};

struct Tag obj_type_tags[] = {
	{"world", DoObjWorld },
	{"poly" , DoObjPoly },
	{"group", DoObjGroup},
	{NULL, NULL}
};

#define OBJ_WORLD  0
#define OBJ_POLY   1
#define OBJ_GROUP  2

int DoObjWorld ( char *s, GNode * node ) {
	return OBJ_WORLD;
}

int DoObjPoly ( char *s, GNode * node ) {
	return OBJ_POLY;
}

int DoObjGroup ( char *s, GNode * node ) {
	return OBJ_GROUP;
}

int DoMaterial ( char *s, GNode * node ) {
	char name[ 1024 ], *nm;
	float rgb[ 3 ], amb[ 3 ], emis[ 3 ], spec[ 3 ], trans;
	int shi;

	if ( 15 != sscanf ( s,
						"%s rgb %f %f %f amb %f %f %f emis %f %f %f spec %f %f %f shi %d trans %f",
						name, &rgb[ 0 ], &rgb[ 1 ], &rgb[ 2 ], &amb[ 0 ], &amb[ 1 ], &amb[ 2 ],
						&emis[ 0 ], &emis[ 1 ], &emis[ 2 ], &spec[ 0 ], &spec[ 1 ], &spec[ 2 ], &shi, &trans ) ) {
		printf( "DoMaterial: Can't parse this MATERIAL: %s\n", s );
	} else {
		nm = name;
		SkipQuotes ( &nm );

#if 0
		/* Get a list for this material */
		matlist[ num_materials ] = glGenLists ( 1 );

		glNewList ( matlist[ num_materials ], GL_COMPILE );
		glMaterialfv ( GL_FRONT_AND_BACK, GL_AMBIENT, amb );
		glMaterialfv ( GL_FRONT_AND_BACK, GL_DIFFUSE, rgb );
		glMaterialfv ( GL_FRONT_AND_BACK, GL_SPECULAR, spec );
		glMaterialfv ( GL_FRONT_AND_BACK, GL_EMISSION, emis );
		glMaterialf ( GL_FRONT_AND_BACK, GL_SHININESS, shi );
		glEndList();
#endif
	}
/*	num_materials++; */

	return PARSE_CONT;
}

int DoObject ( char *s, GNode * node ) {
	int i, obj_type, num_kids;
	char buffer[ 1024 ];


	obj_type = Search ( obj_type_tags, s, node );

	switch ( obj_type ) {
	case OBJ_WORLD:
		if( node->data != NULL ) {
			mesh_set_name( (Mesh*)node->data, TOP_GROUP_NAME );
		}
		break;

	case OBJ_POLY:
		if( node->data != NULL ) {
			if( ((Mesh*)node->data)->name == NULL )
				mesh_set_name( (Mesh*)node->data, "Poly with no name" );
		}
		break;

	case OBJ_GROUP:
		if( node->data != NULL ) {
			mesh_set_name( (Mesh*)node->data, "Group" );
		}
		break;
	}

	need_texture = 0;
	
	/* for each line of input that pertains to this object... */
	while ( NULL != fgets ( buffer, 1024, acfp ) ) {
		if ( Search ( object_tags, buffer, node ) == PARSE_POP ) break;
	}

	num_kids = last_num_kids;

	/* now do each of the object's children */
	for ( i = 0; i < num_kids; i++ ) {
		g_node_append_data( node, mesh_new( ac3dModel ) );
	
		fgets ( buffer, 1024, acfp );
		Search ( top_tags, buffer, g_node_last_child( node ) );
	}

	return PARSE_CONT;
}

int DoACName ( char *s, GNode * node ) {
	
	SkipQuotes ( &s );
/*printf( "DoACName:  obj name %s\n", s ); */
	
	mesh_set_name( (Mesh*)node->data, s );
	
	return PARSE_CONT;
}

int DoData ( char *s, GNode * node ) {
	int i, len;

	len = strtol ( s, NULL, 0 );
	printf( "DoData: WARNING - data string encountered\n" );

	/* len plus one for the newline character at the end of the next line */
	for ( i = 0; i < len +1; i++ )
		fgetc ( acfp );

	return PARSE_CONT;
}


int DoTexture ( char *s, GNode * node ) {

	Texture *t = NULL;
	
	SkipQuotes ( &s );
	
	t = model_get_texture_by_filename( ac3dModel, s );
	if( t != NULL ) {
		mesh_set_texture( (Mesh*)node->data, t );
	} else {
		t = tex_load( s, TEXTURE_FORMAT_UNKNOWN, ac3dModel->fname );
		if( t == NULL ) {
			/* there was an error loading the texture file */
			bb_push_message_f( 3.0, "Texture '%s' could not be loaded", s );
		} else {
			model_add_texture( ac3dModel, t );
			mesh_set_texture( (Mesh*)node->data, t );
		}
	}
	
	
	return PARSE_CONT;
}

int DoTexrep ( char *s, GNode * node ) {
	float texrep[ 2 ];

	if ( 2 != sscanf ( s, "%f %f", &texrep[ 0 ], &texrep[ 1 ] ) ) {
		printf( "DoTexrep: Illegal texrep record: %s\n", s );
	} else {
		mesh_set_texrep( (Mesh*)node->data, texrep[0], texrep[1] );
	}

	return PARSE_CONT;
}

int DoTexoff ( char *s, GNode * node ) {
	float texrep[ 2 ];

	if ( 2 != sscanf ( s, "%f %f", &texrep[ 0 ], &texrep[ 1 ] ) ) {
		printf( "DoTexoff: Illegal texoff record: %s\n", s );
	} else {
		printf( "DoTexoff: texture offsets aren't implemented yet...\n" );
	}

	return PARSE_CONT;
}

int DoRot ( char *s, GNode * node ) {
	float mat[ 4 ][ 4 ];

	mat[ 0 ][ 3 ] = mat[ 1 ][ 3 ] = mat[ 2 ][ 3 ] = mat[ 3 ][ 0 ] = mat[ 3 ][ 1 ] = mat[ 3 ][ 2 ] = 0.0f;
	mat[ 3 ][ 3 ] = 1.0f;

	if ( 9 != sscanf ( s, "%f %f %f %f %f %f %f %f %f",
					   &mat[ 0 ][ 0 ], &mat[ 0 ][ 1 ], &mat[ 0 ][ 2 ],
					   &mat[ 1 ][ 0 ], &mat[ 1 ][ 1 ], &mat[ 1 ][ 2 ],
					   &mat[ 2 ][ 0 ], &mat[ 2 ][ 1 ], &mat[ 2 ][ 2 ] ) ) {
		printf( "DoRot: Illegal rot record: %s\n", s );
	}

	/*	glMultMatrixf (mat); */

	return PARSE_CONT ;
}

int DoLoc ( char *s, GNode * node ) {
	float loc[ 3 ];
	Mesh *mesh;
	
	mesh = (Mesh*)node->data;

	if ( 3 != sscanf ( s, "%f %f %f", &loc[ 0 ], &loc[ 1 ], &loc[ 2 ] ) ) {
		printf( "DoLoc: Illegal loc record: %s\n", s );
	}
	
#ifdef NO_SWAP_YZ
	vector_copy( mesh->loc, loc );
#else
	mesh->loc[0] = loc[0];
	mesh->loc[1] = -1.0 * loc[2];
	mesh->loc[2] = loc[1];
#endif
	
	if( node->parent && node->parent->data ) {
		vector_add( mesh->loc, mesh->loc, ((Mesh*)node->parent->data)->loc );
	}
	
	return PARSE_CONT;
}

int DoUrl ( char *s, GNode * node ) {
	SkipQuotes ( &s );
	return PARSE_CONT;
}

int DoNumvert ( char *s, GNode * node ) {
	char buffer[ 1024 ];
	int i;
	Vertex * vertex;
	float thisX;
	float thisY;
	float thisZ;

	int numVertices = strtol ( s, NULL, 0 );
	
	for ( i = 0; i < numVertices; i++ ) {
		fgets ( buffer, 1024, acfp );

/*printf( "DoNumvert: buffer is: %s", buffer ); */
		vertex = vertex_new();
		
		thisX = thisY = thisZ = 0.0f;
		
		if ( 3 != sscanf( buffer, "%f %f %f", &thisX, &thisY, &thisZ ) ) {
			printf( "DoNumvert: Illegal vertex record: %s\n", buffer );
			free( vertex );
		}
		else {
/*printf( "\tconverted to: %f %f %f\n", thisX, thisY, thisZ ); */
			/* add a new vertex to the model */
#ifdef NO_SWAP_YZ
			vertex->v[ 0 ] = thisX;
			vertex->v[ 1 ] = thisY;
			vertex->v[ 2 ] = thisZ;
#else
			vertex->v[ 0 ] = thisX;
			vertex->v[ 1 ] = -1.0 * thisZ;
			vertex->v[ 2 ] = thisY;
#endif
			/* translate according to the mesh's loc */
			vector_add( vertex->v, vertex->v, ((Mesh*)node->data)->loc );
			
			model_vertex_add( ac3dModel, vertex );
			mesh_vertex_add( (Mesh*)node->data, vertex );
		}

	}

	return PARSE_CONT;
}

int DoNumsurf ( char *s, GNode * node ) {
	int i, ns;
	char buffer[ 1024 ];

	ns = strtol ( s, NULL, 0 );

#if 0
	/* Do first pass to average normals */
	pos = ftell ( acfp );
	acsurfpass = PASS_NORMAL;
	for ( i = 0; i < ns; i++ ) {
		p = ftell ( acfp );
		fgets ( buffer, 1024, acfp );
		Search ( surf_tag, buffer );
	}

	/* Back to beginning of object */
	fseek ( acfp, pos, SEEK_SET, HUH? );

	/* Average normals */
	for ( v = 0; v < nv; v++ ) {
//		vtab[ v ].normal[ 0 ] /= vtab[ v ].n;
//		vtab[ v ].normal[ 1 ] /= vtab[ v ].n;
//		vtab[ v ].normal[ 2 ] /= vtab[ v ].n;
	}

	/* Now render */
	acsurfpass = PASS_RENDER;
	last_mat = ( -1 );
#endif

	for ( i = 0; i < ns; i++ ) {
		fgets ( buffer, 1024, acfp );
		Search ( surf_tag, buffer, node );
	}

	return PARSE_CONT;
}

int DoSurf ( char *s, GNode * node ) {
	char buffer[ 1024 ];

	current_flags = strtol ( s, NULL, 0 );

	while ( NULL != fgets ( buffer, 1024, acfp ) ) {
		if ( Search ( surface_tags, buffer, node ) == PARSE_POP ) break;
	}

	return PARSE_CONT ;
}

int DoMat ( char *s, GNode * node ) {
	int mat;

	mat = strtol ( s, NULL, 0 );

/*	if ( mat != last_mat ) {
		if ( acsurfpass == PASS_RENDER ) glCallList ( matlist[ mat ] );
		last_mat = mat;
	}
*/

	return PARSE_CONT ;
}


int DoRefs ( char *s, GNode * node ) {
	int i;
	int nrefs;
	char buffer[ 1024 ] ;
	int *vlist;
	float *tlistu, *tlistv;

	int vertNum;
	float texCoordU;
	float texCoordV;
	int error = 0;
	
	Mesh *mesh = (Mesh*)node->data;

	nrefs = strtol ( s, NULL, 0 );

	if ( nrefs <= 0 ) return PARSE_POP;

	vlist = ( int * ) malloc ( sizeof( int ) * nrefs );
	tlistu = ( float * ) malloc ( sizeof( float ) * nrefs ) ;
	tlistv = ( float * ) malloc ( sizeof( float ) * nrefs ) ;

	for ( i = 0; i < nrefs; i++ ) {
		fgets ( buffer, 1024, acfp );
	

		vlist[ i ] = vertNum = 0;
		tlistu[ i ] = texCoordU = 0.0f;
		tlistv[ i ] = texCoordV = 0.0f;
		
		if( 3 != sscanf( buffer, "%u %f %f", &vertNum, &texCoordU, &texCoordV ) ) {
			printf( "DoRefs: Illegal ref record: %s\n", buffer );
			if( mesh->name != NULL ) 
				printf( "DoRefs:   in object: %s\n", mesh->name );
			error = 1;
		}
		
		vlist[ i ] = vertNum;
		tlistu[ i ] = texCoordU;
		tlistv[ i ] = texCoordV;

	}


	if ( nrefs >= 2 && !error ) {
		int i;
		Poly * newPoly;

		newPoly = poly_new();
		
		for( i = 0; i < nrefs; i++ ) {
			poly_add_vertex( newPoly, 
				g_slist_nth_data( mesh->vertices, vlist[i] ) );
			
			/* if there was some error in adding the vertex... */
			if( newPoly->num_verts != i+1 ){
				poly_delete( newPoly );
				goto error;
			}

			/*	fabs? ----------v */
			newPoly->tc[ i ]->x = ( tlistu[ i ] );
			newPoly->tc[ i ]->y = ( tlistv[ i ] );
		}

		mesh_polygon_add( mesh, newPoly );
	}
error:
	free ( vlist );
	free ( tlistu );
	free ( tlistv );

	return PARSE_POP;
}

int DoKids ( char *s, GNode * node ) {
	last_num_kids = strtol ( s, NULL, 0 );
	return PARSE_POP;
}







