/* 
 * File:         fmt_md2.c
 * 
 * Description:  loads Quake2 model files (.md2)
 * 
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 
/* 
This code is based on Mete Cerrigan's .md2 loading code.

Note - This code probably won't work on non-Intel systems, 
as the model data is loaded by simply reading in raw 
structs in binary form.  (No effort is made to get the bytes into 
whatever order the local machine uses.)

Damn Intel.  I've been told that the reason for storing 
multi-byte variables (ints, floats etc) in a backwards manner 
is that it provides a slight performance gain.  In my opinion, 
that's not a good enough reason.  
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <gtk/gtk.h>

#include "gui.h"
#include "mesh.h"
#include "model.h"
#include "group.h"
#include "polygon.h"
#include "vertex.h"
#include "texture.h"
#include "bottombar.h"
#include "model_load_save.h"
#include "vector.h"
#include "tools.h"


#ifdef MEMWATCH
#include "memwatch.h"
#endif


#include "fmt_md2.h"

/* PROTOTYPES **************************************************************/

gint md2_load_doc( struct fu_file *file, Model *model );

void md2_set_current_frame( int frm );

int md2_readModel( FILE * file ) ;
void md2_freeModel( void );
void md2_getBoundingBox( float *minmax );
void md2_generateLightNormals( void );
int md2_getAnimationCount( void );
const char *md2_getAnimationName( int animation );
void md2_getAnimationFrames( int animation, int *startFrame, int *endFrame );

void md2_makeFacetNormal( md2_triangle_t *t, int frame, float *fn );
void md2_normalize( float *n );

void md2_populateModel( Model * kl3dModel ) ;

gint md2_prompt_user( void );


/* STRUCTS ******************************************************************/

struct doc_file_fmt fmt_md2 = {
  "Quake2 model (md2)",
  "md2",
  md2_load_doc,
  NULL,
  NULL
};


/* FILE-SCOPE VARS **********************************************************/

#define MD2_DEFAULT_SCALE_FACTOR 0.032

int md2_current_frame;
int md2_total_frames;
float md2_scale_factor = MD2_DEFAULT_SCALE_FACTOR;

md2_model_t *md2Model;	 

float avertexnormals[ NUMVERTEXNORMALS ][ 3 ] = {
#include "fmt_md2_anorms.h"
};



/* FUNCS ********************************************************************/


gint md2_load_doc( struct fu_file *file, Model *model ) {

	int result;

	md2_current_frame = 0;
	md2_total_frames = 0;

// FIXME - check result	

	result = md2_readModel( file->fp );
	md2_generateLightNormals();
	result = md2_prompt_user();
	
	md2_populateModel( model );
	

	md2_freeModel();
	return result;
}


void md2_set_current_frame( int frm ) {

	if ( frm >= 0 && frm < md2_total_frames )
		md2_current_frame = frm;
	else
		md2_current_frame = 0;

}




/*
 * load model
 */
int md2_readModel( FILE * file ) {

	byte buffer[ MD2_MAX_FRAMESIZE ];
	int i;

	md2Model = ( md2_model_t * ) malloc ( sizeof ( md2_model_t ) );
	if ( !md2Model )
		return 0;

	/* initialize model and read header */
	memset ( md2Model, 0, sizeof ( md2_model_t ) );
	fread ( &md2Model->header, sizeof ( md2_header_t ), 1, file );


#if 0
	printf ( "magic:\t\t%d\n", md2Model->header.magic );
	printf ( "version:\t\t%d\n", md2Model->header.version );
	printf ( "skinWidth:\t\t%d\n", md2Model->header.skinWidth );
	printf ( "skinHeight:\t\t%d\n", md2Model->header.skinHeight );
	printf ( "frameSize:\t\t%d\n", md2Model->header.frameSize );
	printf ( "numSkins:\t\t%d\n", md2Model->header.numSkins );
	printf ( "numVertices:\t\t%d\n", md2Model->header.numVertices );
	printf ( "numTexCoords:\t\t%d\n", md2Model->header.numTexCoords );
	printf ( "numTriangles:\t\t%d\n", md2Model->header.numTriangles );
	printf ( "numGlCommands:\t\t%d\n", md2Model->header.numGlCommands );
	printf ( "numFrames:\t\t%d\n", md2Model->header.numFrames );
	printf ( "offsetSkins:\t\t%d\n", md2Model->header.offsetSkins );
	printf ( "offsetTexCoords:\t%d\n", md2Model->header.offsetTexCoords );
	printf ( "offsetTriangles:\t%d\n", md2Model->header.offsetTriangles );
	printf ( "offsetFrames:\t\t%d\n", md2Model->header.offsetFrames );
	printf ( "offsetGlCommands:\t%d\n", md2Model->header.offsetGlCommands );
	printf ( "offsetEnd:\t\t%d\n", md2Model->header.offsetEnd );
#endif

	if ( md2Model->header.magic != ( int ) ( ( '2' << 24 ) + ( 'P' << 16 ) + ( 'D' << 8 ) + 'I' ) ) {
		fclose ( file );
		free ( md2Model );
		return 0;
	}

	/* read skins */
	fseek ( file, md2Model->header.offsetSkins, SEEK_SET );
	if ( md2Model->header.numSkins > 0 ) {
		md2Model->skins = ( md2_skin_t * ) malloc ( sizeof ( md2_skin_t ) * md2Model->header.numSkins );
		if ( !md2Model->skins ) {
			md2_freeModel ();
			return 0;
		}

		for ( i = 0; i < md2Model->header.numSkins; i++ )
			fread ( &md2Model->skins[ i ], sizeof ( md2_skin_t ), 1, file );
	}

	/* read texture coordinates */
	fseek ( file, md2Model->header.offsetTexCoords, SEEK_SET );
	if ( md2Model->header.numTexCoords > 0 ) {
		md2Model->texCoords = ( md2_textureCoordinate_t * ) malloc ( sizeof ( md2_textureCoordinate_t ) * md2Model->header.numTexCoords );
		if ( !md2Model->texCoords ) {
			md2_freeModel ();
			return 0;
		}

		for ( i = 0; i < md2Model->header.numTexCoords; i++ )
			fread ( &md2Model->texCoords[ i ], sizeof ( md2_textureCoordinate_t ), 1, file );
	}

	/* read triangles */
	fseek ( file, md2Model->header.offsetTriangles, SEEK_SET );
	if ( md2Model->header.numTriangles > 0 ) {
		md2Model->triangles = ( md2_triangle_t * ) malloc ( sizeof ( md2_triangle_t ) * md2Model->header.numTriangles );
		if ( !md2Model->triangles ) {
			md2_freeModel ();
			return 0;
		}

		for ( i = 0; i < md2Model->header.numTriangles; i++ )
			fread ( &md2Model->triangles[ i ], sizeof ( md2_triangle_t ), 1, file );
	}


	md2_total_frames = md2Model->header.numFrames;

	/* read alias frames */
	fseek ( file, md2Model->header.offsetFrames, SEEK_SET );
	if ( md2Model->header.numFrames > 0 ) {
		md2Model->frames = ( md2_frame_t * ) malloc ( sizeof ( md2_frame_t ) * md2Model->header.numFrames );
		if ( !md2Model->frames ) {
			md2_freeModel ();
			return 0;
		}

		for ( i = 0; i < md2Model->header.numFrames; i++ ) {
			md2_alias_frame_t *frame = ( md2_alias_frame_t * ) buffer;
			int j;

			md2Model->frames[ i ].vertices = ( md2_triangleVertex_t * ) malloc ( sizeof ( md2_triangleVertex_t ) * md2Model->header.numVertices );
			if ( !md2Model->frames[ i ].vertices ) {
				md2_freeModel ();
				return 0;
			}

			fread ( frame, 1, md2Model->header.frameSize, file );
			strcpy ( md2Model->frames[ i ].name, frame->name );
			for ( j = 0; j < md2Model->header.numVertices; j++ ) {

				md2Model->frames[ i ].vertices[ j ].vertex[ 0 ] = ( ( ( float ) ( ( int ) frame->alias_vertices[ j ].vertex[ 0 ] ) * frame->scale[ 0 ] ) + frame->translate[ 0 ] );
				md2Model->frames[ i ].vertices[ j ].vertex[ 1 ] = ( ( ( float ) ( ( int ) frame->alias_vertices[ j ].vertex[ 1 ] ) * frame->scale[ 1 ] ) + frame->translate[ 1 ] );
				md2Model->frames[ i ].vertices[ j ].vertex[ 2 ] = ( ( ( float ) ( ( int ) frame->alias_vertices[ j ].vertex[ 2 ] ) * frame->scale[ 2 ] ) + frame->translate[ 2 ] );

				md2Model->frames[ i ].vertices[ j ].normal[ 0 ] = avertexnormals[ frame->alias_vertices[ j ].lightNormalIndex ][ 0 ];
				md2Model->frames[ i ].vertices[ j ].normal[ 1 ] = avertexnormals[ frame->alias_vertices[ j ].lightNormalIndex ][ 1 ];
				md2Model->frames[ i ].vertices[ j ].normal[ 2 ] = avertexnormals[ frame->alias_vertices[ j ].lightNormalIndex ][ 2 ];
				//md2Model->frames[i].vertices[j].lightNormalIndex = frame->alias_vertices[j].lightNormalIndex;
			}

		}
	}

	/* read gl commands */
	fseek ( file, md2Model->header.offsetGlCommands, SEEK_SET );
	if ( md2Model->header.numGlCommands ) {
		md2Model->glCommandBuffer = ( int * ) malloc ( sizeof ( int ) * md2Model->header.numGlCommands );
		if ( !md2Model->glCommandBuffer ) {
			md2_freeModel ();
			return 0;
		}

		fread ( md2Model->glCommandBuffer, sizeof ( int ), md2Model->header.numGlCommands, file );
	}

	return 1;
}



/*
 * free model
 */
void md2_freeModel( void ) {
	if ( md2Model ) {
		if ( md2Model->skins )
			free ( md2Model->skins );

		if ( md2Model->texCoords )
			free ( md2Model->texCoords );

		if ( md2Model->triangles )
			free ( md2Model->triangles );

		if ( md2Model->frames ) {
			int i;

			for ( i = 0; i < md2Model->header.numFrames; i++ ) {
				if ( md2Model->frames[ i ].vertices )
					free ( md2Model->frames[ i ].vertices );
			}
			free ( md2Model->frames );
		}

		if ( md2Model->glCommandBuffer )
			free ( md2Model->glCommandBuffer );

		free ( md2Model );
	}
}




/*
 * center model
 *
 */
void md2_getBoundingBox ( float *minmax ) {
	int i;
	float minx, maxx;
	float miny, maxy;
	float minz, maxz;

	minx = miny = minz = 999999.0f;
	maxx = maxy = maxz = -999999.0f;

	/* get bounding box */
	for ( i = 0; i < md2Model->header.numVertices; i++ ) {
		md2_triangleVertex_t *v = &md2Model->frames[ 0 ].vertices[ i ];

		if ( v->vertex[ 0 ] < minx )
			minx = v->vertex[ 0 ];
		else if ( v->vertex[ 0 ] > maxx )
			maxx = v->vertex[ 0 ];

		if ( v->vertex[ 1 ] < miny )
			miny = v->vertex[ 1 ];
		else if ( v->vertex[ 1 ] > maxy )
			maxy = v->vertex[ 1 ];

		if ( v->vertex[ 2 ] < minz )
			minz = v->vertex[ 2 ];
		else if ( v->vertex[ 2 ] > maxz )
			maxz = v->vertex[ 2 ];
	}

	minmax[ 0 ] = minx;
	minmax[ 1 ] = maxx;
	minmax[ 2 ] = miny;
	minmax[ 3 ] = maxy;
	minmax[ 4 ] = minz;
	minmax[ 5 ] = maxz;
}


void md2_makeFacetNormal ( md2_triangle_t *t, int frame, float *fn ) {
	float v1[ 3 ], v2[ 3 ];
	double angle;

	v1[ 0 ] = md2Model->frames[ frame ].vertices[ t->vertexIndices[ 1 ] ].vertex[ 0 ] - md2Model->frames[ frame ].vertices[ t->vertexIndices[ 0 ] ].vertex[ 0 ];
	v1[ 1 ] = md2Model->frames[ frame ].vertices[ t->vertexIndices[ 1 ] ].vertex[ 1 ] - md2Model->frames[ frame ].vertices[ t->vertexIndices[ 0 ] ].vertex[ 1 ];
	v1[ 2 ] = md2Model->frames[ frame ].vertices[ t->vertexIndices[ 1 ] ].vertex[ 2 ] - md2Model->frames[ frame ].vertices[ t->vertexIndices[ 0 ] ].vertex[ 2 ];

	v2[ 0 ] = md2Model->frames[ frame ].vertices[ t->vertexIndices[ 2 ] ].vertex[ 0 ] - md2Model->frames[ frame ].vertices[ t->vertexIndices[ 0 ] ].vertex[ 0 ];
	v2[ 1 ] = md2Model->frames[ frame ].vertices[ t->vertexIndices[ 2 ] ].vertex[ 1 ] - md2Model->frames[ frame ].vertices[ t->vertexIndices[ 0 ] ].vertex[ 1 ];
	v2[ 2 ] = md2Model->frames[ frame ].vertices[ t->vertexIndices[ 2 ] ].vertex[ 2 ] - md2Model->frames[ frame ].vertices[ t->vertexIndices[ 0 ] ].vertex[ 2 ];

	angle = 1;

	fn[ 0 ] = ( v1[ 1 ] * v2[ 2 ] - v1[ 2 ] * v2[ 1 ] ) * ( float ) angle;
	fn[ 1 ] = ( v1[ 2 ] * v2[ 0 ] - v1[ 0 ] * v2[ 2 ] ) * ( float ) angle;
	fn[ 2 ] = ( v1[ 0 ] * v2[ 1 ] - v1[ 1 ] * v2[ 0 ] ) * ( float ) angle;
}



void md2_normalize ( float *n ) {
	float l = ( float ) sqrt ( n[ 0 ] * n[ 0 ] + n[ 1 ] * n[ 1 ] + n[ 2 ] * n[ 2 ] );

	if ( l != 0.0f ) {
		n[ 0 ] /= l;
		n[ 1 ] /= l;
		n[ 2 ] /= l;
	}
}



void md2_generateLightNormals( void ) {
	int i;

	if ( !md2Model )
		return ;


	for ( i = 0; i < md2Model->header.numFrames; i++ ) {
		int j;

		/* clear all normals */
		for ( j = 0; j < md2Model->header.numVertices; j++ ) {
			md2Model->frames[ i ].vertices[ j ].normal[ 0 ] = 0.0f;
			md2Model->frames[ i ].vertices[ j ].normal[ 1 ] = 0.0f;
			md2Model->frames[ i ].vertices[ j ].normal[ 2 ] = 0.0f;
		}

		/* calc normals */
		for ( j = 0; j < md2Model->header.numTriangles; j++ ) {
			int k;
			float facetnormal[ 3 ];

			md2_makeFacetNormal ( &md2Model->triangles[ j ], i, facetnormal );

			for ( k = 0; k < 3; k++ ) {
				md2Model->frames[ i ].vertices[ md2Model->triangles[ j ].vertexIndices[ k ] ].normal[ 0 ] -= facetnormal[ 0 ];
				md2Model->frames[ i ].vertices[ md2Model->triangles[ j ].vertexIndices[ k ] ].normal[ 1 ] += facetnormal[ 2 ];
				md2Model->frames[ i ].vertices[ md2Model->triangles[ j ].vertexIndices[ k ] ].normal[ 2 ] -= facetnormal[ 1 ];
			}
		}

		/* normalize normals */
		for ( j = 0; j < md2Model->header.numVertices; j++ )
			md2_normalize ( md2Model->frames[ i ].vertices[ j ].normal );
	}
}



int md2_getAnimationCount( void ) {
	int i, j, pos;
	int count;
	int lastId;
	char name[ 16 ], last[ 16 ];

	strcpy ( last, md2Model->frames[ 0 ].name );
	pos = strlen ( last ) - 1;
	j = 0;
	while ( last[ pos ] >= '0' && last[ pos ] <= '9' && j < 2 ) {
		pos--;
		j++;
	}
	last[ pos + 1 ] = '\0';

	lastId = 0;
	count = 0;

	for ( i = 0; i <= md2Model->header.numFrames; i++ ) {
		if ( i == md2Model->header.numFrames )
			strcpy ( name, "" );  // some kind of a sentinel
		else
			strcpy ( name, md2Model->frames[ i ].name );
		pos = strlen ( name ) - 1;
		j = 0;
		while ( name[ pos ] >= '0' && name[ pos ] <= '9' && j < 2 ) {
			pos--;
			j++;
		}
		name[ pos + 1 ] = '\0';

		if ( strcmp ( last, name ) ) {
			strcpy ( last, name );
			count++;
		}
	}

	return count;
}



const char * md2_getAnimationName ( int animation ) {
	int i, j, pos;
	int count;
	int lastId;
	static char last[ 32 ];
	char name[ 32 ];

	strcpy ( last, md2Model->frames[ 0 ].name );
	pos = strlen ( last ) - 1;
	j = 0;
	while ( last[ pos ] >= '0' && last[ pos ] <= '9' && j < 2 ) {
		pos--;
		j++;
	}
	last[ pos + 1 ] = '\0';

	lastId = 0;
	count = 0;

	for ( i = 0; i <= md2Model->header.numFrames; i++ ) {
		if ( i == md2Model->header.numFrames )
			strcpy ( name, "" );  // some kind of a sentinel
		else
			strcpy ( name, md2Model->frames[ i ].name );
		pos = strlen ( name ) - 1;
		j = 0;
		while ( name[ pos ] >= '0' && name[ pos ] <= '9' && j < 2 ) {
			pos--;
			j++;
		}
		name[ pos + 1 ] = '\0';

		if ( strcmp ( last, name ) ) {
			if ( count == animation )
				return last;

			strcpy ( last, name );
			count++;
		}
	}

	return 0;
}



void md2_getAnimationFrames ( int animation, int *startFrame, int *endFrame ) {
	int i, j, pos;
	int count, numFrames, frameCount;
	int lastId;
	char name[ 16 ], last[ 16 ];

	strcpy ( last, md2Model->frames[ 0 ].name );
	pos = strlen ( last ) - 1;
	j = 0;
	while ( last[ pos ] >= '0' && last[ pos ] <= '9' && j < 2 ) {
		pos--;
		j++;
	}
	last[ pos + 1 ] = '\0';

	lastId = 0;
	count = 0;
	numFrames = 0;
	frameCount = 0;

	for ( i = 0; i <= md2Model->header.numFrames; i++ ) {
		if ( i == md2Model->header.numFrames )
			strcpy ( name, "" );  // some kind of a sentinel
		else
			strcpy ( name, md2Model->frames[ i ].name );
		pos = strlen ( name ) - 1;
		j = 0;
		while ( name[ pos ] >= '0' && name[ pos ] <= '9' && j < 2 ) {
			pos--;
			j++;
		}
		name[ pos + 1 ] = '\0';

		if ( strcmp ( last, name ) ) {
			strcpy ( last, name );

			if ( count == animation ) {
				*startFrame = frameCount - numFrames;
				*endFrame = frameCount - 1;
				return ;
			}

			count++;
			numFrames = 0;
		}
		frameCount++;
		numFrames++;
	}


	*startFrame = *endFrame = 0;
}


/* FUNCS TO POPULATE KL3D MODEL ********************************************/

void md2_populateModel( Model * kl3dModel ) {

	int i;
	Vertex **kl3dverts;
	md2_frame_t *f = &md2Model->frames[ md2_current_frame ];

	/* Note - model_new has already set up the model for us, and 
	has created an empty mesh and groupnode.  The model's current_mesh 
	and current_node are assumed to be set appropriately.  All we have to 
	do is put stuff in them. */
	
/*
cout << "#Converted from an md2 model by Andrew's magical MD2->AC3D converter!!\n";
cout << "#Here's some info about the original md2 model...\n";
cout << "#magic:\t\t\t" << model->header.magic << endl;
cout << "#version:\t\t" << model->header.version << endl;
cout << "#skinWidth:\t\t" << model->header.skinWidth << endl;
cout << "#skinHeight:\t\t" << model->header.skinHeight << endl;
cout << "#frameSize:\t\t" << model->header.frameSize << endl;
cout << "#numSkins:\t\t" << model->header.numSkins << endl;
cout << "#numVertices:\t\t" << model->header.numVertices << endl;
cout << "#numTexCoords:\t\t" << model->header.numTexCoords << endl;
cout << "#numTriangles:\t\t" << model->header.numTriangles << endl;
cout << "#numGlCommands:\t\t" << model->header.numGlCommands << endl;
cout << "#numFrames:\t\t" << model->header.numFrames << endl;
cout << "#offsetSkins:\t\t" << model->header.offsetSkins << endl;
cout << "#offsetTexCoords:\t" << model->header.offsetTexCoords << endl;
cout << "#offsetTriangles:\t" << model->header.offsetTriangles << endl;
cout << "#offsetFrames:\t\t" << model->header.offsetFrames << endl;
cout << "#offsetGlCommands:\t" << model->header.offsetGlCommands << endl;
cout << "#offsetEnd:\t\t" << model->header.offsetEnd << endl;
*/

	kl3dverts = (Vertex **)malloc( sizeof( Vertex * ) * md2Model->header.numVertices );
	
	mesh_set_name( kl3dModel->current_mesh, f->name );
	
	/* create and add the vertices */
	for ( i = 0; i < md2Model->header.numVertices; i++ ) {
		Vertex *v;
		v = vertex_new();
		vector_copy( v->v, f->vertices[ i ].vertex );
		
		kl3dverts[i] = v;
		model_vertex_add( kl3dModel, v );
		mesh_vertex_add( kl3dModel->current_mesh, v );
	}

	/* create and add the triangles */
	for ( i = 0; i < md2Model->header.numTriangles; i++ ) {
		int j;
		md2_triangle_t *t = &md2Model->triangles[ i ];
		Poly *kl3dpoly;
		
		kl3dpoly = poly_new();
		
		/*
		The vertices for the triangles are given in reverse order, because 
		md2's have cw triangles, but opengl expects ccw triangles
		
		The "1.0 - textureCoord.y" thing is to flip the texture coordinate
		system around to match opengl's... textures created in graphics editors 
		have the origin in the upper left, whereas ogl's is in the lower left.
		*/
		
		for( j = 2; j >= 0; j-- ) {
			poly_add_vertex( kl3dpoly, kl3dverts[t->vertexIndices[ j ]] );
			kl3dpoly->tc[kl3dpoly->num_verts - 1]->x = 
				( float ) md2Model->texCoords[ t->textureIndices[ j ] ].s / 
				( float ) md2Model->header.skinWidth;
			kl3dpoly->tc[kl3dpoly->num_verts - 1]->y = 
				1.0 - ( 
				( float ) md2Model->texCoords[ t->textureIndices[ j ] ].t / 
				( float ) md2Model->header.skinHeight );
		}
		
		mesh_polygon_add( kl3dModel->current_mesh, kl3dpoly );
	}


	if( md2_scale_factor > 0.0f ) {
		/* perform scaling of vertex coordinates */
		tools_vertices_scale_about_origin( kl3dModel, 
			kl3dModel->current_mesh->vertices, md2_scale_factor, TRUE );
	}

	free( kl3dverts );
}




/* GUI STUFF ***************************************************************/

GtkWidget *md2_scale_entry = NULL;
GtkWidget *md2_frame_spinner = NULL;

gint md2_prompt_user( void ) {

	GtkWidget *dialog;
    GtkWidget *hbox, *widget;
	GtkAdjustment *adj;
    gchar buf[32];
	gint dlg_response;
	int result = TRUE;

	dialog = gtk_dialog_new_with_buttons ("MD2 Options",
										  GTK_WINDOW( TopLevelWindow ),
										  GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
										  GTK_STOCK_OK,
										  GTK_RESPONSE_OK,
										  GTK_STOCK_CANCEL,
										  GTK_RESPONSE_CANCEL,
										  NULL);
	
	/* Scaling - frame */
	widget = gtk_frame_new("Scaling");
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), widget, TRUE, TRUE, 0);
	
	/* Scaling - hbox */
	hbox = gtk_hbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(widget), hbox);
	
	/* Scaling - entry box */
	widget = gtk_label_new("Scaling Factor:");
	gtk_box_pack_start(GTK_BOX(hbox), widget, FALSE, FALSE, 0);
	md2_scale_entry = gtk_entry_new();
	sprintf(buf, "%g", md2_scale_factor);
	gtk_entry_set_text(GTK_ENTRY(md2_scale_entry), buf);
	gtk_box_pack_start(GTK_BOX(hbox), md2_scale_entry, TRUE, TRUE, 0);


	/* Select Frame - frame */
	widget = gtk_frame_new("Select Frame");
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), widget, TRUE, TRUE, 0);
	
	/* Select Frame - hbox */
	hbox = gtk_hbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(widget), hbox);
	
	/* Select Frame - entry box */
	widget = gtk_label_new("Frame to load:");
	gtk_box_pack_start(GTK_BOX(hbox), widget, FALSE, FALSE, 0);
	adj = (GtkAdjustment *)gtk_adjustment_new( 
			0., 0., (gdouble)(md2Model->header.numFrames) - 1., 
			1., 1., 1. );
	md2_frame_spinner = gtk_spin_button_new( adj, 1.0, 0 );
	gtk_box_pack_start(GTK_BOX(hbox), md2_frame_spinner, TRUE, TRUE, 0);

	/* set up the OK button as the default widget */
	gtk_dialog_set_default_response( GTK_DIALOG(dialog), GTK_RESPONSE_OK );
	g_signal_connect( G_OBJECT( md2_scale_entry ), "activate",
					G_CALLBACK(gui_dialog_ok_cb), dialog );
	g_signal_connect( G_OBJECT( md2_frame_spinner ), "activate",
					G_CALLBACK(gui_dialog_ok_cb), dialog );

	/* shows the dialog, and since it's modal, prevents rest of 
	   kludge3d from doing stuff */
	gtk_widget_show_all (dialog);
	
	/* runs a gtkmain until the user clicks on something */
	dlg_response = gtk_dialog_run( GTK_DIALOG (dialog) );
	
	switch( dlg_response ) {
	case GTK_RESPONSE_OK:
		md2_scale_factor =
			atof(gtk_entry_get_text(GTK_ENTRY(md2_scale_entry)));
		if( md2_scale_factor < 0.0 )
			md2_scale_factor = -md2_scale_factor;
		md2_set_current_frame( gtk_spin_button_get_value_as_int( 
			GTK_SPIN_BUTTON(md2_frame_spinner) ) );
		break;
	default:
	    md2_scale_factor = -1.;
		md2_current_frame = 0;
		result = FALSE;
		break;
	}
	
	gtk_widget_destroy( dialog );

	md2_scale_entry = NULL;

printf( "md2_scale_factor is %f\n", md2_scale_factor );
printf( "md2_current_frame is %i\n", md2_current_frame );

/* FIXME - possible leak - will the gtkadjustment be properly dereferenced 
when the spinner is destroyed, or do *we* need to do that? */

	return result;
}







/******************************************************************
 * 
 * DEAD CODE
 * 
 *****************************************************************/
#if 0

void md2_dumpModel( float scale ) {
	int i;

	md2_frame_t *f = &md2Model->frames[ md2_current_frame ];

cout << "AC3Db\n";

/*
cout << "#Converted from an md2 model by Andrew's magical MD2->AC3D converter!!\n";
cout << "#Here's some info about the original md2 model...\n";
cout << "#magic:\t\t\t" << model->header.magic << endl;
cout << "#version:\t\t" << model->header.version << endl;
cout << "#skinWidth:\t\t" << model->header.skinWidth << endl;
cout << "#skinHeight:\t\t" << model->header.skinHeight << endl;
cout << "#frameSize:\t\t" << model->header.frameSize << endl;
cout << "#numSkins:\t\t" << model->header.numSkins << endl;
cout << "#numVertices:\t\t" << model->header.numVertices << endl;
cout << "#numTexCoords:\t\t" << model->header.numTexCoords << endl;
cout << "#numTriangles:\t\t" << model->header.numTriangles << endl;
cout << "#numGlCommands:\t\t" << model->header.numGlCommands << endl;
cout << "#numFrames:\t\t" << model->header.numFrames << endl;
cout << "#offsetSkins:\t\t" << model->header.offsetSkins << endl;
cout << "#offsetTexCoords:\t" << model->header.offsetTexCoords << endl;
cout << "#offsetTriangles:\t" << model->header.offsetTriangles << endl;
cout << "#offsetFrames:\t\t" << model->header.offsetFrames << endl;
cout << "#offsetGlCommands:\t" << model->header.offsetGlCommands << endl;
cout << "#offsetEnd:\t\t" << model->header.offsetEnd << endl;
*/
cout << "MATERIAL \"\" rgb 1 1 1  amb 0.2 0.2 0.2  emis 0 0 0  spec 0.5 0.5 0.5  shi 10  trans 0\n";
cout << "OBJECT world\n";
cout << "kids 1\n";
cout << "OBJECT poly\n";
cout << "name \"" << md2_getAnimationName( 0 ) << "\"\n";
cout << "texture \"default.rgb\"\n";
//cout << "loc 0 0 0\n";

// this writes out the mesh
cout << "numvert " << md2Model->header.numVertices << endl;

	for ( i = 0; i < md2Model->header.numVertices; i++ ) {

		// apparently, ac3d seems to use that screwy, retarded coordinate system (ie "y is up")
		// so we'll rotate the vertices around the x axis before we output them
		cout << f->vertices[ i ].vertex[ 0 ] * scale << " "
			 << f->vertices[ i ].vertex[ 2 ] * scale << " "
			 << f->vertices[ i ].vertex[ 1 ] * -1.0 * scale << endl;

	}

// this writes out the texture-mapping information
cout << "numsurf " << md2Model->header.numTriangles << endl;

	for ( i = 0; i < md2Model->header.numTriangles; i++ ) {
		md2_triangle_t *t = &md2Model->triangles[ i ];
		
		cout << "SURF 0x0\n";
		cout << "mat 0\n";
		cout << "refs 3\n";
		
		// the vertices for the triangles are given in reverse order, because 
		// md2's have cw triangles, but opengl expects ccw triangles
		
		// the "1.0 - textureCoord.y" things are to flip the texture coordinate
		// system around to match opengl's... textures created in graphics editors 
		// have the origin in the upper left, whereas ogl's is in the lower left.

		cout << t->vertexIndices[ 2 ] << " ";
		cout << ( float ) md2Model->texCoords[ t->textureIndices[ 2 ] ].s / ( float ) md2Model->header.skinWidth << " ";
		cout << 1.0 - ( ( float ) md2Model->texCoords[ t->textureIndices[ 2 ] ].t / ( float ) md2Model->header.skinHeight ) << endl;
		 
		cout << t->vertexIndices[ 1 ] << " ";
		cout << ( float ) md2Model->texCoords[ t->textureIndices[ 1 ] ].s / ( float ) md2Model->header.skinWidth << " ";
		cout << 1.0 - ( ( float ) md2Model->texCoords[ t->textureIndices[ 1 ] ].t / ( float ) md2Model->header.skinHeight ) << endl;

		cout << t->vertexIndices[ 0 ] << " ";
		cout << ( float ) md2Model->texCoords[ t->textureIndices[ 0 ] ].s / ( float ) md2Model->header.skinWidth << " ";
		cout << 1.0 - ( ( float ) md2Model->texCoords[ t->textureIndices[ 0 ] ].t / ( float ) md2Model->header.skinHeight ) << endl;

	}

cout << "kids 0\n";

}


/*
 * draw normal
 *
 */
void
Md2_Object::_md2_drawModel ( int frame ) {
	int i;

	md2_frame_t *f = &model->frames[ frame ];

	glBegin ( GL_TRIANGLES );

	for ( i = 0; i < model->header.numTriangles; i++ ) {
		md2_triangle_t *t = &model->triangles[ i ];
		//int lightNormalIndex;
		/*
				lightNormalIndex = f->vertices[t->vertexIndices[0]].lightNormalIndex;
				glNormal3f (avertexnormals[lightNormalIndex][0],
					avertexnormals[lightNormalIndex][2],
					-avertexnormals[lightNormalIndex][1]);
		*/
		glNormal3f ( f->vertices[ t->vertexIndices[ 0 ] ].normal[ 0 ],
					 f->vertices[ t->vertexIndices[ 0 ] ].normal[ 2 ],
					 -f->vertices[ t->vertexIndices[ 0 ] ].normal[ 1 ] );
		glTexCoord2f ( ( float ) model->texCoords[ t->textureIndices[ 0 ] ].s / ( float ) model->header.skinWidth,
					   ( float ) model->texCoords[ t->textureIndices[ 0 ] ].t / ( float ) model->header.skinHeight );
		glVertex3f ( f->vertices[ t->vertexIndices[ 0 ] ].vertex[ 0 ],
					 f->vertices[ t->vertexIndices[ 0 ] ].vertex[ 1 ],
					 f->vertices[ t->vertexIndices[ 0 ] ].vertex[ 2 ] );

		/*
				lightNormalIndex = f->vertices[t->vertexIndices[1]].lightNormalIndex;
				glNormal3f (avertexnormals[lightNormalIndex][0],
					avertexnormals[lightNormalIndex][2],
					-avertexnormals[lightNormalIndex][1]);
		*/
		glNormal3f ( f->vertices[ t->vertexIndices[ 1 ] ].normal[ 0 ],
					 f->vertices[ t->vertexIndices[ 1 ] ].normal[ 2 ],
					 -f->vertices[ t->vertexIndices[ 1 ] ].normal[ 1 ] );
		glTexCoord2f ( ( float ) model->texCoords[ t->textureIndices[ 1 ] ].s / ( float ) model->header.skinWidth,
					   ( float ) model->texCoords[ t->textureIndices[ 1 ] ].t / ( float ) model->header.skinHeight );
		glVertex3f ( f->vertices[ t->vertexIndices[ 1 ] ].vertex[ 0 ],
					 f->vertices[ t->vertexIndices[ 1 ] ].vertex[ 1 ],
					 f->vertices[ t->vertexIndices[ 1 ] ].vertex[ 2 ] );
		/*
				lightNormalIndex = f->vertices[t->vertexIndices[2]].lightNormalIndex;
				glNormal3f (avertexnormals[lightNormalIndex][0],
					avertexnormals[lightNormalIndex][2],
					-avertexnormals[lightNormalIndex][1]);
		*/
		glNormal3f ( f->vertices[ t->vertexIndices[ 2 ] ].normal[ 0 ],
					 f->vertices[ t->vertexIndices[ 2 ] ].normal[ 2 ],
					 -f->vertices[ t->vertexIndices[ 2 ] ].normal[ 1 ] );
		glTexCoord2f ( ( float ) model->texCoords[ t->textureIndices[ 2 ] ].s / ( float ) model->header.skinWidth,
					   ( float ) model->texCoords[ t->textureIndices[ 2 ] ].t / ( float ) model->header.skinHeight );
		glVertex3f ( f->vertices[ t->vertexIndices[ 2 ] ].vertex[ 0 ],
					 f->vertices[ t->vertexIndices[ 2 ] ].vertex[ 1 ],
					 f->vertices[ t->vertexIndices[ 2 ] ].vertex[ 2 ] );
	}
	glEnd ();
}


/*
 * draw interpolated
 *
 */
void
Md2_Object::_md2_drawModeli ( int frame1, int frame2, float pol ) {
	int i;

	md2_frame_t *f1 = &model->frames[ frame1 ];
	md2_frame_t *f2 = &model->frames[ frame2 ];

	glBegin ( GL_TRIANGLES );

	for ( i = 0; i < model->header.numTriangles; i++ ) {
		md2_triangle_t *t = &model->triangles[ i ];
		//int lightNormalIndex1, lightNormalIndex2;
		float x1, y1, z1, x2, y2, z2;
		float *n1, *n2;

		glTexCoord2f ( ( float ) model->texCoords[ t->textureIndices[ 0 ] ].s / ( float ) model->header.skinWidth,
					   ( float ) model->texCoords[ t->textureIndices[ 0 ] ].t / ( float ) model->header.skinHeight );
		/*
				lightNormalIndex1 = f1->vertices[t->vertexIndices[0]].lightNormalIndex;
				lightNormalIndex2 = f2->vertices[t->vertexIndices[0]].lightNormalIndex;
				glNormal3f ((1.0f - pol) * avertexnormals[lightNormalIndex1][0] + pol * avertexnormals[lightNormalIndex2][0],
					(1.0f - pol) * avertexnormals[lightNormalIndex1][2] + pol * avertexnormals[lightNormalIndex2][2],
					(1.0f - pol) * -avertexnormals[lightNormalIndex1][1] + pol * -avertexnormals[lightNormalIndex2][1]);
		*/
		n1 = f1->vertices[ t->vertexIndices[ 0 ] ].normal;
		n2 = f2->vertices[ t->vertexIndices[ 0 ] ].normal;
		glNormal3f ( ( 1.0f - pol ) * n1[ 0 ] + pol * n2[ 0 ],
					 ( 1.0f - pol ) * n1[ 2 ] + pol * n2[ 2 ],
					 ( 1.0f - pol ) * -n1[ 1 ] + pol * -n2[ 1 ] );

		x1 = f1->vertices[ t->vertexIndices[ 0 ] ].vertex[ 0 ];
		y1 = f1->vertices[ t->vertexIndices[ 0 ] ].vertex[ 1 ];
		z1 = f1->vertices[ t->vertexIndices[ 0 ] ].vertex[ 2 ];
		x2 = f2->vertices[ t->vertexIndices[ 0 ] ].vertex[ 0 ];
		y2 = f2->vertices[ t->vertexIndices[ 0 ] ].vertex[ 1 ];
		z2 = f2->vertices[ t->vertexIndices[ 0 ] ].vertex[ 2 ];
		glVertex3f ( x1 + pol * ( x2 - x1 ),
					 y1 + pol * ( y2 - y1 ),
					 z1 + pol * ( z2 - z1 ) );

		glTexCoord2f ( ( float ) model->texCoords[ t->textureIndices[ 1 ] ].s / ( float ) model->header.skinWidth,
					   ( float ) model->texCoords[ t->textureIndices[ 1 ] ].t / ( float ) model->header.skinHeight );
		/*
				lightNormalIndex1 = f1->vertices[t->vertexIndices[1]].lightNormalIndex;
				lightNormalIndex2 = f2->vertices[t->vertexIndices[1]].lightNormalIndex;
				glNormal3f ((1.0f - pol) * avertexnormals[lightNormalIndex1][0] + pol * avertexnormals[lightNormalIndex2][0],
					(1.0f - pol) * avertexnormals[lightNormalIndex1][2] + pol * avertexnormals[lightNormalIndex2][2],
					(1.0f - pol) * -avertexnormals[lightNormalIndex1][1] + pol * -avertexnormals[lightNormalIndex2][1]);
		*/
		n1 = f1->vertices[ t->vertexIndices[ 1 ] ].normal;
		n2 = f2->vertices[ t->vertexIndices[ 1 ] ].normal;
		glNormal3f ( ( 1.0f - pol ) * n1[ 0 ] + pol * n2[ 0 ],
					 ( 1.0f - pol ) * n1[ 2 ] + pol * n2[ 2 ],
					 ( 1.0f - pol ) * -n1[ 1 ] + pol * -n2[ 1 ] );

		x1 = f1->vertices[ t->vertexIndices[ 1 ] ].vertex[ 0 ];
		y1 = f1->vertices[ t->vertexIndices[ 1 ] ].vertex[ 1 ];
		z1 = f1->vertices[ t->vertexIndices[ 1 ] ].vertex[ 2 ];
		x2 = f2->vertices[ t->vertexIndices[ 1 ] ].vertex[ 0 ];
		y2 = f2->vertices[ t->vertexIndices[ 1 ] ].vertex[ 1 ];
		z2 = f2->vertices[ t->vertexIndices[ 1 ] ].vertex[ 2 ];
		glVertex3f ( x1 + pol * ( x2 - x1 ),
					 y1 + pol * ( y2 - y1 ),
					 z1 + pol * ( z2 - z1 ) );

		glTexCoord2f ( ( float ) model->texCoords[ t->textureIndices[ 2 ] ].s / ( float ) model->header.skinWidth,
					   ( float ) model->texCoords[ t->textureIndices[ 2 ] ].t / ( float ) model->header.skinHeight );
		/*
				lightNormalIndex1 = f1->vertices[t->vertexIndices[2]].lightNormalIndex;
				lightNormalIndex2 = f2->vertices[t->vertexIndices[2]].lightNormalIndex;
				glNormal3f ((1.0f - pol) * avertexnormals[lightNormalIndex1][0] + pol * avertexnormals[lightNormalIndex2][0],
					(1.0f - pol) * avertexnormals[lightNormalIndex1][2] + pol * avertexnormals[lightNormalIndex2][2],
					(1.0f - pol) * -avertexnormals[lightNormalIndex1][1] + pol * -avertexnormals[lightNormalIndex2][1]);
		*/
		n1 = f1->vertices[ t->vertexIndices[ 2 ] ].normal;
		n2 = f2->vertices[ t->vertexIndices[ 2 ] ].normal;
		glNormal3f ( ( 1.0f - pol ) * n1[ 0 ] + pol * n2[ 0 ],
					 ( 1.0f - pol ) * n1[ 2 ] + pol * n2[ 2 ],
					 ( 1.0f - pol ) * -n1[ 1 ] + pol * -n2[ 1 ] );

		x1 = f1->vertices[ t->vertexIndices[ 2 ] ].vertex[ 0 ];
		y1 = f1->vertices[ t->vertexIndices[ 2 ] ].vertex[ 1 ];
		z1 = f1->vertices[ t->vertexIndices[ 2 ] ].vertex[ 2 ];
		x2 = f2->vertices[ t->vertexIndices[ 2 ] ].vertex[ 0 ];
		y2 = f2->vertices[ t->vertexIndices[ 2 ] ].vertex[ 1 ];
		z2 = f2->vertices[ t->vertexIndices[ 2 ] ].vertex[ 2 ];
		glVertex3f ( x1 + pol * ( x2 - x1 ),
					 y1 + pol * ( y2 - y1 ),
					 z1 + pol * ( z2 - z1 ) );
	}
	glEnd ();
}



/*
 * draw with glcommands
 *
 */
void
Md2_Object::_md2_drawModelg ( int frame ) {
	int i = 0;
	int val = model->glCommandBuffer[ i++ ];

	while ( val != 0 ) {
		int count;

		if ( val > 0 ) {
			glBegin ( GL_TRIANGLE_STRIP );
			count = val;
		} else {
			glBegin ( GL_TRIANGLE_FAN );
			count = -val;
		}

		while ( count-- ) {
			float s = *( float * ) & model->glCommandBuffer[ i++ ];
			float t = *( float * ) & model->glCommandBuffer[ i++ ];
			int index = model->glCommandBuffer[ i++ ];
			//int lightNormalIndex = model->frames[frame].vertices[index].lightNormalIndex;

			glTexCoord2f ( s, t );
			/*
						glNormal3f (avertexnormals[lightNormalIndex][0],
							avertexnormals[lightNormalIndex][2],
							-avertexnormals[lightNormalIndex][1]);
			*/
			glNormal3f ( model->frames[ frame ].vertices[ index ].normal[ 0 ],
						 model->frames[ frame ].vertices[ index ].normal[ 2 ],
						 -model->frames[ frame ].vertices[ index ].normal[ 1 ] );
			glVertex3f ( model->frames[ frame ].vertices[ index ].vertex[ 0 ],
						 model->frames[ frame ].vertices[ index ].vertex[ 1 ],
						 model->frames[ frame ].vertices[ index ].vertex[ 2 ] );
		}

		glEnd ();

		val = model->glCommandBuffer[ i++ ];
	}
}



/*
 * draw with glcommands, interpolated
 *
 */
void
Md2_Object::_md2_drawModelgi ( int frame1, int frame2, float pol ) {
	int i = 0;
	int val = model->glCommandBuffer[ i++ ];

	while ( val != 0 ) {
		int count;

		if ( val > 0 ) {
			glBegin ( GL_TRIANGLE_STRIP );
			count = val;
		} else {
			glBegin ( GL_TRIANGLE_FAN );
			count = -val;
		}

		while ( count-- ) {
			float s = *( float * ) & model->glCommandBuffer[ i++ ];
			float t = *( float * ) & model->glCommandBuffer[ i++ ];
			int index = model->glCommandBuffer[ i++ ];
			//int lightNormalIndex1 = model->frames[frame1].vertices[index].lightNormalIndex;
			//int lightNormalIndex2 = model->frames[frame2].vertices[index].lightNormalIndex;
			float x1, y1, z1, x2, y2, z2;
			float *n1 = model->frames[ frame1 ].vertices[ index ].normal;
			float *n2 = model->frames[ frame2 ].vertices[ index ].normal;

			/* texcoords don't need to be interpolated */
			glTexCoord2f ( s, t );

			/* interpolate light normal */
			/*
						glNormal3f ((1.0f - pol) * avertexnormals[lightNormalIndex1][0] + pol * avertexnormals[lightNormalIndex2][0],
							(1.0f - pol) * avertexnormals[lightNormalIndex1][2] + pol * avertexnormals[lightNormalIndex2][2],
							(1.0f - pol) * -avertexnormals[lightNormalIndex1][1] + pol * -avertexnormals[lightNormalIndex2][1]);
			*/
			glNormal3f ( ( 1.0f - pol ) * n1[ 0 ] + pol * n2[ 0 ],
						 ( 1.0f - pol ) * n1[ 2 ] + pol * n2[ 2 ],
						 ( 1.0f - pol ) * -n1[ 1 ] + pol * -n2[ 1 ] );

			/* interpolate vertices */
			x1 = model->frames[ frame1 ].vertices[ index ].vertex[ 0 ];
			y1 = model->frames[ frame1 ].vertices[ index ].vertex[ 1 ];
			z1 = model->frames[ frame1 ].vertices[ index ].vertex[ 2 ];

			x2 = model->frames[ frame2 ].vertices[ index ].vertex[ 0 ];
			y2 = model->frames[ frame2 ].vertices[ index ].vertex[ 1 ];
			z2 = model->frames[ frame2 ].vertices[ index ].vertex[ 2 ];

			glVertex3f ( x1 + pol * ( x2 - x1 ),
						 y1 + pol * ( y2 - y1 ),
						 z1 + pol * ( z2 - z1 ) );
		}

		glEnd ();

		val = model->glCommandBuffer[ i++ ];
	}
}


/*
 * draw model
 */
void
Md2_Object::md2_drawModel ( int frame1, int frame2, float pol ) {
	if ( g_glcmds ) {
		if ( g_interp )
			_md2_drawModelgi ( frame1, frame2, pol );
		else
			_md2_drawModelg ( frame1 );
	} else {
		if ( g_interp )
			_md2_drawModeli ( frame1, frame2, pol );
		else
			_md2_drawModel ( frame1 );
	}
}

#endif





