/*
 * File:         fmt_quakemap.c
 * 
 * Description:  handles saving of model data in Quake2 map format
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */


/*
 * Ah!  Finally, someone who uses a sane coordinate system!
 * Mr. Carmack, we are in your debt.
 * 
 */

/* Quake2 .map file format handler */

#include <string.h>

#include "globals.h"
#include "mesh.h"
#include "model.h"
#include "polygon.h"
#include "vertex.h"
#include "vector.h"
#include "texture.h"
#include "bottombar.h"
#include "tools.h"

#include "model_load_save.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


#define QUAKEUNITS_PER_METER  32.0f   /* approximate... */
#define TO_QUAKE * QUAKEUNITS_PER_METER


gint quakemap_save_doc(struct fu_file *file, Model *doc);

struct doc_file_fmt fmt_quakemap = {
    "Quake2 map (map)",
    "map",
    NULL,
    quakemap_save_doc,
    NULL
};

void quakemap_write_mesh( GNode *node, gpointer user_data ) ;
void quakemap_write_polygon( struct fu_file *file, Poly *p, Texture *texture ) ;
void quakemap_write_brushsurface( struct fu_file *file, Vertex *v1, Vertex *v2, Vertex *v3, Texture *texture ) ;


/* Save the document to fu_file */
gint quakemap_save_doc(struct fu_file *file, Model *doc) {

    GNode *n;
    n = doc->root;

    fu_truncate(file, 0);

    fu_printf( file, "// This map file was generated by kludge3d.\n" );
    fu_printf( file, "// Kludge3d is open-source software.\n" );
    fu_printf( file, "// You can find it at http://kludge3d.sourceforge.net\n" );
    fu_printf( file, "{\n" );
    fu_printf( file, " \"classname\" \"worldspawn\"\n" );

    g_node_children_foreach( n, G_TRAVERSE_ALL, 
                             quakemap_write_mesh, file );
    
    fu_printf( file, "}\n" );

    return TRUE;
}

void quakemap_write_mesh( GNode *node, gpointer user_data ) {

    Mesh *m;
    GSList *polys;
    
    m = (Mesh *)node->data;
    
    for( polys = m->polygons; polys; polys = polys->next ) {
        Poly *p = (Poly*)polys->data;
        
        quakemap_write_polygon( user_data, p, m->texture );
    }

}

void quakemap_write_polygon( struct fu_file *file, Poly *p, Texture *texture ) {

// This is undoubtedly an inefficient way to convert polygon-based 
// geometry into quake-compatible brushes.  Thing is, we can't just 
// turn our kludge3d meshes into quake brushes, because kludge3d 
// has no requirement that meshes be enclosed, convex solids.
// Quake, OTOH, requires both these conditions.
// Hence, each kludge3d polygon will become a separate quake brush.

// Steps:
// find normal for poly
// make that normal very small (mult by 0.03)
// find avg of polys verts
// subtract smallNormal from avg
// remember that quake surfaces are defined clockwise
//   in total, an N-gon will produce a brush with N+1 surfaces

    float smallNormal[3];
    int i;
    Vertex *avg;
    GSList *verts = NULL;
    
    if( p == NULL || file == NULL )
        return;
    if( p->num_verts < 3 )
        return;

    // if the poly has been manipulated recently, its normal is 
    // likely out-of-date
    poly_find_normal( p );
    
    for( i=0; i < 3; i++ ) {
        smallNormal[i] = p->normal[i] * 0.03;
    }
    

    for( i=0; i < p->num_verts; i++ ) {
        verts = g_slist_append( verts, p->verts[i] );
    }
    
    avg = vertex_new();
    vertices_find_mean( verts, avg );
    vector_sub( avg->v, avg->v, smallNormal );

    fu_printf( file, "{\n" );
    
    // face that the player will actually see
    quakemap_write_brushsurface( file, p->verts[2], p->verts[1], p->verts[0], texture );
    
    // other faces
    for( i=0; i < p->num_verts; i++ ) {
        quakemap_write_brushsurface( file, p->verts[i], p->verts[(i+1) % p->num_verts], avg, texture );
    }
    
    fu_printf( file, "}\n" );
    
    vertex_delete( avg );
}


/*
From a document on the quake map format:

 ( 128 0 0 ) ( 128 1 0 ) ( 128 0 1 ) GROUND1_6 0 0 0 1.0 1.0
   1st Point   2nd Point   3rd Point   Texture  

Now, you're probably wondering what those last five numbers are. 
I've listed what they do below:

x_off     - Texture x-offset (must be multiple of 16)
y_off     - Texture y-offset (must be multiple of 16)
rot_angle - floating point value indicating texture rotation
x_scale   - scales x-dimension of texture (negative value to flip)
y_scale   - scales y-dimension of texture (negative value to flip)

*/

void quakemap_write_brushsurface( struct fu_file *file, Vertex *v1, Vertex *v2, Vertex *v3, Texture *texture ) {

    // remember that quake surfaces are defined clockwise

    // POSSIBLE FUTURE ENHANCEMENT - rint these floats and write out integer data?
    
    fu_printf( file, "( %f %f %f ) ( %f %f %f ) ( %f %f %f ) ", 
        v1->v[0] TO_QUAKE, v1->v[1] TO_QUAKE, v1->v[2] TO_QUAKE,
        v2->v[0] TO_QUAKE, v2->v[1] TO_QUAKE, v2->v[2] TO_QUAKE,
        v3->v[0] TO_QUAKE, v3->v[1] TO_QUAKE, v3->v[2] TO_QUAKE 
    );

    if( texture == NULL ) {
        fu_printf( file, "%s, %i, %i, %i, %f, %f\n", 
            "FIXME_0",
            0,
            0,
            0,
            1.0f,
            1.0f );
    } else {
        // fixme...
        fu_printf( file, "%s, %i, %i, %i, %f, %f\n", 
            "FIXME_0",
            0,
            0,
            0,
            1.0f,
            1.0f );
   }
    
}

