/*
 * File:         fmt_wavefront.c
 * 
 * Description:  handles loading and saving of model data in 
 *               Alias Wavefront format
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */

/* FIXME - I think Wavefront uses backwards coordinates.
 I think that this is the same system used by Microsoft and 3dfx, 
 where -y is up (or something similarly silly). */


/*
 * Portions of this file were ripped from:
 *	- ME3D 3-D Modeler Program   by Sam Revitch
 *
 */

/* Alias Wavefront obj file format handler */

#include <string.h>

#include "globals.h"
#include "mesh.h"
#include "model.h"
#include "group.h"
#include "polygon.h"
#include "vertex.h"
#include "texture.h"
#include "bottombar.h"
#include "tools.h"
#include "gui.h"

#include "model_load_save.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif

gint wavefront_load_doc(struct fu_file *file, Model *doc);
gint wavefront_save_doc(struct fu_file *file, Model *doc);
void wavefront_destroy_priv_data(Model *doc);


struct doc_file_fmt fmt_wavefront = {
	"Alias Wavefront (obj)",
	"obj",
	wavefront_load_doc,
	wavefront_save_doc,
	NULL
};


struct wavefront_state {
	Model *doc;
	struct vertex_prefs *v_prefs;
	struct surface_prefs *s_prefs;
	gint n_vertices, n_surfaces, n_materials;

	GtkWidget *scale_edit_box;
	GLfloat scale_factor;

	GHashTable *mtl_hash;
	GHashTable *group_hash;
	GSList *v_list;		/* list of Vertices */
	GSList *tc_list;		/* list of TexCoords */
	struct fu_file *file;
};

struct wavefront_material {
	gchar *name;
	GLfloat diff[4];
	GLfloat amb[4];
	GLfloat spec[4];
	GLfloat emis[4];
	GLfloat shin;
	Texture *tex;
};


/* functions to prompt for scale factor */
gint wavefront_prompt(struct wavefront_state *ws);


/* load pass 1 functions */
void wavefront_load_texcoord( char *buf, struct wavefront_state *ws ) ;
gint wavefront_load_doc_1(struct fu_file *file, struct wavefront_state *ws);
gint wavefront_load_mtllib(gchar *filename, struct wavefront_state *ws);
void wavefront_dump_mtlhash(gpointer key, gpointer value, gpointer user_data);
void wavefront_dump_grouphash(gpointer key, gpointer value, gpointer user_data);
void wavefront_dump_tclist( gpointer data, gpointer user_data );
void wavefront_parse_face_vertex( char *bp, 
								  int *vertex_index, int *texcoord_index ) ;


/* load pass 2 functions */
gint wavefront_load_doc_2(struct fu_file *file, struct wavefront_state *ws);



/* save funcs */
void wavefront_save_part1( struct fu_file *file, struct wavefront_state *ws ) ;
void wavefront_build_lists( GNode *node, gpointer user_data ) ;
void wavefront_save_vertex( gpointer data, gpointer user_data ) ;
void wavefront_save_texcoord( gpointer data, gpointer user_data ) ;
void wavefront_save_part2( struct fu_file *file, struct wavefront_state *ws ) ;
void wavefront_save_group( GNode *node, gpointer user_data ) ;



gint wavefront_load_doc(struct fu_file *file, Model *doc) {
	struct wavefront_state ws = {
									NULL
								};
	Mesh *m = NULL;

	/* Note - model_new has already created an empty mesh and groupnode.  
	We don't need either, so we'll delete them here. */
	if( doc->root )
		group_delete( doc->root );
	
	ws.doc = doc;
	//  ws.v_prefs = &(doc->config.v_prefs);
	//  ws.s_prefs = &(doc->config.s_prefs);
	ws.scale_factor = 1.0;

	/* Init */
	ws.mtl_hash = g_hash_table_new(g_str_hash, g_str_equal);
	ws.group_hash = g_hash_table_new(g_str_hash, g_str_equal);
	ws.tc_list = NULL;

	/* create the top group */
	doc->root = group_new( doc );
	m = (Mesh*)doc->root->data;
	mesh_set_name( m, TOP_GROUP_NAME );

	g_hash_table_insert(ws.group_hash, m->name, doc->root);


	/* Pass 1 */
	wavefront_load_doc_1(file, &ws);
#ifdef VERBOSE
	printf("%u vertices loaded\n%u material(s) loaded\n",
		   ws.n_vertices, ws.n_materials);
#endif

	/* prompt the user for a scaling factor */
	if(wavefront_prompt(&ws) == FALSE)
		return FALSE;

	/* Pass 2 */
	fu_seek(file, 0);
	wavefront_load_doc_2(file, &ws);
#ifdef VERBOSE
	printf("%u surface(s) loaded\n", ws.n_surfaces);
#endif


	/* Cleanup */
	g_hash_table_foreach(ws.mtl_hash, wavefront_dump_mtlhash, NULL);
	g_hash_table_destroy(ws.mtl_hash);
	g_hash_table_foreach(ws.group_hash, wavefront_dump_grouphash, NULL);
	g_hash_table_destroy(ws.group_hash);

	g_slist_foreach( ws.tc_list, wavefront_dump_tclist, NULL );
	g_slist_free( ws.tc_list );

	/* we've got all the verts stored in the model, 
	   so we can empty this one too */
	g_slist_free( ws.v_list );
	
	return TRUE;
}





void wavefront_load_texcoord( char *buf, struct wavefront_state *ws ) {
	
	TexCoord *tc = NULL;
	float u = 0.0;
	float v = 0.0;
	int status;

	/* extract the texture coordinates */
	status = sscanf( buf, "%f %f", &u, &v );

	if( status > 0 ) {
		tc = (TexCoord *)malloc( sizeof( TexCoord ) );
		tc->x = u;
		tc->y = v;
		ws->tc_list = g_slist_append( ws->tc_list, tc );
	} else {
		printf( "%s: parsing vt entry '%s' failed\n", __FUNCTION__, buf );
	}

}



/*
 * FIRST PASS 
 * This pass reads in all the vertices, and creates empty groups.  That's it. 
 * The surface information will be read in during pass 2.
 */

gint wavefront_load_doc_1(struct fu_file *file, struct wavefront_state *ws) {
	gint pos;
	Vertex *pV;
	GNode *pG;
	double v[3];
	gchar buf[512];

	pG = ws->doc->root;

	while(fu_gets(file, buf, 512) >= 0) {
		switch(buf[0]) {
		case 'v':
			if( buf[1] == 't' ) {
				/* it was a 'vt' (texcoord), not a 'v' */
				wavefront_load_texcoord( &buf[2], ws );
				break;
			}
			
			if((buf[1] != ' ') && (buf[1] != '\t'))
				break;
			/* allocate a vertex struct */
			pV = vertex_new();

			/* extract the vertex coordinates */
			sscanf( &buf[1], "%lf %lf %lf", &v[0], &v[1], &v[2] );
			pV->v[0] = v[0];
			pV->v[1] = v[1];
			pV->v[2] = v[2];

			/* add the new vert to the model.  We'll deal with adding it to a 
			particular mesh later */
			ws->v_list = g_slist_append( ws->v_list, pV );
			model_vertex_add( ws->doc, pV );
			ws->n_vertices++;
			break;

		case 'm':
			/* mtllib reader */
			if(strncmp(buf, "mtllib", 6) != 0)
				break;
			pos = 6;
			while((buf[pos] == ' ') || (buf[pos] == '\t')) pos++;
			wavefront_load_mtllib(&buf[pos], ws);
			break;

		case 'g':
			/* new group */
			if(strncmp(buf, "group", 5) == 0)
				pos = 5;
			else
				pos = 1;
			while((buf[pos] == ' ') || (buf[pos] == '\t')) pos++;
			if(buf[pos] == '\0')
				break;
			if(g_hash_table_lookup(ws->group_hash, &buf[pos]) == NULL) {
				/* create a new group */

				pG = g_node_append_data( ws->doc->root, mesh_new( ws->doc ) );
				mesh_set_name( (Mesh*)pG->data, &buf[pos] );

				g_hash_table_insert(ws->group_hash, ((Mesh*)pG->data)->name, pG);
			}
			break;
		}
	}

	return TRUE;
}

gint wavefront_load_mtllib(gchar *filename, struct wavefront_state *ws) {
	gint pos = -1;
	struct fu_file *mtlfile;
	struct wavefront_material *cur_mtl = NULL;
	gchar buf[512];

	if( !g_path_is_absolute( filename ) ) {
		/* If the mtllib file's name is relative, rather than absolute, then 
		assemble the full file name by concatenating the model file's directory 
		name and the mtllib file's name. 
		We could achieve similar results by chdir-ing to the model's dir, then 
		opening the mtllib file, then chdir-ing back. */
		gchar *full_fname, *model_dirname;
		int full_fname_len;

		model_dirname = g_path_get_dirname( ws->doc->fname );

		full_fname_len = strlen(model_dirname) + strlen(filename) + 
						2; /* 2: 1 for '\0' + 1 for FMT_DELIMITER */
		full_fname = (gchar *) malloc( full_fname_len );
		memset( full_fname, 0, full_fname_len );

		strcpy( full_fname, model_dirname );
		full_fname[strlen(full_fname)] = FMT_DELIMITER;
		strcat( full_fname, filename );

		mtlfile = fu_open( full_fname, FALSE );

		g_free( model_dirname );
		free( full_fname );
	} else {
		mtlfile = fu_open(filename, FALSE);
	}

	if(mtlfile == NULL) {
		g_warning("couldn't open material library '%s'\n", filename);
		return FALSE;
	}

	while(fu_gets(mtlfile, buf, 512) >= 0) {
		if(buf[0] == 'K') {
			if((buf[2] != ' ') && (buf[2] != '\t'))
				continue;
			if(cur_mtl == NULL)
				continue;

			switch(buf[1]) {
				case 'd':
					sscanf(&buf[3], "%f %f %f", 
						&cur_mtl->diff[0], 
						&cur_mtl->diff[1],
						&cur_mtl->diff[2]);
					break;
				case 'a':
					sscanf(&buf[3], "%f %f %f", 
						&cur_mtl->amb[0], 
						&cur_mtl->amb[1],
						&cur_mtl->amb[2]);
					break;
			}
		} else if( buf[0] == 'm' ) {
			if(cur_mtl == NULL)
				continue;
			if( strncmp( buf, "map_Kd", 6 ) != 0 )
				continue;
			pos = 6;
			while((buf[pos] == ' ') || (buf[pos] == '\t')) pos++;
			
			cur_mtl->tex = 
				tex_load( &buf[pos], TEXTURE_FORMAT_UNKNOWN, ws->doc->fname );
			
			if( cur_mtl->tex == NULL ) {
				/* there was an error loading the texture file */
				bb_push_message_f( 3.0, "Texture '%s' could not be loaded", 
					&buf[pos] );
			} else {
				model_add_texture( ws->doc, cur_mtl->tex );
			}
			
		} else {
			switch(buf[0]) {

			case 'n':
				if(strncmp(buf, "newmtl", 6) != 0)
					break;

				pos = 6;
				while((buf[pos] == ' ') || (buf[pos] == '\t')) pos++;

				cur_mtl = g_hash_table_lookup(ws->mtl_hash, &buf[pos]);

				if(cur_mtl == NULL) {
					cur_mtl = (struct wavefront_material *)
								malloc(sizeof(struct wavefront_material));
					memset( cur_mtl, 0, sizeof(struct wavefront_material) );
					cur_mtl->name = (gchar *) malloc(strlen(&buf[pos]) + 1);
					strcpy(cur_mtl->name, &buf[pos]);
					cur_mtl->diff[0] = 0.8;
					cur_mtl->diff[1] = 0.8;
					cur_mtl->diff[2] = 0.8;
					cur_mtl->diff[3] = 1.0;
					cur_mtl->amb[0] = 0.2;
					cur_mtl->amb[1] = 0.2;
					cur_mtl->amb[2] = 0.2;
					cur_mtl->amb[3] = 1.0;
					cur_mtl->spec[0] = 0.0;
					cur_mtl->spec[1] = 0.0;
					cur_mtl->spec[2] = 0.0;
					cur_mtl->spec[3] = 1.0;
					cur_mtl->emis[0] = 0.0;
					cur_mtl->emis[1] = 0.0;
					cur_mtl->emis[2] = 0.0;
					cur_mtl->emis[3] = 1.0;
					cur_mtl->shin = 0.0;
					g_hash_table_insert(ws->mtl_hash, cur_mtl->name, cur_mtl);
					ws->n_materials++;
				}
				break;

			default:
				/* unknown directive */
				break;
			}
		}	/* if(buf[0] == 'K') */

	}	/* while(fu_gets) */

	fu_close(mtlfile);
	return TRUE;
}

void wavefront_dump_mtlhash(gpointer key, gpointer value, gpointer user_data) {
	free(key);
	free(value);
}

void wavefront_dump_grouphash(gpointer key, gpointer value, gpointer user_data) {}

void wavefront_dump_tclist( gpointer data, gpointer user_data ) {
	free(data);
}


gint wavefront_prompt(struct wavefront_state *ws) {

	GtkWidget *dialog;
	GtkWidget *hbox, *widget;
	gchar buf[32];
	gint dlg_response;
	int result = TRUE;

	dialog = gtk_dialog_new_with_buttons ("Alias Wavefront Options",
										  GTK_WINDOW( TopLevelWindow ),
										  GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
										  GTK_STOCK_OK,
										  GTK_RESPONSE_OK,
										  GTK_STOCK_CANCEL,
										  GTK_RESPONSE_CANCEL,
										  NULL);
	
	/* Scaling - frame */
	widget = gtk_frame_new("Scaling");
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), widget, TRUE, TRUE, 0);
	
	/* Scaling - hbox */
	hbox = gtk_hbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(widget), hbox);
	
	/* Scaling - entry box */
	widget = gtk_label_new("Scaling Factor:");
	gtk_box_pack_start(GTK_BOX(hbox), widget, FALSE, FALSE, 0);

	ws->scale_edit_box = gtk_entry_new();
	sprintf(buf, "%g", ws->scale_factor);
	gtk_entry_set_text(GTK_ENTRY(ws->scale_edit_box), buf);
	gtk_box_pack_start(GTK_BOX(hbox), ws->scale_edit_box, TRUE, TRUE, 0);


	/* set up the OK button as the default widget */
	gtk_dialog_set_default_response( GTK_DIALOG(dialog), GTK_RESPONSE_OK );
	g_signal_connect( G_OBJECT( ws->scale_edit_box ), "activate",
					G_CALLBACK(gui_dialog_ok_cb), dialog );
	
	/* shows the dialog, and since it's modal, prevents rest of 
	   kludge3d from doing stuff */
	gtk_widget_show_all (dialog);
	
	/* runs a gtkmain until the user clicks on something */
	dlg_response = gtk_dialog_run( GTK_DIALOG (dialog) );
	
	switch( dlg_response ) {
	case GTK_RESPONSE_OK:
		ws->scale_factor =
			atof(gtk_entry_get_text(GTK_ENTRY(ws->scale_edit_box)));
		if( ws->scale_factor < 0.0 )
			ws->scale_factor = -1.0 * ws->scale_factor;
		break;
	default:
		ws->scale_factor = -1.;
		result = FALSE;
		break;
	}
	
	gtk_widget_destroy( dialog );


#ifdef VERBOSE
	printf("scaling factor: %lf\n", ws->scale_factor);
#endif

	if( ws->scale_factor > 0.0f ) {
		/* perform scaling of vertex coordinates */
		tools_vertices_scale_about_origin( ws->doc, ws->v_list, ws->scale_factor, TRUE );
	}

#ifndef NO_SWAP_YZ
	tools_vertices_rotate_90( ws->doc, ws->v_list, 0 );
	/* hehe... hack */
	tools_vertices_rotate_90( ws->doc, ws->v_list, 0 );
	tools_vertices_rotate_90( ws->doc, ws->v_list, 0 );
#endif

	return result;
}




/** second pass functions **/

void wavefront_parse_face_vertex( char *bp, 
								  int *vertex_index, int *texcoord_index ) {

	int has_tc_index = FALSE;
	char *temp;
	
	if( bp == NULL || vertex_index == NULL || texcoord_index == NULL )
		return;
	
	for( temp = bp; temp[0] != '\0'; ++temp ) {
		if( temp[0] == '/' ) {
			temp[0] = '\0';
			has_tc_index = TRUE;
			temp++;
			break;
		}
	}
	
	*vertex_index = atoi( bp );
	*texcoord_index = -1;
	if( has_tc_index )
		*texcoord_index = atoi( temp );

}


gint wavefront_load_doc_2(struct fu_file *file, struct wavefront_state *ws) {
	int n_links, v_idx, tc_idx, pos;
	struct wavefront_material *cur_mtl = NULL;
	GNode *pG;

	gchar buf[512], *bp;
	int v_indices[POLY_MAX_VERTS];
	int tc_indices[POLY_MAX_VERTS];
	Poly *p = NULL;
	int i;

	pG = ws->doc->root;		/* default group = top */

	while(fu_gets(file, buf, 512) >= 0) {
		switch(buf[0]) {

		case 'f':		/* n-sided facet */
			if((buf[1] != ' ') && (buf[1] != '\t'))
				break;

			n_links = 0;
			p = NULL;

			/* read in the index for each of the vertices */
			bp = strtok(&buf[2], " ");
			while( bp ) {
				if(bp[0] == '\0')
					goto wavefront_lp3_skip;

				wavefront_parse_face_vertex( bp, &v_idx, &tc_idx );
				
				/* TODO: provision for v_idx < 1 */
				if( (v_idx > ws->n_vertices) || (v_idx < 1) )
					return FALSE;

				if( n_links < POLY_MAX_VERTS ) {

					/* wavefront vertices are 1 indexed */
					v_indices[n_links] = v_idx - 1;
					tc_indices[n_links] = tc_idx - 1;

					n_links++;
				}

wavefront_lp3_skip:
				bp = strtok(NULL, " ");
			}


			if( n_links < 2 ) {
				return -1;
			}
			else {
				Vertex *v;
				TexCoord *tc;
				p = poly_new();
				for( i = 0; i < n_links; i++ ) {
					if( v_indices[i] >= 0 ) {
						v = g_slist_nth_data( ws->v_list, v_indices[i] );
						if( v != NULL ) {
							poly_add_vertex( p, v );
						}
else{ printf( "%s: v # %i not found in ws->v_list!\n", __FUNCTION__, v_indices[i] ); }
					}

					if( tc_indices[i] >= 0 ) {
						tc = g_slist_nth_data( ws->tc_list, tc_indices[i] );
						if( tc != NULL ) {
							p->tc[i]->x = tc->x;
							p->tc[i]->y = tc->y;
						}
else{ printf( "%s: tc # %i not found in ws->tc_list!\n", __FUNCTION__, tc_indices[i] ); }
					}

				}

				/* We're done - add the poly to the mesh */
				mesh_polygon_add( (Mesh*)pG->data, p );
			}

			ws->n_surfaces++;
			break;


		case 'g':		/* change group */
			if(strncmp(buf, "group", 5) == 0)
				pos = 5;
			else
				pos = 1;
			while((buf[pos] == ' ') || (buf[pos] == '\t')) pos++;

			if(buf[pos] == '\0') {
				pG = ws->doc->root;
				break;
			}

			/* look up the group name */
			pG = g_hash_table_lookup(ws->group_hash, &buf[pos]);
			if(pG == NULL) {
				g_warning("unknown group '%s' set\n", &buf[pos]);
				pG = ws->doc->root;
			}
			break;


		case 'u':		/* material selector */
			if(strncmp(buf, "usemtl", 6) != 0)
				break;

			pos = 6;
			while((buf[pos] == ' ') || (buf[pos] == '\t')) pos++;

			cur_mtl = g_hash_table_lookup(ws->mtl_hash, &buf[pos]);
			if(cur_mtl == NULL)
				g_warning("wavefront_load_2: unknown material '%s' requested\n",
						  &buf[pos]);
			else {
				/* apply the material to the current group */
				if( cur_mtl->tex )
					mesh_set_texture( (Mesh *)pG->data, cur_mtl->tex );
				/* fixme - apply further material attributes */
			}
			break;
		}
	}

	return TRUE;
}


/* Save the document to fu_file */
gint wavefront_save_doc(struct fu_file *file, Model *doc) {

	struct wavefront_state ws = { NULL };

	fu_truncate(file, 0);

	ws.doc = doc;

	/* Init */
	ws.v_list = NULL;
	ws.tc_list = NULL;

	/* Pass 1 */
	wavefront_save_part1( file, &ws );

	/* Pass 2 */
	wavefront_save_part2( file, &ws );

	/* Cleanup */
	if( ws.v_list != NULL )
		g_slist_free( ws.v_list );
	if( ws.tc_list != NULL )
		g_slist_free( ws.tc_list );

	/* fixme - save material tags for groups to model file, save 
	materials to mtllib file */

	return TRUE;
}

/* builds up the vertex and tc lists, writes them out */
void wavefront_save_part1( struct fu_file *file, struct wavefront_state *ws ) {

	GNode *n;
	
	n = ws->doc->root;

	g_node_children_foreach( n, G_TRAVERSE_ALL, 
							 wavefront_build_lists, ws );
	
	fu_printf( file, "# file generated by kludge3d\n" );
	fu_printf( file, "# model contains %i vertices\n\n", 
			   g_slist_length( ws->doc->verts ) );

	g_slist_foreach( ws->doc->verts, wavefront_save_vertex, file );
	fu_printf( file, "\n" );
	g_slist_foreach( ws->tc_list, wavefront_save_texcoord, file );
	fu_printf( file, "\n" );
}

/* writes out polygon and group data */
void wavefront_save_part2( struct fu_file *file, struct wavefront_state *ws ) {

	GNode *n;
	
	n = ws->doc->root;

	ws->file = file;
	
	g_node_children_foreach( n, G_TRAVERSE_ALL, 
							 wavefront_save_group, ws );

}

void wavefront_build_lists( GNode *node, gpointer user_data ) {

	Mesh *m;
	GSList *polys;
	int i;
	struct wavefront_state *ws = user_data;
	
	m = (Mesh *)node->data;
	
	/* Note - no longer need to build v_list, as we're using ws->doc->verts */
	
	for( polys = m->polygons; polys; polys = polys->next ) {
		Poly *p = (Poly*)polys->data;
		
		for( i = 0; i < p->num_verts; i++ ) {
			ws->tc_list = g_slist_append( ws->tc_list, p->tc[i] );
		}
	}

}

void wavefront_save_vertex( gpointer data, gpointer user_data ) {
	
	struct fu_file *file = (struct fu_file *)user_data;
	Vertex *v = (Vertex *)data;
	
	fu_printf( file, "v %f %f %f\n", v->v[0], v->v[2], -1.0 * v->v[1] );
}

void wavefront_save_texcoord( gpointer data, gpointer user_data ) {
	
	struct fu_file *file = (struct fu_file *)user_data;
	TexCoord *tc = (TexCoord *)data;
	
	fu_printf( file, "vt %f %f\n", tc->x, tc->y );
}


void wavefront_save_group( GNode *node, gpointer user_data ) {
	
	struct wavefront_state *ws;
	struct fu_file *file;
	Mesh *m;
	GSList *polys;
	int i;
	
	ws = (struct wavefront_state *)user_data;
	file = ws->file;
	m = (Mesh*)node->data;
	
	fu_printf( file, "g %s\n", m->name );
	
	for( polys = m->polygons; polys; polys = polys->next ) {
		Poly *p = (Poly*)polys->data;
		
		fu_printf( file, "f" );
		for( i = 0; i < p->num_verts; i++ ) {
			fu_printf( file, " %i/%i",
				g_slist_index( ws->doc->verts, p->verts[i] ) + 1,
				g_slist_index( ws->tc_list, p->tc[i] ) + 1 );
		}
		fu_printf( file, "\n" );
	}
	
}


/* Destroy the contents of doc->fmt_data */
void wavefront_destroy_priv_data(Model *doc) {}


