/*
 * File:         glpreview.c
 * 
 * Description:  handles rendering, opengl stuff, selection testing, etc
 * 
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */


/*
 * Portions of this file are from:
 * 
 * ME3D 3-D Modeler Program
 * Copyright (C) 1998 Sam Revitch
 *
 * G3d modeler program
 * Copyright (C) 2000 Blake E Hegerle
 *
 */


#include <pango/pango.h> /* needed for origin marker text */
#include <math.h>
#include <string.h>
#include "win32stuff.h"

#include "model.h"
#include "polygon.h"
#include "globals.h"
#include "view.h"
#include "view_config.h"
#include "glpreview.h"
#include "notebook.h"
#include "background.h"
#include "bottombar.h"
#include "vector.h"
#include "misc.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif



/* DEFINES, etc. ********************************************************/

/* the minimum and maximum number of entries (primitives) we would like 
   to be able to select in glp_test_click_objects */
#define MIN_SELECTION_BUFFER_LEN 20
#define MAX_SELECTION_BUFFER_LEN 4000

#define DISPLAYLIST_VERTEX 2


// PROTOTYPES ///////////////////////////////////////////////////////////

void glp_texenv( void ) ;
void glp_set_current_tex( Texture *curr_tex ) ;
void glp_ortho_set_size( model_view *view ) ;
void glp_set_viewvolume( model_view *view ) ;
void glp_setup_camera( model_view *view ) ;
void glp_cleanup_camera( model_view *view ) ;
int glp_primitive_type_lookup( int numverts ) ;
void glp_draw_primitive( model_view *view, Poly * p ) ;
void glp_draw_primitives( model_view *view, GSList* polys ) ;
void glp_draw_primitives_textured( model_view *view, GSList * polys,
                                   float texRepX, float texRepY ) ;
void glp_draw_primitive_edit( model_view *view, Poly * t );
void glp_draw_normals( model_view *view, GSList* polys );
void glp_draw_cursor( model_view *view, GLfloat x, GLfloat y,
                      GLfloat z, GLfloat a );
void glp_draw_background( model_view *view ) ;

void glp_draw_vertices( model_view *view, GSList * vertices );
void glp_draw_vertices_as_cubes( model_view *view, GSList* vertices ) ;
void glp_draw_girder( model_view *view, Vertex *p1, Vertex *p2 );

gboolean glp_draw_surfaces_tree( GNode *node, gpointer data );
void glp_draw_mesh( model_view *view, Mesh * m, gint box_it );
void glp_draw_grid( GLfloat *color, GLfloat *mins, GLfloat *maxes, GLfloat spacing );
void glp_build_font( void ) ;
void glp_draw_origin( model_view *view );

void glp_swap_buffers( GdkGLDrawable *gldrawable );

GSList *glp_test_click_objects( model_view *view,
                                GLfloat *z, guint mouse_x, guint mouse_y,
                                guint width, guint height, int closest ) ;


/* FILE-SCOPE VARS ******************************************************/

//Texture *active_texture = NULL;

/* used to retain the gl font displaylist (only need one for each gl context) */
GLuint glp_fontbase = 0;
GLuint glp_displist_vertex = 0;


///////////////////////////////////////////////////////////////////////////
//
// texture funcs
//
///////////////////////////////////////////////////////////////////////////


void glp_texenv( void ) {
    GLfloat env_color[] = {0, 1, 0, 1};
    GLfloat border_color[] = {1, 0, 0, 1};

    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
                     GL_NEAREST );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT );
    glTexEnvi( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL );
    glTexEnvfv( GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, env_color );
    glTexParameterfv( GL_TEXTURE_2D, GL_TEXTURE_BORDER_COLOR, border_color );
    glEnable( GL_TEXTURE_2D );
}


void glp_set_current_tex( Texture *curr_tex ) {

    tex_data *ct;

    if( curr_tex == NULL ) return;

    ct = &(curr_tex->mip[TEXTURE_MIP_LEVEL_OPENGL]);

    glTexImage2D( GL_TEXTURE_2D, 0, ct->depth, ct->width, ct->height, 0,
                  ct->depth == 4 ? GL_RGBA : GL_RGB, GL_UNSIGNED_BYTE,
                  ct->data );
}





//////////////////////////////////////////////////////////////////////////
//
//
// init/setup stuff
//
//
//////////////////////////////////////////////////////////////////////////

void glp_ortho_set_size( model_view *view ) {

//fixme - will probably be phased out, eventually

    // sets the extents for the orthographic views...
    float halfdist;
    float aspect_ratio = 1.0;

    halfdist = (view->dist / 2.0);

    view->minX = - halfdist;
    view->maxX = halfdist;

    view->minY = - halfdist;
    view->maxY = halfdist;

    view->minZ = - halfdist;
    view->maxZ = halfdist;

    if( view->glarea != NULL ) {

        aspect_ratio = (float)(GTK_WIDGET( view->glarea )->allocation.width) /
                       (float)(GTK_WIDGET( view->glarea )->allocation.height);

        switch( view->type ) {
        case VIEW_XY:
            view->minX = view->minY * aspect_ratio;
            view->maxX = view->maxY * aspect_ratio;
            break;
        case VIEW_XZ:
            view->minX = view->minZ * aspect_ratio;
            view->maxX = view->maxZ * aspect_ratio;
            break;
        case VIEW_YZ:
            view->minY = view->minZ * aspect_ratio;
            view->maxY = view->maxZ * aspect_ratio;
            break;
        default:
            break;
        }

    }

}



/*
 *  This func is called once (and only once) for each of the 4 views
 */
void glp_glconfigure( model_view *view ) {

    GLfloat whitelight[] = { 1.0, 1.0, 1.0, 0.5 };
    GLfloat amblight[] = { 0.2, 0.2, 0.2, 1.0 };
    GLfloat lightpos[] = { -5.0, 5.0, -2.0, 1.0 };

    g_return_if_fail( view != NULL );

	g_return_if_fail( gdk_gl_drawable_gl_begin( view->gldrawable, view->glcontext ) );

    /* clear frame color */
    glClearColor( view_config.bkgnd_color[ 0 ], view_config.bkgnd_color[ 1 ],
                  view_config.bkgnd_color[ 2 ], 0.0 );

    /* These are the same under all circumstances */
    glEnable( GL_DEPTH_TEST );
    glClearDepth( 1.0f );
    glDepthFunc( GL_LEQUAL );
    glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );

    /* lighting is the same (for now) */
    glEnable( GL_LIGHT0 );
    glLightfv( GL_LIGHT0, GL_POSITION, lightpos );
    glLightfv( GL_LIGHT0, GL_DIFFUSE, whitelight );
    glLightfv( GL_LIGHT0, GL_AMBIENT, amblight );

    switch( view_config.triangle_order ) {
    case MP_TRI_ORDER_CW:
        glEnable( GL_CULL_FACE );
        glFrontFace( GL_CW );
        break;
    case MP_TRI_ORDER_CCW:
        glEnable( GL_CULL_FACE );
        glFrontFace( GL_CCW );
        break;
    case MP_TRI_ORDER_ALL:
        glDisable( GL_CULL_FACE );
        break;
    default:
        break;
    }

    //FIXME
    //    if ( view->pDoc->config->smooth_mode )
    //        glShadeModel( GL_SMOOTH );
    //    else
    glShadeModel( GL_FLAT );

    glMatrixMode( GL_MODELVIEW );
    glLoadIdentity();

    glp_ortho_set_size( view );

	gdk_gl_drawable_gl_end( view->gldrawable );
}




void glp_set_viewvolume( model_view *view ) {

    // this was made a sep. func. so that it could be
    // used by both the picking routines and glp_glviewport

    float aspect_ratio;

    aspect_ratio = 
           (float)(GTK_WIDGET( view->glarea )->allocation.height) /
           (float)(GTK_WIDGET( view->glarea )->allocation.width);

    if( view->type == VIEW_3D ) {

        GLfloat xd, yd, fov_rad;

        fov_rad = ( view_config.fov * M_PI ) / 360;
        xd = view_config.near_plane * tan( fov_rad );
        yd = xd * aspect_ratio;

        glFrustum( -xd, xd, -yd, yd, view_config.near_plane, view_config.far_plane );

    } else {

        float minX, minY, maxX, maxY;
        
        // set a 2D orthographic projection
        switch( view->type ) {
        case VIEW_XY:
            minX = view->minX;
            maxX = view->maxX;
            minY = view->minX * aspect_ratio;
            maxY = view->maxX * aspect_ratio;
            break;
        case VIEW_XZ:
            minX = view->minX;
            maxX = view->maxX;
            minY = view->minX * aspect_ratio;
            maxY = view->maxX * aspect_ratio;
            break;
        case VIEW_YZ:
            minX = view->minY;
            maxX = view->maxY;
            minY = view->minY * aspect_ratio;
            maxY = view->maxY * aspect_ratio;
            break;
        default:
            break;
        }
        glOrtho( minX, maxX, minY, maxY, 
            -view_config.far_plane, view_config.far_plane );
    }
}


/*
 *  This func is called every time there is a resize event
 */
void glp_glviewport( model_view *view ) {


    // THIS DOES NOT SET THE CAMERA!
    // NO CAMERA-RELATED TRANSFORMS SHOULD GO IN HERE!
    // STOP BEING AN IDIOT, ANDY!

    // this is ONLY to set up the view volume.

    if ( GTK_WIDGET_REALIZED( GTK_WIDGET( view->glarea ) ) &&
            ( GTK_WIDGET( view->glarea ) ->allocation.width > 1 ) &&
            ( GTK_WIDGET( view->glarea ) ->allocation.height > 1 ) ) {

        g_return_if_fail( gdk_gl_drawable_gl_begin( view->gldrawable, view->glcontext ) );

        glViewport( 0, 0, GTK_WIDGET( view->glarea ) ->allocation.width,
                    GTK_WIDGET( view->glarea ) ->allocation.height );

        // switch to projection mode
        glMatrixMode(GL_PROJECTION);

        // reset matrix
        glLoadIdentity();

        glp_set_viewvolume( view );

        glMatrixMode( GL_MODELVIEW );


#ifdef VERBOSE_DEBUG
        g_print( "glp_glviewport %ux%u %f->%f (%f) %f %f %f\n",
                 GTK_WIDGET( view->glarea ) ->allocation.width,
                 GTK_WIDGET( view->glarea ) ->allocation.height,
                 view_config.near_plane, view_config.far_plane, view_config.fov,
                 xd, yd, hwr );
#endif
		
		gdk_gl_drawable_gl_end( view->gldrawable );
    }
}



void glp_setup_camera( model_view *view ) {

    // set up the camera (well, actually the *world*, but you know what I mean)

    glPushMatrix();

    // determines the distance from the look-at point.  In our case,
    // the look-at point is <camX,camY,camZ>, which means that those
    // variable names are a bit misleading...
    glTranslatef( 0, 0, - ( view->dist + view_config.near_plane ) );

    // the nose-up, nose-down angle for the camera
    glRotatef( -( view->pitch ), 1.0, 0.0, 0.0 );

    // you get the idea.
    glRotatef( view->heading, 0.0, 0.0, 1.0 );

    // how well does this work?  I've noticed that at high levels of
    // magnification, the objects tend to be clipped by the camera near/far
    // planes...
    if( view->type != VIEW_3D ) {
        float zoom_factor;
        zoom_factor = view->dist / 10.0f;
        glScalef( zoom_factor, zoom_factor, zoom_factor );
    }

    // finally, translate the camera to its look-at point
    glTranslatef( view->camX, view->camY, view->camZ  );
}


void glp_cleanup_camera( model_view *view ) {
    // should be called after every call to glp_setup_camera()

    glPopMatrix();

}






//////////////////////////////////////////////////////////////////////////
//
//
//  rendering stuff
//
//
//////////////////////////////////////////////////////////////////////////



int glp_primitive_type_lookup( int numverts ) {

    switch( numverts ) {
    case 2:
        return GL_LINES;
    case 3:
        return GL_TRIANGLES;
    case 4:
        return GL_QUADS;
    case 5:
    case 6:
        return GL_TRIANGLE_FAN;
    case 1:
    default:
        return GL_POINTS;
    }
}


void glp_draw_selection_box( model_view* view, float startx, float starty,
                             float endx, float endy ) {

    float depth = 0;

    if ( view == NULL )
        return;
    if( view_config.render_mode != MP_MODE_RENDER )
        return;

    if( !gdk_gl_drawable_gl_begin( view->gldrawable, view->glcontext ) )
        return;

    // push and clear the modelview and projection matrices
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    glMatrixMode( GL_MODELVIEW );
    glPushMatrix();
    glLoadIdentity();

    // fix ogl off-center origin
    glScalef( 2.0, 2.0, 2.0 );
    glTranslatef( -0.5, -0.5, 0.0 );

    // turn on XOR rendering
    glEnable(GL_COLOR_LOGIC_OP);
    glLogicOp(GL_XOR);

    // turn off depth testing
    glDisable( GL_DEPTH_TEST );

    // set color to white ( for XOR rendering )
    glColor3f( 1.0, 1.0, 1.0 );

    glLineWidth( 3.0 );

    glBegin( GL_LINES );

    glVertex3f( startx, starty, depth );
    glVertex3f( endx,   starty, depth );

    glVertex3f( endx,   starty, depth );
    glVertex3f( endx,   endy,   depth );

    glVertex3f( endx,   endy,   depth );
    glVertex3f( startx, endy,   depth );

    glVertex3f( startx, endy,   depth );
    glVertex3f( startx, starty, depth );

    glEnd();


    // paste the buffer to the screen
    glp_swap_buffers( view->gldrawable );


    // erase the lines we just drew
    glBegin( GL_LINES );

    glVertex3f( startx, starty, depth );
    glVertex3f( endx,   starty, depth );

    glVertex3f( endx,   starty, depth );
    glVertex3f( endx,   endy,   depth );

    glVertex3f( endx,   endy,   depth );
    glVertex3f( startx, endy,   depth );

    glVertex3f( startx, endy,   depth );
    glVertex3f( startx, starty, depth );

    glEnd();


    // turn on depth testing
    glEnable( GL_DEPTH_TEST );

    // turn off XOR rendering
    glDisable(GL_COLOR_LOGIC_OP);
    glLineWidth( 1.0 );


    // restore prev. matrix states
    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode( GL_MODELVIEW );
    glPopMatrix();
	
	gdk_gl_drawable_gl_end( view->gldrawable );
}



/* used only when in selection mode for polygons */
void glp_draw_primitive( model_view *view, Poly * p ) {

    int i;

    g_return_if_fail( GTK_WIDGET_VISIBLE( view->gltable ) );
    g_return_if_fail( p != NULL );

    if( p->hidden )
        return;

    glBegin( glp_primitive_type_lookup( p->num_verts ) );

    for( i = 0; i < p->num_verts; i++ ) {
        glVertex3fv( p->verts[ i ]->v );
    }

    glEnd();
}


/* draws a list of polygons - used only when in wireframe or shaded mode */
void glp_draw_primitives( model_view *view, GSList* polys ) {

    Poly * p = NULL;
    GLfloat *color = NULL;
    int i, prev_num_verts = 3;

    g_return_if_fail( GTK_WIDGET_VISIBLE( view->gltable ) );
    
    if( polys == NULL )
        return;

    glBegin( GL_TRIANGLES );

    for( ; polys != NULL; polys = polys->next ) {
        p = (Poly *)polys->data;

        if( p->hidden )
            continue;

        if( prev_num_verts != p->num_verts ) {
            glEnd();
            glBegin( glp_primitive_type_lookup( p->num_verts ) );
			prev_num_verts = p->num_verts;
        }
        if( p->selected )
            color = view_config.poly_sel_color;
        else
            color = view_config.poly_unsel_color;

        glNormal3fv( p->normal );
        glColor3fv( color );

        for( i = 0; i < p->num_verts; i++ ) {
            glVertex3fv( p->verts[ i ]->v );
        }
    }

    glEnd();
}


/* draws a list of polygons - used only when in textured mode */
void glp_draw_primitives_textured( model_view *view, GSList * polys,
                                   float texRepX, float texRepY ) {

    Poly * p = NULL;
    int i, prev_num_verts = 3;

    g_return_if_fail( GTK_WIDGET_VISIBLE( view->gltable ) );
    
    if( polys == NULL )
        return;

    //fixme - this is pointless...
    if( texRepX == 0.0 ) texRepX = 1.0;
    if( texRepY == 0.0 ) texRepY = 1.0;

    glEnable( GL_TEXTURE_2D );
    glBegin( GL_TRIANGLES );

    for( ; polys != NULL; polys = polys->next ) {
        p = (Poly *)polys->data;

        if( p->hidden )
            continue;

        if( prev_num_verts != p->num_verts ) {
            glEnd();
            glBegin( glp_primitive_type_lookup( p->num_verts ) );
			prev_num_verts = p->num_verts;
        }
        for( i = 0; i < p->num_verts; i++ ) {
            glTexCoord2f( p->tc[ i ]->x * texRepX,
                          p->tc[ i ]->y * texRepY );
            glVertex3fv( p->verts[ i ]->v );
        }
    }

    glEnd();
    glDisable( GL_TEXTURE_2D );

}


/* draws the polygon using rectangular prisms instead of lines.
   used to draw partially-completed-poly
 */
void glp_draw_primitive_edit( model_view *view, Poly * p ) {

    Vertex *v;
    int i;

    g_return_if_fail( GTK_WIDGET_VISIBLE( view->gltable ) );
    g_return_if_fail( p != NULL );
    g_return_if_fail( p->num_verts > 0 );

    v = p->verts[0];
    for( i = 1; i < p->num_verts; i++ ) {
        glp_draw_girder( view, v, p->verts[i] );
        v = p->verts[i];
    }

    glp_draw_cursor( view, v->v[0], v->v[1], v->v[2], 0 );
}


void glp_draw_normals( model_view *view, GSList* polys ) {

	Poly * p = NULL;
	float v[3];

	glColor3f( 1.0, 0.0, 0.0 );
	glBegin( GL_LINES );

	while ( polys != NULL ) {
		p = (Poly*)polys->data;
		
		poly_find_center_point( v, p );

		glVertex3fv( v );
		glVertex3f( v[ 0 ] + (p->normal[ 0 ] * view_config.vertex_rad),
					v[ 1 ] + (p->normal[ 1 ] * view_config.vertex_rad),
					v[ 2 ] + (p->normal[ 2 ] * view_config.vertex_rad) );

		polys = polys->next;
	}

	glEnd();
}


void glp_draw_cursor( model_view *view, GLfloat x, GLfloat y,
                      GLfloat z, GLfloat a ) {
    glPushMatrix();

    glTranslatef( x, y, z );
    glRotatef( a, 0, 1, 0 );

    glInterleavedArrays( GL_C4F_N3F_V3F, 0, cursor_glarray );
    glDrawArrays( GL_TRIANGLES, 0, 24 );

    glPopMatrix();

}


void glp_draw_background( model_view *view ) {

    glPushMatrix();

    /* remember, opengl does matrix ops "backwards" */
    glScalef( background_box_size[0],
              background_box_size[1],
              background_box_size[2] );
    glTranslatef( -0.5, -0.5, -0.5 );

    glp_texenv();

    if( background_img_xy != NULL ) {
        glp_set_current_tex( background_img_xy );
        glBegin( GL_QUADS );

        glTexCoord2f( 0.0, 0.0 );
        glVertex3f( 0.0, 0.0, 0.0 );

        glTexCoord2f( 1.0, 0.0 );
        glVertex3f( 1.0, 0.0, 0.0 );

        glTexCoord2f( 1.0, 1.0 );
        glVertex3f( 1.0, 1.0, 0.0 );

        glTexCoord2f( 0.0, 1.0 );
        glVertex3f( 0.0, 1.0, 0.0 );

        glEnd();
    }

    if( background_img_yz != NULL ) {
        glp_set_current_tex( background_img_yz );
        glBegin( GL_QUADS );

        glTexCoord2f( 0.0, 0.0 );
        glVertex3f( 0.0, 0.0, 0.0 );

        glTexCoord2f( 1.0, 0.0 );
        glVertex3f( 0.0, 1.0, 0.0 );

        glTexCoord2f( 1.0, 1.0 );
        glVertex3f( 0.0, 1.0, 1.0 );

        glTexCoord2f( 0.0, 1.0 );
        glVertex3f( 0.0, 0.0, 1.0 );

        glEnd();
    }

    if( background_img_xz != NULL ) {
        glp_set_current_tex( background_img_xz );
        glBegin( GL_QUADS );

        glTexCoord2f( 0.0, 0.0 );
        glVertex3f( 1.0, 0.0, 0.0 );

        glTexCoord2f( 1.0, 0.0 );
        glVertex3f( 0.0, 0.0, 0.0 );

        glTexCoord2f( 1.0, 1.0 );
        glVertex3f( 0.0, 0.0, 1.0 );

        glTexCoord2f( 0.0, 1.0 );
        glVertex3f( 1.0, 0.0, 1.0 );

        glEnd();
    }

    glDisable( GL_TEXTURE_2D );
    glPopMatrix();
}




void glp_draw_vertices( model_view *view, GSList* vertices ) {

	Vertex * v;
	int just_rendered_sel = FALSE;
	int i;
	int is_selecting;
	int is_reversed = FALSE;
	float zero[3] = {0.,0.,0.};
	static float multiplier_table[4][4][3] = {
		/* VIEW_XY */{ {-1., -1.,  0.}, {+1., -1.,  0.}, {+1., +1.,  0.}, {-1., +1.,  0.} },
		/* VIEW_YZ */{ { 0., -1., -1.}, { 0., +1., -1.}, { 0., +1., +1.}, { 0., -1., +1.} },
		/* VIEW_XZ */{ {+1.,  0., +1.}, {-1.,  0., +1.}, {-1.,  0., -1.}, {+1.,  0., -1.} },
		/* VIEW_3D */{ {-1., -1.,  0.}, {+1., -1.,  0.}, {+1., +1.,  0.}, {-1., +1.,  0.} }
	};
	static float multiplier[4][3];
	static int last_view = -1;
	
	if( view_config.triangle_order == MP_TRI_ORDER_CW )
		is_reversed = !is_reversed;
	if( view->flipped )
		is_reversed = !is_reversed;
	
	if( last_view != view->type || view->type == VIEW_3D ) {
		for( i = 0; i < 4; i++ ) {
			vector_mul( multiplier[i], multiplier_table[view->type][i], view_config.vertex_rad );
		}
		last_view = view->type;
	}

	if( view->type == VIEW_3D ) {
		if( view_config.draw_verts_as_cubes )
			return glp_draw_vertices_as_cubes( view, vertices );
		
		/* rotate quads to face camera */
		for( i = 0; i < 4; i++ ) {
			vector_rotate_about_point( multiplier[i], multiplier[i], 0, zero, view->pitch * DEGREES_TO_RADIANS );
			vector_rotate_about_point( multiplier[i], multiplier[i], 2, zero, -1. * view->heading * DEGREES_TO_RADIANS );
		}
	}
	
	is_selecting = (view_config.render_mode == MP_MODE_SELECTION &&
					global_scope == SCOPE_VERTEX);
	
	glColor3fv( view_config.vertex_unsel_color );
	glBegin( GL_QUADS );

	for( ; vertices != NULL; vertices = vertices->next ) {
		v = (Vertex*)vertices->data;

		if( v->hidden )
			continue;

		if( is_selecting ) {
			/* if we are selecting verts, the quads must be rendered in 
			individual glBegin-glEnd pairs */
			glEnd();
			glLoadName( (int)v );
			glBegin( GL_QUADS );
		}

		if( v->selected && !just_rendered_sel ) {
			glColor3fv( view_config.vertex_sel_color );
			just_rendered_sel = TRUE;
		} else if( !v->selected && just_rendered_sel ) {
			glColor3fv( view_config.vertex_unsel_color );
			just_rendered_sel = FALSE;
		}

		if( !is_reversed ) {
		glVertex3f( v->v[ 0 ] + multiplier[0][0], 
					v->v[ 1 ] + multiplier[0][1], 
					v->v[ 2 ] + multiplier[0][2] );
		glVertex3f( v->v[ 0 ] + multiplier[1][0], 
					v->v[ 1 ] + multiplier[1][1], 
					v->v[ 2 ] + multiplier[1][2] );
		glVertex3f( v->v[ 0 ] + multiplier[2][0], 
					v->v[ 1 ] + multiplier[2][1], 
					v->v[ 2 ] + multiplier[2][2] );
		glVertex3f( v->v[ 0 ] + multiplier[3][0], 
					v->v[ 1 ] + multiplier[3][1], 
					v->v[ 2 ] + multiplier[3][2] );
		} else {
		glVertex3f( v->v[ 0 ] + multiplier[3][0], 
					v->v[ 1 ] + multiplier[3][1], 
					v->v[ 2 ] + multiplier[3][2] );
		glVertex3f( v->v[ 0 ] + multiplier[2][0], 
					v->v[ 1 ] + multiplier[2][1], 
					v->v[ 2 ] + multiplier[2][2] );
		glVertex3f( v->v[ 0 ] + multiplier[1][0], 
					v->v[ 1 ] + multiplier[1][1], 
					v->v[ 2 ] + multiplier[1][2] );
		glVertex3f( v->v[ 0 ] + multiplier[0][0], 
					v->v[ 1 ] + multiplier[0][1], 
					v->v[ 2 ] + multiplier[0][2] );
		}
	}

	glEnd();

}


void glp_draw_vertices_as_cubes( model_view *view, GSList* vertices ) {
	/* a slower method of drawing verts.  used only if the user asks for it */

    Vertex * v;
    glInterleavedArrays( GL_N3F_V3F, 0, vertex_glarray );

    for( ; vertices != NULL; vertices = vertices->next ) {
        v = (Vertex*)vertices->data;

        if( v->hidden )
            continue;

        if( view_config.render_mode == MP_MODE_SELECTION &&
                global_scope == SCOPE_VERTEX )
            glLoadName( (int)v );

        if ( v->selected )
            glColor3fv( view_config.vertex_sel_color );
        else
            glColor3fv( view_config.vertex_unsel_color );

        glPushMatrix();
        glTranslatef( v->v[ 0 ], v->v[ 1 ], v->v[ 2 ] );
        glDrawArrays( GL_QUADS, 0, 24 );
        glPopMatrix();
    }

}


void glp_draw_girder( model_view * view, Vertex * p1, Vertex * p2 ) {

    GLfloat length, v1[ 3 ], n[ 2 ], sin_theta, theta;

    if ( p1->v[ 2 ] > p2->v[ 2 ] ) {	    	/* +/- issue with arcsin */
        Vertex *temp = p2;
        p2 = p1;
        p1 = temp;
    }					/* what a pain */

    v1[ 0 ] = p2->v[ 0 ] - p1->v[ 0 ];
    v1[ 1 ] = p2->v[ 1 ] - p1->v[ 1 ];
    v1[ 2 ] = p2->v[ 2 ] - p1->v[ 2 ];

    sin_theta = ( v1[ 0 ] * v1[ 0 ] ) + ( v1[ 1 ] * v1[ 1 ] );
    length = sqrt( sin_theta + ( v1[ 2 ] * v1[ 2 ] ) );
    sin_theta = sqrt( sin_theta );
    if ( sin_theta == 0 ) {
        n[ 0 ] = 1; 		/* doesn't matter */
        n[ 1 ] = 0;
    } else {
        n[ 0 ] = -v1[ 1 ] / sin_theta;
        n[ 1 ] = v1[ 0 ] / sin_theta;
    }
    sin_theta = sin_theta / length;
    theta = asin( sin_theta ) * ( ( GLfloat ) 180 / M_PI );
#ifdef VERBOSE
    printf( "v = <%f, %f, %f> ", v1[ 0 ], v1[ 1 ], v1[ 2 ] );
    printf( "n = <%f, %f, 0> theta = %f\n", n[ 0 ], n[ 1 ], theta );
#endif

    /* Set the remap matrix appropriately */
    glPushMatrix();
    glTranslated( p1->v[ 0 ], p1->v[ 1 ], p1->v[ 2 ] );
    glPushMatrix();
    glRotatef( theta, n[ 0 ], n[ 1 ], 0 );

    girder_glarray[ 17 ] = length;                /* Patch the array */
    girder_glarray[ 23 ] = length;
    girder_glarray[ 41 ] = length;
    girder_glarray[ 47 ] = length;
    girder_glarray[ 65 ] = length;
    girder_glarray[ 71 ] = length;
    girder_glarray[ 89 ] = length;
    girder_glarray[ 95 ] = length;

    glColor3fv( view_config.poly_sel_color );
    glInterleavedArrays( GL_N3F_V3F, 0, girder_glarray );
    glDrawArrays( GL_QUADS, 0, 16 );

    glPopMatrix();
    glPopMatrix();

}




gboolean glp_draw_surfaces_tree( GNode *node, gpointer data ) {
    // recurse through tree

    //FIXME - ac3d format states that translations are cumulative, which
    //        means that I'm going to have to write my own recursive function
    //        (instead of glib's) and push/pop the matrices at the right times
    // update - this in no longer the case... the translations are now flattened
    //          immed. after the model is loaded.  see model_flatten()

    Mesh* m = (Mesh*)node->data;

    glp_draw_mesh( (model_view*)data, m, 0 );

    return FALSE;  /* we want it to continue... */
}


void glp_draw_mesh( model_view *view, Mesh * m, gint box_it ) {

    static GLfloat initializer[ 4 ] = { 0.0, 0.0, 0.0, 1.0 };
    static float default_diff[4] = { 0.8, 0.8, 0.8, 1.0 };
    static float default_amb[4] = { 0.2, 0.2, 0.2, 1.0 };
    static float default_spec[4] = { 0.0, 0.0, 0.0, 1.0 };
    static float default_emis[4] = { 0.0, 0.0, 0.0, 1.0 };
    static float default_shininess = 0.0;

    if( m == NULL )
        return;
	
	if( m->hidden )
		return;

    glPushMatrix();
/*    if( m->location != NULL )
        glTranslatef( m->location->v[0], m->location->v[1], m->location->v[2] );*/


        if( view_config.render_mode == MP_MODE_SELECTION ) {

            /* we are in MP_MODE_SELECTION mode, hence we need only to render in
               flat-shading, with no textures, etc.
            */

            glDisable( GL_LIGHTING );         /* we don't need these things for picking */
            glDisable( GL_COLOR_MATERIAL );
            glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );

            if( global_scope == SCOPE_MESH ) {
                /* if we're selecting meshes... */
                glLoadName( (int)m );
                glp_draw_primitives( view, m->polygons );
            } else if( global_scope == SCOPE_POLYGON ) {
                /* you can't do a glLoadName from w/in a glBegin()/glEnd() block,
                   hence we need to render each triangle sep. in this situation
                */
                GSList * polys = m->polygons;
                while( polys != NULL ) {
                    Poly * t = (Poly *)polys->data;
                    glLoadName( (int)t );
                    glp_draw_primitive( view, t );
                    polys = polys->next;
                }

            } else if( global_scope == SCOPE_VERTEX ) {
                //fixme - should we allow vertices to be selected even if they aren't visible?
                if( view_config.vertices_visible )
                    glp_draw_vertices( view, m->vertices );
            }

        } else if( view_config.render_mode == MP_MODE_RENDER ) {

			switch ( view_config.polygon_mode ) {

            case MP_SHOW_TEXTURED:
                glEnable( GL_LIGHTING );
                glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );

                if( m->texture != NULL ) {
                    glp_set_current_tex( m->texture );
                    glp_texenv();
                    glp_draw_primitives_textured( view, m->polygons, m->texrep_x, m->texrep_y );
                    break;
                }
                /* if this mesh has no texture, fall-through into the wireframe
                   code below...*/

            case MP_SHOW_WIREFRAME:
                /* wire frame mode is not shaded, and isn't depth-tested */
                glDisable( GL_LIGHTING );
                glDisable( GL_DEPTH_TEST );
                glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );

                glp_draw_primitives( view, m->polygons );

                glEnable( GL_DEPTH_TEST );
                break;

            case MP_SHOW_FLAT:
                glEnable( GL_LIGHTING );
                glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );
                glEnable( GL_COLOR_MATERIAL );
                glMaterialfv( GL_FRONT, GL_DIFFUSE, default_diff );
                glMaterialfv( GL_FRONT, GL_AMBIENT, default_amb );
                glMaterialfv( GL_FRONT, GL_SPECULAR, default_spec );
                glMaterialfv( GL_FRONT, GL_EMISSION, default_emis );
                glMaterialf( GL_FRONT, GL_SHININESS, default_shininess );
                glColorMaterial( GL_FRONT, GL_AMBIENT_AND_DIFFUSE );

                glDepthRange (0.001, 1.0);  // part of the overlay stuff

                glp_draw_primitives( view, m->polygons );

                // a test... to draw a wireframe overlay

                /* similar in func to the polygon-offset extension */
                glDepthRange (0.0, 0.999);
                glDisable( GL_LIGHTING );
                glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );
                glp_draw_primitives( view, m->polygons );
                glDepthRange (0.0, 1.0);

                // end of test

                break;


            default:
                break;
            }


            glDisable( GL_LIGHTING );
            glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );

            /* necessary for drawing vertices, etc */
            glEnable( GL_COLOR_MATERIAL );
            glColorMaterial( GL_FRONT, GL_AMBIENT_AND_DIFFUSE );
            glMaterialfv( GL_FRONT, GL_SPECULAR, initializer );
            glMaterialfv( GL_FRONT, GL_EMISSION, initializer );
            glMaterialf( GL_FRONT, GL_SHININESS, 0.0 );

            if( view_config.vertices_visible )
                glp_draw_vertices( view, m->vertices );

			if( view_config.normals_visible )
				glp_draw_normals( view, m->polygons );

        } /* end of if( rendermode == ... ) */

    glPopMatrix();

}



void glp_draw_grid( GLfloat *color, GLfloat *mins, GLfloat *maxes, GLfloat spacing ) {

    GLfloat coord;
    GLfloat multiple;

    glColor3fv( color );
    glBegin( GL_LINES );

    multiple = mins[0] / spacing;
    coord = spacing * rint( multiple );
    for ( ; coord <= maxes[ 0 ]; coord += spacing ) {
        glVertex3f( coord, 0, mins[ 2 ] );
        glVertex3f( coord, 0, maxes[ 2 ] );
        glVertex3f( coord, mins[ 1 ], 0 );
        glVertex3f( coord, maxes[ 1 ], 0 );
    }

    multiple = mins[1] / spacing;
    coord = spacing * rint( multiple );
    for ( ; coord <= maxes[ 1 ]; coord += spacing ) {
        glVertex3f( 0, coord, mins[ 2 ] );
        glVertex3f( 0, coord, maxes[ 2 ] );
        glVertex3f( mins[ 0 ], coord, 0 );
        glVertex3f( maxes[ 0 ], coord, 0 );
    }

    multiple = mins[2] / spacing;
    coord = spacing * rint( multiple );
    for ( ; coord <= maxes[ 2 ]; coord += spacing ) {
        glVertex3f( 0, mins[ 1 ], coord );
        glVertex3f( 0, maxes[ 1 ], coord );
        glVertex3f( mins[ 0 ], 0, coord );
        glVertex3f( maxes[ 0 ], 0, coord );
    }

    glEnd();

}


void glp_build_font( void ) {

	PangoFontDescription *font_desc = NULL;

	/* find a font */
/* hopefully, we no longer need Windows-specific code here...
#ifdef G_OS_WIN32
    font = gdk_font_load( "-*-MS Sans Serif-medium-r-normal-*-*-80-*-*-p-*-iso8859-1" );
#else
	font = gdk_font_load( "-*-fixed-*-*-*-*-10-*" );
#endif
*/
	font_desc = pango_font_description_new ();
	pango_font_description_set_family (font_desc, "sans");
	pango_font_description_set_style (font_desc, PANGO_STYLE_NORMAL);
	pango_font_description_set_variant (font_desc, PANGO_VARIANT_NORMAL);
	pango_font_description_set_weight (font_desc, PANGO_WEIGHT_NORMAL);
	pango_font_description_set_stretch (font_desc, PANGO_STRETCH_NORMAL);
	pango_font_description_set_size (font_desc, 10 );

	/* generate font display list */
	glp_fontbase = glGenLists( 128 );
	gdk_gl_font_use_pango_font( font_desc, 0, 128, glp_fontbase );

	pango_font_description_free( font_desc );
}


/*
 * Display a labelled set of axes at the origin
 */
void glp_draw_origin( model_view *view ) {

    gchar *message;

	/* Draw an origin marker */
    glBegin( GL_LINES );

    glColor3f( 1.0, 1.0, 1.0 );

    /* fixme! -> hardcoded size */
    glVertex3f( 0.0, 0.0, 0.0);
    glVertex3f( 4.0, 0.0, 0.0);

    glVertex3f( 0.0, 0.0, 0.0);
    glVertex3f( 0.0, 4.0, 0.0);

    glVertex3f( 0.0, 0.0, 0.0);
    glVertex3f( 0.0, 0.0, 4.0);

    glEnd();
	
    /* Font rendering code taken from hitchhiker 2000, dp */
    if( ! glp_fontbase ) {
        glp_build_font();
	}

	message = "x";
	glRasterPos3f( 4.2, 0.0, 0.0 );
	glListBase(glp_fontbase);
	glCallLists(strlen(message), GL_UNSIGNED_BYTE, message);
	
	message = "y";
	glRasterPos3f( 0.0, 4.2, 0.0 );
	glListBase(glp_fontbase);
	glCallLists(strlen(message), GL_UNSIGNED_BYTE, message);
	
	message = "z";
	glRasterPos3f( 0.0, 0.0, 4.2 );
	glListBase(glp_fontbase);
	glCallLists(strlen(message), GL_UNSIGNED_BYTE, message);

}



void glp_render( model_view *view ) {

    if ( view == NULL )
        return;

    // HACK
    // it's assumed that, if we're not in render mode, the
    // opengl context has already been set.  I have no idea why
    // setting the context twice causes crashes; my guess is that
    // programs aren't allowed to change contexts while in GL_SELECT mode.

    if( view_config.render_mode == MP_MODE_RENDER )
        if( !gdk_gl_drawable_gl_begin( view->gldrawable, view->glcontext ) )
            return;

    glDisable( GL_LIGHTING );
    glColor3f( 1.0, 1.0, 1.0 );

    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );


    glp_setup_camera( view );

    if( view_config.render_mode == MP_MODE_RENDER ) {

        if( view_config.grid_visible ) {
            glp_draw_grid( view_config.grid_color, view->grid_mins,
                           view->grid_maxes, view_config.grid_spacing );
        }

		/* this ensures that the origin will be drawn in front of the grid */
		glDisable( GL_DEPTH_TEST );
        if( view_config.origin_visible) {
            glp_draw_origin( view );
        }

        glDisable( GL_DEPTH_TEST );

        glp_draw_background( view );

        glEnable( GL_DEPTH_TEST );

		/* render the partial polygon, if there is one */
		if( the_model != NULL && the_model->partially_completed_poly != NULL )
			glp_draw_primitive_edit( view, the_model->partially_completed_poly );

    }


    /* render the surfaces for all meshes */
    if( the_model != NULL && the_model->root != NULL )
        g_node_traverse( the_model->root, G_IN_ORDER, G_TRAVERSE_ALL, -1,
                         glp_draw_surfaces_tree, view );

    glp_cleanup_camera( view );


    if( view_config.render_mode == MP_MODE_RENDER ) {
	    glp_swap_buffers( view->gldrawable );

		/* the following line must only be executed if we're in 
		   rendering mode... see comment above, at top of this func */
		gdk_gl_drawable_gl_end( view->gldrawable );
	}

    view->dirty = FALSE;
}


void glp_expose( model_view * view ) {

    if ( view == NULL )
        return;
/* for now...
    if( view_config.render_mode == MP_MODE_RENDER ) {

        if( !gtk_gl_area_make_current( GTK_GL_AREA( view->glarea ) ) )
            return;
        gtk_gl_area_swapbuffers( GTK_GL_AREA( view->glarea ) );
    }
*/
}


void glp_swap_buffers( GdkGLDrawable *gldrawable ) {
	/* Swap buffers */
	if( gdk_gl_drawable_is_double_buffered(gldrawable) )
		gdk_gl_drawable_swap_buffers(gldrawable);
	else
		glFlush();
}


void glp_rebuild_display_lists( model_view *view ) {
	
	g_return_if_fail( 
		gdk_gl_drawable_gl_begin( view->gldrawable, view->glcontext ) );
	
	if( ! glp_displist_vertex ) {
		glp_displist_vertex = glGenLists( 1 );
	}
	glNewList( glp_displist_vertex, GL_COMPILE );
		
	glEndList();
	
	gdk_gl_drawable_gl_end( view->gldrawable );
}


void glp_set_triangleorder( model_view *view ) {

	g_return_if_fail( 
		gdk_gl_drawable_gl_begin( view->gldrawable, view->glcontext ) );

	switch( view_config.triangle_order ) {

	case MP_TRI_ORDER_CCW:
		glEnable( GL_CULL_FACE );
		glFrontFace( GL_CCW );
		break;

	case MP_TRI_ORDER_CW:
		glEnable( GL_CULL_FACE );
		glFrontFace( GL_CW );
		break;

	case MP_TRI_ORDER_ALL:
		glDisable( GL_CULL_FACE );
		break;

	default:
		break;
	}

	gdk_gl_drawable_gl_end( view->gldrawable );

}


/////////////////////////////////////////////////////////////////////////
//
//
//  funcs for selection of primitives
//
//
//////////////////////////////////////////////////////////////////////////



int glp_unproject( model_view *view,
                   int winx, int winy,
                   GLfloat* objx, GLfloat* objy, GLfloat* objz ) {

    GLint viewport[4];
    GLdouble mvmatrix[16], projmatrix[16];
    GLdouble winz = 0.5;
    GLdouble objd[3];
    int success = FALSE;

    g_return_val_if_fail( gdk_gl_drawable_gl_begin( view->gldrawable, view->glcontext ), FALSE );

    glp_setup_camera( view );

    glGetIntegerv (GL_VIEWPORT, viewport);
    glGetDoublev  (GL_MODELVIEW_MATRIX, mvmatrix);
    glGetDoublev  (GL_PROJECTION_MATRIX, projmatrix);

    success = gluUnProject ( (GLfloat) (winx),
                             (GLfloat) (winy),
                             (GLfloat) (winz),
                             mvmatrix, projmatrix, viewport,
                             &objd[0], &objd[1], &objd[2] );

    *objx = objd[0];
    *objy = objd[1];
    *objz = objd[2];

    glp_cleanup_camera( view );

	gdk_gl_drawable_gl_end( view->gldrawable );

    return success;
}




GSList *glp_test_click_objects( model_view *view,
                                GLfloat *z, guint mouse_x, guint mouse_y,
                                guint width, guint height, int closest ) {

    /*
       This is the picking function... a real workhorse, in that it is used for selection/picking
       of all types of objects, including vertices, polygons, and meshes.  The list that is 
       returned contains pointers to the objects that were under the cursor (or within the 
       selection box).  The type of the objects is determined by the global_scope variable.  The 
       list's contents will need to be cast to the appropriate type later.
       
       mouse_x and y are the screen coordinates, and width and height are the size of the 
       selection box (set both to 1 if you only want stuff immed. under the cursor)
       
       if "closest" is true, then the returned list will contain only one object...
       the closest one
    */

    /* Many thanks to nehe.gamedev.net ! */


    // future improvements - could have an option (like in milkshape) where user
    // can choose, when picking objects after a singleclick, if user wants
    // ALL objects along that ray, or just the closest


    int choose, depth, loop;
    GSList *objs = NULL;
    GLuint *buffer;		// a selection buffer
	int buffer_len;
    GLint	hits;			// The Number Of Objects That We Selected

    // stores dimensions Of the viewport.
    // [0] Is <x>, [1] Is <y>, [2] Is <length>, [3] Is <width>
    GLint	viewport[4];


    g_return_val_if_fail ( view != NULL, NULL );
    g_return_val_if_fail ( view->glarea != NULL, NULL );

    g_return_val_if_fail( gdk_gl_drawable_gl_begin( view->gldrawable, view->glcontext ), NULL );

    glGetIntegerv( GL_VIEWPORT, viewport );

    view_config.render_mode = MP_MODE_SELECTION;

	/* Set up the selection buffer.  Each entry in the buffer requires
	4 32-bit ints.  We would like to be able to *potentially* select every 
	primitive in the model, within reason.  If the model has an extremely 
	large number of primitives, then the user is just SOL, and will only be 
	able to select a subset of the model at a time. */
	/* the number of polys typically can't exceed the number of verts, so 
	this is a good (albeit rough) guess as to how many primitives (of either 
	type) are in the model. */
	buffer_len = the_model->num_verts + 1;
	if( buffer_len < MIN_SELECTION_BUFFER_LEN ) 
		buffer_len = MIN_SELECTION_BUFFER_LEN;
	if( buffer_len > MAX_SELECTION_BUFFER_LEN ) 
		buffer_len = MAX_SELECTION_BUFFER_LEN;
	buffer_len *= 4;
	buffer = (GLuint*)malloc( buffer_len * sizeof( GLuint ) );

	// Tell OpenGL To Use Our Array For Selection
    glSelectBuffer( buffer_len, buffer );

    // Puts OpenGL In Selection Mode. Nothing Will Be Drawn.
    // Object ID's and Extents Are Stored In The Buffer.
    (void) glRenderMode( GL_SELECT );

    glInitNames();                    // Initializes The Name Stack
    glPushName( 0 );                  // Push 0 (At Least One Entry) Onto The Stack

    if( width < 1 ) width = 1;
    if( height < 1 ) height = 1;

    // set up clip frustum //////////////////////////////
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    // This creates a matrix that will zoom up to a small portion of the screen, where the mouse is.
    gluPickMatrix( (GLfloat) mouse_x, (GLfloat) (viewport[3]-mouse_y),
                   (float)width, (float)height, viewport );
    glp_set_viewvolume( view );
    glMatrixMode(GL_MODELVIEW);


    // render ///////////////////////////////////////////
    glp_render( view );


    // restore previous view volume /////////////////////
    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);


    hits = glRenderMode( GL_RENDER );     // Switch To Render Mode, Find Out How Many
    view_config.render_mode = MP_MODE_RENDER;

    bb_push_message_f( 1.5, "Objects selected: %i", hits );
#ifdef VERBOSE
    printf( "glp_tcv: hits is %i\n", hits );
    for( loop = 0; loop < hits; loop++ )
        printf( "         hit vert num %i \n", buffer[loop*4+3] );
#endif

    if( closest ) {
        // if we want only a single object...

        choose = 0;

        if( hits > 0 ) {
            choose = buffer[3];           // Make Our Selection The First Object
            depth = buffer[1];            // Store How Far Away It Is

            for( loop = 1; loop < hits; loop++ ) {
                // If This Object Is Closer To Us Than The One We Have Selected
                if (buffer[loop*4+1] < (GLuint)(depth)) {
                    choose = buffer[loop*4+3];               // Select The Closer Object
                    depth = buffer[loop*4+1];                // Store How Far Away It Is
                }

            }

        }

        if( choose != 0 )
            // if we found an object
            // (choose will have to be cast to the appropriate type later...)
            objs = g_slist_append( objs, (gpointer)(choose) );

    } else {
        // if we want *all* the vertices that were hit...

        if( hits > 0 ) {

            for( loop = 0; loop < hits; loop++ ) {
                objs = g_slist_append( objs, (gpointer)(buffer[loop*4+3]) );
            }

        }


    }

	gdk_gl_drawable_gl_end( view->gldrawable );

	free( buffer );

    return objs;
}




Vertex *glp_test_click_vertex( model_view *view, GLfloat *z,
                               guint mouse_x, guint mouse_y ) {

    GSList *l = NULL;
    Vertex *p = NULL;
    int original_scope;

    /* a hack... needed for add-polygon */
    original_scope = global_scope;
    global_scope = SCOPE_VERTEX;

    l = glp_test_click_objects( view, z, mouse_x, mouse_y, 1, 1, TRUE );

    global_scope = original_scope;

    if( l != NULL && l->data != NULL )
        p = (Vertex*)l->data;

    g_slist_free( l );

    return p;
}


GSList *glp_test_click_vertices( model_view *view, GLfloat *z,
                                 guint x, guint y,
                                 guint width, guint height ) {

    GSList *l = NULL;
    int original_scope;

    /* a hack... just to make sure */
    original_scope = global_scope;
    global_scope = SCOPE_VERTEX;

    l = glp_test_click_objects( view, z, x, y, width, height, FALSE );

    global_scope = original_scope;

    return l;
}


Poly *glp_test_click_polygon( model_view *view, GLfloat *z,
                              guint xc, guint yc ) {

    GSList *l = NULL;
    Poly *p = NULL;
    int original_scope;

    /* a hack... just to make sure */
    original_scope = global_scope;
    global_scope = SCOPE_POLYGON;

    l = glp_test_click_objects( view, z, xc, yc, 1, 1, TRUE );

    global_scope = original_scope;

    if( l != NULL && l->data != NULL )
        p = (Poly *)l->data;

    g_slist_free( l );

    return p;

}


GSList *glp_test_click_polygons( model_view *view, GLfloat *z,
                                 guint x, guint y,
                                 guint width, guint height ) {

    GSList *l = NULL;
    int original_scope;

    /* a hack... just to make sure */
    original_scope = global_scope;
    global_scope = SCOPE_POLYGON;

    l = glp_test_click_objects( view, z, x, y, width, height, FALSE );

    global_scope = original_scope;

    return l;
}



///////////////////////////////////////////////////////////////////////////////
//
//
// ONLY DEAD CODE BELOW THIS POINT
//
//
///////////////////////////////////////////////////////////////////////////////




#if 0

#define glp_ARRAY_SIZE 2048
Poly *glp_test_click_surface( model_view *view,
                              GLfloat *z, guint xc, guint yc ) {

    GLfloat best_z = 9999999.0, bz_vert, m, b, x, y;
    GLfloat array[ glp_ARRAY_SIZE ], *v1, *v2;
    Poly *best_tri = NULL;
    GSList *polygons = NULL;
    int i, j, len, polylen, intersections;

    x = xc;
    y = yc;

    g_return_val_if_fail ( view != NULL, NULL );
    g_return_val_if_fail ( view->glarea != NULL, NULL );
    g_return_val_if_fail ( current_mesh != NULL, NULL );

    g_return_val_if_fail( gtk_gl_area_make_current( GTK_GL_AREA( view->glarea ) ), NULL );
    glFeedbackBuffer( glp_ARRAY_SIZE, GL_3D, array );
    glRenderMode( GL_FEEDBACK );

    // need to flip y axis; X11 origin is upperleft, opengl origin is lowerleft
    y = view->glarea->allocation.height - y;

    glp_setup_camera( view );

    glPushMatrix();
    if( current_mesh->location != NULL )
        glTranslatef( current_mesh->location->v[0],
                      current_mesh->location->v[1],
                      current_mesh->location->v[2] );

    polygons = current_mesh->polygons;

    while ( polygons != NULL ) {
        Poly * t = (Poly *)polygons->data;
        glp_draw_primitive( view, t );

        /* now process the feedback buffer and check for hits */
        len = glRenderMode( GL_FEEDBACK );
        intersections = 0;
        for ( i = 0; i < len; ) {
            g_return_val_if_fail( array[ i ] == GL_POLYGON_TOKEN, NULL );

            i++;
            polylen = ( int ) array[ i ];
            i++;

            bz_vert = 9999999.0;
            v1 = &( array[ 3 * ( polylen - 1 ) + i ] );
            for ( j = 0; j < polylen; j++ ) {
                v2 = v1;
                v1 = &( array[ i ] );
                i += 3;

                if ( array[ 2 ] < bz_vert )
                    bz_vert = array[ 2 ];

                /* try to eliminate the line represented by v1 and v2 */

                /* in order to intersect the test line, the v1<->v2 line */
                /* must have a vertex to both sides of the click vertex */
                if ( !( ( v1[ 0 ] < x ) && ( v2[ 0 ] > x ) ) &&
                        !( ( v2[ 0 ] < x ) && ( v1[ 0 ] > x ) ) )
                    continue;
                /* note that it's now impossible for v1[0] == v2[0] */

                /* the line can be immediately qualified if both are above */
                /* the click vertex */
                if ( ( v1[ 1 ] < y ) && ( v2[ 1 ] < y ) ) {
                    intersections++;
                    continue;
                }

                /* if both vertices are below the click point, the line */
                /* can't intersect */
                if ( ( v1[ 1 ] > y ) && ( v2[ 1 ] > y ) )
                    continue;

                /* now the situation is sticky, the line formula must be used */
                /* to test for intersection */

                m = ( v2[ 1 ] - v1[ 1 ] ) / ( v2[ 0 ] - v1[ 0 ] );
                b = v1[ 1 ] - ( m * v1[ 0 ] );

                if ( y < ( ( m * x ) + b ) )
                    continue;

                intersections++;
            }

            intersections &= 0x1;	/* an odd number of intersections means */
            /* it was clicked on */
            if ( intersections && ( bz_vert < best_z ) ) {
                best_tri = t;
                best_z = bz_vert;
            }
        }

        polygons = polygons->next;
    }	/* polygons != NULL */


    glp_cleanup_camera( view );

    glPopMatrix();
    glRenderMode( GL_RENDER );
    *z = best_z;
    return best_tri;
}

void glp_ortho_viewport( model_view *view ) {


    // sets the extents for the ortho graphic views...

    float halfdist;
    float aspect_ratio = 1.0;


    halfdist = (view->dist / 2.0);

    view->minX = - halfdist;
    view->maxX = halfdist;

    view->minY = - halfdist;
    view->maxY = halfdist;

    view->minZ = - halfdist;
    view->maxZ = halfdist;

    if( view->glarea != NULL ) {

        aspect_ratio = (float)(GTK_WIDGET( view->glarea )->allocation.width) /
                       (float)(GTK_WIDGET( view->glarea )->allocation.height);

        switch( view->type ) {
        case VIEW_XY:
            view->minX = view->minY * aspect_ratio;
            view->maxX = view->maxY * aspect_ratio;
            break;
        case VIEW_XZ:
            view->minX = view->minZ * aspect_ratio;
            view->maxX = view->maxZ * aspect_ratio;
            break;
        case VIEW_YZ:
            view->minY = view->minZ * aspect_ratio;
            view->maxY = view->maxZ * aspect_ratio;
            break;
        default:
            break;
        }

    }

}



#endif



