/* 
 * File:         group_browser.c
 * 
 * Description:  creates and manages the "group" notebook tab, also handles
 * 		 group-related functionality
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 
/* Group browser code */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "config.h"
#include "group_browser.h"
#include "group_btns.h"
#include "scripting_entry.h"
#include "globals.h"
#include "model.h"
#include "group.h"
#include "mesh.h"
#include "vertex.h"
#include "polygon.h"
#include "win32stuff.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif



/*#define VERBOSE 1 */
#undef VERBOSE


/*
funcs that would be useful:
gb_clear_subtree (by groupBtns_action_button_cb)
GtkTreeIter* gb_search( GNode *group ) (by everyone)
*/

/* ENUMS AND STRUCTS ******************************************************/

enum GroupBrowserColumns {
	GB_COLUMN_GROUPNAME = 0,
	GB_COLUMN_GROUPPTR,
	GB_COLUMN_GROUPVISIBLE,
	GB_NUM_COLUMNS
};


struct groupBrowser_struct {

	GtkTreeStore *store;	/* mirrors the model's tree data */
	GtkWidget *treeView;	/* displays the contents of the above treestore */

    GNode *selected_grp;
}
groupBrowser[] = { { NULL } };


/* FILE-SCOPE VARS **********************************************************/

GNode *nodeToSearchFor = NULL;  /* used by the model-search funcs */


/* PROTOTYPES ***************************************************************/

void gb_clear( void ) ;
void gb_build( GNode *group, gpointer parentIter ) ;
void gb_set_selected_and_expand( void ) ;
GtkTreeIter *gb_search_model( GNode *group ) ;
gboolean gb_search_model_foreach( 
	GtkTreeModel *model,
	GtkTreePath *path,
	GtkTreeIter *iter,
	gpointer data ) ;

void gb_selection_changed_cb( GtkTreeSelection *selection, gpointer data ) ;
void gb_cell_toggled_cb(
		GtkCellRendererToggle *cell,
		gchar *path_str,
		gpointer data);
void gb_drag_data_get_cb( 
	GtkWidget *widget,
	GdkDragContext *drag_context,
	GtkSelectionData *data,
	guint info,
	guint time,
	gpointer user_data );
void gb_drag_data_recvd_cb( 
	GtkWidget *widget,
	GdkDragContext *drag_context,
	gint x,
	gint y,
	GtkSelectionData *data,
	guint info,
	guint time,
	gpointer user_data );



/* FUNCS ********************************************************************/

GtkWidget * create_group_browser( void ) {

    GtkWidget *vbox;
	GtkWidget *scrollwindow;

    vbox = gtk_vbox_new( FALSE, 4 );
    gtk_container_set_border_width( GTK_CONTAINER( vbox ), 4 );


	/* set up tree data */
	groupBrowser->store = gtk_tree_store_new( GB_NUM_COLUMNS,
											G_TYPE_STRING,
											G_TYPE_POINTER,
											G_TYPE_BOOLEAN );
	
	/* set up tree viewer */
	groupBrowser->treeView = 
		gtk_tree_view_new_with_model( GTK_TREE_MODEL( groupBrowser->store ) );

	scrollwindow = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW(scrollwindow),
									GTK_POLICY_AUTOMATIC,
									GTK_POLICY_AUTOMATIC );
	gtk_container_add( GTK_CONTAINER(scrollwindow), groupBrowser->treeView );
	gtk_box_pack_start( GTK_BOX( vbox ), scrollwindow,
						TRUE, TRUE, 0 );

	/* set up the cell renderer(s) for our tree viewer */
	{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes( "Group/Mesh name",
														renderer,
														"text", 
														GB_COLUMN_GROUPNAME,
														NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW(groupBrowser->treeView), column );


	renderer = gtk_cell_renderer_toggle_new();
	g_object_set_data( G_OBJECT(renderer), "column", (gint *)GB_COLUMN_GROUPVISIBLE );
	g_signal_connect( renderer, "toggled", G_CALLBACK(gb_cell_toggled_cb), 
					  GTK_TREE_MODEL( groupBrowser->store ) );
	column = gtk_tree_view_column_new_with_attributes( "Visible",
														renderer,
														"active", 
														GB_COLUMN_GROUPVISIBLE,
														NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW(groupBrowser->treeView), column );
	}
	
	/* set up the selection handler for our tree viewer */
	{
	GtkTreeSelection *select;

	select = gtk_tree_view_get_selection( GTK_TREE_VIEW(groupBrowser->treeView) );
	gtk_tree_selection_set_mode( select, GTK_SELECTION_SINGLE );
	g_signal_connect( G_OBJECT(select), "changed",
						G_CALLBACK(gb_selection_changed_cb),
						NULL );
	}
	
	/* set up drag-and-drop for our tree viewer */
	{
	static const GtkTargetEntry type_accept_list[] = {
		{ "kludge3d/model-group-ptr", 0, 0 }
	};

	gtk_drag_source_set( 
		groupBrowser->treeView, 
		GDK_BUTTON1_MASK, 
		type_accept_list, 1, 
		GDK_ACTION_COPY | GDK_ACTION_MOVE );
	gtk_drag_dest_set( 
		groupBrowser->treeView, 
		GTK_DEST_DEFAULT_ALL, 
		type_accept_list, 1, 
		GDK_ACTION_COPY | GDK_ACTION_MOVE );

	g_signal_connect( G_OBJECT( groupBrowser->treeView ), "drag-data-get",
						G_CALLBACK( gb_drag_data_get_cb ), NULL );
	g_signal_connect( G_OBJECT( groupBrowser->treeView ), "drag-data-received",
						G_CALLBACK( gb_drag_data_recvd_cb ), NULL );
	

	/* This is a work-around for a bug in gtk_tree_view_get_path_at_pos.
	gtk_tree_view_get_path_at_pos doesn't take into account the y offset 
	that the column headers add.  As a consequence, drag *drop* events will, 
	from the user's point of view, appear to occur over the wrong row (one row 
	higher than the user actually dropped onto). 

	By turning off the column headers, hopefully drop events will work 
	correctly.

	Note that there are some extra DND-related functions associated with 
	GtkTreeViews.  They aren't documented (of course... dammit), so I'm not 
	using them.  However, if using said functions avoids this problem, perhaps 
	I'll make some more changes to this DND stuff in order to make use of them.
	*/
	gtk_tree_view_set_headers_visible(
		GTK_TREE_VIEW( groupBrowser->treeView ), FALSE );
	}


    gtk_box_pack_start( GTK_BOX( vbox ), create_group_btns(), FALSE, FALSE, 0 );
    gtk_box_pack_start( GTK_BOX( vbox ), create_scripting_entry(), FALSE, FALSE, 0 );

	/* We'll set up model_set_currents here.  There's no really 
		appropriate place for this, so here is as good as anywhere else... 
		Note that model_set_currents must called before gb_rebuild, as 
		gb_rebuild uses the current_node to determine which row should 
		start out as the "selected" one. */
	g_signal_connect( notificationObj, "notify::model-structure-changed", 
		G_CALLBACK(model_set_currents), NULL );
	/* set up a signal handler so that gb_rebuild gets called when the model's 
		structure changes */
	g_signal_connect( notificationObj, "notify::model-structure-changed", 
		G_CALLBACK(gb_rebuild), NULL );

	gtk_widget_show_all( vbox );

    return vbox;
}


gboolean gb_rebuild( void ) {
	gb_clear();
	gb_build( the_model->root, NULL );
	gb_set_selected_and_expand();
	return FALSE;
}

void gb_clear( void ) {
	
	GtkTreeIter iter;

	/* get an iterator pointing to the top level */
	if( gtk_tree_model_get_iter_first( GTK_TREE_MODEL(groupBrowser->store), &iter ) ) {
		/* now remove everything at the top level (there should be only one 
		  item at the top level, the "World" group, but just to be safe...) */
		while( gtk_tree_store_remove( groupBrowser->store, &iter ) ) {
		}
	} else {
		/* tree store was empty... nothing to do */
	}

}

void gb_build( GNode *group, gpointer parentIter ) {

	GtkTreeIter groupIter;

	gtk_tree_store_append( groupBrowser->store, &groupIter, 
		(GtkTreeIter *)parentIter );
	gtk_tree_store_set( groupBrowser->store, &groupIter, 
		GB_COLUMN_GROUPNAME, ((Mesh*)group->data)->name, 
		GB_COLUMN_GROUPPTR, group, 
		GB_COLUMN_GROUPVISIBLE, (int)(!((Mesh*)group->data)->hidden), 
		-1 );

	/* recurse */
	g_node_children_foreach( group, G_TRAVERSE_ALL, gb_build, &groupIter );
}

void gb_set_selected_and_expand( void ) {
	GtkTreeIter *iter;
	
	iter = gb_search_model( the_model->current_node );
	
	if( iter ) {
		GtkTreePath *path = gtk_tree_model_get_path( 
			GTK_TREE_MODEL( groupBrowser->store ), iter );

#ifdef VERBOSE
		char * pathstring = gtk_tree_path_to_string( path );
		printf( "group %x was found at path %s \n", 
			(unsigned int )the_model->current_node, pathstring );
		g_free( pathstring );
#endif
		gtk_tree_view_expand_to_path( 
			GTK_TREE_VIEW( groupBrowser->treeView ), path );
		/* ...or expand_all?  I don't know which would be more appropriate. */
		
		/* gtk_tree_selection_select_iter doesn't set the selection correctly 
		unless the treeview is visible (at the top of a stack of notebook tabs, 
		for example).  The following function, however, works as expected. */
		gtk_tree_view_set_cursor( GTK_TREE_VIEW( groupBrowser->treeView ),
                                             path, NULL, FALSE );

		gtk_tree_path_free( path );
		gtk_tree_iter_free( iter );
	} else {
		printf( "group %x not found in model\n", 
			(unsigned int )the_model->current_node );
		gtk_tree_view_expand_all( GTK_TREE_VIEW( groupBrowser->treeView ) );
	}
	
}

GtkTreeIter *gb_search_model( GNode *group ) {
	/* It baffles me that the gtk people didn't implement something like this. 
	"Search a data structure?  Why *ever* would one want to do that?" */
	
	GtkTreeIter *result = NULL;
	
	/* I don't like having to use file-scope variables like this, but 
	there's really no other way.  gtk_tree_model_foreach only lets you 
	pass in one user variable. */
	nodeToSearchFor = group;
	gtk_tree_model_foreach( GTK_TREE_MODEL( groupBrowser->store ),
		gb_search_model_foreach, &result );
	
	return result;
}

gboolean gb_search_model_foreach( 
	GtkTreeModel *model,
	GtkTreePath *path,
	GtkTreeIter *iter,
	gpointer data ) 
{
	gpointer groupPtr;
	GtkTreeIter *temp, **result;

	gtk_tree_model_get( model, iter, 
		GB_COLUMN_GROUPPTR, &groupPtr, 
		-1 );
	if( groupPtr == nodeToSearchFor ) {
		temp = gtk_tree_iter_copy( iter );
		result = data;
		*result = temp;
		return TRUE;
	}
	return FALSE;
}

/* CALLBACKS ************************************************************/

void gb_selection_changed_cb( GtkTreeSelection *selection, gpointer data ) {

	GtkTreeIter iter;
	GtkTreeModel *model;
	gchar *groupname;
	gpointer groupPtr;

	if( gtk_tree_selection_get_selected( selection, &model, &iter ) ) {
		gtk_tree_model_get( model, &iter, 
			GB_COLUMN_GROUPNAME, &groupname, 
			GB_COLUMN_GROUPPTR, &groupPtr, 
			-1 );
#ifdef VERBOSE
		g_print( "You selected the group with the name '%s'\n", groupname );
		g_print( "You selected the group with address %x\n", (unsigned int)groupPtr );
#endif
		g_free( groupname );

		the_model->current_node = (GNode *)groupPtr;
		the_model->current_mesh = (Mesh*)((GNode *)groupPtr)->data;
		
		groupBtns_button_set_sensitive( GROUPACTION_NEW, TRUE );
		
		if( the_model->current_node == the_model->root ) {
			groupBtns_button_set_sensitive( GROUPACTION_FOCUS, FALSE );
			groupBtns_button_set_sensitive( GROUPACTION_EDIT, FALSE );
			groupBtns_button_set_sensitive( GROUPACTION_DUPE, FALSE );
			groupBtns_button_set_sensitive( GROUPACTION_DELETE, FALSE );  /* can't delete top group */
			groupBtns_button_set_sensitive( GROUPACTION_MERGE, FALSE );
			groupBtns_button_set_sensitive( GROUPACTION_SPLIT, FALSE );
		} else {
			groupBtns_button_set_sensitive( GROUPACTION_FOCUS, TRUE );
			groupBtns_button_set_sensitive( GROUPACTION_EDIT, TRUE );
			groupBtns_button_set_sensitive( GROUPACTION_DUPE, TRUE );
			groupBtns_button_set_sensitive( GROUPACTION_DELETE, TRUE );
			groupBtns_button_set_sensitive( GROUPACTION_MERGE, TRUE );
			groupBtns_button_set_sensitive( GROUPACTION_SPLIT, TRUE );
		}

	}

}


/* called when user toggles visibility on a group/mesh */
void gb_cell_toggled_cb(
		GtkCellRendererToggle *cell,
		gchar *path_str,
		gpointer data)
{
	GtkTreeModel *model = (GtkTreeModel *)data;
	GtkTreePath *path = gtk_tree_path_new_from_string( path_str );
	GtkTreeIter iter;
	gboolean toggle_item;
	gint *column;
	gpointer groupPtr;
	
	column = g_object_get_data( G_OBJECT(cell), "column" );
	
	/* get toggled iter */
	gtk_tree_model_get_iter( model, &iter, path );
	gtk_tree_model_get( model, &iter, 
		column, &toggle_item, 
		GB_COLUMN_GROUPPTR, &groupPtr, 
		-1 );
	
	/* do something with the value */
	toggle_item ^= 1;
	
	/* Note - the reason we attached the column number to the renderer 
	and retrieved that value here (rather than just assuming we're talking 
	about a specific column) is because it is possible that, in the future, 
	more toggle-able columns will be added to the browser.  If more toggles 
	are added, all we have to do to handle them is to add a switch statement 
	here for the various column numbers.  This function would then be able 
	to service all of those toggles. */

	((Mesh*)((GNode *)groupPtr)->data)->hidden = !((char)toggle_item);
	
	/* set new value */
	gtk_tree_store_set( GTK_TREE_STORE( model), &iter, column,
						toggle_item, -1 );
	
	/* clean up */
	gtk_tree_path_free( path );

	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}


/* called when a drag is initiated */
void gb_drag_data_get_cb( 
	GtkWidget *widget,
	GdkDragContext *drag_context,
	GtkSelectionData *data,
	guint info,
	guint time,
	gpointer user_data )
{
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	GtkTreeModel *model;
	gchar *groupname;
	gpointer groupPtr;

	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );

	if( gtk_tree_selection_get_selected( selection, &model, &iter ) ) {
		gtk_tree_model_get( model, &iter, 
			GB_COLUMN_GROUPNAME, &groupname, 
			GB_COLUMN_GROUPPTR, &groupPtr, 
			-1 );
#ifdef VERBOSE
		g_print( "You are about to drag group '%s', address %x\n", 
			groupname, (unsigned int)groupPtr );
#endif
		g_free( groupname );
		
		gtk_selection_data_set( 
			data,
			gdk_atom_intern( "kludge3d/model-group-ptr", FALSE ),
/* FIXME - is 32 right?  or should it be 8? */			32, 
			( guchar * ) &groupPtr, 
			sizeof( groupPtr ) );
	}
}


/* called when a drag is completed */
void gb_drag_data_recvd_cb( 
	GtkWidget *widget,
	GdkDragContext *drag_context,
	gint x,
	gint y,
	GtkSelectionData *data,
	guint info,
	guint time,
	gpointer user_data )
{
	GtkTreePath *path;

	if( gtk_tree_view_get_path_at_pos( 
			GTK_TREE_VIEW( widget ), 
			x, y, 
			&path, NULL, NULL, NULL ) ) 
	{
		gpointer groupPtr;
		GtkTreeIter iter;
		GNode *destGroup;
		GNode *srcGroup = *( GNode ** )data->data;
		
#ifdef VERBOSE
		gchar *pathString = gtk_tree_path_to_string( path );
		printf( "you tried to drop group %x on path %s\n", 
			(unsigned int)srcGroup, pathString );
		g_free( pathString );
#endif

		gtk_tree_model_get_iter( 
			GTK_TREE_MODEL(groupBrowser->store), &iter, path );
		gtk_tree_path_free( path );
		
		gtk_tree_model_get( GTK_TREE_MODEL(groupBrowser->store), &iter, 
			GB_COLUMN_GROUPPTR, &groupPtr, 
			-1 );
		destGroup = (GNode *)groupPtr;

		if( group_reparent( the_model, srcGroup, destGroup ) == FALSE )
			printf( "There was an error while relocating a group\n" );
		else {
			g_signal_emit_by_name( notificationObj, 
				"notify::model-structure-changed", NULL );
			g_signal_emit_by_name( notificationObj, 
				"notify::model-appearance-changed", NULL );
		}

	} else {
#ifdef VERBOSE
		printf( "tree view path could not be determined\n" );
		printf( "\tcursor position (%i %i) is probably outside widget\n",
				x, y );
#endif
	}
	
}



