/* 
 * File:         group_btns.c
 * 
 * Description:  creates and manages the group-notebook-tab buttons, 
 *               also handles their callbacks
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 
/* Group buttons code */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "config.h"
#include "group_browser.h"
#include "group_btns.h"
#include "globals.h"
#include "model.h"
#include "group.h"
#include "mesh.h"
#include "vertex.h"
#include "polygon.h"
#include "notebook.h"
#include "undo.h"
#include "win32stuff.h"
#include "selection.h"
#include "gui.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


/* button pixmaps */
#include "pixmaps/group_new.xpm"
#include "pixmaps/group_new_is.xpm"
#include "pixmaps/group_focus.xpm"
#include "pixmaps/group_focus_is.xpm"
#include "pixmaps/group_duplicate.xpm"
#include "pixmaps/group_duplicate_is.xpm"
#include "pixmaps/group_ungroup.xpm"
#include "pixmaps/group_ungroup_is.xpm"
#include "pixmaps/group_delete.xpm"
#include "pixmaps/group_delete_is.xpm"
#include "pixmaps/group_edit.xpm"
#include "pixmaps/group_edit_is.xpm"


/*#define VERBOSE 1 */
#undef VERBOSE

/* PROTOTYPES *************************************************************/

void groupBtns_action_button_cb( GtkWidget *widget, gint idx );

void groupBtns_new_group_dlg( GNode *parent );
void groupBtns_new_group_cb( GtkWidget *widget, gpointer data );
void groupBtns_new_group_dlg_close( GtkWidget *widget, gpointer data );

void groupBtns_edit_dlg( GNode *group );
void groupBtns_edit_cb( GtkWidget *widget, gpointer data );
void groupBtns_edit_dlg_close( GtkWidget *widget, gpointer data );


/* ENUMS AND STRUCTS ******************************************************/

struct groupBtns_struct {

    guint group_button_states; 		/* bitfield */

    GtkWidget *buttons[ GROUPACTION_NBUTTONS ];
    GtkWidget *pixwidgets[ GROUPACTION_NBUTTONS ];

    GdkPixbuf *pixbuf[ GROUPACTION_NBUTTONS ][ 2 ];
}
groupBtns_s[] = { { 0 } };


struct groupBtns_button_data {
    gchar **pix;
    gchar **pix_is;
    gchar *tooltip;
};

struct groupBtns_button_data groupBtns_bd[] = {
    { group_new_xpm,       group_new_is_xpm,       "Create Empty Group" },
    { group_focus_xpm,     group_focus_is_xpm,     "Select Group's Vertices/Polygons" },
    { group_edit_xpm,      group_edit_is_xpm,      "Edit Group" },
    { group_duplicate_xpm, group_duplicate_is_xpm, "Duplicate Group" },
    { group_delete_xpm,    group_delete_is_xpm,    "Delete Group" },
    { group_duplicate_xpm, group_duplicate_is_xpm, "Merge Group with Children" },
    { group_ungroup_xpm,   group_ungroup_is_xpm,   "Create New Group From Sel. Polys" }
};


struct groupBtns_newgroup {
    GtkWidget *MainWin;
    GtkWidget *n_ent;
//    GtkWidget *x_ent, *y_ent, *z_ent;
    GNode *parent;
}
groupBtns_ng[] = { { NULL } };


struct groupBtns_editgroup {
    GtkWidget *MainWin;
    GtkWidget *n_ent;
//    GtkWidget *x_ent, *y_ent, *z_ent;
    GtkWidget *tex_repeat_entry_x;
    GtkWidget *tex_repeat_entry_y;
    GNode *group;
}
groupBtns_eg[] = { { NULL } };



/* INITIALIZATION FUNCS ***************************************************/

GtkWidget *create_group_btns( void ) {

    GtkWidget *hbox;
    GtkTooltips *tips;
    int i;

	/* this hbox is for the action buttons */
    hbox = gtk_hbox_new( FALSE, 0 );
    gtk_widget_show( hbox );

	tips = gtk_tooltips_new();

    /* action buttons for group tab */
    for ( i = 0; i < GROUPACTION_NBUTTONS; i++ ) {
        /* button itself */
        groupBtns_s->buttons[ i ] = gtk_button_new();
        gtk_box_pack_start( GTK_BOX( hbox ), groupBtns_s->buttons[ i ], TRUE, TRUE, 0 );
        g_signal_connect( G_OBJECT( groupBtns_s->buttons[ i ] ), "clicked",
                            G_CALLBACK( groupBtns_action_button_cb ), ( gpointer ) i );
        gtk_widget_show( groupBtns_s->buttons[ i ] );

        /* tooltip */
        gtk_tooltips_set_tip( tips, groupBtns_s->buttons[ i ],
                              groupBtns_bd[ i ].tooltip, NULL );

		/* sensitive pixbuf */
		groupBtns_s->pixbuf[ i ][ 0 ] = 
			gdk_pixbuf_new_from_xpm_data( 
				( const gchar ** ) groupBtns_bd[ i ].pix );

		/* insensitive pixbuf */
		groupBtns_s->pixbuf[ i ][ 1 ] = 
			gdk_pixbuf_new_from_xpm_data( 
				( const gchar ** ) groupBtns_bd[ i ].pix_is );

		/* create GtkImage widgets */
		groupBtns_s->pixwidgets[ i ] = gtk_image_new();
		gtk_widget_show( groupBtns_s->pixwidgets[ i ] );

		/* add the GtkImage to the button */
        gtk_container_add( GTK_CONTAINER( groupBtns_s->buttons[ i ] ),
                           groupBtns_s->pixwidgets[ i ] );

		/* this takes care of putting a pixbuf in the GtkImage, 
		   and makes the button insensitive */
		groupBtns_button_set_sensitive( i, FALSE );
    }

	gtk_tooltips_enable( tips );

    groupBtns_s->group_button_states = 0;

	return hbox;
}



/* CALLBACKS, ETC *********************************************************/


void groupBtns_action_button_cb( GtkWidget *widget, gint idx ) {

    g_return_if_fail( the_model->current_node != NULL );

    switch ( idx ) {
    case GROUPACTION_NEW:
        groupBtns_new_group_dlg( the_model->current_node );
        break;
    case GROUPACTION_FOCUS:
        /* we're selecting all the verts or polys in the current mesh */
        if( the_model->current_mesh != NULL ) {
			switch( global_scope ) {
			case SCOPE_VERTEX:
				sel_vert_unselect_all( the_model );
				sel_vert_select_list( the_model->current_mesh->vertices );
				action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
				break;
			case SCOPE_POLYGON:
				sel_poly_unselect_all( the_model );
				sel_poly_select_list( the_model->current_mesh->polygons );
				action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
				break;
			default:
				break;
			}
        }
		g_signal_emit_by_name( notificationObj, 
			"notify::model-appearance-changed", NULL );
        break;
    case GROUPACTION_EDIT:
        groupBtns_edit_dlg( the_model->current_node );
        break;
    case GROUPACTION_DUPE:
        if( the_model->current_node != NULL ){
            group_duplicate( the_model->current_node );
			action_do( the_model, ACTION_UNKNOWN, NULL, NULL, NULL, NULL, NULL, NULL );
			action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
        }
		g_signal_emit_by_name( notificationObj, 
			"notify::model-structure-changed", NULL );
		g_signal_emit_by_name( notificationObj, 
			"notify::model-appearance-changed", NULL );
        break;
    case GROUPACTION_DELETE:
        if( the_model->current_node != NULL && 
			the_model->current_node != the_model->root )
		{
            /*gb_remove_subtree( the_model->current_node );*/
            group_delete( the_model->current_node );
			action_do( the_model, ACTION_UNKNOWN, NULL, NULL, NULL, NULL, NULL, NULL );
			action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
        }
		g_signal_emit_by_name( notificationObj, 
			"notify::model-structure-changed", NULL );
		g_signal_emit_by_name( notificationObj, 
			"notify::model-appearance-changed", NULL );
        break;
    case GROUPACTION_MERGE:
        if( the_model->current_node != NULL ){
            group_merge( the_model, the_model->current_node );
			action_do( the_model, ACTION_UNKNOWN, NULL, NULL, NULL, NULL, NULL, NULL );
			action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
        }
		g_signal_emit_by_name( notificationObj, 
			"notify::model-structure-changed", NULL );
		g_signal_emit_by_name( notificationObj, 
			"notify::model-appearance-changed", NULL );
        break;
    case GROUPACTION_SPLIT:
        if( the_model->current_node != NULL ){
            group_new_group_from_polys( the_model, the_model->current_node, the_model->selected_polys );
			action_do( the_model, ACTION_UNKNOWN, NULL, NULL, NULL, NULL, NULL, NULL );
			action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
        }
		g_signal_emit_by_name( notificationObj, 
			"notify::model-structure-changed", NULL );
		g_signal_emit_by_name( notificationObj, 
			"notify::model-appearance-changed", NULL );
        break;
    }

}

void groupBtns_button_set_sensitive( gint button_idx, gint sensitive ) {
    g_return_if_fail( button_idx < GROUPACTION_NBUTTONS );

    if ( sensitive ) {
        if ( groupBtns_s->group_button_states & ( 1 << button_idx ) )
            return ;
        groupBtns_s->group_button_states |= ( 1 << button_idx );
    } else {
        if ( !( groupBtns_s->group_button_states & ( 1 << button_idx ) ) )
            return ;
        groupBtns_s->group_button_states &= ~( 1 << button_idx );
    }

	gtk_image_set_from_pixbuf( 
		GTK_IMAGE( groupBtns_s->pixwidgets[ button_idx ] ),
		groupBtns_s->pixbuf[ button_idx ][ sensitive ? 0 : 1 ] );
	gtk_widget_set_sensitive( groupBtns_s->buttons[ button_idx ], sensitive );
}



void groupBtns_new_group_dlg( GNode *parent ) {

	GtkWidget *dialog;
	GtkWidget * widget, *hbox;
	gint dlg_response;

	dialog = gtk_dialog_new_with_buttons ("New Group Options",
										  GTK_WINDOW( TopLevelWindow ),
										  GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
										  GTK_STOCK_OK,
										  GTK_RESPONSE_OK,
										  GTK_STOCK_CANCEL,
										  GTK_RESPONSE_CANCEL,
										  NULL);

	hbox = gtk_hbox_new( FALSE, 4 );
	gtk_box_pack_start( GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, FALSE, FALSE, 0 );
	
	widget = gtk_label_new( "Name:" );
	gtk_box_pack_start( GTK_BOX( hbox ), widget, FALSE, FALSE, 0 );

	groupBtns_ng->n_ent = gtk_entry_new();
	gtk_box_pack_start( GTK_BOX( hbox ), groupBtns_ng->n_ent, TRUE, TRUE, 0 );

	gtk_entry_set_text( GTK_ENTRY( groupBtns_ng->n_ent ),
						"New Group" );

	groupBtns_ng->parent = parent;

	/* set up the OK button as the default widget */
	gtk_dialog_set_default_response( GTK_DIALOG(dialog), GTK_RESPONSE_OK );
	g_signal_connect( G_OBJECT( groupBtns_ng->n_ent ), "activate",
					G_CALLBACK(gui_dialog_ok_cb), dialog );
	
	/* shows the dialog, and since it's modal, prevents rest of 
	   kludge3d from doing stuff */
	gtk_widget_show_all (dialog);
	
	/* runs a gtkmain until the user clicks on something */
	dlg_response = gtk_dialog_run( GTK_DIALOG (dialog) );
	
	switch( dlg_response ) {
	case GTK_RESPONSE_OK:
		groupBtns_new_group_cb( NULL, NULL );
		break;
	default:
		break;
	}
	
	gtk_widget_destroy( dialog );

}

void groupBtns_new_group_cb( GtkWidget *widget, gpointer data ) {
    const gchar * text;
    GNode *group;
    
/*    gtk_widget_hide( groupBtns_ng->MainWin );*/

    text = gtk_entry_get_text( GTK_ENTRY( groupBtns_ng->n_ent ) );

    group = group_new( the_model );
	group_reparent( the_model, group, groupBtns_ng->parent );

#ifdef VERBOSE
printf( "new group: %x parent %x children %x data %x \n", 
group, 
group->parent, 
group->children, 
group->data );
#endif

    mesh_set_name( (Mesh *)group->data, (char *)text );

	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );

	g_signal_emit_by_name( notificationObj, 
		"notify::model-structure-changed", NULL );
}

void groupBtns_new_group_dlg_close( GtkWidget *widget, gpointer data ) {
    gtk_widget_hide( groupBtns_ng->MainWin );
}


void groupBtns_edit_dlg( GNode *group ) {

	GtkWidget *dialog;
	GtkWidget * widget, *hbox, *tab;
	gint dlg_response;

	dialog = gtk_dialog_new_with_buttons ("Edit Group",
										  GTK_WINDOW( TopLevelWindow ),
										  GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
										  GTK_STOCK_OK,
										  GTK_RESPONSE_OK,
										  GTK_STOCK_CANCEL,
										  GTK_RESPONSE_CANCEL,
										  NULL);

	hbox = gtk_hbox_new( FALSE, 4 );
	gtk_box_pack_start( GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, FALSE, FALSE, 0 );
	
	widget = gtk_label_new( "Name:" );
	gtk_box_pack_start( GTK_BOX( hbox ), widget, FALSE, FALSE, 0 );

	groupBtns_eg->n_ent = gtk_entry_new();
	gtk_box_pack_start( GTK_BOX( hbox ), groupBtns_eg->n_ent, TRUE, TRUE, 0 );

	/* TEXTURE REPEAT STUFF */
	
	widget = gtk_frame_new( "Texture Repeat" );
	gtk_box_pack_start( GTK_BOX(GTK_DIALOG(dialog)->vbox), widget, TRUE, TRUE, 0 );
	
	tab = gtk_table_new( 3, 2, FALSE );
	gtk_container_add( GTK_CONTAINER( widget ), tab );
	
	widget = gtk_label_new( "U:" );
	gtk_table_attach_defaults( GTK_TABLE( tab ), widget, 0, 1, 0, 1 );
	
	groupBtns_eg->tex_repeat_entry_x = gtk_entry_new();
	gtk_entry_set_max_length( GTK_ENTRY( groupBtns_eg->tex_repeat_entry_x ), 10 );
	gtk_table_attach_defaults( GTK_TABLE( tab ), groupBtns_eg->tex_repeat_entry_x, 1, 2, 0, 1 );
	
	widget = gtk_label_new( "V:" );
	gtk_table_attach_defaults( GTK_TABLE( tab ), widget, 0, 1, 1, 2 );
	
	groupBtns_eg->tex_repeat_entry_y = gtk_entry_new();
	gtk_entry_set_max_length( GTK_ENTRY( groupBtns_eg->tex_repeat_entry_y ), 10 );
	gtk_table_attach_defaults( GTK_TABLE( tab ), groupBtns_eg->tex_repeat_entry_y, 1, 2, 1, 2 );

	{
	
	Mesh* m = (Mesh*)group->data;
	if( m != NULL ) {
		char s[32]; 
		memset( s, 0, 32);
		
		gtk_entry_set_text( GTK_ENTRY( groupBtns_eg->n_ent ), m->name );

		g_snprintf( s, 31, "%f", m->texrep_x );
		gtk_entry_set_text( GTK_ENTRY( groupBtns_eg->tex_repeat_entry_x ), s );
		g_snprintf( s, 31, "%f", m->texrep_y );
		gtk_entry_set_text( GTK_ENTRY( groupBtns_eg->tex_repeat_entry_y ), s );

	} else {
		gtk_entry_set_text( GTK_ENTRY( groupBtns_eg->n_ent ), "An error occurred" );
		gtk_entry_set_text( GTK_ENTRY( groupBtns_eg->tex_repeat_entry_x ), "1.0" );
		gtk_entry_set_text( GTK_ENTRY( groupBtns_eg->tex_repeat_entry_y ), "1.0" );
	}
	
	}

	groupBtns_eg->group = group;


	/* set up the OK button as the default widget */
	gtk_dialog_set_default_response( GTK_DIALOG(dialog), GTK_RESPONSE_OK );
	g_signal_connect( G_OBJECT( groupBtns_eg->n_ent ), "activate",
					G_CALLBACK(gui_dialog_ok_cb), dialog );
	g_signal_connect( G_OBJECT( groupBtns_eg->tex_repeat_entry_x ), "activate",
					G_CALLBACK(gui_dialog_ok_cb), dialog );
	g_signal_connect( G_OBJECT( groupBtns_eg->tex_repeat_entry_y ), "activate",
					G_CALLBACK(gui_dialog_ok_cb), dialog );
	
	/* shows the dialog, and since it's modal, prevents rest of 
	   kludge3d from doing stuff */
	gtk_widget_show_all (dialog);
	
	/* runs a gtkmain until the user clicks on something */
	dlg_response = gtk_dialog_run( GTK_DIALOG (dialog) );
	
	switch( dlg_response ) {
	case GTK_RESPONSE_OK:
		groupBtns_edit_cb( NULL, NULL );
		break;
	default:
		break;
	}
	
	gtk_widget_destroy( dialog );

}

void groupBtns_edit_cb( GtkWidget *widget, gpointer data ) {
    float x, y;
    const gchar * text;
    
/*    gtk_widget_hide( groupBtns_eg->MainWin );*/

    /* get name */
	text = gtk_entry_get_text( GTK_ENTRY( groupBtns_eg->n_ent ) );

    mesh_set_name( (Mesh *)groupBtns_eg->group->data, (char *)text );

    /* get texture repeat values */
	text = gtk_entry_get_text( GTK_ENTRY( groupBtns_eg->tex_repeat_entry_x ) );
    x = strtod( text, (char **)NULL );
    text = gtk_entry_get_text( GTK_ENTRY( groupBtns_eg->tex_repeat_entry_y ) );
    y = strtod( text, (char **)NULL );

    mesh_set_texrep( the_model->current_mesh, x, y );

	g_signal_emit_by_name( notificationObj, 
		"notify::model-structure-changed", NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}

void groupBtns_edit_dlg_close( GtkWidget *widget, gpointer data ) {
    gtk_widget_hide( groupBtns_eg->MainWin );
}



