/* 
 * File:		 gui.c
 * 
 * Description:  creates main window and populates it
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <netdb.h>	/* for gethostbyname */
#include <gtk/gtk.h>
#include <gtk/gtkgl.h>

#include "gui.h"
#include "config.h"
#include "toolbar.h"
#include "bottombar.h"
#include "frame.h"
#include "menu.h"
#include "notebook.h"
#include "view.h"
#include "model_load_save.h"	/* for DND loading */
#include "documents.h"			/* for DND loading */

#ifdef MEMWATCH
#include "memwatch.h"
#endif

/* GLOBALS *************************************************************/

GtkWidget *TopLevelWindow = NULL;


/* PROTOTYPES **********************************************************/

gboolean gui_quit( GtkWidget *widget, GdkEvent *event, gpointer data ) ;
void gui_drag_data_recvd_cb( 
	GtkWidget *widget,
	GdkDragContext *drag_context,
	gint x,
	gint y,
	GtkSelectionData *selection_data,
	guint info,
	guint time,
	gpointer user_data );


/* FUNCS ***************************************************************/

void gui_init( int *argc, char ***argv ) {
	gtk_init( argc, argv );

	gtk_gl_init( argc, argv );

/*	gdk_rgb_set_verbose( TRUE );*/

	gtk_widget_set_default_colormap (gdk_rgb_get_colormap());
	
}

void gui_create_main_window( void ) {
	GtkWidget * vbox;
	GtkWidget * hbox;

	TopLevelWindow = gtk_window_new( GTK_WINDOW_TOPLEVEL );
	gtk_widget_set_name( TopLevelWindow, "kludge3d" );
	gtk_widget_set_size_request( TopLevelWindow, 800, 600 );

	vbox = gtk_vbox_new( FALSE, 0 );
	hbox = gtk_hbox_new( FALSE, 0 );

	gtk_container_add( GTK_CONTAINER( TopLevelWindow ), vbox );
	gtk_widget_show( vbox );

	g_signal_connect( G_OBJECT( TopLevelWindow ), "delete-event",
						G_CALLBACK( gui_quit ), NULL );
	g_signal_connect( G_OBJECT( TopLevelWindow ), "destroy-event",
						G_CALLBACK( gui_quit ), NULL );
	
	gtk_widget_realize( TopLevelWindow );

	/* Part of a work-around for sharing OpenGL contexts amongst the views.
	See view.c's view_create_dummy_drawable for an explanation.  Note that 
	this MUST come before the views are created. */
	gtk_box_pack_start( GTK_BOX( vbox ), view_create_dummy_drawable(), FALSE, FALSE, 0 );

	gtk_box_pack_start( GTK_BOX( vbox ), get_menu_bar( TopLevelWindow ), FALSE, FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( vbox ), create_toolbar(), FALSE, FALSE, 0 );

	gtk_box_pack_start( GTK_BOX( hbox ), create_frames(), TRUE, TRUE, 0 );
	gtk_box_pack_start( GTK_BOX( hbox ), create_notebook(), FALSE, FALSE, 0 );
	gtk_widget_show( hbox );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, TRUE, TRUE, 0 );

	gtk_box_pack_start( GTK_BOX( vbox ), create_bottombar(), FALSE, FALSE, 0 );

	/* DRAG AND DROP */
	{
	static const GtkTargetEntry type_accept_list[] = {
		{ "text/uri-list", 0, 0 }
	};

	gtk_drag_dest_set( 
		TopLevelWindow, 
		GTK_DEST_DEFAULT_ALL, 
		type_accept_list, 1, 
		GDK_ACTION_COPY | GDK_ACTION_MOVE );

	g_signal_connect( G_OBJECT( TopLevelWindow ), "drag-data-received",
						G_CALLBACK( gui_drag_data_recvd_cb ), NULL );
	
	}

	gtk_widget_show( TopLevelWindow );

}


void gui_run( void ) {
	gtk_main();
}


gboolean gui_quit( GtkWidget *widget, GdkEvent *event, gpointer data ) {

	gtk_main_quit();

	return FALSE;
}


/* called when a drag is completed */
void gui_drag_data_recvd_cb( 
	GtkWidget *widget,
	GdkDragContext *drag_context,
	gint x,
	gint y,
	GtkSelectionData *selection_data,
	guint info,
	guint time,
	gpointer user_data )
{
	gchar *text = selection_data->data;
	char hostname[512];
	char completehostname[512];
	char filename[512];
	int fnlen;
		
	if( gethostname( hostname, 511 ) == 0 )
	{
		/* gethostname doesn't always return the full name... */
		struct hostent *ent;

		ent = gethostbyname( hostname );
		if( ent ) {
			strncpy( completehostname, ent->h_name, 511 );
			completehostname[511] = '\0';
		} else
			completehostname[0] = '\0';
	}
	else
	{
		g_warning("gethostname() failed - using 'localhost'\n");
		strncpy( hostname, "localhost", 511 );
		strncpy( completehostname, "localhost", 511 );
	}
	
	while( *text != '\0' ) {
		if( strncmp( text, "file://", 7 ) == 0 ) {
			text += 7;

			if( *text != '/' ) {
				/* hostname has been included */
				if( strncmp( text, completehostname, strlen( completehostname ) ) == 0 ) {
					/* completehostname matches */
					text += strlen( completehostname );
				} else if( strncmp( text, hostname, strlen( hostname ) ) == 0 ) {
					/* hostname matches */
					text += strlen( hostname );
				} else if( strncmp( text, "localhost", strlen( "localhost" ) ) == 0 ) {
					/* "localhost" matches */
					text += strlen( "localhost" );
				} else {
					/* hostname does not match; move on to next line */
					continue;
				}
			}

			/* if next char still isn't a /, move on to next line */
			if( *text != '/' )
				continue;

			fnlen = 0;
			while( text[fnlen] != '\n' && text[fnlen] != '\0' )
				fnlen ++;
			if( text[fnlen] == '\n' )
				fnlen --;
			if( fnlen > 0 ) {
				Model *new_model;
				strncpy( filename, text, fnlen < 511 ? fnlen : 511 );
				filename[fnlen] = '\0';
fprintf( stderr, "filename is %s\n", filename );
				new_model = model_load(filename, -1);

				if( new_model == NULL ) {
					bb_push_message_f( 3.0, "Failed to load file %s", filename );
				} else {
					docs_add_model( new_model );
					docs_switch_model( new_model );

					bb_push_message_f( 2.5, "Loaded file %s", filename );
					break;
				}
				
			}
		}
		while( *text != '\0' && *text != '\n' )
			text ++;
		if( *text == '\n' )
			text ++;
	}
}


/* used by other modules */
void gui_dialog_ok_cb( GtkWidget *entry, gpointer dialog ) {
	g_return_if_fail(GTK_IS_DIALOG(dialog));
	gtk_dialog_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
}

