/* 
 * File:		 log_window.c
 * 
 * Description:  creates and manages the log window
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <gtk/gtk.h>

#include "log_window.h"
#include "scripting_entry.h"
#include "globals.h"
#include "gui.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


/* The log stream will be polled once every LOG_TIMEOUT_VALUE milliseconds */
#define LOG_TIMEOUT_VALUE 500


/* FILE SCOPE VARS ******************************************************/

FILE *redirected_stdout_fp = NULL;
FILE *saved_stdout_fp = NULL;
FILE *log_reader_fp = NULL;
int log_reader_fdesc = -1;

GtkTextBuffer *log_textbuffer = NULL;
GtkTextMark *log_textbuffer_endmark = NULL;
GtkWidget *log_window = NULL;


/* PROTOTYPES ***********************************************************/

gboolean log_cleanup( gpointer data ) ;
void log_redirect_stdout( void ) ;
void log_close_stdout( void ) ;
void log_open_redirected_output( void ) ;
gboolean log_read_timeout_cb( gpointer data ) ;
int log_isready( uint fd ) ;

void log_textbuffer_setup( void ) ;
void log_textbuffer_append( char * str ) ;
void log_window_hide( GtkWidget *widget, gpointer data ) ;
void log_window_scroll_to_bottom( 
	GtkTextBuffer *textbuffer, gpointer data ) ;


/* FUNCS ****************************************************************/

void log_init( void ) {
	
	log_textbuffer_setup();
	log_redirect_stdout();
	log_open_redirected_output();

	g_signal_connect( 
		notificationObj, "notify::exit", G_CALLBACK(log_cleanup), NULL );
}


gboolean log_cleanup( gpointer data ) {
	log_close_stdout();
	return FALSE;
}


void log_redirect_stdout( void ) {

	char logfilename[512];

	snprintf( logfilename, 511, "%s/.kludge3d/log", getenv("HOME") );

	redirected_stdout_fp = fopen( logfilename, "w" );
	if( redirected_stdout_fp == NULL ) {
		printf( "Unable to open %s for writing\n", logfilename );
		return;
	}
	/* turn off buffering */
	setvbuf( redirected_stdout_fp, NULL, _IONBF, 0 );
	printf( "kludge3d - starting log\n" );

	/* change stdout to point to our log stream */
	fflush( stdout );
	saved_stdout_fp = stdout;
	stdout = redirected_stdout_fp;

}


void log_close_stdout( void ) {
	if( saved_stdout_fp )
		stdout = saved_stdout_fp;
	if( redirected_stdout_fp )
		fclose( redirected_stdout_fp );
}


void log_open_redirected_output( void ) {
	char logfilename[512];

	snprintf( logfilename, 511, "%s/.kludge3d/log", getenv("HOME") );

	log_reader_fdesc = open( logfilename, O_RDONLY );
	if( log_reader_fdesc == -1 ) {
		printf( "Unable to open %s for reading\n", logfilename );
		return;
	}
	
	log_reader_fp = fdopen( log_reader_fdesc, "r" );
	if( log_reader_fp == NULL ) {
		printf( "Unable to open fd %i for reading\n", log_reader_fdesc );
		return;
	}

	gtk_timeout_add( LOG_TIMEOUT_VALUE, log_read_timeout_cb, NULL );
}


gboolean log_read_timeout_cb( gpointer data ) {
	char buf[512];
	int no_input = FALSE;
	
	if( !log_isready( log_reader_fdesc ) )
		return TRUE; /* important */
	
	while( fgets( buf, 511, log_reader_fp ) && !no_input ) {
		if( strlen( buf ) <= 0 )
			no_input = TRUE;
		else {
			log_textbuffer_append( buf );
#ifdef VERBOSE
			fprintf( stderr, "Log appended the following - %s\n", buf );
#endif
		}
	}

	return TRUE; /* important */
}


int log_isready( uint fd ) {
	static struct timeval tval={0,0};
	fd_set readset;

	FD_ZERO(&readset);
	FD_SET(fd, &readset);
	if (select(fd+1, &readset, NULL, NULL, &tval) < 0) {
		perror("Error in select loop");
		exit(-1);
	}
	if (FD_ISSET(fd, &readset)) return 1;
	else return 0;
}


/* TEXTBUFFER STUFF *****************************************************/

void log_textbuffer_setup( void ) {
	
	GtkTextIter iter;

	log_textbuffer = gtk_text_buffer_new( NULL );

	/* put a mark at the end of the buffer, and make sure that it will 
	always be to the right of any text that is appended to the buffer */
	gtk_text_buffer_get_end_iter( log_textbuffer, &iter );
	log_textbuffer_endmark = 
		gtk_text_buffer_create_mark( log_textbuffer, "endmark", &iter, FALSE );
}


void log_textbuffer_append( char * str ) {
	GtkTextIter iter;
	gtk_text_buffer_get_iter_at_mark( 
		log_textbuffer, &iter, log_textbuffer_endmark );
	gtk_text_buffer_insert( log_textbuffer, &iter, str, strlen( str ) );
}


/* GUI STUFF ************************************************************/

GtkWidget * create_log_view( void ) {
	GtkWidget *textview, *sw;
	
	textview = gtk_text_view_new_with_buffer( log_textbuffer );
	sw = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW(sw),
									GTK_POLICY_AUTOMATIC,
									GTK_POLICY_AUTOMATIC);
	gtk_container_add( GTK_CONTAINER(sw), textview );

	gtk_text_view_set_wrap_mode( GTK_TEXT_VIEW(textview), GTK_WRAP_WORD );
	gtk_text_view_set_editable( GTK_TEXT_VIEW(textview), FALSE );
	gtk_text_view_set_cursor_visible( GTK_TEXT_VIEW(textview), FALSE );
	
	/* every time the textbuffer is changed, scroll to the bottom */
	g_signal_connect( G_OBJECT( log_textbuffer ), "changed",
						G_CALLBACK( log_window_scroll_to_bottom ), textview );
	
	gtk_widget_show_all( sw );
	return sw;
}


void show_log_window( void ) {
	GtkWidget *vbox, *button;
	
	if( log_window == NULL ) {
		log_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_container_set_border_width( GTK_CONTAINER(log_window), 5 );
		gtk_window_set_title(GTK_WINDOW(log_window), "Kludge3d Scripting Log");

		/* make the window a child of the TLW */
		gtk_window_set_transient_for( GTK_WINDOW(log_window), GTK_WINDOW(TopLevelWindow) );
		/* ensure that the window will go away when the program exits */
		gtk_window_set_destroy_with_parent( GTK_WINDOW(log_window), TRUE );

		/* set a minimum size */
		gtk_widget_set_size_request( log_window, 300, 200 );
		
		vbox = gtk_vbox_new( FALSE, 0 );
		gtk_container_add(GTK_CONTAINER(log_window), vbox );
		
        gtk_box_pack_start( GTK_BOX( vbox ), 
			create_log_view(), TRUE, TRUE, 0 );
		
		gtk_box_pack_start( GTK_BOX( vbox ), 
			create_scripting_entry(), FALSE, FALSE, 5 );
		
		button = gtk_button_new_with_label( "Close" );
        g_signal_connect( G_OBJECT( button ), "clicked",
                            G_CALLBACK( log_window_hide ), NULL );
        gtk_box_pack_start( GTK_BOX( vbox ), button, FALSE, FALSE, 0 );

		/* catches event generated when user clicks on the window manager's 
		close-window button ('X' for most WM's) */
		g_signal_connect( G_OBJECT( log_window ), "delete-event",
							G_CALLBACK( log_window_hide ), NULL );
		
		gtk_widget_show_all( log_window );
	}
	gtk_widget_show( log_window );
}


void log_window_hide( GtkWidget *widget, gpointer data ) {
	gtk_widget_hide( log_window );
}


void log_window_scroll_to_bottom( 
	GtkTextBuffer *textbuffer, gpointer data ) 
{
	/* 
	It is important that gtk_text_view_scroll_to_mark be used, rather 
	than gtk_text_view_scroll_to_iter.  gtk_text_view_scroll_to_iter 
	operates on the textview immediately, whereas gtk_text_view_scroll_to_mark 
	will cause the scrolling to occur *after* the textbuffer's/textview's size 
	has been updated.
	Here's what would happen if gtk_text_view_scroll_to_iter were used: 
	If multiple 'changed' events are sent in a row (as the result of several 
	lines being read in), the textview would only scroll to the first new line, 
	rather than to the bottom of the text area. 
	*/

	GtkTextView *textview = GTK_TEXT_VIEW( data );
	
	gtk_text_view_scroll_mark_onscreen( textview, log_textbuffer_endmark );
}


