/* 
 * File:         main.c
 * 
 * Description:  main function for kludge3d, creates main window and 
 * 		 populates it
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <gtk/gtk.h>

#include "config.h"
#include "globals.h"
#include "gui.h"
#include "model.h"
#include "model_load_save.h"
#include "documents.h"
#include "undo.h"
#include "prefs.h"
#include "log_window.h"
#include "misc.h"

#ifdef USE_PYTHON
#include "pyinit.h"
#endif

#ifdef MEMWATCH
#include "memwatch.h"
#endif

/* GLOBALS *************************************************************/

gpointer notificationObj = NULL;


/* FILE SCOPE VARS ****************************************************/


/* PROTOTYPES *********************************************************/

gboolean main_cleanup( gpointer data ) ;
gboolean main_load_model_cb( gpointer data ) ;
void main_initialize_model( int argc, char **argv ) ;
void main_initialize_notificationObj( void ) ;
void main_create_directory( void );


/* FUNCS **************************************************************/

gboolean main_cleanup( gpointer data ) {

#ifdef USE_PYTHON
	stop_python();
#endif
	
	docs_cleanup();
	the_model = NULL;

	return FALSE;
}

gboolean main_load_model_cb( gpointer data ) {
	Model *new_model;

	new_model = model_load( (char *)data, -1 );
	if( new_model == NULL ) {
		/* TODO: message box */
		printf("model_load failed\n");
	} else {
		
		/* kludge3d always starts with a new/blank model.  replace it with 
		the model that the user has loaded from the command line */
		docs_remove_model( the_model );
		
		docs_add_model( new_model );
		docs_switch_model( new_model );
	}

	return FALSE;	/* important */
}

void main_initialize_model( int argc, char **argv ) {
	Model *m;

    if( argc == 2 ) {
        /* user has specified a filename on command line */

        // FIXME - better command-line parsing
		
		/* Model loading must be done via a timeout because 
		   some model loaders (wavefront) make use of gui elements
		   (gtk_main is the real problem) and hence *must* be called 
		   only *after* the MAIN gtk_main has been entered.
		   The use of the timeout will ensure that the MAIN gtk_main 
		   has been entered.
		 */
		gtk_timeout_add( 50, main_load_model_cb, argv[1] );
    }

	m = model_new();
	docs_add_model( m );
	docs_switch_model( m );
}

void main_initialize_notificationObj( void ) {
	GType param_types[1];
	
	param_types[0] = G_TYPE_POINTER;
	
	/* add new signals to gobject */
	g_signal_newv( 
		"switched-model", 
		G_TYPE_OBJECT, 
		G_SIGNAL_ACTION, 
		NULL, /* class closure */
		NULL, /* accumulator func, called after every call to a closure */
		NULL, /* accumulator data */
		g_cclosure_marshal_VOID__POINTER, 
		G_TYPE_NONE, /* return_type */
		1, /* num params */
		param_types );
	g_signal_newv( 
		"added-model", 
		G_TYPE_OBJECT, 
		G_SIGNAL_ACTION, 
		NULL, /* class closure */
		NULL, /* accumulator func, called after every call to a closure */
		NULL, /* accumulator data */
		g_cclosure_marshal_VOID__POINTER, 
		G_TYPE_NONE, /* return_type */
		1, /* num params */
		param_types );
	g_signal_newv( 
		"removed-model", 
		G_TYPE_OBJECT, 
		G_SIGNAL_ACTION, 
		NULL, /* class closure */
		NULL, /* accumulator func, called after every call to a closure */
		NULL, /* accumulator data */
		g_cclosure_marshal_VOID__POINTER, 
		G_TYPE_NONE, /* return_type */
		1, /* num params */
		param_types );
	
	
	/* create notificationObj */
	notificationObj = g_object_new( G_TYPE_OBJECT, NULL );
	
}


#define DIR_MODE (S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH)	/* same as 0755 */
void main_create_directory( void ) {
	/* checks to see if a ~/.kludge3d directory exists, and creates one 
	   if needed */

	char dirname[512];

	snprintf( dirname, 511, "%s/.kludge3d", getenv("HOME") );
	
	if( !file_exists( dirname ) ) {
		if( mkdir( dirname, DIR_MODE ) != 0 ) {
			fprintf( stderr, "Failed to create directory %s\n", dirname );
			return;
		} else {
			printf( "Created directory %s\n", dirname );
		}
	}
}


int main( int argc, char **argv ) {

    main_create_directory();
	
	gui_init( &argc, &argv );
	
	main_initialize_notificationObj();
	prefs_load();
	log_init();

	g_signal_connect( 
		notificationObj, "notify::exit", G_CALLBACK(main_cleanup), NULL );

	gui_create_main_window();
	
	undo_init();

    main_initialize_model( argc, argv );

#ifdef USE_PYTHON
    start_python();
#endif

	gui_run();

	g_signal_emit_by_name( notificationObj, "notify::exit", NULL );
	
    return 0;
}
