/*
 * File:         mat_edit.c
 * 
 * Description:  Allows user to change a material's properties (things like 
 *               diffuse, ambient, etc colors)
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */


/*
 * Portions of this file are from:
 * 
 * ME3D 3-D Modeler Program
 * Copyright (C) 1998 Sam Revitch
 * 
 * The original file was a complete mess, though.  I've cleaned it up a bit; 
 * "refactored" is probably an appropriate term.  Much of the duplicate 
 * code has been consolidated.
 * 
 * I've also ported the code to GTK2; this was difficult, as Revitch used many 
 * deprecated features of GTK1.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <gtk/gtk.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include "mat_edit.h"
#include "mat_preview.h"
#include "material.h"
#include "globals.h"
#include "gui.h"


#ifdef MEMWATCH
#include "memwatch.h"
#endif


#define MATEDIT_ZERO_THRESHOLD 0.00001


typedef struct _MatEditData MatEditData;
struct _MatEditData {
	GtkWidget *MainWin;
	GtkWidget *MatPreview;
	GtkWidget *MatNotebook;
	GtkWidget *MatPicker;
	GtkWidget *MatPickerBox;
	GtkAdjustment *SliderAdj[MAT_INDEX_NUM_INDICES];
	GtkWidget *SwatchWidget[MAT_INDEX_NUM_INDICES];
	GdkPixbuf *SwatchPixbuf[MAT_INDEX_NUM_INDICES];
	guint cont_update, active_preview, freeze, lock_intensity;
	/* base color */
	float bc[4][4];
	/* base intensities; 4th element not used, but needed for matedit_normalize_attribute */
	float bi[4];
	/* previous color, needed for color picker->cancel */
	GLfloat pc[4];
	Material *material;
};

MatEditData *GlobalMatEdit = NULL;


MatEditData *matedit_new( void );
GtkWidget *mat_edit_create_slider( 
	char *str, int index, MatEditData *med );
gboolean matedit_cleanup( gpointer data ) ;
void matedit_hide(MatEditData *med);
void matedit_set_material( MatEditData *med, Material *material ) ;
void matedit_update_color_swatch( 
	GdkPixbuf *pixbuf, GtkWidget *parent, GLfloat *color );
void matedit_recompute_base_intensities(MatEditData *med);
float matedit_normalize_attribute( float dest[4], float src[4] ) ;
void matedit_toggle_intensity_lock(GtkWidget *widget, MatEditData *med);

void matedit_colorpicker_show(GtkWidget *widget, guint color_id);
void matedit_colorpicker_changed_cb(GtkWidget *widget, MatEditData *med);
void matedit_colorpicker_ok(GtkWidget *widget, MatEditData *med);
void matedit_colorpicker_close(GtkWidget *widget, MatEditData *med);
void matedit_colorpicker_close_2(GtkWidget *widget, GdkEvent *event,
				 MatEditData *med);

void matedit_color_changed(GtkWidget *widget, MatEditData *med);
void matedit_slider_changed(GtkAdjustment *adj, int index );

void matedit_set_preview_value(MatEditData *med, GLfloat *v, int i ) ;

void matedit_kill_dialog(GtkWidget *widget, GdkEvent *event,
			 MatEditData *med);
void matedit_ok(GtkWidget *widget, MatEditData *med);
void matedit_apply(GtkWidget *widget, MatEditData *med);
void matedit_cancel(GtkWidget *widget, MatEditData *med);

void matedit_gdkcolor_to_floats( GdkColor *gdkcolor, float *color ) ;
void matedit_floats_to_gdkcolor( GdkColor *gdkcolor, float *color ) ;
void matedit_get_color( MatEditData *med, int index, float color[4] ) ;
float matedit_get_shininess( MatEditData *med ) ;


MatEditData *matedit_new( void )
{
	MatEditData *med;
	GtkWidget *widget, *vbox, *hbox;
	GtkWidget *slider_vbox;

	med = (MatEditData *) malloc(sizeof(MatEditData));
	memset(med, 0, sizeof(MatEditData));

	med->cont_update = TRUE;
	
	/* Before we go any farther, check to make sure that we can create a 
	valid mat_preview. */
	med->MatPreview = mat_preview_new();
	if( med->MatPreview == NULL ) {
		free( med );
		return NULL;
	}

	/* begin dialog box */
	med->MainWin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(med->MainWin), "Material Properties");

	/* make the med window a child of the TLW */
	gtk_window_set_transient_for( GTK_WINDOW(med->MainWin), GTK_WINDOW(TopLevelWindow) );
	/* ensure that the med window will go away when the program exits */
	gtk_window_set_destroy_with_parent( GTK_WINDOW(med->MainWin), TRUE );

	g_signal_connect(G_OBJECT(med->MainWin), "delete_event",
			G_CALLBACK(matedit_kill_dialog), (gpointer) med);
	gtk_container_set_border_width(GTK_CONTAINER(med->MainWin), 4);

	vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(med->MainWin), vbox);
	gtk_widget_show(vbox);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	gtk_widget_show(hbox);

	widget = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(widget), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(hbox), widget, TRUE, TRUE, 0);
	gtk_widget_show(widget);

	gtk_container_add(GTK_CONTAINER(widget), med->MatPreview);
	gtk_widget_show(med->MatPreview);

	med->MatNotebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(med->MatNotebook), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(hbox), med->MatNotebook, FALSE, FALSE, 0);
	gtk_widget_show(med->MatNotebook);


	/** Color Picker Tab **/
	widget = gtk_label_new("Quick-Set Color Picker");
	med->MatPicker = gtk_color_selection_new();
	gtk_color_selection_set_has_opacity_control(GTK_COLOR_SELECTION(med->MatPicker), TRUE);
	g_signal_connect(G_OBJECT(med->MatPicker), "color_changed",
				 G_CALLBACK(matedit_color_changed), (gpointer) med);

	gtk_notebook_append_page(GTK_NOTEBOOK(med->MatNotebook),
				 med->MatPicker, widget);
	gtk_widget_show(widget);
	gtk_widget_show(med->MatPicker);


	/** Levels Tab **/
	slider_vbox = gtk_vbox_new( FALSE, 4 );
	gtk_notebook_append_page( GTK_NOTEBOOK(med->MatNotebook), 
		slider_vbox, gtk_label_new("Individual Colors") );

	gtk_box_pack_start( GTK_BOX(slider_vbox), 
		mat_edit_create_slider( "Diffuse:", MAT_INDEX_DIFFUSE, med ), 
		FALSE, FALSE, 0 );
	gtk_box_pack_start( GTK_BOX(slider_vbox), 
		mat_edit_create_slider( "Ambient:", MAT_INDEX_AMBIENT, med ), 
		FALSE, FALSE, 0 );
	gtk_box_pack_start( GTK_BOX(slider_vbox), 
		mat_edit_create_slider( "Emission:", MAT_INDEX_EMISSION, med ), 
		FALSE, FALSE, 0 );

	widget = gtk_check_button_new_with_label("Lock Intensities");
	g_signal_connect(G_OBJECT(widget), "toggled",
					 G_CALLBACK( matedit_toggle_intensity_lock),
					 (gpointer) med);
	gtk_box_pack_start( GTK_BOX(slider_vbox), widget, FALSE, FALSE, 0 );
	gtk_widget_show(widget);
	gtk_widget_show( slider_vbox );


	/** Reflection Tab **/
	slider_vbox = gtk_vbox_new( FALSE, 4 );
	gtk_notebook_append_page( GTK_NOTEBOOK(med->MatNotebook), 
		slider_vbox, gtk_label_new("Reflection") );

	gtk_box_pack_start( GTK_BOX(slider_vbox), 
		mat_edit_create_slider( "Specular:", MAT_INDEX_SPECULAR, med ), 
		FALSE, FALSE, 0 );
	gtk_box_pack_start( GTK_BOX(slider_vbox), 
		mat_edit_create_slider( "Shininess:", MAT_INDEX_SHININESS, med ), 
		FALSE, FALSE, 0 );

	gtk_widget_show( slider_vbox );


	/** OK/cancel buttons **/
	hbox = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);
	widget = gtk_button_new_with_label("OK");
	GTK_WIDGET_SET_FLAGS(widget, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(hbox), widget, TRUE, TRUE, 0);
	g_signal_connect(G_OBJECT(widget), "clicked",
				 G_CALLBACK(matedit_ok), (gpointer) med);
	gtk_widget_show(widget);
	gtk_widget_grab_default(widget);
	widget = gtk_button_new_with_label("Apply");
	GTK_WIDGET_SET_FLAGS(widget, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(hbox), widget, TRUE, TRUE, 0);
	g_signal_connect(G_OBJECT(widget), "clicked",
				 G_CALLBACK(matedit_apply), (gpointer) med);
	gtk_widget_show(widget);
	widget = gtk_button_new_with_label("Cancel");
	GTK_WIDGET_SET_FLAGS(widget, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(hbox), widget, TRUE, TRUE, 0);
	g_signal_connect(G_OBJECT(widget), "clicked",
				 G_CALLBACK(matedit_cancel), (gpointer) med);
	gtk_widget_show(widget);
	/* end of dialog box */


	/** Ancillary floating color picker **/
	med->MatPickerBox = gtk_color_selection_dialog_new("Select Color");
	g_signal_connect(G_OBJECT(med->MatPickerBox), "delete_event",
				 G_CALLBACK(matedit_colorpicker_close_2), (gpointer) med);
	g_signal_connect(
		G_OBJECT(GTK_COLOR_SELECTION_DIALOG(med->MatPickerBox)->colorsel),
		"color_changed", G_CALLBACK(matedit_colorpicker_changed_cb), (gpointer) med);
	g_signal_connect(
		G_OBJECT(GTK_COLOR_SELECTION_DIALOG(med->MatPickerBox)->ok_button),
		"clicked", G_CALLBACK(matedit_colorpicker_ok), (gpointer) med);
	g_signal_connect(
		G_OBJECT(GTK_COLOR_SELECTION_DIALOG(med->MatPickerBox)->cancel_button),
		"clicked", G_CALLBACK(matedit_colorpicker_close), (gpointer) med);

	return med;
}


GtkWidget *mat_edit_create_slider( 
	char *str, int index, MatEditData *med ) 
{
	GtkWidget *widget, *hbox;
	GtkAdjustment *adj;

	hbox = gtk_hbox_new(FALSE, 4);

	widget = gtk_label_new( str );
	gtk_box_pack_start( GTK_BOX(hbox), widget, TRUE, TRUE, 0 );
	gtk_widget_show(widget);

	/* ADJUSTMENT */
	if( index != MAT_INDEX_SHININESS ) {
		adj = (GtkAdjustment *)
			gtk_adjustment_new( 0.0, 0.0, 1.0, 0.01, 0.01, 0.01 );
	} else {
		adj = (GtkAdjustment *)
			gtk_adjustment_new( 0.0, 0.0, 128.0, 1.0, 1.0, 1.0 );
	}
	g_object_set_data(G_OBJECT(adj), "MatEditStruct", (gpointer) med);
	g_signal_connect(G_OBJECT(adj), "value_changed",
					 G_CALLBACK(matedit_slider_changed), (gpointer) index);
	med->SliderAdj[index] = adj;

	/* SCALE / SLIDER */
	widget = gtk_hscale_new(adj);
	if(med->cont_update)
		gtk_range_set_update_policy(GTK_RANGE(widget), GTK_UPDATE_CONTINUOUS);
	else
		gtk_range_set_update_policy(GTK_RANGE(widget), GTK_UPDATE_DISCONTINUOUS);
	gtk_range_set_inverted( GTK_RANGE(widget), index == MAT_INDEX_SHININESS );
	gtk_scale_set_draw_value(GTK_SCALE(widget), FALSE);
	gtk_scale_set_digits( GTK_SCALE(widget), 3 );
	gtk_box_pack_start( GTK_BOX(hbox), widget, FALSE, FALSE, 0 );
	gtk_widget_set_size_request( widget, 160, 20 );
	gtk_widget_show(widget);

	/* SPINBUTTON */
	widget = gtk_spin_button_new( adj, 0.1, 3 );
	gtk_box_pack_start( GTK_BOX(hbox), widget, FALSE, FALSE, 0 );
	gtk_widget_set_size_request( widget, 50, 20 );
	gtk_widget_show(widget);

	/* BUTTON & COLOR-SWATCH */
	if( index != MAT_INDEX_SHININESS ) {
		GdkPixbuf *pixbuf;
		
		widget = gtk_button_new();
		g_object_set_data(G_OBJECT(widget), "MatEditStruct", (gpointer) med);
		g_signal_connect(G_OBJECT(widget), "clicked",
						 G_CALLBACK(matedit_colorpicker_show), (gpointer) index);
		gtk_box_pack_start( GTK_BOX(hbox), widget, FALSE, FALSE, 0 );

		pixbuf = gdk_pixbuf_new( GDK_COLORSPACE_RGB, FALSE, 8, 32, 20 );
		gdk_pixbuf_fill( pixbuf, 0x000000ff );
		med->SwatchWidget[index] = gtk_image_new_from_pixbuf( pixbuf );
		med->SwatchPixbuf[index] = pixbuf;
		g_object_unref( pixbuf );

		gtk_container_add(GTK_CONTAINER(widget), med->SwatchWidget[index]);
		gtk_widget_show(med->SwatchWidget[index]);
		
		gtk_widget_show(widget);
	}
	
	gtk_widget_show_all( hbox );
	return hbox;
}


void matedit_show( Material *material )
{
	if( GlobalMatEdit == NULL ) {
		GlobalMatEdit = matedit_new();
		g_signal_connect( notificationObj, "notify::exit", 
			G_CALLBACK(matedit_cleanup), NULL );
	}

	/* if it is still NULL, something went wrong */
	if( GlobalMatEdit == NULL ) {
		fprintf( stderr, "%s : unable to create material editor\n", __FUNCTION__ );
		return;
	}

	matedit_set_material( GlobalMatEdit, material );

	gtk_widget_show(GlobalMatEdit->MainWin);
}


gboolean matedit_cleanup( gpointer data ) {
	if( GlobalMatEdit != NULL ) {
		free( GlobalMatEdit );
		GlobalMatEdit = NULL;
	}
	
	return FALSE;
}


void matedit_hide(MatEditData *med)
{
	g_return_if_fail(med != NULL);
	g_return_if_fail(med->MainWin != NULL);
	gtk_widget_hide(med->MatPickerBox);
	gtk_widget_hide(med->MainWin);
}


void matedit_set_material( MatEditData *med, Material *material ) {
	int i;
	float v;
	GtkAdjustment *adj;
	GdkColor gdkcolor;
	
	med->material = material;
	
	material_copy_to_array( med->material, med->bc );
	
	/* for all 4 elems of med->bc: normalize; use return value to set slider 
	value */
	for( i = 0; i < 4; i++ ) {
		v = matedit_normalize_attribute( med->bc[i], med->bc[i] );
		adj = med->SliderAdj[i];
		adj->value = v;
		g_signal_emit_by_name( G_OBJECT(adj), "value_changed" );
	}

	/* set gdkcolor to med->bc[MAT_INDEX_DIFFUSE] */
	matedit_floats_to_gdkcolor( &gdkcolor, med->bc[MAT_INDEX_DIFFUSE] );
	/* ... and update the color picker */
	gtk_color_selection_set_current_color( 
		GTK_COLOR_SELECTION(med->MatPicker),
		&gdkcolor );
	
	/* set shininess slider */
	adj = med->SliderAdj[MAT_INDEX_SHININESS];
	adj->value = material->shininess;
	g_signal_emit_by_name( G_OBJECT(adj), "value_changed" );
	
	matedit_recompute_base_intensities( med );
}


void matedit_update_color_swatch( 
	GdkPixbuf *pixbuf, GtkWidget *parent, GLfloat *color )
{
	guint32 fill_color = 0;
	guchar c[3];
	
	if( pixbuf == NULL || parent == NULL || color == NULL ) return;
	
//fprintf( stderr, "%s : color is r %3.3f g %3.3f b %3.3f\n", __FUNCTION__, color[0], color[1], color[2] );
	c[0] = (guchar) (255.0 * color[0]);
	c[1] = (guchar) (255.0 * color[1]);
	c[2] = (guchar) (255.0 * color[2]);
	
	fill_color = c[0] << 24 | c[1] << 16 | c[2] << 8 | 0xff << 0;
//fprintf( stderr, "%s : fill_color is %x\n", __FUNCTION__, (guint)fill_color );
	
	gdk_pixbuf_fill( pixbuf, fill_color );
	gtk_widget_queue_draw( parent );
}


void matedit_recompute_base_intensities(MatEditData *med)
{
	int i;
	GtkAdjustment *adj;

	for( i = 0; i < 3; i++ ) {
		adj = med->SliderAdj[i];
		if( !adj ) 
			med->bi[i] = 0.0;
		else
			med->bi[i] = adj->value;
	}

	matedit_normalize_attribute( med->bi, med->bi );
}


float matedit_normalize_attribute( float dest[4], float src[4] ) {
	/* This code was duplicated several times in the original mat_edit 
	(from me3d), so I consolidated it here.  If it turns out that this 
	normalization is unnecessary or incorrect, it will be easier to disable it 
	if it is all in one place (rather than scattered all over the file). */

	float max;

	max = src[0];
	if(src[1] > max)
		max = src[1];
	if(src[2] > max)
		max = src[2];

	if( fabs( max ) < MATEDIT_ZERO_THRESHOLD ) {
		dest[0] = dest[1] = dest[2] = MATEDIT_ZERO_THRESHOLD + 0.001;
	} else {
		dest[0] = src[0] / max;
		dest[1] = src[1] / max;
		dest[2] = src[2] / max;
	}
	dest[3] = src[3];
	
	return max;
}


void matedit_toggle_intensity_lock(GtkWidget *widget, MatEditData *med)
{
	if(GTK_TOGGLE_BUTTON(widget)->active)
		med->lock_intensity = TRUE;
	else
		med->lock_intensity = FALSE;
}


/* what to do when the integrated color picker changes */
void matedit_color_changed(GtkWidget *widget, MatEditData *med)
{
	float g_color[4], alpha, max;
	GLfloat m_color[4];
	GtkAdjustment *adj;
	GdkColor gdkcolor;
	guint16 alpha16;
	int i;

	if(med->freeze)
		return;

	gtk_color_selection_get_current_color( GTK_COLOR_SELECTION(med->MatPicker), 
		&gdkcolor );
	matedit_gdkcolor_to_floats( &gdkcolor, g_color );

	alpha16 = gtk_color_selection_get_current_alpha( GTK_COLOR_SELECTION(med->MatPicker) );
	alpha = ((float) alpha16) / 65535.0;

	g_color[3] = alpha;
	
/*
fprintf( stderr, "%s - ", __FUNCTION__ );
for(i=0;i<4;i++) {
fprintf( stderr, "%3.3f ", g_color[i] );
}
fprintf( stderr, "\n" );
*/

	max = matedit_normalize_attribute( g_color, g_color );

	med->bc[0][0] = g_color[0];
	med->bc[0][1] = g_color[1];
	med->bc[0][2] = g_color[2];
	med->bc[0][3] = alpha;
	med->bc[1][0] = g_color[0];
	med->bc[1][1] = g_color[1];
	med->bc[1][2] = g_color[2];
	med->bc[1][3] = 1.0;
	med->bc[2][0] = g_color[0];
	med->bc[2][1] = g_color[1];
	med->bc[2][2] = g_color[2];
	med->bc[2][3] = 1.0;

	med->freeze = TRUE;

	for( i = 0; i < 3; i++ ) {
		adj = med->SliderAdj[i];
		if(!med->lock_intensity) {
			adj->value = max * med->bi[i];
			g_signal_emit_by_name(G_OBJECT(adj),
						"value_changed");
		} else {
			m_color[0] = med->bc[i][0] * adj->value;
			m_color[1] = med->bc[i][1] * adj->value;
			m_color[2] = med->bc[i][2] * adj->value;
			m_color[3] = med->bc[i][3];
			matedit_update_color_swatch(
				med->SwatchPixbuf[i], med->SwatchWidget[i], m_color);
			matedit_set_preview_value(med, m_color, i);
		}
	}

	med->freeze = FALSE;
	
}


/* what to do when the intensity sliders/spinbuttons change */
void matedit_slider_changed(GtkAdjustment *adj, int index )
{
	GLfloat m_color[4];
	float v;
	MatEditData *med;

	med = (MatEditData *) g_object_get_data(G_OBJECT(adj), "MatEditStruct");

	v = (float) adj->value;
	if( index == MAT_INDEX_SHININESS ) {
		mat_preview_set_shininess( med->MatPreview, v );
	} else {
		m_color[0] = med->bc[index][0] * v;
		m_color[1] = med->bc[index][1] * v;
		m_color[2] = med->bc[index][2] * v;
		m_color[3] = med->bc[index][3];
		matedit_update_color_swatch(
			med->SwatchPixbuf[index], med->SwatchWidget[index], m_color);
		matedit_set_preview_value( med, m_color, index );
	}
	if(!med->freeze) {
		matedit_recompute_base_intensities( med );
	}
}


void matedit_set_preview_value(MatEditData *med, GLfloat *v, int i ) {

	switch( i ) {
		case MAT_INDEX_DIFFUSE:
			mat_preview_set_diffuse( med->MatPreview, v );
			break;
		case MAT_INDEX_AMBIENT:
			mat_preview_set_ambient( med->MatPreview, v );
			break;
		case MAT_INDEX_EMISSION:
			mat_preview_set_emission( med->MatPreview, v );
			break;
		case MAT_INDEX_SPECULAR:
			mat_preview_set_specular( med->MatPreview, v );
			break;
		case MAT_INDEX_SHININESS:
		default:
			break;
	}
	
}



/* functions dealing with the floating color picker */
void matedit_colorpicker_show(GtkWidget *widget, guint color_id)
{
	MatEditData *med;
	float g_color[4];
	GtkAdjustment *adj;
	GdkColor gdkcolor;

	med = (MatEditData *) g_object_get_data(G_OBJECT(widget),
								 "MatEditStruct");

	med->active_preview = color_id;

	adj = med->SliderAdj[color_id];

	med->pc[0] = g_color[0] = med->bc[color_id][0] * adj->value;
	med->pc[1] = g_color[1] = med->bc[color_id][1] * adj->value;
	med->pc[2] = g_color[2] = med->bc[color_id][2] * adj->value;
	med->pc[3] = g_color[3] = med->bc[color_id][3];
	
	matedit_floats_to_gdkcolor( &gdkcolor, g_color );
	
	gtk_color_selection_set_current_color( 
		GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(med->MatPickerBox)->colorsel),
		&gdkcolor );
	
	gtk_color_selection_set_current_alpha( 
		GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(med->MatPickerBox)->colorsel),
		(guint16)(g_color[3] * 65535.) );

	if(color_id == MAT_INDEX_DIFFUSE) {		/* Opacity is only setable on the diffuse color */
		gtk_color_selection_set_has_opacity_control(
			GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(med->MatPickerBox)->colorsel),
			TRUE);
	} else {
		gtk_color_selection_set_has_opacity_control(
			GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(med->MatPickerBox)->colorsel),
			FALSE);
	}
	gtk_widget_show(med->MatPickerBox);
}

void matedit_colorpicker_changed_cb(GtkWidget *widget, MatEditData *med)
{
	float g_color[4] = { 0.0, 0.0, 0.0, 1.0 }, max;
	float alpha = 1.0;
	GtkAdjustment *adj;
	GdkColor gdkcolor;
	guint16 alpha16;

	gtk_color_selection_get_current_color( 
		GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(med->MatPickerBox)->colorsel), 
		&gdkcolor );
	matedit_gdkcolor_to_floats( &gdkcolor, g_color );

	alpha16 = gtk_color_selection_get_current_alpha( 
		GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(med->MatPickerBox)->colorsel) );
	alpha = ((float) alpha16) / 65535.0;

	g_color[3] = alpha;

	matedit_update_color_swatch(
		med->SwatchPixbuf[med->active_preview], 
		med->SwatchWidget[med->active_preview], g_color);

	max = matedit_normalize_attribute( med->bc[med->active_preview], g_color );
	
	adj = med->SliderAdj[med->active_preview];
	if((!med->lock_intensity) || (med->active_preview == MAT_INDEX_SPECULAR))
		adj->value = max;
	g_signal_emit_by_name(G_OBJECT(adj), "value_changed");
}

void matedit_colorpicker_ok(GtkWidget *widget, MatEditData *med)
{
	gtk_widget_hide(med->MatPickerBox);
}

void matedit_colorpicker_close(GtkWidget *widget, MatEditData *med)
{
	GLfloat max;
	GtkAdjustment *adj;

	matedit_update_color_swatch(
		med->SwatchPixbuf[med->active_preview], 
		med->SwatchWidget[med->active_preview], med->pc);
	max = matedit_normalize_attribute( med->bc[med->active_preview], med->pc );

	adj = med->SliderAdj[med->active_preview];
	adj->value = (gfloat) max;
	g_signal_emit_by_name(G_OBJECT(adj), "value_changed");

	gtk_widget_hide(med->MatPickerBox);
}

void matedit_colorpicker_close_2(GtkWidget *widget, GdkEvent *event,
				 MatEditData *med)
{
	matedit_colorpicker_close(widget, med);
}

void matedit_kill_dialog(GtkWidget *widget, GdkEvent *event,
			 MatEditData *med)
{
	matedit_hide(med);
}


/* OK/Apply/Cancel handlers */
void matedit_ok(GtkWidget *widget, MatEditData *med)
{
	matedit_apply(widget, med);
	matedit_hide(med);
}

void matedit_apply(GtkWidget *widget, MatEditData *med)
{
	g_return_if_fail(med != NULL);
	
	/* copy the material data back into the med->material */
	matedit_get_color( med, MAT_INDEX_DIFFUSE, med->material->diffuse );
	matedit_get_color( med, MAT_INDEX_AMBIENT, med->material->ambient );
	matedit_get_color( med, MAT_INDEX_EMISSION, med->material->emission );
	matedit_get_color( med, MAT_INDEX_SPECULAR, med->material->specular );
	med->material->shininess = matedit_get_shininess( med );
	
}

void matedit_cancel(GtkWidget *widget, MatEditData *med)
{
	matedit_hide(med);
}



void matedit_gdkcolor_to_floats( GdkColor *gdkcolor, float *color ) {
	color[0] = ((float) gdkcolor->red) / 65535.0;
	color[1] = ((float) gdkcolor->green) / 65535.0;
	color[2] = ((float) gdkcolor->blue) / 65535.0;
}


void matedit_floats_to_gdkcolor( GdkColor *gdkcolor, float *color ) {
	gdkcolor->red =   (int)( color[0] * 65535.0 );
	gdkcolor->green = (int)( color[1] * 65535.0 );
	gdkcolor->blue =  (int)( color[2] * 65535.0 );
}


/* Retrieves color info from the material editor, in a form that is more 
generally useful (esp to other parts of kludge3d). */
void matedit_get_color( MatEditData *med, int index, float color[4] ) {
	float v;
	GtkAdjustment *adj;
	
	adj = med->SliderAdj[index];
	v = adj->value;
	
	color[0] = med->bc[index][0] * v;
	color[1] = med->bc[index][1] * v;
	color[2] = med->bc[index][2] * v;
	color[3] = med->bc[index][3];
}


/* Retrieves shininess info from the material editor. */
float matedit_get_shininess( MatEditData *med ) {

	return med->SliderAdj[MAT_INDEX_SHININESS]->value;
}






/***************************************************************************

DEAD CODE

***************************************************************************/


#if 0
void matedit_redraw_color_preview(GtkWidget *widget, GLfloat *color)
{
	gint x, y, i, wid, heig, f, n;
	guchar c[3], cc[3 * 2], *sample_buf;
	gdouble o;

	g_return_if_fail(widget != NULL);
	g_return_if_fail(color != NULL);

	wid = widget->requisition.width;
	heig = widget->requisition.height;

	g_return_if_fail(wid != 0);

	sample_buf = g_new(guchar, 3 * wid);

	c[0] = (guchar) (255.0 * color[0]);
	c[1] = (guchar) (255.0 * color[1]);
	c[2] = (guchar) (255.0 * color[2]);

	o = color[3];

	cc[0] = (guchar) ((1.0 - o) * 192 + (o * (gdouble) c[0]));
	cc[1] = (guchar) ((1.0 - o) * 192 + (o * (gdouble) c[1]));
	cc[2] = (guchar) ((1.0 - o) * 192 + (o * (gdouble) c[2]));
	cc[3] = (guchar) ((1.0 - o) * 128 + (o * (gdouble) c[0]));
	cc[4] = (guchar) ((1.0 - o) * 128 + (o * (gdouble) c[1]));
	cc[5] = (guchar) ((1.0 - o) * 128 + (o * (gdouble) c[2]));

	for (y = 0; y < heig; y++)
	{
		i = 0;
		for (x = 0; x < wid; x++)
		{
			f = 3 * (((x % 32) < 16) ^ ((y % 32) < 16));

			for (n = 0; n < 3; n++)
	sample_buf[i++] = cc[n + f];
		}

		gtk_preview_draw_row(GTK_PREVIEW(widget), sample_buf, 0, y, wid);
	}

	gtk_widget_draw(widget, NULL);
	g_free(sample_buf);
}


void matedit_drop_material(GtkWidget *widget, MatEditData *med)
{
	matedit_color_assimilate(med);
}

/* bonus copy function */
void preview_copy_material(GtkGLTexPreview *gtpv, gpointer data)
{
	GLfloat buf[4];
	GtkGLTexPreview *gtpv_dest = GTK_GL_TEX_PREVIEW(data);

	gtk_gl_tex_preview_get_diffuse(gtpv, buf);
	gtk_gl_tex_preview_set_diffuse(gtpv_dest, buf);

	gtk_gl_tex_preview_get_ambient(gtpv, buf);
	gtk_gl_tex_preview_set_ambient(gtpv_dest, buf);

	gtk_gl_tex_preview_get_specular(gtpv, buf);
	gtk_gl_tex_preview_set_specular(gtpv_dest, buf);

	gtk_gl_tex_preview_get_emission(gtpv, buf);
	gtk_gl_tex_preview_set_emission(gtpv_dest, buf);

	gtk_gl_tex_preview_get_shininess(gtpv, buf);
	gtk_gl_tex_preview_set_shininess(gtpv_dest, buf[0]);

	gtk_gl_tex_preview_redraw(gtpv_dest);
}


/* utility functions: color_assimilate, redraw_color_preview, recompute_bi */
void matedit_color_assimilate(MatEditData *med)
{
	/* integrates material data from the preview widget into the UI */
	GtkAdjustment *adj;
	gdouble g_color[4], max;
	GLfloat m_color[4];

	med->freeze = TRUE;

	adj = gtk_range_get_adjustment(GTK_RANGE(med->MatSlider[0]));
	gtk_gl_tex_preview_get_diffuse(GTK_GL_TEX_PREVIEW(med->MatPreview), m_color);

	g_color[0] = m_color[0];
	g_color[1] = m_color[1];
	g_color[2] = m_color[2];
	g_color[3] = m_color[3];
	gtk_color_selection_set_color(GTK_COLOR_SELECTION(med->MatPicker), g_color);

	max = m_color[0];
	if(m_color[1] > max)
		max = m_color[1];
	if(m_color[2] > max)
		max = m_color[2];
	if(max == 0)
		med->bc[0][0] = med->bc[0][1] = med->bc[0][2] = 1;
	else {
		med->bc[0][0] = m_color[0] / max;
		med->bc[0][1] = m_color[1] / max;
		med->bc[0][2] = m_color[2] / max;
	}
	med->bc[0][3] = m_color[3];
	adj->value = max;
	g_signal_emit_by_name(G_OBJECT(adj),
				"value_changed");

	adj = gtk_range_get_adjustment(GTK_RANGE(med->MatSlider[1]));
	gtk_gl_tex_preview_get_ambient(GTK_GL_TEX_PREVIEW(med->MatPreview), m_color);
	max = m_color[0];
	if(m_color[1] > max)
		max = m_color[1];
	if(m_color[2] > max)
		max = m_color[2];
	if(max == 0)
		med->bc[1][0] = med->bc[1][1] = med->bc[1][2] = 1;
	else {
		med->bc[1][0] = m_color[0] / max;
		med->bc[1][1] = m_color[1] / max;
		med->bc[1][2] = m_color[2] / max;
	}
	med->bc[1][3] = 1.0;
	adj->value = max;
	g_signal_emit_by_name(G_OBJECT(adj),
				"value_changed");

	adj = gtk_range_get_adjustment(GTK_RANGE(med->MatSlider[2]));
	gtk_gl_tex_preview_get_emission(GTK_GL_TEX_PREVIEW(med->MatPreview), m_color);
	max = m_color[0];
	if(m_color[1] > max)
		max = m_color[1];
	if(m_color[2] > max)
		max = m_color[2];
	if(max == 0)
		med->bc[2][0] = med->bc[2][1] = med->bc[2][2] = 1;
	else {
		med->bc[2][0] = m_color[0] / max;
		med->bc[2][1] = m_color[1] / max;
		med->bc[2][2] = m_color[2] / max;
	}
	med->bc[2][3] = 1.0;
	adj->value = max;
	g_signal_emit_by_name(G_OBJECT(adj),
				"value_changed");

	adj = gtk_range_get_adjustment(GTK_RANGE(med->MatSlider[3]));
	gtk_gl_tex_preview_get_specular(GTK_GL_TEX_PREVIEW(med->MatPreview), m_color);
	max = m_color[0];
	if(m_color[1] > max)
		max = m_color[1];
	if(m_color[2] > max)
		max = m_color[2];
	if(max == 0)
		med->bc[3][0] = med->bc[3][1] = med->bc[3][2] = 1;
	else {
		med->bc[3][0] = m_color[0] / max;
		med->bc[3][1] = m_color[1] / max;
		med->bc[3][2] = m_color[2] / max;
	}
	med->bc[3][3] = 1.0;
	adj->value = max;
	g_signal_emit_by_name(G_OBJECT(adj),
				"value_changed");

	adj = gtk_range_get_adjustment(GTK_RANGE(med->MatSlider[4]));
	gtk_gl_tex_preview_get_shininess(GTK_GL_TEX_PREVIEW(med->MatPreview), m_color);
	adj->value = 1 - (m_color[0] / 128);		/* 1.0 is shiniest, 0.0 is least */
	g_signal_emit_by_name(G_OBJECT(adj),
				"value_changed");

	med->freeze = FALSE;
	matedit_recompute_base_intensities(med);
}

#endif
