/*
 * File:        mat_preview.c
 * 
 * Description:	displays a preview of the material attributes passed to it
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */

/* Note: this file is based on:
 * simple.c, written by Naofumi Yasufuku	<naofumi@users.sourceforge.net>
 * ...which is one of the example programs packaged with gtkglext.
 * I grabbed the checkered-background interleaved array (backer_square) from 
 * me3d's gtk_gl_tex_preview.
 * 
 * This widget is intended as a simple replacement for me3d's 
 * gtk_gl_tex_preview.  The ultimate goal is to make it easier to port me3d's 
 * mat_edit to kludge3d.
 * 
 * Note: Unlike me3d's gtk_gl_tex_preview, this widget is for output only.
 * You can not *retrieve* the material attributes from the widget; you can 
 * only *set* the attributes.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>
#include <gtk/gtkgl.h>
#ifdef G_OS_WIN32
#define WIN32_LEAN_AND_MEAN 1
#include <windows.h>
#endif
#include <GL/gl.h>
#include <GL/glu.h>

#include "mat_preview.h"
#include "material.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


#define MAT_PREVIEW_SPHERE_DISPLIST 1
#define MAT_PREVIEW_DRAW_CHECKERED_BACKGROUND 1


static GLfloat backer_square[] = {
  0.50, 0.50, 0.50, 1.0,  0.0, 0.0, -1.0,  -100.0,  100.0, 0.0,
  0.50, 0.50, 0.50, 1.0,  0.0, 0.0, -1.0,  -100.0,    0.0, 0.0,
  0.50, 0.50, 0.50, 1.0,  0.0, 0.0, -1.0,     0.0,    0.0, 0.0,
  0.50, 0.50, 0.50, 1.0,  0.0, 0.0, -1.0,     0.0,  100.0, 0.0,
  0.75, 0.75, 0.75, 1.0,  0.0, 0.0, -1.0,     0.0,  100.0, 0.0,
  0.75, 0.75, 0.75, 1.0,  0.0, 0.0, -1.0,     0.0,    0.0, 0.0,
  0.75, 0.75, 0.75, 1.0,  0.0, 0.0, -1.0,   100.0,    0.0, 0.0,
  0.75, 0.75, 0.75, 1.0,  0.0, 0.0, -1.0,   100.0,  100.0, 0.0,
  0.75, 0.75, 0.75, 1.0,  0.0, 0.0, -1.0,  -100.0,    0.0, 0.0,
  0.75, 0.75, 0.75, 1.0,  0.0, 0.0, -1.0,  -100.0, -100.0, 0.0,
  0.75, 0.75, 0.75, 1.0,  0.0, 0.0, -1.0,     0.0, -100.0, 0.0,
  0.75, 0.75, 0.75, 1.0,  0.0, 0.0, -1.0,     0.0,    0.0, 0.0,
  0.50, 0.50, 0.50, 1.0,  0.0, 0.0, -1.0,     0.0,    0.0, 0.0,
  0.50, 0.50, 0.50, 1.0,  0.0, 0.0, -1.0,     0.0, -100.0, 0.0,
  0.50, 0.50, 0.50, 1.0,  0.0, 0.0, -1.0,   100.0, -100.0, 0.0,
  0.50, 0.50, 0.50, 1.0,  0.0, 0.0, -1.0,   100.0,    0.0, 0.0,
};		/* C4F_N3F_V3F with CCW polygons */



gboolean matpreview_free_attr( 
	GtkWidget *widget, gpointer data );


static void
matpreview_realize( GtkWidget *widget, gpointer data )
{
	GdkGLContext *glcontext = gtk_widget_get_gl_context (widget);
	GdkGLDrawable *gldrawable = gtk_widget_get_gl_drawable (widget);

	GLUquadricObj *qobj;
	static GLfloat light_diffuse[] = {1.0, 1.0, 1.0, 1.0};
	static GLfloat light_position[] = {1.0, 1.0, 1.0, 0.0};
	GLfloat aspect = 1.0;
	int width, height;

	/*** OpenGL BEGIN ***/
	if (!gdk_gl_drawable_gl_begin (gldrawable, glcontext))
		return;

	width = widget->allocation.width;
	height = widget->allocation.height;

	qobj = gluNewQuadric ();
	gluQuadricDrawStyle (qobj, GLU_FILL);
	glNewList (MAT_PREVIEW_SPHERE_DISPLIST, GL_COMPILE);
	gluSphere (qobj, 1.0, 20, 20);
	glEndList ();

	glLightfv (GL_LIGHT0, GL_DIFFUSE, light_diffuse);
	glLightfv (GL_LIGHT0, GL_POSITION, light_position);
	glEnable (GL_LIGHTING);
	glEnable (GL_LIGHT0);

	glEnable (GL_DEPTH_TEST);
	
	glClearColor (0.50, 0.50, 0.50, 1.0);
	glClearDepth (1.0);

	glViewport (0, 0, width, height);
	aspect = (GLfloat)width / (GLfloat)height;

	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();
	gluPerspective (40.0, (GLdouble)aspect, 1.0, 10.0);

	glMatrixMode (GL_MODELVIEW);
	glLoadIdentity ();
	gluLookAt( 0.0, 0.0, 3.0,
				0.0, 0.0, 0.0,
				0.0, 1.0, 0.0 );
	glTranslatef (0.0, 0.0, -2.0);

	gdk_gl_drawable_gl_end (gldrawable);
	/*** OpenGL END ***/
}

static gboolean
matpreview_configure_event( 
	GtkWidget *widget, GdkEventConfigure *event, gpointer data )
{
	GdkGLContext *glcontext = gtk_widget_get_gl_context (widget);
	GdkGLDrawable *gldrawable = gtk_widget_get_gl_drawable (widget);

	/*** OpenGL BEGIN ***/
	if (!gdk_gl_drawable_gl_begin (gldrawable, glcontext))
		return FALSE;

	glViewport (0, 0,
		widget->allocation.width, widget->allocation.height);

	gdk_gl_drawable_gl_end (gldrawable);
	/*** OpenGL END ***/

	return TRUE;
}

static gboolean
matpreview_expose_event( 
	GtkWidget *widget, GdkEventExpose *event, gpointer data )
{
	GdkGLContext *glcontext = gtk_widget_get_gl_context (widget);
	GdkGLDrawable *gldrawable = gtk_widget_get_gl_drawable (widget);
	Material *mp_attr;

	/*** OpenGL BEGIN ***/
	if (!gdk_gl_drawable_gl_begin (gldrawable, glcontext))
		return FALSE;

#if MAT_PREVIEW_DRAW_CHECKERED_BACKGROUND
	glClear (GL_DEPTH_BUFFER_BIT);
	glDisable(GL_LIGHTING);
	glInterleavedArrays(GL_C4F_N3F_V3F, 0, backer_square);
	glDrawArrays(GL_QUADS, 0, 16);
	glEnable(GL_LIGHTING);
#else
	glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
#endif

	mp_attr = (Material *)
		g_object_get_data( G_OBJECT( widget ), "user-data" );	

    glMaterialfv(GL_FRONT, GL_AMBIENT, mp_attr->ambient);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, mp_attr->diffuse);
    glMaterialfv(GL_FRONT, GL_SPECULAR, mp_attr->specular);
    glMaterialfv(GL_FRONT, GL_EMISSION, mp_attr->emission);
    glMaterialf(GL_FRONT, GL_SHININESS, mp_attr->shininess);

	glCallList ( MAT_PREVIEW_SPHERE_DISPLIST );

	if (gdk_gl_drawable_is_double_buffered (gldrawable))
		gdk_gl_drawable_swap_buffers (gldrawable);
	else
		glFlush ();

	gdk_gl_drawable_gl_end (gldrawable);
	/*** OpenGL END ***/

	return TRUE;
}


GtkWidget *mat_preview_new( void )
{
	GdkGLConfig *glconfig;
	GtkWidget *drawing_area;
	Material *mp_attr;
	
	/*
	 * Configure OpenGL-capable visual.
	 */

	/* Try double-buffered visual */
	glconfig = gdk_gl_config_new_by_mode (
		GDK_GL_MODE_RGB		|
		GDK_GL_MODE_DEPTH	|
		GDK_GL_MODE_DOUBLE);

	if (glconfig == NULL) {
		fprintf( stderr, "*** Cannot find the double-buffered visual.\n");
		fprintf( stderr, "*** Trying single-buffered visual.\n");

		/* Try single-buffered visual */
		glconfig = gdk_gl_config_new_by_mode (
			GDK_GL_MODE_RGB	 |
			GDK_GL_MODE_DEPTH );
		if (glconfig == NULL) {
			fprintf( stderr, "*** No appropriate OpenGL-capable visual found.\n" );
			return NULL;
		}
	}

	/*
	 * Drawing area for drawing OpenGL scene.
	 */

	drawing_area = gtk_drawing_area_new ();
	gtk_widget_set_size_request (drawing_area, 200, 200);

	/* Set OpenGL-capability to the widget. */
	gtk_widget_set_gl_capability (drawing_area,
		glconfig,
		NULL,
		TRUE,
		GDK_GL_RGBA_TYPE);
	
	mp_attr = material_new();
	
	g_object_set_data( G_OBJECT( drawing_area ), "user-data", mp_attr );
	g_signal_connect (G_OBJECT (drawing_area), "destroy",
				G_CALLBACK (matpreview_free_attr), mp_attr );
	
	g_signal_connect_after (G_OBJECT (drawing_area), "realize",
				G_CALLBACK (matpreview_realize), NULL);
	g_signal_connect (G_OBJECT (drawing_area), "configure_event",
				G_CALLBACK (matpreview_configure_event), NULL);
	g_signal_connect (G_OBJECT (drawing_area), "expose_event",
				G_CALLBACK (matpreview_expose_event), NULL);

	gtk_widget_show (drawing_area);

	return drawing_area;
}


void mat_preview_set_all( GtkWidget *widget, Material *material ) {
	Material *mp_attr;
	
	mp_attr = (Material *)
		g_object_get_data( G_OBJECT( widget ), "user-data" );	
	if( mp_attr == NULL )
		return;

	material_copy( mp_attr, material );
	
	gtk_widget_queue_draw( widget );
}


void mat_preview_set_ambient( GtkWidget *widget, float *ambient ) {
	Material *mp_attr;
	
	mp_attr = (Material *)
		g_object_get_data( G_OBJECT( widget ), "user-data" );	
	if( mp_attr == NULL )
		return;

	mp_attr->ambient[0] = ambient[0];
	mp_attr->ambient[1] = ambient[1];
	mp_attr->ambient[2] = ambient[2];
	mp_attr->ambient[3] = ambient[3];

	gtk_widget_queue_draw( widget );
}


void mat_preview_set_diffuse( GtkWidget *widget, float *diffuse ) {
	Material *mp_attr;
	
	mp_attr = (Material *)
		g_object_get_data( G_OBJECT( widget ), "user-data" );	
	if( mp_attr == NULL )
		return;
	
	mp_attr->diffuse[0] = diffuse[0];
	mp_attr->diffuse[1] = diffuse[1];
	mp_attr->diffuse[2] = diffuse[2];
	mp_attr->diffuse[3] = diffuse[3];

	gtk_widget_queue_draw( widget );
}


void mat_preview_set_specular( GtkWidget *widget, float *specular ) {
	Material *mp_attr;
	
	mp_attr = (Material *)
		g_object_get_data( G_OBJECT( widget ), "user-data" );	
	if( mp_attr == NULL )
		return;
	
	mp_attr->specular[0] = specular[0];
	mp_attr->specular[1] = specular[1];
	mp_attr->specular[2] = specular[2];
	mp_attr->specular[3] = specular[3];

	gtk_widget_queue_draw( widget );
}


void mat_preview_set_emission( GtkWidget *widget, float *emission ) {
	Material *mp_attr;
	
	mp_attr = (Material *)
		g_object_get_data( G_OBJECT( widget ), "user-data" );	
	if( mp_attr == NULL )
		return;
	
	mp_attr->emission[0] = emission[0];
	mp_attr->emission[1] = emission[1];
	mp_attr->emission[2] = emission[2];
	mp_attr->emission[3] = emission[3];

	gtk_widget_queue_draw( widget );
}


void mat_preview_set_shininess( GtkWidget *widget, float shininess ) {
	Material *mp_attr;
	
	mp_attr = (Material *)
		g_object_get_data( G_OBJECT( widget ), "user-data" );	
	if( mp_attr == NULL )
		return;

	mp_attr->shininess = shininess;

	gtk_widget_queue_draw( widget );
}


gboolean matpreview_free_attr( 
	GtkWidget *widget, gpointer data ) 
{
	free( data );
	return FALSE;
}


/**************************************************************************

DEAD CODE

***************************************************************************/

#if 0

static void
print_gl_config_attrib (GdkGLConfig *glconfig,
		const gchar *attrib_str,
		int					attrib,
		gboolean		 is_boolean)
{
	int value;

	g_print ("%s = ", attrib_str);
	if (gdk_gl_config_get_attrib (glconfig, attrib, &value))
		{
			if (is_boolean)
				g_print ("%s\n", value == TRUE ? "TRUE" : "FALSE");
			else
				g_print ("%d\n", value);
		}
	else
		g_print ("*** Cannot get %s attribute value\n", attrib_str);
}

static void
examine_gl_config_attrib (GdkGLConfig *glconfig)
{
	g_print ("\nOpenGL visual configurations :\n\n");

	g_print ("gdk_gl_config_is_rgba (glconfig) = %s\n",
					 gdk_gl_config_is_rgba (glconfig) ? "TRUE" : "FALSE");
	g_print ("gdk_gl_config_is_double_buffered (glconfig) = %s\n",
					 gdk_gl_config_is_double_buffered (glconfig) ? "TRUE" : "FALSE");
	g_print ("gdk_gl_config_is_stereo (glconfig) = %s\n",
					 gdk_gl_config_is_stereo (glconfig) ? "TRUE" : "FALSE");
	g_print ("gdk_gl_config_has_alpha (glconfig) = %s\n",
					 gdk_gl_config_has_alpha (glconfig) ? "TRUE" : "FALSE");
	g_print ("gdk_gl_config_has_depth_buffer (glconfig) = %s\n",
					 gdk_gl_config_has_depth_buffer (glconfig) ? "TRUE" : "FALSE");
	g_print ("gdk_gl_config_has_stencil_buffer (glconfig) = %s\n",
					 gdk_gl_config_has_stencil_buffer (glconfig) ? "TRUE" : "FALSE");
	g_print ("gdk_gl_config_has_accum_buffer (glconfig) = %s\n",
					 gdk_gl_config_has_accum_buffer (glconfig) ? "TRUE" : "FALSE");

	g_print ("\n");

	print_gl_config_attrib (glconfig, "GDK_GL_USE_GL",					 GDK_GL_USE_GL,					 TRUE);
	print_gl_config_attrib (glconfig, "GDK_GL_BUFFER_SIZE",			GDK_GL_BUFFER_SIZE,			FALSE);
	print_gl_config_attrib (glconfig, "GDK_GL_LEVEL",						GDK_GL_LEVEL,						FALSE);
	print_gl_config_attrib (glconfig, "GDK_GL_RGBA",						 GDK_GL_RGBA,						 TRUE);
	print_gl_config_attrib (glconfig, "GDK_GL_DOUBLEBUFFER",		 GDK_GL_DOUBLEBUFFER,		 TRUE);
	print_gl_config_attrib (glconfig, "GDK_GL_STEREO",					 GDK_GL_STEREO,					 TRUE);
	print_gl_config_attrib (glconfig, "GDK_GL_AUX_BUFFERS",			GDK_GL_AUX_BUFFERS,			FALSE);
	print_gl_config_attrib (glconfig, "GDK_GL_RED_SIZE",				 GDK_GL_RED_SIZE,				 FALSE);
	print_gl_config_attrib (glconfig, "GDK_GL_GREEN_SIZE",			 GDK_GL_GREEN_SIZE,			 FALSE);
	print_gl_config_attrib (glconfig, "GDK_GL_BLUE_SIZE",				GDK_GL_BLUE_SIZE,				FALSE);
	print_gl_config_attrib (glconfig, "GDK_GL_ALPHA_SIZE",			 GDK_GL_ALPHA_SIZE,			 FALSE);
	print_gl_config_attrib (glconfig, "GDK_GL_DEPTH_SIZE",			 GDK_GL_DEPTH_SIZE,			 FALSE);
	print_gl_config_attrib (glconfig, "GDK_GL_STENCIL_SIZE",		 GDK_GL_STENCIL_SIZE,		 FALSE);
	print_gl_config_attrib (glconfig, "GDK_GL_ACCUM_RED_SIZE",	 GDK_GL_ACCUM_RED_SIZE,	 FALSE);
	print_gl_config_attrib (glconfig, "GDK_GL_ACCUM_GREEN_SIZE", GDK_GL_ACCUM_GREEN_SIZE, FALSE);
	print_gl_config_attrib (glconfig, "GDK_GL_ACCUM_BLUE_SIZE",	GDK_GL_ACCUM_BLUE_SIZE,	FALSE);
	print_gl_config_attrib (glconfig, "GDK_GL_ACCUM_ALPHA_SIZE", GDK_GL_ACCUM_ALPHA_SIZE, FALSE);

	g_print ("\n");
}


#endif






