/* 
 * File:         mesh.c
 * 
 * Description:  funcs for mesh manipulation
 * 
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "mesh.h"
#include "vertex.h"
#include "polygon.h"
#include "undo.h"
#include "primitive.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


/* PROTOTYPES ************************************************************/



/* FUNCS *****************************************************************/

Mesh *mesh_new( Model *model ) {
    Mesh * m;
    m = ( Mesh * ) malloc( sizeof( Mesh ) );
    memset( m, 0, sizeof( Mesh ) );
    
    m->vertices = NULL;
    m->polygons = NULL;
    m->texture = NULL;
    m->texrep_x = 1.0;
    m->texrep_y = 1.0;

    m->name = NULL;
    
    m->trans = NULL;
/*    m->location = NULL;*/

    m->list = 0;
    m->dirty = 1;
	
	m->model = model;

    mesh_set_name( m, "NewMesh" );
    
    action_do( m->model, ACTION_MESH_ADD, NULL, NULL, NULL, NULL, NULL, NULL );
    return m;
}


Mesh *mesh_dup( Mesh *m ) {

    Mesh *copy;
    GSList *l;

    if( m == NULL )
        return NULL;
    
    copy = mesh_new( m->model );

    /* copy the vertices */
    for( l = m->vertices; l; l = l->next ) {
/*FIXME - this will screw up the undo stuff */
        copy->vertices = g_slist_append( copy->vertices, vertex_dup( l->data ) );
    }
    
    /* copy the polygons */
    for( l = m->polygons; l; l = l->next ) {
/*FIXME - this will screw up the undo stuff */
        int i, n;
        Poly *new_poly;
        Poly *p = (Poly*)l->data;
        
        new_poly = poly_dup( p );
        
        /* point the new_poly's vertices to the copy's, not m's */
        for( i = 0; i < new_poly->num_verts; i++ ) {
            n = g_slist_index( m->vertices, p->verts[i] );
            new_poly->verts[i] = g_slist_nth_data( copy->vertices, n );
        }
        
        /* set the poly's mesh */
        new_poly->mesh = copy;
        
        copy->polygons = g_slist_append( copy->polygons, new_poly );
    }
    
    copy->texture = m->texture;
    copy->texrep_x = m->texrep_x;
    copy->texrep_y = m->texrep_y;

    mesh_set_name( copy, m->name );

    /*fixme - copy trans and location?*/
    
    copy->list = 0; /* what is list used for? */
    copy->dirty = 1;

    // FIXME - we'll have to rewrite this function... break it into 
    // add-vert, add-poly actions
    action_do( copy->model, ACTION_UNKNOWN, NULL, NULL, NULL, NULL, NULL, NULL );
    
    return copy;
}


void mesh_delete( Mesh *m ) {

    if( m == NULL ) return;
    
    if( m->trans != NULL ) {
        free( m->trans );
        m->trans = NULL;
    }
/*    if( m->location != NULL ) {
        free( m->location );
        m->location = NULL;
    }*/

    if( m->polygons != NULL ) {
        GSList * l = m->polygons;
    
        for( ; l != NULL; l = m->polygons ) {
            mesh_polygon_remove( m, (Poly *)l->data );
        }

        g_slist_free( m->polygons );
        m->polygons = NULL;
    }
    if( m->vertices != NULL ) {
        g_slist_free( m->vertices );
        m->vertices = NULL;
    }

    if( m->name != NULL ) {
        free( m->name );
        m->name = NULL;
    }

    m->texture = NULL;
    m->list = 0;
    m->dirty = 1;

    action_do( m->model, ACTION_MESH_DEL, NULL, NULL, NULL, NULL, NULL, NULL );
    free( m );
}


/* Takes all the stuff in m2 and moves it to m1.  m2 is destroyed. */
void mesh_merge( Mesh *m1, Mesh *m2 ) {

    GSList *l;
    
    if( m1 == NULL || m2 == NULL )
        return;
    
    /* transfer the vertices */
    m1->vertices = g_slist_concat( m1->vertices, m2->vertices );
    
    /* transfer the polygons */
    m1->polygons = g_slist_concat( m1->polygons, m2->polygons );
    
    m2->vertices = NULL;
    m2->polygons = NULL;
    
    mesh_delete( m2 );
    
    /* now, re-set "mesh" in m2's polys and verts */
//    for( l = m1->vertices; l; l = l->next ) {
//        ((Vertex*)l->data)->mesh = m1;
//    }
    for( l = m1->polygons; l; l = l->next ) {
        ((Poly*)l->data)->mesh = m1;
    }
    
    // FIXME - we'll have to rewrite this function... break it into 
    // add-vert, remove-vert, reparent-poly actions
    action_do( m1->model, ACTION_UNKNOWN, NULL, NULL, NULL, NULL, NULL, NULL );
}


void mesh_set_transform( Mesh *m, Trans *new_trans ) {
    if( m != NULL && new_trans != NULL )
        m->trans = new_trans;
}

#if 0
void mesh_set_location( Mesh *m, Vertex* loc ) {
    if( m != NULL && loc != NULL )
        m->location = loc;
}
#endif


void mesh_set_texture( Mesh *m, Texture *nt ) {

    // a mesh's texture *can* be set to NULL...

    if( m != NULL )
        m->texture = nt;
}


void mesh_set_texrep( Mesh *m, float x, float y ) {
    
    if( m != NULL ) {
        m->texrep_x = x;
        m->texrep_y = y;
    }
}


void mesh_set_name( Mesh *m, char *name ) {
    
    if( m != NULL && name != NULL ) {
        if( m->name != NULL )
            free( m->name );
        m->name = strdup( name );
    }
    
}


void mesh_vertex_add( Mesh *m, Vertex *v ) {

	if( m == NULL || v == NULL )
		return;
	if( mesh_has_vertex( m, v ) )
		return;

	m->vertices = g_slist_append( m->vertices, ( gpointer ) v );
	action_do( m->model, ACTION_VERTEX_ADD_TO_MESH, NULL, v, m, NULL, NULL, NULL );
	primitive_ref( (Primitive*) v );

}


/* returns a boolean indicating success */
int mesh_vertex_remove( Mesh *m, Vertex *v ) {

    GSList * cl;
    Poly *p;

    if( m == NULL || m->vertices == NULL ) 
        return FALSE;
    if( v == NULL ) 
        return FALSE;

    cl = g_slist_find( m->vertices, ( gpointer ) v );
    if( cl == NULL )
        return FALSE;

	/* delete any of the polygons that make use of this vert... */
	/* not using a 'for' loop b/c we're modifying the list as we traverse it */
	cl = m->polygons;
	while( cl != NULL ) {
		p = ( Poly * ) cl->data;
		cl = cl->next;
		if( poly_has_vertex( p, v ) ) {
			mesh_polygon_remove( m, p );
		}
	}

    m->vertices = g_slist_remove( m->vertices, v );
    action_do( m->model, ACTION_VERTEX_REMOVE_MESH, NULL, v, m, NULL, NULL, NULL );
	primitive_unref( (Primitive*) v );
	
	return TRUE;
}


void mesh_polygon_add( Mesh *m, Poly *p ) {

    if( m != NULL && p != NULL ) {

		mesh_polygon_add_verts_to_mesh( m, p );

        /* fixme - should update the normal every time the vertices 
           in the poly are moved, but that's not exactly efficient...
         */
        poly_find_normal( p );
        
        m->polygons = g_slist_append( m->polygons, ( gpointer ) p );
        p->mesh = m;

        action_do( m->model, ACTION_POLY_ADD_TO_MESH, NULL, p, m, NULL, NULL, NULL );
		primitive_ref( (Primitive*) p );
    }

}


/* returns a boolean indicating success */
int mesh_polygon_remove( Mesh *m, Poly *p ) {

    GSList * cl;

    if( m == NULL || m->polygons == NULL )
        return FALSE;
    if( p == NULL )
        return FALSE;

    cl = g_slist_find( m->polygons, ( gpointer ) p );
    if( cl == NULL )
        return FALSE;

	poly_remove( p );

    m->polygons = g_slist_remove( m->polygons, cl->data );
    p->mesh = NULL;

    action_do( m->model, ACTION_POLY_REMOVE_MESH, NULL, p, m, NULL, NULL, NULL );
	primitive_unref( (Primitive*) p );

    return TRUE;
}


void mesh_polygon_reparent( Poly *p, Mesh *m ) {

    GSList * cl;

	if( m == NULL )
		return;
	if( p == NULL )
		return;
	if( p->mesh == m )
		return;
	if( p->mesh == NULL ) {
		fprintf( stderr, "warning: in %s - poly has NULL mesh\n", __FUNCTION__ );
		return;
	}

	/* Note - can't use mesh_polygon_remove b/c it calls poly_remove.
	Can't use mesh_polygon_add b/c it calls primitive_ref. */
	
	/* step one: remove poly from the old mesh */
	cl = g_slist_find( p->mesh->polygons, ( gpointer ) p );
	if( cl != NULL ) {
		p->mesh->polygons = g_slist_remove( p->mesh->polygons, cl->data );
	}
	p->mesh = NULL;

	/* step two: add poly to the new mesh */
	mesh_polygon_add_verts_to_mesh( m, p );
	m->polygons = g_slist_append( m->polygons, ( gpointer ) p );
	p->mesh = m;

	action_do( m->model, ACTION_UNKNOWN, NULL, NULL, NULL, NULL, NULL, NULL );
}


void mesh_polygon_add_verts_to_mesh( Mesh *m, Poly *p ) {
	int i;

	if( m == NULL )
		return;
	if( p == NULL )
		return;

	/* for each of the vertices in this polygon, 
	   make sure it actually exists in mesh m */
	for( i = 0; i < p->num_verts; i++ ) {
		Vertex* v = NULL;

		v = p->verts[i];
		if( v == NULL )
			return;
			
		/* if the mesh doesn't contain this particular vert... */
		if( ! mesh_has_vertex( m, v ) ) {
			/* ...add it */
			mesh_vertex_add( m, v );
		}
	}
}



int mesh_is_using_vertex( Mesh *m, Vertex *v ) {
	/* check all polys to see if v is being used by any of them */
	
	GSList *l;

	if( m == NULL )
		return FALSE;
	if( v == NULL )
		return FALSE;
	
	for( l = m->polygons; l; l = l->next ) {
		if( poly_has_vertex( (Poly*)l->data, v ) )
			return TRUE;
	}
	
	return FALSE;
}


int mesh_has_vertex( Mesh *m, Vertex *v ) {

	GSList *l;

	if( m == NULL )
		return FALSE;
	if( v == NULL )
		return FALSE;
	
	/* check to see if the mesh already has this vert */
	l = g_slist_find( m->vertices, v );

	return( l != NULL );

}

/*

DEAD CODE

*/


/* Old way of doing things - back when verts were stored by mesh */
#if 0
                // if the mesh doesn't contain this particular (non-null) 
                // vertex, then let's see if it contains one nearby
                
                l = g_slist_find_custom( m->vertices, v, 
                                        (GCompareFunc) vertices_isNotEqual );

                if( l == NULL ) {
                    // if we still can't find an exact or a close vertex,
                    // we should make a copy and add that to our mesh
                    v = vertex_dup( v );
                    mesh_vertex_add( m, v );
                } else {
                    // if we've found a vertex in this mesh that's nearby...
                    v = (Vertex*)l->data;
                }
                p->verts[i] = v;

/* New way of doing things - now verts are stored in a master list... meshes 
only have pointers */
#endif
