/* 
 * File:         model.c
 * 
 * Description:  handles model-related stuff
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 


#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>

#include "transform.h"
#include "globals.h"
#include "mesh.h"
#include "model.h"
#include "group.h"
#include "polygon.h"
#include "vertex.h"
#include "texture.h"
#include "undo.h"
#include "selection.h"
#include "primitive.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif



/* GLOBALS ***************************************************************/

Model *the_model = NULL;


/* PROTOTYPES ************************************************************/

void model_vertex_remove_all( Model * ) ;
gboolean model_vertex_remove_tree( GNode *node, gpointer data ) ;
gboolean model_polygon_remove_tree( GNode *node, gpointer data ) ;
gboolean model_remove_texture_tree( GNode *node, gpointer data ) ;


/* MODEL STUFF ***********************************************************/

Model * model_new( void ) {
    Model *m;

	m = (Model *)malloc( sizeof(Model) );
    memset( m, 0, sizeof( Model ) );
	
	undo_disable( m );

	m->root = group_new( m );
    mesh_set_name( (Mesh*)m->root->data, TOP_GROUP_NAME );
    
    m->current_node = group_new( m );
    g_node_append( m->root, m->current_node );
    m->current_mesh = (Mesh*)m->current_node->data;
    mesh_set_name( m->current_mesh, "Default Group" );

	model_set_name( m, "Untitled" );

	return m;
}

void model_delete( Model *model ) {

	if( model == NULL ) return;
	
	action_clear_stack( model );
	undo_disable(model);

    if( model->name ) free( model->name );
    if( model->fname ) free( model->fname );
	
	model->current_mesh = NULL;
    model->current_node = NULL;
    
    sel_vert_unselect_all( model );
    sel_poly_unselect_all( model );
    sel_vert_show_all( model );
    sel_poly_show_all( model );
    
    model_vertex_remove_all( model );
    model->verts = NULL; // just in case
    
    model_remove_all_textures( model );
    model->textures = NULL; // just in case
    
    group_delete( model->root );
    model->root = NULL;

    memset( model, 0, sizeof( Model ) );
	free( model );

}


gboolean model_set_currents( void ) {
	return model_set_currents_m( the_model );
}


gboolean model_set_currents_m( Model *model ) {
	if( model && model->root && model->root->children ) {
#ifdef VERBOSE
printf("model, root, and children are ok\n");
#endif
		if( (Mesh*)model->root->children->data ) {
#ifdef VERBOSE
printf("children's data is ok\n");
#endif
			model->current_node = model->root->children;
			model->current_mesh = (Mesh*)model->current_node->data;
#ifdef VERBOSE
printf("current_mesh name is now %s\n", model->current_mesh->name);
#endif
		}
	}
	return FALSE;
}


void model_set_name( Model *model, char *name ) {
	if( model == NULL || name == NULL ) return;
	
	if( model->name )
		free( model->name );
	model->name = strdup( name );
}


void model_set_filename( Model *model, char *fname ) {
	if( model == NULL || fname == NULL ) return;
	
	if( model->fname )
		free( model->fname );
	model->fname = strdup( fname );
}


/* TEXTURE MANAGEMENT ****************************************************/

gboolean model_remove_texture_tree( GNode *node, gpointer data ) {
    // recurse through tree
    
    Mesh* m = (Mesh*)node->data;

    if( m && ( m->texture == (Texture*)data  ||  data == NULL ) )
        m->texture = NULL;
    
    return FALSE;  /* we want it to continue... */
}

void model_remove_all_textures( Model *model ) {

    GSList * tl;
    
    if( model->root != NULL ) {
        g_node_traverse( model->root, G_POST_ORDER, G_TRAVERSE_ALL, -1,
                         model_remove_texture_tree, (gpointer)NULL );
    }

    for( tl = model->textures; tl != NULL; tl = tl->next ) {
        tex_delete( (Texture*) tl->data );
    }
    g_slist_free( model->textures );
    
}

void model_add_texture( Model *model, Texture *nt ) {
    
    if( nt != NULL )
        model->textures = g_slist_append( model->textures, nt );
}


void model_remove_texture( Model *model, Texture *tt ) {

    if( tt == NULL ) return;

    if( model->root != NULL ) {
        g_node_traverse( model->root, G_POST_ORDER, G_TRAVERSE_ALL, -1,
                         model_remove_texture_tree, (gpointer)tt );
    }

    model->textures = g_slist_remove( model->textures, ( gpointer ) tt );
    tex_delete( tt );
}


Texture * model_get_texture_by_filename( Model *model, char* filename ) {
    
    GSList * tl = NULL;
    
    if( filename == NULL ) return NULL;
    
    for( tl = model->textures; tl != NULL; tl = tl->next ) {
        Texture * t = (Texture*) tl->data;
        if( strcmp( t->filename, filename ) == 0 )
            return t;
    }
    
    
    return NULL;
}


/* VERTEX MANAGEMENT *****************************************************/

void model_vertex_add( Model *model, Vertex *v ) {

	if( v == NULL ) return;

	model->verts = g_slist_prepend( model->verts, v );
	model->num_verts++;
	v->model = model;
	
	action_do( model, ACTION_VERTEX_ADD_TO_MODEL, NULL, v, NULL, NULL, NULL, NULL );
	primitive_ref( (Primitive*) v );
}


void model_vertex_remove( Model *model, Vertex *v ) {

	if( v == NULL ) return;

	if( model->root != NULL ) {
		g_node_traverse( model->root, G_POST_ORDER, G_TRAVERSE_ALL, -1,
						 model_vertex_remove_tree, (gpointer)v );
	}
	
	vertex_remove( v );
	model->verts = g_slist_remove( model->verts, v );
	model->num_verts--;
	v->model = NULL;

	action_do( model, ACTION_VERTEX_REMOVE_MODEL, NULL, v, NULL, NULL, NULL, NULL );
	primitive_unref( (Primitive*) v );
}


gboolean model_vertex_remove_tree( GNode *node, gpointer data ) {
    // recurse through tree
    
    Mesh* m = (Mesh*)node->data;

    if( m == NULL )
        return FALSE;  /* we want it to continue... */

    mesh_vertex_remove( m, (Vertex*)data );
    return FALSE;
}


void model_vertex_remove_all( Model *model ) {

    while( model->verts != NULL ) {
        model_vertex_remove( model, (Vertex *) model->verts->data );
    }
    g_slist_free( model->verts );
}


/* POLYGON MANAGEMENT ****************************************************/

void model_polygon_remove( Model *model, Poly *t ) {

    /* traverses the model tree, looking for the Poly */
    
    if( t == NULL ) return;

    if( model->root != NULL ) {
        g_node_traverse( model->root, G_POST_ORDER, G_TRAVERSE_ALL, -1,
                         model_polygon_remove_tree, (gpointer)t );
    }

}


gboolean model_polygon_remove_tree( GNode *node, gpointer data ) {
    // recurse through tree
    
    Mesh* m = (Mesh*)node->data;

    if( m == NULL )
        return FALSE;  /* we want it to continue... */

    return mesh_polygon_remove( m, (Poly *)data );
}


/* MISC UTILITY FUNCTIONS *************************************************/

#if 0
/* a misnomer... this func just removes all the locs from the model, 
   it does not affect the organizational tree of the model 
*/
void model_flatten( GNode *n, float parentx, float parenty, float parentz ) {

    Mesh* m = NULL;
    GNode * current_node = n;
    
    if( current_node == NULL )
        return;
    
    
    for( ; current_node != NULL; current_node = current_node->next ) {
    
        float x = parentx;
        float y = parenty;
        float z = parentz;
        
        m = (Mesh*)(current_node->data);
        
        if( m != NULL ) {
            if( m->location != NULL ) {
                x += m->location->v[0];
                y += m->location->v[1];
                z += m->location->v[2];

                free( m->location );
                m->location = NULL;
            }
            if( m->vertices != NULL ) {
                // for each vertex...
                GSList * l = m->vertices;
                for( ; l; l = l->next ) {
                    Vertex* p = (Vertex*)(l->data);
                    p->v[0] += x;
                    p->v[1] += y;
                    p->v[2] += z;
                }
            
            }
            
        }
        
        if( current_node->children != NULL ) {
            model_flatten( current_node->children, x , y , z  );
        }
        
    }

}
#endif


void model_remove_unused_vertices( Model *model ) {
	GSList *l;
	Vertex *v;

	if( model == NULL ) return;
	
	for( l = model->verts; l;  ) {
		v = (Vertex *)l->data;
		if( !vertex_is_in_use( v ) ) {
			model_vertex_remove( model, v );
			l = model->verts;
		} else {
			l = l->next;
		}
	}
}


