/* 
 * File:         model_load_save.c
 * 
 * Description:  Abstracted interface to the different fmt readers and writers
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 
/*
 * Some code in this file is from:
 *   - ME3D 3-D Modeler Program
 *     the ME3D program is Copyright (C) 1998 Sam Revitch
 *
 */

/* Document load/save code */

#include <string.h>

#include "undo.h"		/* for action_clear_stack(), etc */
#include "bottombar.h"

#include "model_load_save.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif

extern struct doc_file_fmt fmt_ac3d;
extern struct doc_file_fmt fmt_f3ds;
extern struct doc_file_fmt fmt_quakemap;
extern struct doc_file_fmt fmt_md2;
extern struct doc_file_fmt fmt_wavefront;

struct doc_file_fmt *fmt_list[] = {	/* KEEP ALPHABETICAL */
    &fmt_ac3d,
#ifdef HAVE_LIB3DS
    &fmt_f3ds,
#endif
    &fmt_quakemap,
	&fmt_md2,
    &fmt_wavefront,
    NULL
};



gint model_ls_examine_extension(const gchar *filename, gint load);


/*** Load Functions ***/

Model *model_load(const gchar *filename, gint fmt_idx)
{
  struct fu_file *file;
  Model *model = NULL;
  gint ret;

  g_return_val_if_fail(filename != NULL, FALSE);
  g_return_val_if_fail(fmt_idx >= -1, FALSE);

  if(fmt_idx == -1)
    fmt_idx = model_ls_examine_extension(filename, TRUE);
  if(fmt_idx == -1)
    return NULL;

  file = fu_open(filename, FALSE);
  if(file == NULL)
    return NULL;

  model = model_new();

  action_clear_stack(model);
  undo_disable(model);

  model_set_filename( model, (char*)filename );

  {
  /* set the model name to be the basename */
  char *basename = g_path_get_basename(filename);
  model_set_name( model, basename );
  g_free( basename );
  }

  ret = (*(fmt_list[fmt_idx]->load_file))(file, model);
  fu_close(file);
  if(ret == FALSE) {
	model_delete( model );
    return NULL;
  }

  /* only set the format ID if there's a save function */
  if(fmt_list[fmt_idx]->save_file != NULL)
    model->fmt_idx = fmt_idx;
  else
    model->fmt_idx = -1;
  
  return model;
}



/*** Save Functions ***/

gint model_save(Model *doc, const gchar *filename, 
				int overwrite, int *file_exists )
{
	struct fu_file *file;
	gchar *old_fname = NULL;
	gint ret, name_changed = FALSE, old_fmt_idx;

	g_return_val_if_fail(doc != NULL, FALSE);

	if((filename == NULL) && (doc->fname == NULL))
		return FALSE;

	*file_exists = FALSE;

	if(filename != NULL) {
		if(doc->fname == NULL) {
			name_changed = TRUE;
		} else {
			if(strcmp(doc->fname, filename) != 0) {
				old_fname = strdup( doc->fname );
				name_changed = TRUE;
			}
		}
	} else {
		filename = doc->fname;
	}

	old_fmt_idx = doc->fmt_idx;
	if(doc->fmt_idx == -1) {
		doc->fmt_idx = model_ls_examine_extension(filename, FALSE);
		if(doc->fmt_idx == -1)
			goto failed;		/* unknown file type */
	}

	file = fu_open(filename, TRUE);
	if(file == NULL)
		goto failed;

	/* check to see if file already exists */
	if(fu_file_size(file) > 0) {
		if(name_changed && !overwrite) {
			*file_exists = TRUE;
			goto failed;
		}

		/* overwrite==TRUE means truncate and save to file */
		fu_truncate(file, 0);
	}

	/* We need to set the model's filename before we try to call the 
	   model's save function.  This is because some format-saving-functions 
	   may make use of external libraries, and those libraries may expect a 
	   filename rather than a file pointer when saving. */
	if( name_changed ) {
		char *basename = g_path_get_basename(filename);
		model_set_name( doc, basename );
		g_free( basename );

		model_set_filename( doc, (char*)filename );
	}

	ret = (*(fmt_list[doc->fmt_idx]->save_file))(file, doc);

	fu_close(file);
	file = NULL;

	if(ret) {
//		doc->unsaved_changes = FALSE;
//		doc_set_name(doc, bn);
	}

	/* if the save failed, AND we changed the name before we tried to save,
	 then we should set the name back to what it was before */
	if(!ret && name_changed) {
		/* replace the name with the old one */
		char *basename = g_path_get_basename(old_fname);
		model_set_name( doc, basename );
		g_free( basename );

		model_set_filename( doc, old_fname );

		doc->fmt_idx = old_fmt_idx;
	}

	if( old_fname != NULL )
		free( old_fname );

	return ret;

failed:
	if( file != NULL )
		fu_close( file );
	if( old_fname != NULL )
		free( old_fname );
	return FALSE;
}


void model_ls_clean_miscdata(Model *doc)
{
	if( (doc->fmt_idx > -1) &&
		(fmt_list[doc->fmt_idx]->destroy_priv_data != NULL) )
	{
		(*(fmt_list[doc->fmt_idx]->destroy_priv_data))(doc);
	}
	doc->fmt_idx = -1;
}


gint model_ls_examine_extension(const gchar *filename, gint load)
{
  gint i, j, l, m;

  m = strlen(filename);
  for(i = 0; (fmt_list[i] != NULL); i++)
  {
    if(load && (fmt_list[i]->load_file == NULL))
       continue;
    if(!load && (fmt_list[i]->save_file == NULL))
       continue;

    l = strlen(fmt_list[i]->extension);
    if(l >= m)
      continue;

    for(j = 0; j < l; j++)
    {
      if(fmt_list[i]->extension[l - j] != filename[m - j])
	break;
    }

    if((j == l) && (filename[m-l - 1] == '.'))
      return i;
  }
  return -1;
}

