/* 
 * File:         polygon.c
 * 
 * Description:  functions to handle things like polygon creation
 * 		 and selection
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 
#include <stdlib.h>
#include <string.h>  /* why the hell is memset in string.h ? */

#include "polygon.h"
#include "vector.h"
#include "undo.h"
#include "selection.h"
#include "primitive.h"
#include "misc.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif

/*
PROTOTYPES
*/
void poly_free( Poly *p );
void poly_free( Poly *p ) { free( p ); }


Poly * poly_new( void ) {
    Poly * result = (Poly*) malloc( sizeof( Poly ) );

    if( result == NULL ) return NULL;

    memset( result, 0, sizeof( Poly ) );

	primitive_set_delete_cb( (Primitive*)result, poly_free );
	((Primitive*)result)->type = PRIM_TYPE_POLY;

    result->normal[0] = 0.0f;
    result->normal[1] = 0.0f;
    result->normal[2] = 1.0f;

    return result;
}


void poly_delete( Poly *p ) {
	if( !p ) return;

	/* Referencing the poly here avoids double-frees.  In situations where the 
	poly has not yet been added to a mesh, but *has* had verts added to it, 
	calling poly_remove will release all references to the poly, causing it 
	to be freed.  Thus, calling free() on it here would result in a double-free.
	So, instead of calling poly_remove and free, we'll do this: */

	primitive_ref( (Primitive*)p );
		poly_remove( p );

		/* mostly for debugging/just-in-case; the refcount really *should* be 
		one at this point */
		if( p->primitive.ref_count != 1 ) {
			printf( "error in %s: not all references to poly %x "
					"have been released\n",
				__FUNCTION__, (unsigned int)p );
			free( p );
		}
	primitive_unref( (Primitive*)p );
	
}


void poly_remove( Poly *p ) {

    if( !p ) return;

	sel_poly_unselect( p, TRUE );
	sel_poly_show( p );
    
	while( p->num_verts > 0 ) {
		poly_remove_vertex( p, p->verts[0] );
	}

}


Poly * poly_dup( Poly *p ) {
    Poly *copy;
    int i;
    
    if( p == NULL )
        return NULL;
    
    copy = poly_new();

    for( i = 0; i < p->num_verts; i++ ) {
        poly_add_vertex( copy, p->verts[i] );
        copy->tc[i]->x = p->tc[i]->x;
        copy->tc[i]->y = p->tc[i]->y;
    }

    copy->selected = FALSE;
    copy->hidden = FALSE;
    copy->mesh = NULL;
    return copy;
}


void poly_add_vertex( Poly *p, Vertex *v ) {
	poly_add_vertex_nth( p, v, -1 );
}


/* Adds vertex to the polygon at a specific index (used by the undo system).
   If you just want to add the vertex to the end of the poly's list, you 
   should pass in -1 as the index.  poly_add_vertex should be used instead of 
   this func in almost every case. */
void poly_add_vertex_nth( Poly *p, Vertex *v, int vert_index ) {
    Vertex ** tempV;
    TexCoord ** tempTC;
	int i, j;
    
    if( p == NULL || v == NULL ) return;
    if( poly_has_vertex( p, v ) ) return;
    if( p->num_verts == POLY_MAX_VERTS ) return;
    if( vert_index > p->num_verts ) return;

	if( vert_index < 0 )
		vert_index = p->num_verts;
	
    tempV = malloc( sizeof( Vertex * ) * ( p->num_verts + 1 ) );
    tempTC = malloc( sizeof( TexCoord * ) * ( p->num_verts + 1 ) );

	/* copy the lists of pointers */
	for( i = 0, j = 0; i <= p->num_verts; i++ ) {
		if( i != vert_index ) {
			tempV[i] = p->verts[j];
			tempTC[i] = p->tc[j];
			j++;
		} else {
			tempV[i] = v;
			tempTC[i] = malloc( sizeof( TexCoord ) );
			tempTC[i]->x = 0.0;
			tempTC[i]->y = 0.0;
		}
	}

    if( p->verts ) {
        free( p->verts );
    }
    if( p->tc ) {
        free( p->tc );
    }

    p->verts = tempV;
    p->tc = tempTC;
    p->num_verts++;
    
	/* check to make sure that we have a Model* that we can use... 
	there are a few instances when we will not have one:
		- when creating polys while a model is being loaded (in which case it 
		is OK to omit a call to action_do, as undo will be disabled at such a 
		time anyway)
		- when creating a poly by clicking on things (view.c) (in which case 
		it is NOT OK to omit a call to action_do). we can use v's model in 
		this case
	*/
	if( p->mesh ) {
		action_do( p->mesh->model, ACTION_VERTEX_ADD_TO_POLY, NULL, v, NULL, NULL, NULL, p );
	} else {
		if( v->model ) {
			action_do( v->model, ACTION_VERTEX_ADD_TO_POLY, NULL, v, NULL, NULL, NULL, p );
		} else {
printf( "%s : %i - couldn't find a model to use for action_do\n", __FILE__, __LINE__ );
		}
	}

	primitive_ref( (Primitive*) v );
	
	vertex_add_poly( v, p );
	
	/* if we are part of a mesh, notify the mesh that we have gained a vert */
	if( p->mesh ) {
		mesh_polygon_add_verts_to_mesh( p->mesh, p );
	}
}


void poly_remove_vertex( Poly *p, Vertex *v ) {
	Vertex ** tempV;
	TexCoord ** tempTC;
	int i, j;
	int vert_index = -1;
	
	if( p == NULL || v == NULL ) return;
	if( p->num_verts <= 0 ) return;
	if( !poly_has_vertex( p, v ) ) return;
	
	if( p->num_verts == 1 ) {
		/* special case: one vert left */
		tempV = NULL;
		tempTC = NULL;
		if( p->tc[0] )
			free( p->tc[0] );
		vert_index = 0;
	} else {
		tempV = malloc( sizeof( Vertex * ) * ( p->num_verts - 1 ) );
		tempTC = malloc( sizeof( TexCoord * ) * ( p->num_verts - 1 ) );
	
		/* copy the lists of pointers */
		for( i = 0, j = 0; i < p->num_verts; i++ ) {
			if( v != p->verts[i] ) {
				tempV[j] = p->verts[i];
				tempTC[j] = p->tc[i];
				j++;
			} else {
				vert_index = i;
				if( p->tc[i] )
					free( p->tc[i] );
			}
		}
	}

	free( p->verts );
	free( p->tc );
	p->verts = tempV;
	p->tc = tempTC;
	p->num_verts--;
	
	/* check to make sure that we have a Model* that we can use... 
	there are a few instances when we will not have one:
		- when discarding polys while a model is being loaded (in which case it 
		is OK to omit a call to action_do, as undo will be disabled at such a 
		time anyway)
		- when user aborts a poly they were creating (view.c) (in which case 
		it is NOT OK to omit a call to action_do). we can use v's model in 
		this case
	*/
	if( p->mesh ) {
		action_do( p->mesh->model, ACTION_VERTEX_REMOVE_POLY, NULL, v, NULL, NULL, &vert_index, p );
	} else {
		if( v->model ) {
			action_do( v->model, ACTION_VERTEX_REMOVE_POLY, NULL, v, NULL, NULL, &vert_index, p );
		} else {
printf( "%s : %i - couldn't find a model to use for action_do\n", __FILE__, __LINE__ );
		}
	}
	primitive_unref( (Primitive*) v );

	vertex_remove_poly( v, p );

	/* I'm tempted to notify p's mesh (if it has one) that it may no longer 
	need vertex v.  However, that would not be correct behavior (think about 
	it for a bit, if you don't immed. understand). */
}


void poly_replace_vertex( Poly *p, Vertex *v, int vert_index ) {
	
	Vertex *old_vert;
	
	if( p == NULL || v == NULL ) return;
	if( p->num_verts <= 0 ) return;
	if( vert_index > p->num_verts || vert_index < 0 ) return;
	if( poly_has_vertex( p, v ) ) return;
	
	old_vert = p->verts[vert_index];

	primitive_ref( (Primitive*) v );
	vertex_add_poly( v, p );

	primitive_unref( (Primitive*) old_vert );
	vertex_remove_poly( old_vert, p );

	p->verts[vert_index] = v;
	/* tex coord remains untouched */
	
	/* we'll use v's model.  it doesn't matter. */
	action_do( v->model, ACTION_VERTEX_SWITCH_POLY, NULL, old_vert, NULL, NULL, &vert_index, p );

	/* if we are part of a mesh, notify the mesh that we have gained a vert */
	if( p->mesh ) {
		mesh_polygon_add_verts_to_mesh( p->mesh, p );
	}
}


int poly_has_vertex( Poly *p, Vertex *v ) {
    int i;
    for( i = 0; i < p->num_verts; i++ ) {
        if( p->verts[i] == v ) {
            return TRUE;
        }
    }
    return FALSE;
}


int poly_has_edge( Poly *p, Vertex *v1, Vertex *v2 ) {

	if( v1 == v2 ) return FALSE;

	if( poly_has_vertex( p, v1 ) && poly_has_vertex( p, v2 ) ) {
		/* we now know that both v1 and v2 are in p, but are they adjacent? */
		int difference, v1index, v2index;

		if( p->num_verts <= 3 )
			return TRUE;
		v1index = array_find( (void **)p->verts, p->num_verts, v1 );
		v2index = array_find( (void **)p->verts, p->num_verts, v2 );
		difference = abs( v1index - v2index );
		if( difference == 1 || difference == (p->num_verts - 1) )
			return TRUE;
	}
	return FALSE;
}


int polys_count_vertex_usage( GSList *polys, Vertex *v ) {
    int count = 0;
    
    for( ; polys; polys = polys->next ) {
        if( poly_has_vertex( (Poly*)polys->data, v ) ) {
            count++;
        }
    }
    return count;
}


int polys_count_edge_usage( GSList *polys, Vertex *v1, Vertex *v2 ) {
    int count = 0;
    
    for( ; polys; polys = polys->next ) {
        if( poly_has_edge( (Poly*)polys->data, v1, v2 ) ) {
            count++;
        }
    }
    return count;
}


void poly_reverse_winding( Poly * p ) {

    Vertex *vertices[ POLY_MAX_VERTS ];
    TexCoord *tc[ POLY_MAX_VERTS ];
    int i;
    
    if( p == NULL )
        return;

    for( i = 0; i < p->num_verts; i++ ) {
        vertices[i] = p->verts[i];
        tc[i] = p->tc[i];
    }

    for( i = 0; i < p->num_verts; i++ ) {
        p->verts[i] = vertices[p->num_verts - 1 - i];
        p->tc[i] = tc[p->num_verts - 1 - i];
    }

	action_do( p->mesh->model, ACTION_POLY_REVERSE_WINDING, NULL, p, NULL, NULL, NULL, NULL );
}


void poly_find_normal( Poly * p ) {

    GLfloat v1[ 3 ];
    GLfloat v2[ 3 ];
    
    if( p == NULL )
        return;

    if( p->num_verts < 3 )
        return;

    /* Calculate Vector from point 1 to 0 */
    vector_sub( v1, p->verts[0]->v, p->verts[1]->v );
    vector_normalize( v1 );

    /* Calculate Vector from point 2 to 1 */
    vector_sub( v2, p->verts[1]->v, p->verts[2]->v );
    vector_normalize( v2 );
    
    /* the cross product will get us the vector perpendicular to the poly */
    vector_cross_prod( p->normal, v1, v2 );
	vector_normalize( p->normal );
}


/* stores result in v */
void poly_find_center_point( float *v, Poly * p ) {
	
	int i;
	
    if( p == NULL || v == NULL )
        return;
	
	vector_zero( v );
	
	for( i = 0; i < p->num_verts; i++ ) {
		vector_add( v, v, p->verts[i]->v );
	}
	vector_mul( v, v, 1. / (float)p->num_verts );
}


GSList * polys_get_verts_unique( GSList *polys ) {

    GSList *verts = NULL;
    int i;
    Poly *p;
    
    for( ; polys != NULL; polys = polys->next ) {
        p = (Poly*)polys->data;
        for( i = 0; i < p->num_verts; i++ ) {
            if( g_slist_index( verts, p->verts[i] ) == -1 )
                verts = g_slist_append( verts, p->verts[i] );
        }
    }
    
    return verts;
}


GSList * polys_get_verts( GSList *polys ) {

    GSList *verts = NULL;
    int i;
    Poly *p;
    
    for( ; polys != NULL; polys = polys->next ) {
        p = (Poly*)polys->data;
        for( i = 0; i < p->num_verts; i++ ) {
            verts = g_slist_append( verts, p->verts[i] );
        }
    }
    
    return verts;
}



TexCoord * poly_get_texcoord( Poly *p, Vertex *v ) {
	int vertindex;
	
	if( p == NULL || v == NULL ) return NULL;
	if( p->num_verts < 1 ) return NULL;
	
	vertindex = array_find( (void **)p->verts, p->num_verts, v );
	if( vertindex < 0 )
		return NULL;
	return p->tc[vertindex];
}


void texcoord_change_values( TexCoord *tc, float u, float v ) {
	
	if( tc == NULL ) return;
	
	tc->x = u;
	tc->y = v;
}






