/*
 * File:        pref_man.c
 * 
 * Description:	handles the preferences dialog
 * 
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */

#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <gtk/gtk.h>

#include "pref_man.h"
#include "prefs.h"
#include "gui.h"
#include "globals.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


#define PREFMAN_MAX_NUMERIC_ENTRY_LEN 10
#define PREFMAN_MAX_STRING_ENTRY_LEN 128


/* FILE-SCOPE VARS **********************************************************/

GHashTable *pref_man_page_hash = NULL;
GtkWidget *pref_man_notebook = NULL;
GtkWidget *pref_man_apply_button = NULL;


/* PROTOTYPES ***************************************************************/

GtkWidget *pref_man_lookup_page( char *pref_name ) ;
void pref_man_populate( void ) ;
void pref_man_get_foreach( gpointer value, gpointer user_data ) ;

void pref_man_add_pref_bool( Pref *pref );
void pref_man_add_pref_int( Pref *pref );
void pref_man_add_pref_float( Pref *pref );
void pref_man_add_pref_vector( Pref *pref );
void pref_man_add_pref_string( Pref *pref );

void pref_man_bool_cb( GtkWidget *w, gpointer data ) ;
void pref_man_int_cb( GtkWidget *w, gpointer data ) ;
void pref_man_float_cb( GtkWidget *w, gpointer data ) ;
void pref_man_vector_cb( GtkWidget *w, gpointer data ) ;
void pref_man_string_cb( GtkWidget *w, gpointer data ) ;

char *pref_man_extract_group_name( char *name ) ;
char *pref_man_extract_label_name( char *name ) ;




/* FUNCS ********************************************************************/

void pref_man_show( void ) {
	
	GtkWidget *dialog;
	gint dlg_response;
	int close_dialog = FALSE;
	
	pref_man_page_hash = 
		g_hash_table_new_full( 
			g_str_hash, g_str_equal,
			g_free, NULL );
	
	dialog = gtk_dialog_new_with_buttons ("Kludge3d Preferences",
										  GTK_WINDOW( TopLevelWindow ),
										  GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
										  GTK_STOCK_OK,
										  GTK_RESPONSE_OK,
										  GTK_STOCK_APPLY,
										  GTK_RESPONSE_APPLY,
										  GTK_STOCK_CANCEL,
										  GTK_RESPONSE_CANCEL,
										  NULL);
//	gtk_container_set_border_width( GTK_CONTAINER( dialog ), 10 );

	/* create a push button, but do not add it to the dialog or display it.
	we are simply using it as a notification object. */
	pref_man_apply_button = gtk_button_new();
	

	/* NOTEBOOK ********************************/
	pref_man_notebook = gtk_notebook_new();
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), pref_man_notebook, TRUE, TRUE, 0);

	gtk_widget_show_all( dialog );

	/* NOTEBOOK PAGES **************************/
	pref_man_populate();

	/* RUN DIALOG ******************************/
	while( !close_dialog ) {
		/* runs a gtkmain until the user clicks on something */
		dlg_response = gtk_dialog_run( GTK_DIALOG (dialog) );
		
		switch( dlg_response ) {
		case GTK_RESPONSE_OK:
			close_dialog = TRUE;
			/* fall through to apply, below */
		case GTK_RESPONSE_APPLY:
			gtk_button_clicked( GTK_BUTTON( pref_man_apply_button ) );
			g_signal_emit_by_name( notificationObj, 
									"notify::preferences-changed", NULL );
			break;
		default:
			close_dialog = TRUE;
			break;
		}
	}
	
	gtk_widget_destroy( dialog );

	gtk_widget_destroy( pref_man_apply_button );
	pref_man_apply_button = NULL;

	g_hash_table_destroy( pref_man_page_hash );
	pref_man_page_hash = NULL;
}


/* FUNCS FOR POPULATING NOTEBOOK ********************************************/

GtkWidget *pref_man_lookup_page( char *pref_name ) {
	GtkWidget *vbox, *label;
	char * group_name;

	group_name = pref_man_extract_group_name( pref_name );
	vbox = g_hash_table_lookup( pref_man_page_hash, group_name );

	if( !vbox ) {
		vbox = gtk_vbox_new( FALSE, 0 );
		gtk_widget_show( vbox );
		
		label = gtk_label_new( group_name );
		gtk_widget_show( label );
		
		gtk_notebook_append_page( 
			GTK_NOTEBOOK( pref_man_notebook ), 
			vbox, 
			label );
		
		/* do not free group_name now, as it must survive for the lifetime 
		of the hash table.  it will be freed when the hash table is deleted. */
		g_hash_table_insert( pref_man_page_hash, group_name, vbox );
	} else {

		g_free( group_name );
	}
	
	return vbox;
}


void pref_man_populate( void ) {
	g_slist_foreach( prefs_list, pref_man_get_foreach, NULL );
}


void pref_man_get_foreach( gpointer value, gpointer user_data ) {
	Pref *pref = (Pref *)value;
	
	switch( pref->type ) {
		case PREF_TYPE_BOOL:
			pref_man_add_pref_bool( pref );
			break;
		case PREF_TYPE_INT:
			pref_man_add_pref_int( pref );
			break;
		case PREF_TYPE_FLOAT:
			pref_man_add_pref_float( pref );
			break;
		case PREF_TYPE_VECTOR:
			pref_man_add_pref_vector( pref );
			break;
		case PREF_TYPE_STRING:
			pref_man_add_pref_string( pref );
			break;
		default:
			break;
	}
}


void pref_man_add_pref_bool( Pref *pref ) {
	GtkWidget *button;
	GtkWidget *vbox = pref_man_lookup_page( pref->name );
	char *label_name;
	
	label_name = pref_man_extract_label_name( pref->name );
	button = gtk_check_button_new_with_label( label_name );
	g_free( label_name );

	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(button), 
		pref_get_bool( pref->name, FALSE ) );
	
	g_object_set_data( G_OBJECT( button ), "user-data", pref );
	
	g_signal_connect( 
		pref_man_apply_button, "clicked", 
		G_CALLBACK( pref_man_bool_cb ), button );
	
	gtk_widget_show( button );
	gtk_box_pack_start( GTK_BOX( vbox ), button, FALSE, FALSE, 0 );
}


void pref_man_add_pref_int( Pref *pref ) {
	GtkWidget *entry, *hbox, *label;
	GtkWidget *vbox = pref_man_lookup_page( pref->name );
	char *label_name;
	char s[32];
	
	memset( s, 0, 32);
	
	label_name = pref_man_extract_label_name( pref->name );
	label = gtk_label_new( label_name );
	g_free( label_name );

	hbox = gtk_hbox_new( FALSE, 0 );

	entry = gtk_entry_new();
	gtk_entry_set_max_length( GTK_ENTRY( entry ), PREFMAN_MAX_NUMERIC_ENTRY_LEN );
	g_snprintf( s, 31, "%i", pref_get_int( pref->name, -1 ) );
	gtk_entry_set_text( GTK_ENTRY( entry ), s );
	
	gtk_box_pack_start( GTK_BOX( hbox ), label, TRUE, TRUE, 10 );
	gtk_box_pack_start( GTK_BOX( hbox ), entry, FALSE, FALSE, 0 );
	
	g_object_set_data( G_OBJECT( entry ), "user-data", pref );
	
	g_signal_connect( 
		pref_man_apply_button, "clicked", 
		G_CALLBACK( pref_man_int_cb ), entry );
	
	gtk_widget_show_all( hbox );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, FALSE, 0 );
}


void pref_man_add_pref_float( Pref *pref ) {
	GtkWidget *entry, *hbox, *label;
	GtkWidget *vbox = pref_man_lookup_page( pref->name );
	char *label_name;
	char s[32];
	
	memset( s, 0, 32);
	
	label_name = pref_man_extract_label_name( pref->name );
	label = gtk_label_new( label_name );
	g_free( label_name );

	hbox = gtk_hbox_new( FALSE, 0 );

	entry = gtk_entry_new();
	gtk_entry_set_max_length( GTK_ENTRY( entry ), PREFMAN_MAX_NUMERIC_ENTRY_LEN );
	g_snprintf( s, 31, "%f", pref_get_float( pref->name, -1.0 ) );
	gtk_entry_set_text( GTK_ENTRY( entry ), s );
	
	gtk_box_pack_start( GTK_BOX( hbox ), label, TRUE, TRUE, 10 );
	gtk_box_pack_start( GTK_BOX( hbox ), entry, FALSE, FALSE, 0 );
	
	g_object_set_data( G_OBJECT( entry ), "user-data", pref );
	
	g_signal_connect( 
		pref_man_apply_button, "clicked", 
		G_CALLBACK( pref_man_float_cb ), entry );
	
	gtk_widget_show_all( hbox );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, FALSE, 0 );
}


void pref_man_add_pref_vector( Pref *pref ) {
	float *v, zero[] = {0.,0.,0.};
	int i;
	GtkWidget *entry, *hbox, *label;
	GtkWidget *vbox = pref_man_lookup_page( pref->name );
	char *label_name;
	char s[32];
	
	memset( s, 0, 32);
	
	v = pref_get_vector( pref->name, zero );
	
	hbox = gtk_hbox_new( FALSE, 0 );

	label_name = pref_man_extract_label_name( pref->name );
	label = gtk_label_new( label_name );
	g_free( label_name );
	gtk_box_pack_start( GTK_BOX( hbox ), label, TRUE, TRUE, 10 );

	for( i = 0; i < 3; i++ ) {
		entry = gtk_entry_new();
		gtk_entry_set_max_length( GTK_ENTRY( entry ), PREFMAN_MAX_NUMERIC_ENTRY_LEN );
		g_snprintf( s, 31, "%f", v[i] );
		gtk_entry_set_text( GTK_ENTRY( entry ), s );

		gtk_box_pack_start( GTK_BOX( hbox ), entry, FALSE, FALSE, 0 );
	}

	g_object_set_data( G_OBJECT( hbox ), "user-data", pref );
	
	g_signal_connect( 
		pref_man_apply_button, "clicked", 
		G_CALLBACK( pref_man_vector_cb ), hbox );
	
	gtk_widget_show_all( hbox );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, FALSE, 0 );
}


void pref_man_add_pref_string( Pref *pref ) {
	GtkWidget *entry, *hbox, *label;
	GtkWidget *vbox = pref_man_lookup_page( pref->name );
	char *label_name;
	
	label_name = pref_man_extract_label_name( pref->name );
	label = gtk_label_new( label_name );
	g_free( label_name );

	hbox = gtk_hbox_new( FALSE, 0 );

	entry = gtk_entry_new();
	gtk_entry_set_max_length( GTK_ENTRY( entry ), PREFMAN_MAX_STRING_ENTRY_LEN );
	gtk_entry_set_text( GTK_ENTRY( entry ), 
						pref_get_string( pref->name, "uh oh" ) );
	
	gtk_box_pack_start( GTK_BOX( hbox ), label, FALSE, FALSE, 10 );
	gtk_box_pack_start( GTK_BOX( hbox ), entry, TRUE, TRUE, 0 );
	
	g_object_set_data( G_OBJECT( entry ), "user-data", pref );
	
	g_signal_connect( 
		pref_man_apply_button, "clicked", 
		G_CALLBACK( pref_man_string_cb ), entry );
	
	gtk_widget_show_all( hbox );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, FALSE, 0 );
}


/* CALLBACKS **************************************************************/

void pref_man_bool_cb( GtkWidget *w, gpointer data ) {
	GtkWidget *button = (GtkWidget *)data;
	Pref *pref;
	
	pref = (Pref *)g_object_get_data( G_OBJECT( button ), "user-data" );
	pref_set_bool( pref->name, 
				   gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(button) ) );
}


void pref_man_int_cb( GtkWidget *w, gpointer data ) {
	GtkWidget *entry = (GtkWidget *)data;
	Pref *pref;
	int val;
	const gchar * text;
	
	text = gtk_entry_get_text( GTK_ENTRY( entry ) );
	val = atoi( text );
	
	pref = (Pref *)g_object_get_data( G_OBJECT( entry ), "user-data" );
	pref_set_int( pref->name, val );
}


void pref_man_float_cb( GtkWidget *w, gpointer data ) {
	GtkWidget *entry = (GtkWidget *)data;
	Pref *pref;
	float val;
	const gchar * text;
	
	text = gtk_entry_get_text( GTK_ENTRY( entry ) );
	val = strtod( text, (char **)NULL );
	
	pref = (Pref *)g_object_get_data( G_OBJECT( entry ), "user-data" );
	pref_set_float( pref->name, val );
}


void pref_man_vector_cb( GtkWidget *w, gpointer data ) {
	GtkWidget *hbox = (GtkWidget *)data;
	GList *children, *l;
	Pref *pref;
	float v[3];
	int i;
	const gchar * text;
	
	children = gtk_container_get_children( GTK_CONTAINER( hbox ) );
	/* the first widget in the hbox should be a label, so we will skip it.
	we are interested only in the entries. */
	l = children->next;
	for( i = 0; l && i < 3; l = l->next ) {
		text = gtk_entry_get_text( GTK_ENTRY( l->data ) );
		v[i++] = strtod( text, (char **)NULL );
	}
	g_list_free( children );
	
	pref = (Pref *)g_object_get_data( G_OBJECT( hbox ), "user-data" );
	pref_set_vector( pref->name, v );
}


void pref_man_string_cb( GtkWidget *w, gpointer data ) {
	GtkWidget *entry = (GtkWidget *)data;
	Pref *pref;
	const gchar * text;
	
	text = gtk_entry_get_text( GTK_ENTRY( entry ) );
	
	pref = (Pref *)g_object_get_data( G_OBJECT( entry ), "user-data" );
	pref_set_string( pref->name, (char*)text );
}



/* MISC *******************************************************************/

char *pref_man_extract_group_name( char *name ) {
	
	char *double_colon_offset;
	double_colon_offset = strstr( name, "::" );
	if( double_colon_offset == NULL ) {
		return g_strdup( name );
	}
	return g_strndup( name, double_colon_offset - name );
}


char *pref_man_extract_label_name( char *name ) {
	
	char *double_colon_offset;
	double_colon_offset = strstr( name, "::" );
	if( double_colon_offset == NULL ) {
		return g_strdup( name );
	}
	double_colon_offset += 2;
	return g_strdup( double_colon_offset );
}


