/* 
 * File:         prefs.c
 * 
 * Description:  prefs stuff
 * 
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */


#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <glib.h>
#include <string.h>

#include "prefs.h"
#include "vector.h"
#include "globals.h"


#ifdef MEMWATCH
#include "memwatch.h"
#endif


#define PREF_VERSION 1
#define PREF_INTERNAL_STRING "INTERNAL"


/* GLOBALS ******************************************************************/

GSList *prefs_list = NULL;


/* FILE-SCOPE VARS ******************************************************/

GHashTable *prefs_hash = NULL;


/* PROTOTYPES ***************************************************************/

void pref_read( char *prefstring ) ;
void pref_write_foreach( gpointer value, gpointer user_data ) ;
Pref * pref_new( void ) ;
void pref_delete( Pref *p ) ;
void pref_set_name( Pref *p, char *name ) ;
void pref_switch_type( Pref *p, int newtype ) ;
Pref * pref_lookup( char * name ) ;
gboolean prefs_exit_cb( gpointer data ) ;
void pref_delete_foreach( gpointer key, gpointer value, gpointer user_data ) ;


/* FUNCS ********************************************************************/

void prefs_load( void ) {
	FILE *fp;
	char filename[512];
	char *status;
	char prefstring[1024];
	
	if( prefs_hash == NULL ) {
		prefs_hash = g_hash_table_new( g_str_hash, g_str_equal );
		
		/* set up a listener for the notify::exit signal */
		g_signal_connect( 
			notificationObj, "notify::exit", G_CALLBACK(prefs_exit_cb), NULL );
	}

	snprintf( filename, 511, "%s/.kludge3d/prefs.txt", getenv("HOME") );

	fp = fopen( filename, "r" );
	if( fp == NULL ) {
		fprintf( stderr, "Unable to open file %s for reading.\n", filename );
		return;
	}
	
	do {
		memset( prefstring, '\0', sizeof( prefstring ) );
		status = fgets( prefstring, 1023, fp );
		if( status != NULL ) {
			pref_read( prefstring );
		}
	} while( status != NULL );
	
	fclose( fp );
}


void pref_read( char *prefstring ) {

	Pref *pref;
	char *prefname, *preftypestring, *prefdata;
	int preftype;

	prefname = strchr( prefstring, '\"' );
	if( prefname == NULL ) return;
	/* skip the double-quote */
	prefname++;
	
	preftypestring = strchr( prefname, '\"' );
	if( preftypestring == NULL ) return;
	/* null-terminate the prefname string section */
	preftypestring[0] = '\0';
	/* skip null and following space */
	preftypestring += 2;
	
	prefdata = strchr( preftypestring, ' ' );
	if( prefdata == NULL ) return;
	/* null-terminate the preftypestring string section */
	prefdata[0] = '\0';
	/* skip null */
	prefdata++;
	
	preftype = atoi( preftypestring );

	/* check to see if it is an INTERNAL preference */
	if( strncmp( prefname, PREF_INTERNAL_STRING, 
				 strlen(PREF_INTERNAL_STRING) ) == 0 )
	{
		return;
	}
	
	pref = pref_lookup( prefname );
	pref_switch_type( pref, preftype );

	switch( preftype ) {
		case PREF_TYPE_BOOL:
		case PREF_TYPE_INT:
			sscanf( prefdata, "%i", &(pref->data_int) );
			break;
		case PREF_TYPE_FLOAT:
			sscanf( prefdata, "%f", &(pref->data_float) );
			break;
		case PREF_TYPE_VECTOR:
			sscanf( prefdata, "<%f %f %f>", 
				&(pref->data_vector[0]), 
				&(pref->data_vector[1]), 
				&(pref->data_vector[2]) );
			break;
		case PREF_TYPE_STRING:
			/* remove the newline at the end */
			if( prefdata[strlen( prefdata ) - 1] == '\n' )
				prefdata[strlen( prefdata ) - 1] = '\0';
			pref_set_string( prefname, prefdata );
			break;
		default:
			break;
	}
}


void prefs_save( void ) {
	FILE *fp;
	char filename[512];
	
	snprintf( filename, 511, "%s/.kludge3d/prefs.txt", getenv("HOME") );

	fp = fopen( filename, "w" );
	if( fp == NULL ) {
		fprintf( stderr, "Unable to open file %s for writing.\n", filename );
		return;
	}

	fprintf( fp, "\"%s::version\" %i %i\n", PREF_INTERNAL_STRING, 
		PREF_TYPE_INT, PREF_VERSION );

	/* we would like them written out in-order, so we'll use prefs_list 
	rather than the hash table */
	g_slist_foreach( prefs_list, pref_write_foreach, fp );
	
	fclose( fp );
}


void pref_write_foreach( gpointer value, gpointer user_data ) {
	FILE *fp = (FILE *)user_data;
	Pref *pref = (Pref *)value;
	
	fprintf( fp, "\"%s\" %i ", pref->name, pref->type );
	
	switch( pref->type ) {
		case PREF_TYPE_BOOL:
		case PREF_TYPE_INT:
			fprintf( fp, "%i", pref->data_int );
			break;
		case PREF_TYPE_FLOAT:
			fprintf( fp, "%f", pref->data_float );
			break;
		case PREF_TYPE_VECTOR:
			fprintf( fp, "<%f %f %f>", 
				pref->data_vector[0], 
				pref->data_vector[1], 
				pref->data_vector[2] );
			break;
		case PREF_TYPE_STRING:
			fprintf( fp, "%s", pref->data_string );
			break;
		default:
			break;
	}

	fprintf( fp, "\n" );
}


Pref * pref_new( void ) {
	Pref *result;
	
	result = (Pref*)malloc( sizeof( Pref ) );
	if( result == NULL ) return NULL;
	
	memset( result, 0, sizeof( Pref ) );
	
	result->type = PREF_TYPE_UNKNOWN;
	
	return result;
}


void pref_delete( Pref *p ) {
	if( p == NULL ) return;
	
	if( p->name )
		free( p->name );

	if( p->type == PREF_TYPE_STRING && p->data_string )
		free( p->data_string );
	
	free( p );
}


void pref_set_name( Pref *p, char *name ) {
	
	if( p->name )
		free( p->name );
	
	p->name = strdup( name );
}


void pref_switch_type( Pref *p, int newtype ) {
	
	if( p->type == PREF_TYPE_STRING && p->data_string ) {
		free( p->data_string );
		p->data_string = NULL;
	}
	
	p->type = newtype;
}


Pref * pref_lookup( char * name ) {
	/* if pref doesn't exist in hash table, create one and add it to the table.
	in either case, return the Pref when exiting */
	
	Pref * result;
	
	result = g_hash_table_lookup( prefs_hash, name );
	if( result )
		return result;

	result = pref_new();
	pref_set_name( result, name );

	/* is is important that we use result->name (not name) as the key, 
	because any string used as a key for a g_hash_table *MUST* be 
	'alive' and unmodified for the life of the hash table */
	g_hash_table_insert( prefs_hash, result->name, result );
	
	/* add the pref to the list too */
	prefs_list = g_slist_append( prefs_list, result );
	
	return result;
}


void pref_set_bool( char * name, int data ) {
	Pref *p = pref_lookup( name );
	pref_switch_type( p, PREF_TYPE_BOOL );
	p->data_int = (data != FALSE);
}


int pref_get_bool( char * name, int default_value ) {
	Pref *p = pref_lookup( name );
	if( p->type == PREF_TYPE_UNKNOWN ) {
		pref_set_bool( name, default_value );
		return (default_value != FALSE);
	} else
		return p->data_int;
}


void pref_set_int( char * name, int data ) {
	Pref *p = pref_lookup( name );
	pref_switch_type( p, PREF_TYPE_INT );
	p->data_int = data;
}


int pref_get_int( char * name, int default_value ) {
	Pref *p = pref_lookup( name );
	if( p->type == PREF_TYPE_UNKNOWN ) {
		pref_set_int( name, default_value );
		return default_value;
	} else
		return p->data_int;
}


void pref_set_float( char * name, float data ) {
	Pref *p = pref_lookup( name );
	pref_switch_type( p, PREF_TYPE_FLOAT );
	p->data_float = data;
}


float pref_get_float( char * name, float default_value ) {
	Pref *p = pref_lookup( name );
	if( p->type == PREF_TYPE_UNKNOWN ) {
		pref_set_float( name, default_value );
		return default_value;
	} else
		return p->data_float;
}


void pref_set_vector( char * name, float * data ) {
	Pref *p = pref_lookup( name );
	pref_switch_type( p, PREF_TYPE_VECTOR );
	vector_copy( p->data_vector, data );
}


float * pref_get_vector( char * name, float * default_value ) {
	Pref *p = pref_lookup( name );
	if( p->type == PREF_TYPE_UNKNOWN ) {
		pref_set_vector( name, default_value );
		return default_value;
	} else
		return p->data_vector;
}


void pref_set_string( char * name, char * data ) {
	Pref *p = pref_lookup( name );
	pref_switch_type( p, PREF_TYPE_STRING );
	p->data_string = strdup( data );
}


char * pref_get_string( char * name, char * default_value ) {
	Pref *p = pref_lookup( name );
	if( p->type == PREF_TYPE_UNKNOWN ) {
		pref_set_string( name, default_value );
		return default_value;
	} else
		return p->data_string;
}


gboolean prefs_exit_cb( gpointer data ) {

	prefs_save();
	
	g_hash_table_foreach( prefs_hash, pref_delete_foreach, NULL );
	g_hash_table_destroy( prefs_hash );
	prefs_hash = NULL;
	
	g_slist_free( prefs_list );
	prefs_list = NULL;
	
	return FALSE;
}


void pref_delete_foreach( gpointer key, gpointer value, gpointer user_data ) {
	pref_delete( (Pref *)value );
}



