/* 
 * File:		 pywrapper.c
 * 
 * Description:  Thin C wrapper for functions we want to call from the 
 *               scripting languages.
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */

#include <Python.h>
#include <glib.h>

#include "vector.h"
#include "vertex.h"
#include "polygon.h"
#include "mesh.h"
#include "model.h"
#include "geo.h"
#include "tools.h"
#include "pyinit.h"
#include "pywrapper.h"
#include "selection.h"
#include "misc.h"
#include "undo.h"

#include "win32stuff.h"


/* PROTOTYPES ***********************************************************/


/* FILE-SCOPE VARS ******************************************************/

char py_init_string[] = 
"\n\
from pykludge3d import *\n\
import pykludge3d\n\
import sys\n\
sys.path = sys.path + ['.']\n\
\n\
def extract_function_args( func ):\n\
	argstring = ''\n\
	for i in range( func.func_code.co_argcount ):\n\
		argstring += func.func_code.co_varnames[i]\n\
		argstring += ' '\n\
	return argstring\n\
# end of extract_function_args\n\
\n\
def register_function( modulename, func ):\n\
	register_func_with_kludge3d( modulename, func.__name__, func.__doc__, extract_function_args( func ) )\n\
# end of register_function\n\
\n\
# We'll add this function to the pykludge3d module, so that the scripts can \n\
# access it.  I'm not sure if this is an 'acceptable' thing to do in Python.\n\
pykludge3d.register_function = register_function\n\
";


/* FUNCS ****************************************************************/

/* "Registers" a Python function with kludge3d.  This info will be used to 
 * set up buttons and dialogs for the function, so that it can be called 
 * without having to type the stuff in by hand.
 */
void register_func_with_kludge3d( char *modulename, char *funcname, char *helpstring, char *argstring ) {
	char buf[256];
	
	snprintf( buf, 255, "%s.%s", modulename, funcname );
	scriptbox_add_func( buf, helpstring, argstring );
}


/* Inserts a new vertex into the current mesh.
 *  Returns a pointer to the newly-created vertex.
 */
Vertex *vert_insert(double x, double y, double z) {

  Vertex *p = NULL;

  p = vertex_new();
  p->v[0] = (GLdouble) x;
  p->v[1] = (GLdouble) y;
  p->v[2] = (GLdouble) z;

  model_vertex_add( the_model, p );
  mesh_vertex_add( the_model->current_mesh, p );

	return p;
}

/* Moves verts along the vector defined by <x,y,z>
 *
 */
void translate_verts( GSList *verts, double x, double y, double z ) {
	float delta[3];
	delta[0] = x;
	delta[1] = y;
	delta[2] = z;
	tools_vertices_move( the_model, verts, x, y, z );
	action_do( the_model, ACTION_VERTEX_MOVE, verts, NULL, NULL, delta, NULL, NULL );
}


/* Moves currently selected verts along the vector defined by <x,y,z>
 *
 */
void translate_sel_verts( double x, double y, double z ) {
	translate_verts( the_model->selected_verts, x, y, z );
}


/* Rotates verts around coordinate axes
 *
 * Rotations occur around the x-axis first, then y, then z.
 * The angle is in degrees.
 */
void rotate_verts( GSList *verts, double x, double y, double z ) {
  tools_vertices_rotate(the_model, verts, 0, x * M_PI / 180.0);
  tools_vertices_rotate(the_model, verts, 1, y * M_PI / 180.0);
  tools_vertices_rotate(the_model, verts, 2, z * M_PI / 180.0);
}


/* Rotates currently selected verts around coordinate axes
 *
 * Rotations occur around the x-axis first, then y, then z.
 * The angle is in degrees.
 */
void rotate_sel_verts( double x, double y, double z ) {
  rotate_verts( the_model->selected_verts, x, y, z );
}


/* Rotates verts around specified point px,py,pz
 *
 * Rotations occur around the x-axis first, then y, then z.
 * The angle is in degrees.
 */
void rotate_verts_about_point(GSList *verts, 
						double x, double y, double z,
                        double px, double py, double pz) 
{
  float pt[3];

  pt[0] = px;
  pt[1] = py;
  pt[2] = pz;
  tools_vertices_rotate_about_point(the_model, verts, 0, pt, x * M_PI / 180.0, TRUE );
  tools_vertices_rotate_about_point(the_model, verts, 1, pt, y * M_PI / 180.0, TRUE );
  tools_vertices_rotate_about_point(the_model, verts, 2, pt, z * M_PI / 180.0, TRUE );
}


/* Rotates currently selected verts around specified point px,py,pz
 *
 * Rotations occur around the x-axis first, then y, then z.
 * The angle is in degrees.
 */
void rotate_sel_verts_about_point(
						double x, double y, double z,
                        double px, double py, double pz) 
{
	rotate_verts_about_point( the_model->selected_verts, x, y, z, px, py, pz );
}


/* Scales verts about their midpoint by factor X,Y,Z.  To scale uniformly, 
 * use the same value for all 3 factor arguments.
 */
void scale_verts( GSList *verts, 
				double factorX, double factorY, double factorZ ) 
{
	float factor[3];
	factor[0] = factorX;
	factor[1] = factorY;
	factor[2] = factorZ;
	tools_vertices_scale_about_center( the_model, verts, factor, TRUE );
}


/* Scales the selected verts about their midpoint by factor X,Y,Z.  To scale 
 * uniformly, use the same value for all 3 factor arguments.
 */
void scale_sel_verts( 
				double factorX, double factorY, double factorZ ) 
{
	scale_verts( the_model->selected_verts, factorX, factorY, factorZ );
}


/* Returns the selected vertices
 *
 */
GSList * get_selected_verts( void ) {
	return g_slist_copy( the_model->selected_verts );
}


/* Selects the verts in the list
 *
 */
void select_verts( GSList *verts ) {
	
	sel_vert_select_list( verts );
}


/* Unselects all of the vertices.
 *
 */
void unselect_verts_all( void ) {
	sel_vert_unselect_all( the_model );
}


/* Returns a list of all verts within the box specified by min xyz, max xyz.
 * Each element of min xyz should be less than its corresponding 
 * element in max xyz.
 */
GSList *get_verts_box( double min_x, double min_y, double min_z,
                       double max_x, double max_y, double max_z ) 
{

	GSList * verts = NULL, *l;
	
	if( ! (min_x < max_x) ) return NULL;
	if( ! (min_y < max_y) ) return NULL;
	if( ! (min_z < max_z) ) return NULL;
	
	for( l = the_model->verts; l; l = l->next ) {
		Vertex *v = (Vertex*)l->data;
		if( min_x < v->v[0] && v->v[0] < max_x )
			if( min_y < v->v[1] && v->v[1] < max_y )
				if( min_z < v->v[2] && v->v[2] < max_z )
					verts = g_slist_append( verts, v );
	}
	
	return verts;
}


/* Returns a list of all verts within the box specified by min xyz, max xyz.
 * Each element of min xyz should be less than its corresponding 
 * element in max xyz.
 */
GSList *get_verts_sphere( double x, double y, double z,
                          double radius ) 
{

	GSList * verts = NULL, *l;
	float sphereCenter[3];
	
	sphereCenter[0] = (float)x;
	sphereCenter[1] = (float)y;
	sphereCenter[2] = (float)z;
	
	for( l = the_model->verts; l; l = l->next ) {
		Vertex *v = (Vertex*)l->data;
		if( vector_dist( sphereCenter, v->v ) < radius )
			verts = g_slist_append( verts, v );
	}
	
	return verts;
}


/* Returns a list of the vertices used by the given polygons.  No vertex will 
 * appear twice in the resulting list.
 */
GSList * get_verts_in_polys( GSList *polys ) {
	return polys_get_verts_unique( polys );
}


/* Inserts a new poly into the current mesh.  This func will search for 
 * verts near the specified verts, and (if such verts are not found)
 * will create new verts as needed.
 */
Poly * poly_insert( GSList * verts ) {
	
	GSList *l;
	Poly *p;
	Mesh *m = the_model->current_mesh;
	
	if( !verts ) return NULL;
	if( !m ) return NULL;
	
	p = poly_new();
	
	for( ; verts; verts = verts->next ) {
		Vertex *v = (Vertex*)verts->data;
	
		/* try to find a vert near this one */
		l = g_slist_find_custom( the_model->verts, v, 
		                        (GCompareFunc) vertices_isNotEqual );
	
		if( l == NULL ) {
			/* if we still can't find an exact or a close vertex,
			   we should make a copy and add that to our model and mesh */
			v = vertex_dup( v );
			model_vertex_add( the_model, v );
			mesh_vertex_add( m, v );
		} else {
			/* if we've found a vertex that's nearby... */
			v = (Vertex*)l->data;
		}
		
		poly_add_vertex( p, v );
	}
	
	if( p->num_verts > 2 ) {
		mesh_polygon_add( m, p );
	} else {
		poly_delete( p );
		p = NULL;
		printf( "Aborted polygon in %s, poly had too few verts\n", 
			__FUNCTION__ );
		/* we'll just let the verts stay where they are... 
		there's no way of knowing if it's OK to delete them or not. */
	}
	
	return p;
}


/* DEPRECATED Inserts a new 3-sided poly into the current mesh, using the 
 * 3 given coordinates as vertex locations.  Vertices are created as needed.
 */
Poly * poly_insert_tri( 
	double p1x, double p1y, double p1z, 
	double p2x, double p2y, double p2z, 
	double p3x, double p3y, double p3z ) 
{
	Vertex *v1, *v2, *v3;
	GSList *l = NULL;
	Poly *p;

	/* arrays? loops? what are those? :) */
	v1 = vertex_new();
	v2 = vertex_new();
	v3 = vertex_new();
	
	v1->v[0] = p1x;
	v1->v[1] = p1y;
	v1->v[2] = p1z;
	v2->v[0] = p2x;
	v2->v[1] = p2y;
	v2->v[2] = p2z;
	v3->v[0] = p3x;
	v3->v[1] = p3y;
	v3->v[2] = p3z;
	
	l = g_slist_append( l, v1 );
	l = g_slist_append( l, v2 );
	l = g_slist_append( l, v3 );
	
	p = poly_insert( l );
	
	vertex_delete( v1 );
	vertex_delete( v2 );
	vertex_delete( v3 );
	g_slist_free( l );
	
	return p;
}


/* DEPRECATED Inserts a new 4-sided poly into the current mesh, using the 
 * 4 given coordinates as vertex locations.  Vertices are created as needed.
 */
Poly * poly_insert_quad( 
	double p1x, double p1y, double p1z, 
	double p2x, double p2y, double p2z, 
	double p3x, double p3y, double p3z, 
	double p4x, double p4y, double p4z ) 
{
	Vertex *v1, *v2, *v3, *v4;
	GSList *l = NULL;
	Poly *p;

	/* arrays? loops? what are those? :) */
	v1 = vertex_new();
	v2 = vertex_new();
	v3 = vertex_new();
	v4 = vertex_new();
	
	v1->v[0] = p1x;
	v1->v[1] = p1y;
	v1->v[2] = p1z;
	v2->v[0] = p2x;
	v2->v[1] = p2y;
	v2->v[2] = p2z;
	v3->v[0] = p3x;
	v3->v[1] = p3y;
	v3->v[2] = p3z;
	v4->v[0] = p4x;
	v4->v[1] = p4y;
	v4->v[2] = p4z;
	
	l = g_slist_append( l, v1 );
	l = g_slist_append( l, v2 );
	l = g_slist_append( l, v3 );
	l = g_slist_append( l, v4 );
	
	p = poly_insert( l );
	
	vertex_delete( v1 );
	vertex_delete( v2 );
	vertex_delete( v3 );
	vertex_delete( v4 );
	g_slist_free( l );

	return p;
}


/* Returns the selected polys
 *
 */
GSList * get_selected_polys( void ) {
	return g_slist_copy( the_model->selected_polys );
}


/* Extrudes the polys in the list.  If 'connected' is true, polys will remain 
 * connected after being extruded; otherwise, polys will be extruded 
 * individually.  Returns the list of newly-created polys, with the members 
 * of 'polys' at the head of the list.
 */
GSList * polys_extrude( GSList *polys, int connected ) {
	return connected ? 
		tools_polys_extrude_connected( the_model, polys ) :
		tools_polys_extrude( the_model, polys );
}


/* Returns the normal of the poly.  It is called like so: 
 * tuple = get_poly_normal( p, None ) */
void get_poly_normal( Poly * p, float out[3] ) {
	vector_zero( out );
	if( p == NULL ) return;
	poly_find_normal( p );
	vector_copy( out, p->normal );
}


void start_python( void ) {
	Py_Initialize();
	initpykludge3d();
	PyRun_SimpleString( py_init_string );

	{
	char buffer[1024];
	char filename[1024];

	sprintf( buffer, "sys.path = sys.path + ['%s/.kludge3d']", getenv("HOME") );
	PyRun_SimpleString(buffer);

	sprintf( filename, "%s/.kludge3d/base.py", getenv("HOME") );
	if( file_exists( filename ) ) {
		sprintf( buffer, "execfile('%s')", filename );
		PyRun_SimpleString(buffer);
	}
	}
}

void stop_python( void ) {
  Py_Finalize();
}
