/* 
 * File:		 scripting_entry.c
 * 
 * Description:  creates and manages the python scripting text entry widget.
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h> /* for keyboard values */

#include "config.h"
#include "scripting_entry.h"
#include "globals.h"
#include "model.h"
#include "group.h"
#include "mesh.h"
#include "vertex.h"
#include "polygon.h"
#include "undo.h"
#include "win32stuff.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


#ifdef USE_PYTHON
#include <Python.h>
#endif

#ifdef USE_PYTHON
#define SE_FRAME_TEXT	"Python Scripting Entry"
#else
#define SE_FRAME_TEXT	"Manual Vertex Entry"
#endif


void process_entry_cb(GtkWidget* widget, gpointer data) ;
int se_last_char_is_colon( char *str ) ;
gboolean se_key_press_cb( GtkWidget *widget, 
						  GdkEventKey *event, gpointer data ) ;



GtkWidget * create_scripting_entry( void ) {

	GtkWidget *widget, *frame;
	
	frame = gtk_frame_new( SE_FRAME_TEXT );
	
	/* This text entry field will be used to accept input for the python 
	   interpreter, or (if kludge3d was not compiled with python support)
	   to enter coordinates for new vertices.
	 */
	widget = gtk_entry_new();
	g_signal_connect( G_OBJECT( widget ), "activate",
			G_CALLBACK( process_entry_cb ), (gpointer) widget );
	g_signal_connect( G_OBJECT( widget ), "key-press-event",
			G_CALLBACK( se_key_press_cb ), (gpointer) widget );

	gtk_container_add( GTK_CONTAINER( frame ), widget );
	gtk_widget_show_all( frame );
	return frame;
}


#ifdef USE_PYTHON

void process_entry_cb(GtkWidget* widget, gpointer data) {
	gchar *str;
	static char * buf = NULL;
	static int unfinished = FALSE;

	str = gtk_editable_get_chars (GTK_EDITABLE(data), 0, -1);

	/* Here's the low-down:  Python isn't happy if you send it a multi-line 
	program segment one-line-at-a-time.  We need to buffer the multi-line 
	segments until they are complete, and then send them.
	- if the last non-whitespace char in the line is a colon ':', 
	- save everything the user enters 
	- ...*until* they enter a line with no tab chars at the beginning
	*/
	
	/* if there is no tab at the beginning */
	if( str[0] != '\t' ) {
		/* and we are unfinished */
		if( unfinished ) {
			/* send everything to the interpreter, reset 'unfinished', 
			clear the buffer */
			PyRun_SimpleString( buf );
			unfinished = FALSE;
			free( buf );
			buf = NULL;
		}
	}
	
	/* if there was a colon at the end of the new line OR we're still 
	waiting for the user to finish their program segment */
	if( se_last_char_is_colon( str ) || unfinished ) {
		/* append the string to the end of our buffer */
		char * temp;
		temp = malloc( (buf == NULL ? 0 : strlen( buf )) + 
					   strlen( str ) + 2 );
		sprintf( temp, "%s%s\n", (buf == NULL ? "" : buf), str );
		free( buf );
		buf = temp;
		unfinished = TRUE;
	} else {
		/* the command is a simple one-liner and we can just send it to the 
		interpreter */
		PyRun_SimpleString(str);
	}
	
	gtk_entry_set_text(GTK_ENTRY(data), "");
	g_free(str);

	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );

	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}

#else

void process_entry_cb(GtkWidget* widget, gpointer data) {

	gchar *text;
	gchar **tok;
	gchar *pch;
	int nstr;

	Vertex *p = NULL;

	text = gtk_editable_get_chars (GTK_EDITABLE(data), 0, -1);
	/* how do we validate this? This WILL blow up at the moment */

	g_strstrip(text);
	
	p = vertex_new();

	pch = strtok(text, " ,][");
	for (nstr = 0; nstr < 3 && pch != NULL; nstr++ ) {
	//printf("%s\n", pch);
	p->v[nstr] = (GLdouble) g_strtod(pch, NULL);

	pch = strtok(NULL, " ,][");
	}

	if (nstr != 3)
	return;

	model_vertex_add( the_model, p );
	mesh_vertex_add( the_model->current_mesh, p );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );

	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );

	gtk_entry_set_text(GTK_ENTRY(data), "");
	
}

#endif

int se_last_char_is_colon( char *str ) {
	int len, i;
	if( str == NULL ) return FALSE;
	
	len = strlen( str );
	if( len <= 0 ) return FALSE;
	
	for( i = len - 1; i >= 0; i-- ) {
		if( str[i] != '\t' && str[i] != ' ' )
			break;
	}
	return( str[i] == ':' );
}


gboolean se_key_press_cb( GtkWidget *widget, 
						  GdkEventKey *event, gpointer data ) 
{
	
	if( event->keyval == GDK_Tab ) {
		int pos = gtk_editable_get_position( GTK_EDITABLE( widget ) );
		gtk_editable_insert_text( 
			GTK_EDITABLE( widget ), "\t", 1, &pos );
		gtk_editable_set_position( GTK_EDITABLE( widget ), pos );
		return TRUE;
	}
	return FALSE;
}

