/* 
 * File:		 selection.c
 * 
 * Description:  manages selected primitives; also handles hidden primitives
 * 
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */

#include <glib.h>

#include "selection.h"
#include "geo.h"
#include "model.h"
#include "undo.h"
#include "globals.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


/* SELECTION, VERTS *****************************************************/

void sel_vert_select( Vertex* v, int make_undo_entry ) {

	if( v != NULL && !g_slist_find( v->model->selected_verts, v ) ) {
		v->model->selected_verts = g_slist_append( v->model->selected_verts, v );
		v->selected = TRUE;

		if( make_undo_entry ) {
			GSList *l = NULL;
			l = g_slist_append( l, v );
			action_do( v->model, ACTION_VERTEX_SEL, l, NULL, NULL, NULL, NULL, NULL );
			g_slist_free( l );
		}
	}
}


void sel_vert_select_list( GSList *l ) {
	
	if( l == NULL || l->data == NULL ) return;
	
	action_do( ((Vertex*)l->data)->model, ACTION_VERTEX_SEL, l, NULL, NULL, NULL, NULL, NULL );

	for( ; l != NULL; l = l->next ) {
		if( l->data != NULL ) 
			sel_vert_select( (Vertex*)l->data, FALSE );
	}
}


void sel_vert_unselect( Vertex *v, int make_undo_entry ) {

	if( v != NULL ) {
		if( v->selected ) {
			v->selected = FALSE;
			v->model->selected_verts = g_slist_remove( v->model->selected_verts, v );

			if( make_undo_entry ) {
				GSList *l = NULL;
				l = g_slist_append( l, v );
				action_do( v->model, ACTION_VERTEX_UNSEL, l, NULL, NULL, NULL, NULL, NULL );
				g_slist_free( l );
			}
		}
	}
}


void sel_vert_unselect_list( GSList *l ) {

	if( l == NULL || l->data == NULL ) return;
	
	action_do( ((Vertex*)l->data)->model, ACTION_VERTEX_UNSEL, l, NULL, NULL, NULL, NULL, NULL );

	for( ; l != NULL; l = l->next ) {
		if( l->data != NULL ) 
			sel_vert_unselect( (Vertex*)l->data, FALSE );
	}
}


void sel_vert_unselect_all( Model *model ) {

	GSList * l = model->selected_verts;
	
	action_do( model, ACTION_VERTEX_UNSEL, model->selected_verts, NULL, NULL, NULL, NULL, NULL );

	for( ; l != NULL; l = l->next ) {
		((Vertex*)l->data)->selected = FALSE;
	}
	
	g_slist_free( model->selected_verts );
	model->selected_verts = NULL;
}


/* VISIBILITY, VERTS ****************************************************/

void sel_vert_hide_selected( Model *model ) {

	GSList *cl, *templist;
	Vertex * v;
	
	templist = g_slist_copy( model->selected_verts );
	sel_vert_unselect_all( model );

	/* go through all the formerly-selected vertices and hide them */
	for( cl = templist; cl != NULL; cl = cl->next ) {
		v = ( Vertex * ) cl->data;
		v->hidden = TRUE;
	}
	
	model->hidden_verts = g_slist_concat( model->hidden_verts, templist );
}


void sel_vert_show_all( Model *model ) {

	GSList *cl;
	Vertex * p;
	
	for ( cl = model->hidden_verts; cl != NULL; cl = cl->next ) {
		/* go through all the hidden vertices and show them */
		p = ( Vertex * ) cl->data;
		p->hidden = FALSE;
	}
	
	g_slist_free( model->hidden_verts );
	model->hidden_verts = NULL;
}


void sel_vert_show( Vertex *v ) {
	if( v == NULL )
		return;
	
	if( v->hidden ) {
		v->hidden = FALSE;
		v->model->hidden_verts = g_slist_remove( v->model->hidden_verts, v );
	}
}



/* SELECTION, POLYS *****************************************************/

void sel_poly_select( Poly* p, int make_undo_entry ) {

	if( p != NULL && !g_slist_find( p->mesh->model->selected_polys, p ) ) {
		p->mesh->model->selected_polys = g_slist_append( p->mesh->model->selected_polys, p );
		p->selected = TRUE;

		if( make_undo_entry ) {
			GSList *l = NULL;
			l = g_slist_append( l, p );
			action_do( p->mesh->model, ACTION_POLY_SEL, l, NULL, NULL, NULL, NULL, NULL );
			g_slist_free( l );
		}
	}
}


void sel_poly_select_list( GSList *l ) {

	if( l == NULL || l->data == NULL ) return;
	
	action_do( ((Poly*)l->data)->mesh->model, ACTION_POLY_SEL, l, NULL, NULL, NULL, NULL, NULL );

	for( ; l != NULL; l = l->next ) {
		if( l->data != NULL ) 
			sel_poly_select( (Poly*)l->data, FALSE );
	}
}


void sel_poly_unselect( Poly* p, int make_undo_entry ) {

	if( p != NULL ) {
		if( p->selected ) {
			p->selected = FALSE;
			p->mesh->model->selected_polys = g_slist_remove( p->mesh->model->selected_polys, p );

			if( make_undo_entry ) {
				GSList *l = NULL;
				l = g_slist_append( l, p );
				action_do( p->mesh->model, ACTION_POLY_UNSEL, l, NULL, NULL, NULL, NULL, NULL );
				g_slist_free( l );
			}
		}
	}
}


void sel_poly_unselect_list( GSList *l ) {

	if( l == NULL || l->data == NULL ) return;
	
	action_do( ((Poly*)l->data)->mesh->model, ACTION_POLY_UNSEL, l, NULL, NULL, NULL, NULL, NULL );

	for( ; l != NULL; l = l->next ) {
		if( l->data != NULL ) 
			sel_poly_unselect( (Poly*)l->data, FALSE );
	}
}


void sel_poly_unselect_all( Model *model ) {

	GSList * l = model->selected_polys;
	
	action_do( model, ACTION_POLY_UNSEL, model->selected_polys, NULL, NULL, NULL, NULL, NULL );

	for( ; l != NULL; l = l->next ) {
		((Poly*)l->data)->selected = FALSE;
	}
	
	g_slist_free( model->selected_polys );
	model->selected_polys = NULL;
}


/* VISIBILITY, POLYS ****************************************************/

void sel_poly_hide_selected( Model *model ) {

	GSList *cl, *templist;
	Poly * p;
	
	templist = g_slist_copy( model->selected_polys );
	sel_poly_unselect_all( model );

	/* go through all the formerly-selected polys and hide them */
	for ( cl = templist; cl != NULL; cl = cl->next ) {
		p = ( Poly * ) cl->data;
		p->hidden = TRUE;
	}
	
	model->hidden_polys = g_slist_concat( model->hidden_polys, templist );
}


void sel_poly_show_all( Model *model ) {

	GSList *cl;
	Poly * p;
	
	for ( cl = model->hidden_polys; cl != NULL; cl = cl->next ) {
		/* go through all the hidden polys and show them */
		p = ( Poly * ) cl->data;
		p->hidden = FALSE;
	}
	
	g_slist_free( model->hidden_polys );
	model->hidden_polys = NULL;
}


void sel_poly_show( Poly* p ) {

	if( p != NULL ) {
		if( p->hidden ) {
			p->hidden = FALSE;
			p->mesh->model->hidden_polys = g_slist_remove( p->mesh->model->hidden_polys, p );
		}
	}
}


