/* 
 * File:         tex_app.c
 * 
 * Description:  dialog for editing the u-v coords of polys
 * 
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <gdk/gdkkeysyms.h> /* for keyboard values */
#include <gdk-pixbuf/gdk-pixbuf.h>


#include "tex_app.h"
#include "geo.h"
#include "texture.h"
#include "polygon.h"
#include "globals.h"
#include "model.h"
#include "gui.h"
#include "vector.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


#define TEXAPP_VERTEX_RADIUS_F     0.01
#define TEXAPP_VERTEX_RADIUS_I     2
#define TEXAPP_LINE_WIDTH          1

/* This define is used to determine the number of msecs that the 
   texapp dialog will wait before forcing a redraw of the glp views.
   Multiple calls to texapp_refresh_views() will result in only a 
   single call to view_redraw_all() */
#define TEXAPP_REDRAW_TIMEOUT 200 

/* STRUCTS ****************************************************************/

struct _tex_app {
    GtkWidget *dialog, *vbox, *hbox;
    GtkWidget *button, *drawing_area;
	GtkWidget *warning_label;
}
tex_app_dialog = { NULL, NULL, NULL, NULL, NULL /* that should be enough */ };

struct _TCNode {
    TexCoord *tc;
    Vertex *v;
    int selected;
    GSList *neighbors;
    int has_been_drawn; /* used to avoid unneccessary redraws */
};
typedef struct _TCNode TCNode;


/* FILE-SCOPE VARS ********************************************************/

GSList *current_tcnodes = NULL;  /* list of tcnodes, created when dlg is shown */

float texapp_prev_x = -1.0;
float texapp_prev_y = -1.0;
int texapp_drag_x = -1;
int texapp_drag_y = -1;
int texapp_hasdragged = FALSE;

int texapp_queued_to_draw = FALSE;

int texapp_polys_from_several_groups = FALSE;

Texture *texapp_texture = NULL;
float texapp_texrep_x = 1.0;
float texapp_texrep_y = 1.0;
GdkPixbuf *texapp_pixbuf = NULL;

/* PROTOTYPES *************************************************************/

void texapp_hide( GtkWidget *widget, gpointer data ) ;
int texapp_destroyed( GtkWidget *widget, GdkEvent *event, gpointer data ) ;
gint texapp_show_help_dialog( GtkWidget *widget, gpointer data ) ;
void texapp_pick_texture( GSList *polys ) ;
void texapp_create_tiled_pixbuf( void ) ;

void tv_to_widget( float tv_x, float tv_y, int* w_x, int* w_y, int w_width, int w_height ) ;
void widget_to_tv( int w_x, int w_y, float* tv_x, float* tv_y, int w_width, int w_height ) ;
gint texapp_refresh_views_cb( gpointer cbdata ) ;
void texapp_refresh_views( void ) ;
void texapp_draw_tcnodes( GtkWidget * widget ) ;
void texapp_draw_texture( GtkWidget * widget ) ;
gint texapp_expose_event( GtkWidget *widget, GdkEventExpose *event, gpointer data ) ;
gint texapp_configure_event( GtkWidget *widget, GdkEvent *event, gpointer data ) ;
void texapp_remap_xy_cb( GtkWidget *widget, gpointer data ) ;
void texapp_remap_yz_cb( GtkWidget *widget, gpointer data ) ;
void texapp_remap_xz_cb( GtkWidget *widget, gpointer data ) ;
void texapp_remap_cyl_z_cb( GtkWidget *widget, gpointer data ) ;
void texapp_remap_cyl_x_cb( GtkWidget *widget, gpointer data ) ;
void texapp_remap_cyl_y_cb( GtkWidget *widget, gpointer data ) ;
void texapp_flip_h_cb( GtkWidget *widget, gpointer data ) ;
void texapp_flip_v_cb( GtkWidget *widget, gpointer data ) ;
gboolean texapp_key_press_cb( GtkWidget *widget, GdkEventKey *event, gpointer data ) ;
gboolean texapp_mouse_down( GtkWidget *widget, GdkEventButton *event,
                            gpointer data ) ;
gboolean texapp_mouse_motion( GtkWidget *widget, GdkEventMotion *event,
                              gpointer data ) ;
gboolean texapp_mouse_up( GtkWidget *widget, GdkEventButton *event,
                          gpointer data ) ;
void texapp_scaleall_cb( GtkWidget *widget, gpointer data ) ;
void texapp_moveall_cb( GtkWidget *widget, gpointer data ) ;
int texapp_switched_model_cb( gpointer no, Model *m, gpointer data ) ;

TCNode *texapp_pick_node( float u, float v ) ;
void texapp_select_area( float start_x, float start_y, float end_x, float end_y ) ;
void texapp_unselect_all( void ) ;
void texapp_add_tcnodes( Poly *p ) ;
void texapp_delete_tcnodes( void ) ;
void texapp_create_tcnodes( void ) ;
void texapp_remap_tcnodes( int uindex, int vindex ) ;
void texapp_remap_cylindrical_tcnodes( int uindex, int vindex, int axis_index ) ;
void texapp_scale_tcnodes( float , float ) ;
void texapp_move_tcnodes( float delta_u, float delta_v ) ;
void texapp_mirror_tcnodes_horiz( void ) ;
void texapp_mirror_tcnodes_vert( void ) ;



/* DIALOG CREATION AND MANAGEMENT ***************************************/

void texapp_hide( GtkWidget *widget, gpointer data ) {

    texapp_delete_tcnodes();

    gtk_widget_hide( tex_app_dialog.dialog );
}


int texapp_destroyed( GtkWidget *widget, GdkEvent *event, gpointer data ) {

	texapp_delete_tcnodes();
	
	printf( "texapp was destroyed\n" );

	return FALSE;
}


void show_tex_app( void ) {

    /* when invoked, this dialog allows user to edit tex coords for 
       the selected polys */

    GtkWidget *button;
	GtkWidget *frame, *vbox, *table, *label;
	GtkWidget *outermost_vbox;

    texapp_create_tcnodes();
    
    if( tex_app_dialog.dialog == NULL ) {
    
        tex_app_dialog.dialog = gtk_window_new(GTK_WINDOW_TOPLEVEL);
        gtk_container_set_border_width(GTK_CONTAINER(tex_app_dialog.dialog), 5);
        gtk_window_set_title(GTK_WINDOW(tex_app_dialog.dialog), "Edit Texture Coordinates");
    
		/* make the window a child of the TLW */
		gtk_window_set_transient_for( GTK_WINDOW(tex_app_dialog.dialog), GTK_WINDOW(TopLevelWindow) );
		/* ensure that the window will go away when the program exits */
		gtk_window_set_destroy_with_parent( GTK_WINDOW(tex_app_dialog.dialog), TRUE );

		outermost_vbox = gtk_vbox_new( FALSE, 10 );
        gtk_container_add(GTK_CONTAINER(tex_app_dialog.dialog), outermost_vbox );

		/* BOXES *********************************************************/
		
		tex_app_dialog.hbox = gtk_hbox_new( FALSE, 0 );
		gtk_box_pack_end( GTK_BOX( outermost_vbox ), tex_app_dialog.hbox, FALSE, FALSE, 0 );

		tex_app_dialog.vbox = gtk_vbox_new( FALSE, 0 );
		
		/* WARNING MESSAGE ***********************************************/
		
		tex_app_dialog.warning_label = gtk_label_new( 
			"Warning: You are attempting to edit the \n"
			"texture coordinates on polygons from multiple \n"
			"meshes.  Note that the meshes might not share \n"
			"the same texture attributes, so be careful." );
        gtk_box_pack_start( GTK_BOX( tex_app_dialog.vbox ), tex_app_dialog.warning_label, TRUE, FALSE, 0 );

        /* PLANAR TEXTURE REMAPPING ************************************/
		
		frame = gtk_frame_new( "Planar Texture Remapping" );
		gtk_container_set_border_width( GTK_CONTAINER( frame ), 10 );
        gtk_box_pack_start( GTK_BOX( tex_app_dialog.vbox ), frame, TRUE, FALSE, 0 );

		vbox = gtk_vbox_new( FALSE, 0 );
		gtk_container_add( GTK_CONTAINER( frame ), vbox );

		
		button = gtk_button_new_with_label( "Remap XY" );
        g_signal_connect( G_OBJECT( button ), "clicked",
                            G_CALLBACK( texapp_remap_xy_cb ), NULL );
        gtk_box_pack_start( GTK_BOX( vbox ), button, TRUE, FALSE, 0 );

        button = gtk_button_new_with_label( "Remap YZ" );
        g_signal_connect( G_OBJECT( button ), "clicked",
                            G_CALLBACK( texapp_remap_yz_cb ), NULL );
        gtk_box_pack_start( GTK_BOX( vbox ), button, TRUE, FALSE, 0 );

        button = gtk_button_new_with_label( "Remap XZ" );
        g_signal_connect( G_OBJECT( button ), "clicked",
                            G_CALLBACK( texapp_remap_xz_cb ), NULL );
        gtk_box_pack_start( GTK_BOX( vbox ), button, TRUE, FALSE, 0 );


		button = gtk_button_new_with_label( "Cylindrical Remap Z" );
        g_signal_connect( G_OBJECT( button ), "clicked",
                            G_CALLBACK( texapp_remap_cyl_z_cb ), NULL );
        gtk_box_pack_start( GTK_BOX( vbox ), button, TRUE, FALSE, 0 );

		button = gtk_button_new_with_label( "Cylindrical Remap X" );
        g_signal_connect( G_OBJECT( button ), "clicked",
                            G_CALLBACK( texapp_remap_cyl_x_cb ), NULL );
        gtk_box_pack_start( GTK_BOX( vbox ), button, TRUE, FALSE, 0 );

		button = gtk_button_new_with_label( "Cylindrical Remap Y" );
        g_signal_connect( G_OBJECT( button ), "clicked",
                            G_CALLBACK( texapp_remap_cyl_y_cb ), NULL );
        gtk_box_pack_start( GTK_BOX( vbox ), button, TRUE, FALSE, 0 );

		/* ARROW BUTTONS - MOVE ****************************************/
		
		frame = gtk_frame_new( "Move/Scale All Points" );
		gtk_container_set_border_width( GTK_CONTAINER( frame ), 10 );
		gtk_box_pack_start( GTK_BOX( tex_app_dialog.vbox ), frame, TRUE, FALSE, 0 );

		vbox = gtk_vbox_new( FALSE, 0 );
		gtk_container_add( GTK_CONTAINER( frame ), vbox );

		table = gtk_table_new( 3, 3, FALSE );
		
		button = gtk_button_new();
		gtk_container_add( GTK_CONTAINER( button ), 
			gtk_arrow_new( GTK_ARROW_UP, GTK_SHADOW_NONE ) );
		g_signal_connect( G_OBJECT( button ), "clicked",
							G_CALLBACK( texapp_moveall_cb ), (gpointer)GDK_Up );
		gtk_table_attach_defaults( GTK_TABLE(table), button, 
									1, 2, 0, 1 );

		button = gtk_button_new();
		gtk_container_add( GTK_CONTAINER( button ), 
			gtk_arrow_new( GTK_ARROW_LEFT, GTK_SHADOW_NONE ) );
		g_signal_connect( G_OBJECT( button ), "clicked",
							G_CALLBACK( texapp_moveall_cb ), (gpointer)GDK_Left );
		gtk_table_attach_defaults( GTK_TABLE(table), button, 
									0, 1, 1, 2 );

		button = gtk_button_new();
		gtk_container_add( GTK_CONTAINER( button ), 
			gtk_arrow_new( GTK_ARROW_RIGHT, GTK_SHADOW_NONE ) );
		g_signal_connect( G_OBJECT( button ), "clicked",
							G_CALLBACK( texapp_moveall_cb ), (gpointer)GDK_Right );
		gtk_table_attach_defaults( GTK_TABLE(table), button, 
									2, 3, 1, 2 );

		button = gtk_button_new();
		gtk_container_add( GTK_CONTAINER( button ), 
			gtk_arrow_new( GTK_ARROW_DOWN, GTK_SHADOW_NONE ) );
		g_signal_connect( G_OBJECT( button ), "clicked",
							G_CALLBACK( texapp_moveall_cb ), (gpointer)GDK_Down );
		gtk_table_attach_defaults( GTK_TABLE(table), button, 
									1, 2, 2, 3 );
		
		label = gtk_label_new( "Move" );
		gtk_table_attach_defaults( GTK_TABLE(table), label, 
									1, 2, 1, 2 );

		gtk_box_pack_start( GTK_BOX( vbox ), table, TRUE, FALSE, 0 );

		/* ARROW BUTTONS - SCALE ***************************************/
		
		table = gtk_table_new( 3, 3, FALSE );
		
		button = gtk_button_new();
		gtk_container_add( GTK_CONTAINER( button ), 
			gtk_arrow_new( GTK_ARROW_UP, GTK_SHADOW_NONE ) );
		g_signal_connect( G_OBJECT( button ), "clicked",
							G_CALLBACK( texapp_scaleall_cb ), (gpointer)GDK_Up );
		gtk_table_attach_defaults( GTK_TABLE(table), button, 
									1, 2, 0, 1 );

		button = gtk_button_new();
		gtk_container_add( GTK_CONTAINER( button ), 
			gtk_arrow_new( GTK_ARROW_LEFT, GTK_SHADOW_NONE ) );
		g_signal_connect( G_OBJECT( button ), "clicked",
							G_CALLBACK( texapp_scaleall_cb ), (gpointer)GDK_Left );
		gtk_table_attach_defaults( GTK_TABLE(table), button, 
									0, 1, 1, 2 );

		button = gtk_button_new();
		gtk_container_add( GTK_CONTAINER( button ), 
			gtk_arrow_new( GTK_ARROW_RIGHT, GTK_SHADOW_NONE ) );
		g_signal_connect( G_OBJECT( button ), "clicked",
							G_CALLBACK( texapp_scaleall_cb ), (gpointer)GDK_Right );
		gtk_table_attach_defaults( GTK_TABLE(table), button, 
									2, 3, 1, 2 );

		button = gtk_button_new();
		gtk_container_add( GTK_CONTAINER( button ), 
			gtk_arrow_new( GTK_ARROW_DOWN, GTK_SHADOW_NONE ) );
		g_signal_connect( G_OBJECT( button ), "clicked",
							G_CALLBACK( texapp_scaleall_cb ), (gpointer)GDK_Down );
		gtk_table_attach_defaults( GTK_TABLE(table), button, 
									1, 2, 2, 3 );

		label = gtk_label_new( "Scale" );
		gtk_table_attach_defaults( GTK_TABLE(table), label, 
									1, 2, 1, 2 );

		gtk_box_pack_start( GTK_BOX( vbox ), table, TRUE, FALSE, 0 );

		
		/* OTHER MANIP BUTTONS ********************************************/
		
		frame = gtk_frame_new( "Misc." );
		gtk_container_set_border_width( GTK_CONTAINER( frame ), 10 );
        gtk_box_pack_start( GTK_BOX( tex_app_dialog.vbox ), frame, TRUE, FALSE, 0 );

		vbox = gtk_vbox_new( FALSE, 0 );
		gtk_container_add( GTK_CONTAINER( frame ), vbox );

		
		button = gtk_button_new_with_label( "Flip All Horizontally" );
        g_signal_connect( G_OBJECT( button ), "clicked",
                            G_CALLBACK( texapp_flip_h_cb ), NULL );
        gtk_box_pack_start( GTK_BOX( vbox ), button, TRUE, FALSE, 0 );

		button = gtk_button_new_with_label( "Flip All Vertically" );
        g_signal_connect( G_OBJECT( button ), "clicked",
                            G_CALLBACK( texapp_flip_v_cb ), NULL );
        gtk_box_pack_start( GTK_BOX( vbox ), button, TRUE, FALSE, 0 );

		
		/* MISC DIALOG BUTTONS ********************************************/

        button = gtk_button_new_with_label( "Help" );
        g_signal_connect( G_OBJECT( button ), "clicked",
                            G_CALLBACK( texapp_show_help_dialog ), NULL );
        gtk_box_pack_start( GTK_BOX( tex_app_dialog.vbox ), button, TRUE, FALSE, 0 );

        tex_app_dialog.button = gtk_button_new_with_label( "Close" );
        g_signal_connect( G_OBJECT( tex_app_dialog.button ), "clicked",
                            G_CALLBACK( texapp_hide ), NULL );
        gtk_box_pack_start( GTK_BOX( tex_app_dialog.vbox ), tex_app_dialog.button, TRUE, FALSE, 0 );


		/* DRAWING AREA **************************************************/

        frame = gtk_frame_new( "Texture Preview" );
		gtk_container_set_border_width( GTK_CONTAINER( frame ), 10 );
    
        tex_app_dialog.drawing_area = gtk_drawing_area_new();
        gtk_widget_set_events( GTK_WIDGET( tex_app_dialog.drawing_area ), 
                               GDK_EXPOSURE_MASK|
                               GDK_BUTTON_PRESS_MASK|
                               GDK_BUTTON_RELEASE_MASK|
                               GDK_POINTER_MOTION_MASK|
                               GDK_POINTER_MOTION_HINT_MASK );
        gtk_widget_set_size_request( tex_app_dialog.drawing_area, TEXTURE_MIP_LEVEL_LARGE_SIZE, TEXTURE_MIP_LEVEL_LARGE_SIZE );
        gtk_container_add( GTK_CONTAINER( frame ),
                           tex_app_dialog.drawing_area );
        g_signal_connect ( G_OBJECT ( tex_app_dialog.drawing_area ),
                             "expose_event",
                             G_CALLBACK( texapp_expose_event ), NULL );
        g_signal_connect ( G_OBJECT ( tex_app_dialog.drawing_area ),
                             "configure_event",
                             G_CALLBACK( texapp_configure_event ), NULL );
        g_signal_connect( G_OBJECT( tex_app_dialog.drawing_area ), "button_press_event",
                            G_CALLBACK( texapp_mouse_down ), NULL );
        g_signal_connect( G_OBJECT( tex_app_dialog.drawing_area ), "motion_notify_event",
                            G_CALLBACK( texapp_mouse_motion ), NULL );
        g_signal_connect( G_OBJECT( tex_app_dialog.drawing_area ), "button_release_event",
                            G_CALLBACK( texapp_mouse_up ), NULL );

        g_signal_connect( G_OBJECT( tex_app_dialog.dialog ), "key-press-event",
                            G_CALLBACK( texapp_key_press_cb ), NULL );

		/* catches event generated when user clicks on the window manager's 
		close-window button ('X' for most WM's) */
		g_signal_connect( G_OBJECT( tex_app_dialog.dialog ), "delete-event",
							G_CALLBACK( texapp_hide ), NULL );

		/* catches event generated if user sends the 'destroy' event 
		to this dialog.  some window managers are capable of this, but not 
		all? */
		g_signal_connect( G_OBJECT( tex_app_dialog.dialog ), "destroy-event",
							G_CALLBACK( texapp_destroyed ), NULL );

    
        gtk_box_pack_start( GTK_BOX( tex_app_dialog.hbox ), frame, FALSE, FALSE, 0 );
        gtk_box_pack_start( GTK_BOX( tex_app_dialog.hbox ), tex_app_dialog.vbox, FALSE, FALSE, 0 );

		gtk_widget_show_all( outermost_vbox );


		/* register a listener for the switched-model and removed-model 
		signals */
		g_signal_connect( notificationObj, "switched-model", 
						G_CALLBACK(texapp_switched_model_cb), NULL );
		g_signal_connect( notificationObj, "removed-model", 
						G_CALLBACK(texapp_switched_model_cb), NULL );
		

    }

	if( texapp_polys_from_several_groups ) {
		gtk_widget_show( tex_app_dialog.warning_label );
	} else {
		gtk_widget_hide( tex_app_dialog.warning_label );
	}

	texapp_pick_texture( the_model->selected_polys );
	texapp_create_tiled_pixbuf();
    
    gtk_widget_show( tex_app_dialog.dialog );
    texapp_draw_texture( tex_app_dialog.drawing_area );

}


gint texapp_show_help_dialog( GtkWidget *widget, gpointer data ) {
	GtkWidget * dialog, *label, *button;

	dialog = gtk_dialog_new();

	button = gtk_button_new_with_label( "OK" );
	gtk_box_pack_start( GTK_BOX( GTK_DIALOG( dialog ) ->action_area ), button,
						FALSE, FALSE, 5 );
	g_signal_connect_swapped( G_OBJECT( button ), "clicked",
					  G_CALLBACK( gtk_widget_destroy ), dialog );
	gtk_widget_show( button );

	label = gtk_label_new( 
		"Mouse actions: \n"
		"\tButton 1\n\t\tToggle selection for point under cursor; \n"
		"\t\tclick and drag to select a rectangular area\n"
		"\tButton 2 & 3\n\t\tMove selected points\n"
		"Keyboard shortcuts: \n"
		"\tArrow keys\n\t\tMove all points; hold Shift to scale instead" );
/*	gtk_label_set_justify( GTK_LABEL( label ), GTK_JUSTIFY_CENTER );*/
	gtk_widget_show( label );
	gtk_box_pack_start( GTK_BOX( GTK_DIALOG( dialog ) ->vbox ), label,
						FALSE, FALSE, 5 );

	gtk_widget_show( dialog );

	return TRUE;
}


void texapp_pick_texture( GSList *polys ) {

	/* Given a list of polys, find the most frequently-occuring mesh and 
	remember it.  This most-popular-mesh's texture & tex-repeat vals will 
	be the ones we use. */

	GSList *spl;
	Mesh *mpm = NULL;
	int mpm_count = 0;
	GHashTable *mhash = NULL;
	
	mhash = g_hash_table_new( NULL, NULL );
	
	for( spl = polys; spl; spl = spl->next ) {
		Poly *p = (Poly*)spl->data;
		gpointer countp;
		int count;
		
		countp = g_hash_table_lookup( mhash, p->mesh );
		count = (int)countp; /* damnable whining compilers */
		count++;
		
		g_hash_table_replace( mhash, p->mesh, (gpointer)count );
		
		if( mpm == NULL || count > mpm_count ) {
			mpm = p->mesh;
			mpm_count = count;
		}
		
	}
	
	g_hash_table_destroy( mhash );
	
	if( mpm != NULL ) {
		texapp_texture = mpm->texture;
		texapp_texrep_x = mpm->texrep_x;
		texapp_texrep_y = mpm->texrep_y;
	} else {
		texapp_texture = NULL;
		texapp_texrep_x = 1.0;
		texapp_texrep_y = 1.0;
	}
}



void texapp_create_tiled_pixbuf( void ) {
	
	/* Creates a pixbuf from the original texture for display in the 
	tex-coord editing window.  Since the polys' mesh may have texture-repeat 
	values other than 1, we will scale (and possibly tile) the result 
	to match what the user would see in the kludge3d opengl views. */
	
	tex_data *td = NULL;
	float tile_w, tile_h;
	int tile_w_i, tile_h_i;
	float dest_x, dest_y;
	GdkPixbuf *tile = NULL, *dest;
	
	if( texapp_pixbuf ) {
		g_object_unref( texapp_pixbuf );
		texapp_pixbuf = NULL;
	}

	if( texapp_texture == NULL ) 
		return;
	
	td = &(texapp_texture->mip[TEXTURE_MIP_LEVEL_LARGE]);
	
	/* if the texture repeat is 1 for both u and v, skip all of this 
	tiling/scaling nonsense and bail */
	if( fabs( 1.0 - texapp_texrep_x ) < 0.01 && 
	    fabs( 1.0 - texapp_texrep_y ) < 0.01 ) 
	{
		texapp_pixbuf = gdk_pixbuf_new_from_data( 
			td->data, GDK_COLORSPACE_RGB, 
			(td->depth == 4), 8, 
			td->width, td->height, 
			td->width * td->depth, /* rowstride */
			NULL, NULL /* no destructor func is specified */
			);
		return;
	}
	
	texapp_pixbuf = gdk_pixbuf_new( GDK_COLORSPACE_RGB, FALSE, 8, 
									td->width, td->height );
	dest = texapp_pixbuf;
	
	tile_w = (float)( td->width ) / texapp_texrep_x;
	tile_h = (float)( td->height ) / texapp_texrep_y;
	tile_w_i = (int)tile_w;
	tile_h_i = (int)tile_h;

	tile = gdk_pixbuf_scale_simple( texapp_texture->pixbuf, 
				tile_w_i, tile_h_i, GDK_INTERP_HYPER );
	
	for( dest_x = 0.; dest_x < (float)( td->width ); dest_x += tile_w ) {
		for( dest_y = 0.; dest_y < (float)( td->height ); dest_y += tile_h ) {
			int w, h;
			w = td->width - (int)dest_x;
			h = td->height - (int)dest_y;
			gdk_pixbuf_copy_area( tile, 0, 0, 
								  w < tile_w_i ? w : tile_w_i, 
								  h < tile_h_i ? h : tile_h_i, 
								  dest, (int)dest_x, (int)dest_y );
		}
	}
	
	g_object_unref( tile );
	
}


/* FUNCS ******************************************************************/

void tv_to_widget( float tv_x, float tv_y, int* w_x, int* w_y, int w_width, int w_height ) {

    *w_x = (int)( fabs(tv_x) * (float)w_width );
    *w_y = (int)( fabs(tv_y) * (float)w_height );
    
    // opengl coords -> X11 coords
    *w_y = w_height - *w_y;
}
void widget_to_tv( int w_x, int w_y, float* tv_x, float* tv_y, int w_width, int w_height ) {

    // X11 coords -> opengl coords
    w_y = w_height - w_y;
    
    *tv_x = (float)w_x / (float)w_width;
    *tv_y = (float)w_y / (float)w_height;
}


gint texapp_refresh_views_cb( gpointer cbdata ) {

    /* a callback to actually perform the draw */
    
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
    texapp_queued_to_draw = FALSE;

    /* important... */
    return FALSE;
}


void texapp_refresh_views( void ) {

    /* performing the redraws in this manner will hopefully 
       avoid clobbering the view_redraw_all() function when a texcoord is 
       dragged.  
     */
    
    if( !texapp_queued_to_draw ) {
        texapp_queued_to_draw = TRUE;
        gtk_timeout_add( TEXAPP_REDRAW_TIMEOUT, texapp_refresh_views_cb, NULL );
    }

}



void texapp_draw_tcnodes( GtkWidget * widget ) {

    GdkGC * gc;
    GdkColor color;
    GdkColormap *colormap;
    
    GSList *l, *neighbors;
    int w_t_x, w_t_y;
    int w_n_x, w_n_y;
    
    int widget_w, widget_h;
    
    widget_w = widget->allocation.width;
    widget_h = widget->allocation.height;
    
    gc = gdk_gc_new( widget->window );
    colormap = gdk_drawable_get_colormap( widget->window );
    gdk_color_white( colormap, &color );
    gdk_gc_set_foreground( gc, &color );
    gdk_color_black( colormap, &color );
    gdk_gc_set_background( gc, &color );
    gdk_gc_set_line_attributes( gc, TEXAPP_LINE_WIDTH, 
        GDK_LINE_DOUBLE_DASH, GDK_CAP_PROJECTING, GDK_JOIN_MITER );


/*
printf( "tv's : %f %f - %f %f - %f %f\n",
t->tv[0].x, t->tv[0].y,
t->tv[1].x, t->tv[1].y,
t->tv[2].x, t->tv[2].y );
*/

    
    for( l = current_tcnodes; l; l = l->next ) {
        TCNode *t = (TCNode*)l->data;

        tv_to_widget( t->tc->x, t->tc->y, &w_t_x, &w_t_y, widget_w, widget_h );
    
        for( neighbors = t->neighbors; neighbors; neighbors = neighbors->next ) {
            TCNode *n = (TCNode*)neighbors->data;
        
            if( !(n->has_been_drawn) ) {
                tv_to_widget( n->tc->x, n->tc->y, &w_n_x, &w_n_y, widget_w, widget_h );
    
                gdk_draw_line( widget->window, gc, w_t_x, w_t_y, w_n_x, w_n_y );
            }
        }
        

        t->has_been_drawn = TRUE;
    }
    
   gdk_gc_set_line_attributes( gc, TEXAPP_LINE_WIDTH, 
       GDK_LINE_SOLID, GDK_CAP_PROJECTING, GDK_JOIN_MITER );

    /* the time saved by not redrawing each line multiple times 
       more than justifies the extra time spent (in the loop below)
       to clear the redraw flags
     */
    for( l = current_tcnodes; l; l = l->next ) {
        TCNode *t = (TCNode*)l->data;

        /* time to draw the vertex */
        tv_to_widget( t->tc->x, t->tc->y, &w_t_x, &w_t_y, widget_w, widget_h );
        gdk_draw_rectangle( widget->window, gc,
                            t->selected,
                            w_t_x - TEXAPP_VERTEX_RADIUS_I,
                            w_t_y - TEXAPP_VERTEX_RADIUS_I,
                            TEXAPP_VERTEX_RADIUS_I << 1,
                            TEXAPP_VERTEX_RADIUS_I << 1 );

        /* clean up for next time */
        t->has_been_drawn = FALSE;
    }

    gdk_gc_unref( gc );
}



void texapp_draw_texture( GtkWidget * widget ) {
    
//FIXME - let it be known that this can only handle fixed-size textures
// (which is ok, for now)

	if( texapp_pixbuf ) {
		
		gdk_pixbuf_render_to_drawable( 
			texapp_pixbuf,
			widget->window,
			widget->style->fg_gc[ GTK_WIDGET_STATE ( widget ) ],
			0, 0, 
			0, 0, 
			gdk_pixbuf_get_width( texapp_pixbuf ), 
			gdk_pixbuf_get_height( texapp_pixbuf ), 
			GDK_RGB_DITHER_NORMAL, 0, 0 );
		
	} else {
        gdk_draw_rectangle( widget->window,
                            widget->style->fg_gc[ GTK_WIDGET_STATE ( widget ) ],
                            TRUE, 0, 0,
                            widget->allocation.width,
                            widget->allocation.height );
    }
    
    texapp_draw_tcnodes( widget );
    

}


gint texapp_expose_event( GtkWidget *widget, GdkEventExpose *event, gpointer data ) {

    texapp_draw_texture( widget );
    
    return 0;
}

gint texapp_configure_event( GtkWidget *widget, GdkEvent *event, gpointer data ) {

    if( texapp_texture != NULL ) {

        gtk_widget_queue_draw( tex_app_dialog.drawing_area );
        gdk_flush();
    }

    return TRUE;
}


/* USER EVENT CALLBACKS *************************************************/


void texapp_remap_xy_cb( GtkWidget *widget, gpointer data ) {

    texapp_remap_tcnodes( 0, 1 );

    texapp_draw_texture( tex_app_dialog.drawing_area );
    texapp_refresh_views();
}

void texapp_remap_yz_cb( GtkWidget *widget, gpointer data ) {

    texapp_remap_tcnodes( 1, 2 );

    texapp_draw_texture( tex_app_dialog.drawing_area );
    texapp_refresh_views();
}

void texapp_remap_xz_cb( GtkWidget *widget, gpointer data ) {

    texapp_remap_tcnodes( 0, 2 );

    texapp_draw_texture( tex_app_dialog.drawing_area );
    texapp_refresh_views();
}


void texapp_remap_cyl_z_cb( GtkWidget *widget, gpointer data ) {

    texapp_remap_cylindrical_tcnodes( 0, 1, 2 );

    texapp_draw_texture( tex_app_dialog.drawing_area );
    texapp_refresh_views();
}

void texapp_remap_cyl_x_cb( GtkWidget *widget, gpointer data ) {

    texapp_remap_cylindrical_tcnodes( 1, 2, 0 );

    texapp_draw_texture( tex_app_dialog.drawing_area );
    texapp_refresh_views();
}

void texapp_remap_cyl_y_cb( GtkWidget *widget, gpointer data ) {

    texapp_remap_cylindrical_tcnodes( 0, 2, 1 );

    texapp_draw_texture( tex_app_dialog.drawing_area );
    texapp_refresh_views();
}


void texapp_flip_h_cb( GtkWidget *widget, gpointer data ) {

	texapp_mirror_tcnodes_horiz();

	texapp_draw_texture( tex_app_dialog.drawing_area );
	texapp_refresh_views();
}

void texapp_flip_v_cb( GtkWidget *widget, gpointer data ) {

	texapp_mirror_tcnodes_vert();

	texapp_draw_texture( tex_app_dialog.drawing_area );
	texapp_refresh_views();
}


gboolean texapp_key_press_cb( GtkWidget *widget, GdkEventKey *event, gpointer data ) {

	switch( event->keyval ) {

		case '+': case '=':
			texapp_scale_tcnodes( 1.01, 1.01 );
			break;
		case '-': case '_':
			texapp_scale_tcnodes( 0.99, 0.99 );
			break;
		default:
			if( event->state & GDK_SHIFT_MASK )
				texapp_scaleall_cb( NULL, (gpointer)event->keyval );
			else
				texapp_moveall_cb( NULL, (gpointer)event->keyval );
			break;
	}

	texapp_draw_texture( tex_app_dialog.drawing_area );
	texapp_refresh_views();
	
	/* returning TRUE indicates that the event has been handled and that 
	it should not be passed on to other handlers.  This is important, since 
	we don't want the widgets to change focus every time the arrow keys are 
	pressed. */
	return TRUE;
}



gboolean texapp_mouse_down( GtkWidget *widget, GdkEventButton *event,
                            gpointer data ) {

    texapp_prev_x = texapp_prev_y = -1.0;
    
    if( event->button == 1 ) {
        
        texapp_drag_x = event->x;
        texapp_drag_y = event->y;
        texapp_hasdragged = FALSE;
        
    }

    return TRUE;
}




gboolean texapp_mouse_motion( GtkWidget *widget, GdkEventMotion *event,
                              gpointer data ) {

    int x, y;
    GdkModifierType state;
    float current_x = 0.0;
    float current_y = 0.0;
    float delta_u, delta_v;
    GSList *l = NULL;


    if ( event->is_hint )
        gdk_window_get_pointer ( event->window, &x, &y, &state );
    else {
        x = event->x;
        y = event->y;
        state = event->state;
    }
    
    if( x < 0 || x > widget->allocation.width ||
        y < 0 || y > widget->allocation.height )
        return TRUE;

    widget_to_tv( x, y, &current_x, &current_y, widget->allocation.width, widget->allocation.height );

    if( texapp_prev_x < 0 || texapp_prev_y < 0 ) {
        texapp_prev_x = current_x;
        texapp_prev_y = current_y;
    }
    
    delta_u = current_x - texapp_prev_x;
    delta_v = current_y - texapp_prev_y;
    
    if( state & GDK_BUTTON1_MASK ) {
        /* select */
        texapp_hasdragged = TRUE;

    } else if( state & GDK_BUTTON2_MASK || state & GDK_BUTTON3_MASK ) {
        /* move */
        for( l = current_tcnodes; l; l = l->next ) {
            TCNode *t = (TCNode*)l->data;
            
            if( t->selected ) {
                t->tc->x += delta_u;
                t->tc->y += delta_v;
            }
        }
        texapp_draw_texture( widget );
    }

    texapp_prev_x = current_x;
    texapp_prev_y = current_y;

    return TRUE;
}



gboolean texapp_mouse_up( GtkWidget *widget, GdkEventButton *event,
                          gpointer data ) {

    float start_x, start_y, end_x, end_y;
    TCNode *n;

    texapp_prev_x = texapp_prev_y = -1.0;
    widget_to_tv( event->x, event->y, &end_x, &end_y, widget->allocation.width, widget->allocation.height );

    if( event->button == 1 ) {
        
        if( texapp_hasdragged ) {
            
            widget_to_tv( texapp_drag_x, texapp_drag_y, &start_x, &start_y, widget->allocation.width, widget->allocation.height );
            
            texapp_select_area( start_x, start_y, end_x, end_y );
            
            texapp_hasdragged = FALSE;

        } else {

            n = texapp_pick_node( end_x, end_y );
            if( n ) {
                n->selected = !n->selected;
            } else {
                texapp_unselect_all();
            }
        }
    }

    texapp_draw_texture( widget );
    texapp_refresh_views();
    return TRUE;
}


void texapp_scaleall_cb( GtkWidget *widget, gpointer data ) {

	switch( (int)data ) {

		case GDK_Left:
			texapp_scale_tcnodes( 0.99, 1.00 );
			break;
		case GDK_Right:
			texapp_scale_tcnodes( 1.01, 1.00 );
			break;
		case GDK_Up:
			texapp_scale_tcnodes( 1.00, 1.01 );
			break;
		case GDK_Down:
			texapp_scale_tcnodes( 1.00, 0.99 );
			break;
		default:
			break;
	}

	texapp_draw_texture( tex_app_dialog.drawing_area );
	texapp_refresh_views();
}


void texapp_moveall_cb( GtkWidget *widget, gpointer data ) {

	switch( (int)data ) {

		case GDK_Left:
			texapp_move_tcnodes( -0.01, 0.0 );
			break;
		case GDK_Right:
			texapp_move_tcnodes( 0.01, 0.0 );
			break;
		case GDK_Up:
			texapp_move_tcnodes( 0.0, 0.01 );
			break;
		case GDK_Down:
			texapp_move_tcnodes( 0.0, -0.01 );
			break;
		default:
			break;
	}

	texapp_draw_texture( tex_app_dialog.drawing_area );
	texapp_refresh_views();
}


int texapp_switched_model_cb( gpointer no, Model *m, gpointer data ) {
	/* The switched-model signal is emitted whenever the user switches models.
	The texapp dialog should be hidden when this happens.  It would be very 
	confusing if the user were able to edit tex coords on a model that is 
	no longer visible.  This func is also used for the removed-model signal. */
	
	if( tex_app_dialog.dialog != NULL ) {
		texapp_hide( NULL, NULL );
	}
	return FALSE;
}

/* SELECTION FUNCS ******************************************************/


TCNode *texapp_pick_node( float u, float v ) {
    
    GSList *tcnl;
    TCNode *closest_node = NULL;
    float closest_delta_u, closest_delta_v;
    float current_delta_u, current_delta_v;
    float current_delta;
    float closest_delta = TEXAPP_VERTEX_RADIUS_F;
    
printf( "----\n" );

    for( tcnl = current_tcnodes ; tcnl; tcnl = tcnl->next ) {
        TCNode *t = (TCNode*)tcnl->data;

printf( "looking for <%f %f>, considering <%f %f>\n", u, v, t->tc->x, t->tc->y );
        
        current_delta_u = fabs( u - t->tc->x );
        current_delta_v = fabs( v - t->tc->y );
        
        current_delta = sqrt( current_delta_u*current_delta_u + 
                              current_delta_v*current_delta_v );

        if( closest_node ) {
        closest_delta_u = fabs( u - closest_node->tc->x );
        closest_delta_v = fabs( v - closest_node->tc->y );
        
        closest_delta = sqrt( closest_delta_u*closest_delta_u + 
                              closest_delta_v*closest_delta_v );
        }

        if( current_delta < TEXAPP_VERTEX_RADIUS_F && 
            current_delta < closest_delta ) {
            closest_node = t;
        }

    }
    
    return closest_node;
}


void texapp_select_area( float start_x, float start_y, float end_x, float end_y ) {
    
    GSList *tcnl;
    
    if( start_x > end_x ) {
        texapp_select_area( end_x, start_y, start_x, end_y );
        return;
    }
    if( start_y > end_y ) {
        texapp_select_area( start_x, end_y, end_x, start_y );
        return;
    }

    for( tcnl = current_tcnodes ; tcnl; tcnl = tcnl->next ) {
        TCNode *t = (TCNode*)tcnl->data;
        
        t->selected = FALSE;
        
        if( t->tc->x > start_x && t->tc->x < end_x )
            if( t->tc->y > start_y && t->tc->y < end_y )
                t->selected = TRUE;
    }
    
}

void texapp_unselect_all( void ) {
    
    GSList *tcnl;

    for( tcnl = current_tcnodes ; tcnl; tcnl = tcnl->next ) {
        TCNode *t = (TCNode*)tcnl->data;
        
        t->selected = FALSE;
    }
    
}


/* NODE MANAGEMENT FUNCS ************************************************/


void texapp_add_tcnodes( Poly *p ) {

    /* creates tcnodes given Poly p, sets up their neighbors, and adds them */

    TCNode *tcns[POLY_MAX_VERTS];
    int i, j;
    
    if( p == NULL ) return;
    
    /* take care of the easy stuff... */
    for( i = 0; i < p->num_verts; i++ ) {
        tcns[i] = (TCNode*)malloc( sizeof( TCNode ) );
        
        tcns[i]->v = p->verts[i];
        tcns[i]->tc = p->tc[i];
        tcns[i]->selected = FALSE;
        tcns[i]->neighbors = NULL;
        tcns[i]->has_been_drawn = FALSE;
    }
    
    /* set up the neighbor list, and add to the current_tcnodes */
    for( i = 0; i < p->num_verts; i++ ) {
        for( j = 0; j < p->num_verts; j++ ) {
            if( i != j ) {
                tcns[i]->neighbors = g_slist_append( tcns[i]->neighbors, tcns[j] );
            }
        }
        current_tcnodes = g_slist_append( current_tcnodes, tcns[i] );
    }

}


void texapp_delete_tcnodes( void ) {

    GSList *tcnl;
    
    for( tcnl = current_tcnodes; tcnl; tcnl = tcnl->next ) {
        TCNode *n = (TCNode*)tcnl->data;
        
        if( n )
            free( n );
    }
    
    g_slist_free( current_tcnodes );
    current_tcnodes = NULL;
}

void texapp_create_tcnodes( void ) {

    GSList *spl;
	Mesh *m = NULL;
    
    texapp_delete_tcnodes();
	
	texapp_polys_from_several_groups = FALSE;
    
    for( spl = the_model->selected_polys; spl; spl = spl->next ) {
        Poly *p = (Poly*)spl->data;
        
		if( m == NULL ) {
			m = p->mesh;
		} else {
			if( m != p->mesh )
				texapp_polys_from_several_groups = TRUE;
		}
		
        texapp_add_tcnodes( p );
    }
}


/* NODE MANIPULATION FUNCS **********************************************/


void texapp_remap_tcnodes( int uindex, int vindex ) {

    GSList *tcnl = NULL;
    float min_x, min_y, max_x, max_y;
    float scale_x, scale_y;
    
    min_x = min_y = 10000000.0;  /* fixme - is ten million too little? */
    max_x = max_y = -10000000.0;  /* fixme - is neg. ten million too big? */

    for( tcnl = current_tcnodes; tcnl; tcnl = tcnl->next ) {
        TCNode * n = (TCNode*)tcnl->data;
        
        if( n->v->v[uindex] < min_x ) min_x = n->v->v[uindex];
        if( n->v->v[vindex] < min_y ) min_y = n->v->v[vindex];

        if( n->v->v[uindex] > max_x ) max_x = n->v->v[uindex];
        if( n->v->v[vindex] > max_y ) max_y = n->v->v[vindex];
        
        /* set these now, just for convenience */
        n->tc->x = n->v->v[uindex];
        n->tc->y = n->v->v[vindex];
    }
    
    scale_x = 1.0 / (max_x - min_x);
    scale_y = 1.0 / (max_y - min_y);

    /* final loop, to actually perform the mapping */
    for( tcnl = current_tcnodes; tcnl; tcnl = tcnl->next ) {
        TCNode * n = (TCNode*)tcnl->data;
        
        n->tc->x -= min_x;
        n->tc->x *= scale_x;
        n->tc->y -= min_y;
        n->tc->y *= scale_y;
    }

}


void texapp_remap_cylindrical_tcnodes( int uindex, int vindex, int axis_index ) {

	GSList *tcnl = NULL, *vlist = NULL;
	Vertex avg_vert;
	float *avg_float;
	float min_x, min_y, max_x, max_y;
	float scale_x, scale_y;
	float PIx2 = 2.0 * M_PI;
	
	min_x = min_y = 10000000.0;  /* fixme - is ten million too little? */
	max_x = max_y = -10000000.0;  /* fixme - is neg. ten million too big? */

	/* build up a list of the vertices */
	for( tcnl = current_tcnodes; tcnl; tcnl = tcnl->next ) {
		vlist = g_slist_append( vlist, ((TCNode*)tcnl->data)->v );
	}
	/* use vlist to find the average */
	vertices_find_mean( vlist, &avg_vert );
	g_slist_free( vlist );
	avg_float = avg_vert.v;
	avg_float[axis_index] = 0.;
	
	for( tcnl = current_tcnodes; tcnl; tcnl = tcnl->next ) {
		TCNode * n = (TCNode*)tcnl->data;
		
		if( n->v->v[axis_index] < min_y ) min_y = n->v->v[axis_index];

		if( n->v->v[axis_index] > max_y ) max_y = n->v->v[axis_index];
		
		/* set this now, just for convenience */
		n->tc->y = n->v->v[axis_index];
	}
	
	scale_y = 1.0 / (max_y - min_y);

	/* final loop, to actually perform the mapping */
	for( tcnl = current_tcnodes; tcnl; tcnl = tcnl->next ) {
		TCNode * n = (TCNode*)tcnl->data;
		float delta[3], uvspace_delta[3], len=0;
		vector_copy( delta, n->v->v );
		delta[axis_index] = 0.;
		vector_sub( delta, delta, avg_float );
		len = vector_normalize( delta );
		if( len > 0.001 ) {
			uvspace_delta[0] = delta[uindex];
			uvspace_delta[1] = delta[vindex];
			uvspace_delta[2] = 0.;
			n->tc->x = vector_find_heading( uvspace_delta ) / PIx2;
		} else {
			n->tc->x = 0.5;
		}
		
		n->tc->y -= min_y;
		n->tc->y *= scale_y;
	}

}


void texapp_scale_tcnodes( float uscale, float vscale ) {

    GSList *tcnl = NULL;
    
    for( tcnl = current_tcnodes; tcnl; tcnl = tcnl->next ) {
        TCNode * n = (TCNode*)tcnl->data;
        
        n->tc->x *= uscale;
        n->tc->y *= vscale;
    }

}


void texapp_move_tcnodes( float delta_u, float delta_v ) {

    GSList *tcnl = NULL;
    
    for( tcnl = current_tcnodes; tcnl; tcnl = tcnl->next ) {
        TCNode * n = (TCNode*)tcnl->data;
        
        n->tc->x += delta_u;
        n->tc->y += delta_v;
    }

}


void texapp_mirror_tcnodes_horiz( void ) {
	texapp_scale_tcnodes( -1.0, 1.0 );
	texapp_move_tcnodes( 1.0, 0.0 );
}


void texapp_mirror_tcnodes_vert( void ) {
	texapp_scale_tcnodes( 1.0, -1.0 );
	texapp_move_tcnodes( 0.0, 1.0 );
}



////////////////////////////////////////////////////////////////////////////
//
// DEAD CODE
//
////////////////////////////////////////////////////////////////////////////



#if 0

gint texapp_uv_adj_cb( GtkWidget *widget, int whichcoord ) {

//printf( "uvcoord: %f -> %f\n", *uv_val, ( double ) GTK_ADJUSTMENT( widget ) ->value );
printf( "tv's : %f %f - %f %f - %f %f\n",
selected_tri->tv[0]->x, selected_tri->tv[0]->y,
selected_tri->tv[1]->x, selected_tri->tv[1]->y,
selected_tri->tv[2]->x, selected_tri->tv[2]->y );

    switch( whichcoord ) {
        case 0:
            selected_tri->tv[0]->x = ( double ) GTK_ADJUSTMENT( widget ) ->value;
            break;
        case 1:
            selected_tri->tv[0]->y = ( double ) GTK_ADJUSTMENT( widget ) ->value;
            break;
        case 2:
            selected_tri->tv[1]->x = ( double ) GTK_ADJUSTMENT( widget ) ->value;
            break;
        case 3:
            selected_tri->tv[1]->y = ( double ) GTK_ADJUSTMENT( widget ) ->value;
            break;
        case 4:
            selected_tri->tv[2]->x = ( double ) GTK_ADJUSTMENT( widget ) ->value;
            break;
        case 5:
            selected_tri->tv[2]->y = ( double ) GTK_ADJUSTMENT( widget ) ->value;
            break;
        default:
            break;
    }

    texapp_draw_texture( tex_app_dialog.drawing_area );
    texapp_refresh_views();

    return TRUE;
}
#endif
