/* 
 * File:         
 * 
 * Description:  
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gdk/gdk.h>
#include <gdk/gdk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "tex_gdkpixbuf.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


#define VERBOSE 1


int gpb_load( FILE *fp, Texture *tex ) ;

struct texture_file_fmt tex_fmt_gdkpixbuf = {
	"GdkPixbuf (handles most formats)",
	NULL,
	gpb_load
};


/**********************************************************************
* LOAD
**********************************************************************/
int gpb_load( FILE *fp, Texture *tex ) {
	
	GdkPixbuf * pixbuf;
	gint width, height, n_channels, bytespersample, rowstride;
	guchar *pbpixels;
	tex_data *dest;
	int total_length;
	
	g_return_val_if_fail( tex != NULL, TRUE );
	
	dest = &(tex->original);

	/* First, create & load the pixbuf */
	pixbuf = gdk_pixbuf_new_from_file( tex->filename, NULL );
	if( pixbuf == NULL ) {
		return TRUE;
	}

#ifdef VERBOSE
	printf( "loaded '%s'\n", tex->filename );
#endif

	if( gdk_pixbuf_get_colorspace( pixbuf ) != GDK_COLORSPACE_RGB ) {
		/* this should never really happen, but... */
		g_object_unref( pixbuf );
		return TRUE;
	}
	
	width = gdk_pixbuf_get_width( pixbuf );
	height = gdk_pixbuf_get_height( pixbuf );
	n_channels = gdk_pixbuf_get_n_channels( pixbuf );
	bytespersample = gdk_pixbuf_get_bits_per_sample( pixbuf ) / 8;
	rowstride = gdk_pixbuf_get_rowstride( pixbuf );

	if( bytespersample != 1 ) {
		/* we don't like multi-byte color samples */
		g_object_unref( pixbuf );
		return TRUE;
	}
	
	if( n_channels < 3 || n_channels > 4 ) {
		/* we don't like monochromatic or tetrachromatic images */
		g_object_unref( pixbuf );
		return TRUE;
	}
	
#ifdef VERBOSE
	printf( "%i x %i\n", width, height );
	printf( "n channels  %i\n", gdk_pixbuf_get_n_channels( pixbuf ) );
	printf( "has alpha   %i\n", gdk_pixbuf_get_has_alpha( pixbuf ) );
	printf( "bits/sample %i\n", gdk_pixbuf_get_bits_per_sample( pixbuf ) );
	printf( "rowstride   %i\n", gdk_pixbuf_get_rowstride( pixbuf ) );
#endif
	
	pbpixels = gdk_pixbuf_get_pixels( pixbuf );
	
	dest->width = width;
	dest->height = height;
	dest->depth = n_channels;
	total_length = width * height * n_channels;
	dest->data = (guchar*)malloc( total_length );
	
	/* copy the pixel data */
	if( rowstride == width * n_channels ) {
		memcpy( dest->data, pbpixels, total_length );
	} else if( rowstride > width * n_channels ) {
		/* If rowstride is larger than expected (ie there is extra stuff at the 
		end of each row) we can not simply copy the data verbatim.  Most images 
		do not have this problem.  It seems most prevalent among JPEGs; roughly 
		30% of the JPEG files I've tested have extra stuff at the end of each 
		row.  One possible explanation for this extra data is that the image 
		authoring program was attempting to align the beginning of each row on 
		integer boundaries, and padded the end of each row with extra bytes so 
		that everything lined up. */
		
		int src_row_start, dest_row_start, dest_rowstride, row, src_i, dest_i;
		dest_rowstride = width * n_channels;
		
		/* for each row... */
		for( row = 0; row < height; row++ ) {
			src_row_start = row * rowstride;
			dest_row_start = row * dest_rowstride;
			for( src_i = src_row_start, dest_i = dest_row_start;
				dest_i < dest_row_start + dest_rowstride;
				src_i++, dest_i++ )
			{
				dest->data[dest_i] = pbpixels[src_i];
			}
		}
	} else {
		/* if rowstride is smaller than expected, bail */
		g_object_unref( pixbuf );
		return TRUE;
	}
	

	tex->pixbuf = pixbuf;

	return FALSE;
}


