/* 
 * File:         tex_man.c
 * 
 * Description:  funcs for creating the "textures" notebook tab.  the texture
 * 		 load dialog stuff was moved into here too.
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "tex_man.h"
#include "model.h"
#include "mesh.h"
#include "texture.h"
#include "tex_app.h"
#include "bottombar.h"
#include "globals.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


#define VERBOSE 1

#define TEX_SIZE_ICON 16


/* ENUMS AND STRUCTS ******************************************************/

enum TextureManagerColumns {
	TM_COLUMN_TEXNAME = 0,
	TM_COLUMN_TEXPTR,
	TM_COLUMN_TEXICON,
	TM_NUM_COLUMNS
};


struct textureManager_struct {
	GtkListStore *store;	/* mirrors the model's texture list */
	GtkWidget *listView;	/* displays the contents of the above liststore */

	GtkWidget *frame;
	GtkWidget *drawing_area;

	GtkWidget *apply;
	GtkWidget *delete;
	GtkWidget *edit_texcoords;
}
textureManager;


/* FILE-SCOPE VARS **********************************************************/

Texture *texman_current_tex = NULL;
GtkFileSelection *texman_load_filesel = NULL;
unsigned char *texture_image_filename = NULL;


/* PROTOTYPES ***************************************************************/

GtkWidget *create_texture_preview( void ) ;
gboolean texman_cleanup( gpointer data ) ;
GtkWidget *create_texman_buttons( void ) ;
void texman_buttons_set_sensitive( int truth ) ;
void tm_clear( void ) ;
void tm_build( void ) ;
void tm_set_selected( void ) ;
void texman_add_tex( Texture *tex ) ;
void texman_remove_tex( Texture *tex ) ;

void tm_selection_changed_cb( GtkTreeSelection *selection, gpointer data ) ;
void texman_draw_texture( GtkWidget * widget ) ;
gint texman_expose_event( GtkWidget *widget, GdkEventExpose *event, gpointer data ) ;
gint texman_configure_event( GtkWidget *widget, GdkEvent *event, gpointer data ) ;
gint texman_delete_cb( GtkWidget *widget, gpointer data ) ;
gint texman_apply_tex_cb( GtkWidget *widget, gpointer data ) ;
gint texman_edit_tcs_cb( GtkWidget *widget, gpointer data ) ;

gint tex_load_ok_cb( GtkWidget *widget, gpointer data ) ;
gint tex_load_show_dlg( GtkWidget *widget, gpointer data ) ;
gint tex_load_hide_dlg( GtkWidget *widget, gpointer data ) ;


/* FUNCS ********************************************************************/

GtkWidget * create_tex_manager( void ) {

	GtkWidget *vbox;
	GtkWidget *scrollwindow;
	
	vbox = gtk_vbox_new( FALSE, 0 );

	/* create the texture preview widget */
	gtk_box_pack_start( GTK_BOX( vbox ), create_texture_preview(), 
						FALSE, FALSE, 0 );


	/* set up list data */
	textureManager.store = gtk_list_store_new( TM_NUM_COLUMNS,
												G_TYPE_STRING,
												G_TYPE_POINTER,
												G_TYPE_OBJECT );

	/* set up list viewer */
	textureManager.listView = 
		gtk_tree_view_new_with_model( GTK_TREE_MODEL( textureManager.store ) );
	
	scrollwindow = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW(scrollwindow),
									GTK_POLICY_AUTOMATIC,
									GTK_POLICY_AUTOMATIC );
	gtk_container_add( GTK_CONTAINER(scrollwindow), textureManager.listView );
	gtk_box_pack_start( GTK_BOX( vbox ), scrollwindow,
						TRUE, TRUE, 0 );
	
	/* set up the cell renderer(s) for our list viewer */
	{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	
	renderer = gtk_cell_renderer_pixbuf_new ();
	column = gtk_tree_view_column_new_with_attributes( "Icon",
														renderer,
														"pixbuf", 
														TM_COLUMN_TEXICON,
														NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW(textureManager.listView), column );

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes( "Texture name",
														renderer,
														"text", 
														TM_COLUMN_TEXNAME,
														NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW(textureManager.listView), column );
	}
	
	/* set up the selection handler for our list viewer */
	{
	GtkTreeSelection *select;

	select = gtk_tree_view_get_selection( GTK_TREE_VIEW(textureManager.listView) );
	gtk_tree_selection_set_mode( select, GTK_SELECTION_SINGLE );
	g_signal_connect( G_OBJECT(select), "changed",
						G_CALLBACK(tm_selection_changed_cb),
						NULL );
	}
	

	/* create the texture management buttons (load, delete, etc) */
	gtk_box_pack_start( GTK_BOX( vbox ), create_texman_buttons(), 
						FALSE, FALSE, 5 );

	texman_buttons_set_sensitive( FALSE );

	/* set up a signal handler for the ::model-texturelist-changed signal */
	g_signal_connect( 
		notificationObj, "notify::model-texturelist-changed", 
		G_CALLBACK(texman_rebuild_list), NULL );
	
	/* set up a signal handler for the ::exit signal */
	g_signal_connect( 
		notificationObj, "notify::exit", 
		G_CALLBACK(texman_cleanup), NULL );
	
	gtk_widget_show_all( vbox );

	return vbox;
}


gboolean texman_cleanup( gpointer data ) {
	if( texture_image_filename )
		free( texture_image_filename );

	return FALSE;
}


GtkWidget *create_texture_preview( void ) {

	textureManager.frame = gtk_frame_new( "Texture Preview" );
/*	gtk_widget_set_size_request( textureManager.frame, 
		TEXTURE_MIP_LEVEL_MEDIUM_SIZE + 10, 
		TEXTURE_MIP_LEVEL_MEDIUM_SIZE + 10 );*/
	gtk_widget_show( textureManager.frame );
	
	textureManager.drawing_area = gtk_drawing_area_new();
	gtk_widget_set_size_request( textureManager.drawing_area, 
		TEXTURE_MIP_LEVEL_MEDIUM_SIZE, 
		TEXTURE_MIP_LEVEL_MEDIUM_SIZE );
	gtk_container_add( GTK_CONTAINER( textureManager.frame ),
						textureManager.drawing_area );

	g_signal_connect ( G_OBJECT ( textureManager.drawing_area ),
						"expose_event",
						G_CALLBACK( texman_expose_event ), NULL );
	g_signal_connect ( G_OBJECT ( textureManager.drawing_area ),
						"configure_event",
						G_CALLBACK( texman_configure_event ), NULL );

	gtk_widget_show( textureManager.drawing_area );
	
	return textureManager.frame;
}


GtkWidget *create_texman_buttons( void ) {

	GtkWidget *button;
	GtkWidget *table;

	table = gtk_table_new( 3, 3, FALSE );
	gtk_table_set_row_spacings( GTK_TABLE( table ), 5 );
	gtk_table_set_col_spacings( GTK_TABLE( table ), 5 );
	gtk_widget_show( table );
	
	button = gtk_button_new_with_label( "Load..." );
	gtk_table_attach_defaults( GTK_TABLE( table ),
								button,
								0, 1, 0, 1 );
	g_signal_connect( G_OBJECT( button ), "clicked",
						G_CALLBACK( tex_load_show_dlg ), NULL );
	gtk_widget_show( button );
	
	button = gtk_button_new_with_label( "Delete" );
	textureManager.delete = button;
	gtk_table_attach_defaults( GTK_TABLE( table ), button,
								2, 3, 0, 1 );
	g_signal_connect( G_OBJECT( button ), "clicked",
						G_CALLBACK( texman_delete_cb ), NULL );
	gtk_widget_show( button );
	
	button = gtk_button_new_with_label( "Apply Texture to Current Mesh" );
	gtk_widget_set_sensitive( button, FALSE );
	g_signal_connect( G_OBJECT( button ), "clicked",
						G_CALLBACK( texman_apply_tex_cb ), NULL );
	textureManager.apply = button;
	gtk_table_attach_defaults( GTK_TABLE( table ), button,
								0, 3, 1, 2 );
	gtk_widget_show( button );
	
	button = gtk_button_new_with_label( "Edit Tex Coords on Sel. Polys" );
	g_signal_connect( G_OBJECT( button ), "clicked",
						G_CALLBACK( texman_edit_tcs_cb ), NULL );
	textureManager.edit_texcoords = button;
	gtk_table_attach_defaults( GTK_TABLE( table ), button,
								0, 3, 2, 3 );
	gtk_widget_show( button );

	return table;
}


void texman_buttons_set_sensitive( int truth ) {
    gtk_widget_set_sensitive( textureManager.delete, truth );
    gtk_widget_set_sensitive( textureManager.apply, truth );
}


/* used when deleting the model / freeing all the textures
   or when loading a model that already has a list of textures */
gboolean texman_rebuild_list( void ) {
	tm_clear();
	tm_build();
	tm_set_selected();
	return FALSE;
}


void tm_clear( void ) {
	
	GtkTreeIter iter;

	/* get an iterator pointing to the top level */
	if( gtk_tree_model_get_iter_first( GTK_TREE_MODEL(textureManager.store), &iter ) ) {
		/* now remove everything */
		while( gtk_list_store_remove( textureManager.store, &iter ) ) {
		}
	} else {
		/* list store was empty... nothing to do */
	}
	texman_current_tex = NULL;
}

void tm_build( void ) {
	GSList *ch;

	for ( ch = the_model->textures; ch != NULL; ch = ch->next ) {
		texman_add_tex( ( Texture * ) ch->data );
	}
}

void tm_set_selected( void ) {
	/* set texman_current_tex to the first entry (if there is one) and 
	make sure that it is highlighted */

	GtkTreeIter iter;
	gpointer texPtr;

	/* get an iterator pointing to the top level */
	if( gtk_tree_model_get_iter_first( GTK_TREE_MODEL(textureManager.store), &iter ) ) {
		GtkTreePath *path = gtk_tree_model_get_path( 
			GTK_TREE_MODEL( textureManager.store ), &iter );

		/* highlight the correct row */
		gtk_tree_view_set_cursor( GTK_TREE_VIEW( textureManager.listView ),
                                             path, NULL, FALSE );
		gtk_tree_path_free( path );

		/* set texman_current_tex appropriately */
		gtk_tree_model_get( GTK_TREE_MODEL(textureManager.store), &iter, 
			TM_COLUMN_TEXPTR, &texPtr, 
			-1 );
		texman_current_tex = (Texture*)texPtr;
		
	} else {
		texman_current_tex = NULL;
	}

	gtk_widget_queue_draw( textureManager.drawing_area );
}

void texman_add_tex( Texture *tex ) {
	GtkTreeIter iter;
	GdkPixbuf *pixbuf;
	
	if( tex == NULL ) return;
	
	/* scale the pixbuf down to 'icon' size */
	pixbuf = gdk_pixbuf_scale_simple( 
		tex->pixbuf, TEX_SIZE_ICON, TEX_SIZE_ICON, GDK_INTERP_BILINEAR );
	
	gtk_list_store_append( textureManager.store, &iter );
	gtk_list_store_set( textureManager.store, &iter, 
		TM_COLUMN_TEXNAME, tex->filename, 
		TM_COLUMN_TEXPTR, tex, 
		TM_COLUMN_TEXICON, pixbuf, 
		-1 );
	
	g_object_unref( pixbuf );
}

void texman_remove_tex( Texture *tex ) {
	g_signal_emit_by_name( notificationObj, 
		"notify::model-texturelist-changed", NULL );
}


/* CALLBACKS **************************************************************/

/* TEXTURE MANAGER STUFF *****/

void tm_selection_changed_cb( GtkTreeSelection *selection, gpointer data ) {

	GtkTreeIter iter;
	GtkTreeModel *model;
	gchar *texname;
	gpointer texPtr;

	if( gtk_tree_selection_get_selected( selection, &model, &iter ) ) {
		gtk_tree_model_get( model, &iter, 
			TM_COLUMN_TEXNAME, &texname, 
			TM_COLUMN_TEXPTR, &texPtr, 
			-1 );

#ifdef VERBOSE
		g_print( "You selected the tex with the name '%s'\n", texname );
		g_print( "You selected the tex with address %x\n", 
					(unsigned int)texPtr );
#endif
		g_free( texname );

		texman_current_tex = (Texture*) texPtr;

		gtk_widget_queue_draw( textureManager.drawing_area );
		texman_buttons_set_sensitive( TRUE );
	} else {
		texman_current_tex = NULL;
		gtk_widget_queue_draw( textureManager.drawing_area );
		texman_buttons_set_sensitive( FALSE );
	}
}


void texman_draw_texture( GtkWidget * widget ) {
    
    int tex_psx, tex_psy;

    if ( texman_current_tex ) {
        
        tex_data *td;
        td = &(texman_current_tex->mip[TEXTURE_MIP_LEVEL_MEDIUM]);
        tex_psx = ( widget->allocation.width - td->width ) >> 1;
        tex_psy = ( widget->allocation.height - td->height ) >> 1;

        gdk_draw_rgb_image( widget->window,
                            widget->style->fg_gc[ GTK_WIDGET_STATE ( widget ) ],
                            tex_psx, tex_psy,
                            td->width,
                            td->height,
                            GDK_RGB_DITHER_NORMAL, td->data,
                            td->width * td->depth );
    } else {
        gdk_draw_rectangle( widget->window,
                            widget->style->fg_gc[ GTK_WIDGET_STATE ( widget ) ],
                            TRUE, 0, 0,
                            widget->allocation.width,
                            widget->allocation.height );
    }
    
}


gint texman_expose_event( GtkWidget *widget, GdkEventExpose *event, gpointer data ) {

    texman_draw_texture( widget );
    
    return 0;
}

gint texman_configure_event( GtkWidget *widget, GdkEvent *event, gpointer data ) {

    gtk_widget_queue_draw( textureManager.drawing_area );
    gdk_flush();

    return TRUE;
}


gint texman_delete_cb( GtkWidget *widget, gpointer data ) {

	Texture *temp_tex;
	
	if( texman_current_tex == NULL )
		return FALSE;
	
	temp_tex = texman_current_tex;
	texman_current_tex = NULL;

	model_remove_texture( the_model, temp_tex );
	texman_remove_tex( temp_tex );
	
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );

	return TRUE;
}


gint texman_apply_tex_cb( GtkWidget *widget, gpointer data ) {

    if( the_model->current_mesh == NULL || texman_current_tex == NULL )
        return FALSE;
    
    mesh_set_texture( the_model->current_mesh, texman_current_tex );
    
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );

    return TRUE;
}

gint texman_edit_tcs_cb( GtkWidget *widget, gpointer data ) {

    if( the_model->current_mesh == NULL || the_model->selected_polys == NULL )
        return FALSE;
    
    show_tex_app( );
    
    return TRUE;
}


/* TEXTURE-LOAD DIALOG STUFF *****/

gint tex_load_ok_cb( GtkWidget *widget, gpointer data ) {
    const gchar * filename;
    Texture *new_tex;

    filename = gtk_file_selection_get_filename( texman_load_filesel );

    new_tex = tex_load( filename, TEXTURE_FORMAT_UNKNOWN, the_model->fname );
	if( new_tex == NULL ) {
		bb_push_message_f( 3.0, "Texture '%s' could not be loaded", filename );
		return TRUE;
	}

    model_add_texture( the_model, new_tex );
    texman_add_tex( new_tex );

    if ( texture_image_filename )
        free( texture_image_filename );
    texture_image_filename = strdup( filename );

    return TRUE;
}


/* show file selector */
gint tex_load_show_dlg( GtkWidget *widget, gpointer data ) {

	if( texman_load_filesel == NULL ) {
		texman_load_filesel = 
			(GtkFileSelection *)gtk_file_selection_new( "Load texture..." );

		g_signal_connect( G_OBJECT( texman_load_filesel->cancel_button ), 
							"clicked", G_CALLBACK( tex_load_hide_dlg ), 
							texman_load_filesel );

		g_signal_connect( G_OBJECT( texman_load_filesel->ok_button ), 
							"clicked", G_CALLBACK( tex_load_ok_cb ), 
							texman_load_filesel );
		g_signal_connect( G_OBJECT( texman_load_filesel->ok_button ), 
							"clicked", G_CALLBACK( tex_load_hide_dlg ), 
							texman_load_filesel );
	}

	if( texture_image_filename )
		gtk_file_selection_set_filename( 
			texman_load_filesel, texture_image_filename );

	gtk_widget_show( GTK_WIDGET( texman_load_filesel ) );

	return TRUE;
}


gint tex_load_hide_dlg( GtkWidget *widget, gpointer data ) {
	gtk_widget_hide( GTK_WIDGET( data ) );

	return TRUE;
}






////////////////////////////////////////////////////////////////////////////
//
// DEAD CODE
//
////////////////////////////////////////////////////////////////////////////

#if 0

gint texman_set2d_cb( GtkWidget *widget, gpointer data ) {
    GSList * cl;
    Poly *t;

    if( the_model->current_mesh == NULL || texman_current_tex == NULL )
        return FALSE;
    
    for ( cl = the_model->current_mesh->polygons; cl != NULL; cl = cl->next ) {
        t = ( Poly * ) cl->data;
#if 0
        t->tv[ 0 ].x = t->vertices[ 0 ]->v[ 0 ];    // / 100;
        t->tv[ 0 ].y = t->vertices[ 0 ]->v[ 1 ];    // / 100;
        t->tv[ 1 ].x = t->vertices[ 1 ]->v[ 0 ];    // / 100;
        t->tv[ 1 ].y = t->vertices[ 1 ]->v[ 1 ];    // / 100;
        t->tv[ 2 ].x = t->vertices[ 2 ]->v[ 0 ];    // / 100;
        t->tv[ 2 ].y = t->vertices[ 2 ]->v[ 1 ];    // / 100;
#else
        // hack...
        t->tv[ 0 ].x = 0.0;
        t->tv[ 0 ].y = 0.0;
        t->tv[ 1 ].x = 1.0;
        t->tv[ 1 ].y = 0.0;
        t->tv[ 2 ].x = 1.0;
        t->tv[ 2 ].y = 1.0;
#endif
        
    }

    view_redraw_all();

    return TRUE;
}


void texman_rename_cb( GtkWidget *widget, gpointer data ) {

    char * tmp = gtk_entry_get_text( GTK_ENTRY( textureManager.entry ) );

    if ( texman_current_tex->name )
        free( texman_current_tex->name );
    texman_current_tex->name = malloc( strlen( tmp ) + 1 );
    strcpy( texman_current_tex->name, tmp );

    gtk_clist_set_text( GTK_CLIST( textureManager.clist ),
                        tex_last_row, 1, texman_current_tex->name );
}



#endif



