/* 
 * File:		 tex_pcx.c
 * 
 * Description:  loads pcx files, incl. 8-bit (paletted) and 24-bit.
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 
/*
 The PCX_Header struct and the readpcximage function are from 
 the linux game development center:
 http://sunsite.auc.dk/lgdc/tutorials/courbot_col2b.html

 Everything else is by me, Andrew Sampson.  Also, I added 
 support for 24-bit pcx files (which accounts for over half the 
 code here).
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "tex_pcx.h"
#include "texture.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif

#define PCX_PALETTE_LEN 768


/* PROTOTYPES *************************************************************/

int pcx_load( FILE *fp, Texture *tex ) ;
void readpcximage(FILE * file, int size);

void pcx_copy_to_tex( tex_data *td, int width, int height, int color_layers ) ;
void pcx_copy_paletted( tex_data *td, int width, int height ) ;
void pcx_copy_RGB( tex_data *td, int width, int height ) ;

/* STRUCTS ****************************************************************/

struct texture_file_fmt tex_fmt_pcx = {
	"PCX Format (pcx)",
	"pcx",
	pcx_load
};

struct PCX_Header
{
	char signature;
	char version;
	char encoding;
	char bytes_per_pixel;
	unsigned short int xmin;
	unsigned short int ymin;
	unsigned short int xmax;
	unsigned short int ymax;
	unsigned short int vres;
	unsigned short int hres;
	char palette[48];
	char reserved;
	char color_layers;
	unsigned short int bytes_per_line;
	unsigned short int palette_type;
	char unused[58];
};


/* FILE-SCOPE VARS ********************************************************/

unsigned char *pcx_palette;
unsigned char *pcx_buffer;


/* FUNCS ******************************************************************/

int pcx_load( FILE *fp, Texture *tex ) {
	
	struct PCX_Header header;
	int height;
	int width;

	pcx_palette = (unsigned char *)malloc( PCX_PALETTE_LEN );
	pcx_buffer = NULL;
	
	fseek( fp,0,SEEK_SET );
	fread( &header, sizeof(struct PCX_Header), 1, fp );   /* read the header */

	/* Check if this file is in pcx format */
	if((header.signature!=0x0a)||(header.version!=5))
		return( TRUE );

#ifdef VERBOSE
	printf( "signature %i version %i encoding %i bytes_per_pixel %i\n"
			"color_layers %i bytes_per_line %i palette_type %i \n", 
	(unsigned int) header.signature, 
	(unsigned int) header.version, 
	(unsigned int) header.encoding, 
	(unsigned int) header.bytes_per_pixel, 
	(unsigned int) header.color_layers, 
	(unsigned int) header.bytes_per_line, 
	(unsigned int) header.palette_type );
#endif

	/* set height and width */
	width = header.xmax+1-header.xmin;
	height = header.ymax+1-header.ymin;
	
	/* Allocate the sprite buffer */
	pcx_buffer = (unsigned char *)malloc( width*height*header.color_layers );
	
	/* Read the image */
	readpcximage( fp, width*height*header.color_layers );

/* FIXME - check color_layers and palette_type before bothering to read in a palette */
	/* Get the palette */
	fseek(fp,-PCX_PALETTE_LEN,SEEK_END);
	fread(pcx_palette,1,PCX_PALETTE_LEN,fp);

	/* PCX succesfully read! */
	
	/* this is an svgalib-specific thing... palette entries
	   must be 0-63, not 0-255 */
/*	{
	int i;
	for( i=0; i<PCX_PALETTE_LEN; i++ ) 
		pcx_palette[i] = pcx_palette[i]>>2;
	}
*/
	
	tex->original.width = width;
	tex->original.height = height;
	tex->original.depth = 3;
	tex->original.data = (guchar*) malloc( width * height * 3 );

	pcx_copy_to_tex( &(tex->original), width, height, header.color_layers );


	free( pcx_palette );
	free( pcx_buffer );
	
	return FALSE;
}

void readpcximage(FILE * file, int size) {
	unsigned char buf;
	unsigned int counter;
	int i=0;
	while( i < size ) { /* Image not entirely read? */
		/* Get one byte */
		fread(&buf,1,1,file);
		/* Check the 2 most significant bits */
		if ((buf&192)==192) {		 /* We have 11xxxxxx */
			counter=(buf&63);		 /* Number of times to repeat next byte */
			fread(&buf,1,1,file);	 /* Get next byte */
			for(;counter>0;counter--) /* and copy it counter times */
			{
				pcx_buffer[i]=buf;
				i++;				  /* increase the number of bytes written */
			}
		} else {
			/* Just copy the byte */
			pcx_buffer[i]=buf;
			i++;   /* Increase the number of bytes written */
		}
	}
}


void pcx_copy_to_tex( tex_data *td, int width, int height, int color_layers ) {
	
	if( color_layers == 1 )
		pcx_copy_paletted( td, width, height );
	else if( color_layers == 3 )
		pcx_copy_RGB( td, width, height );
}


void pcx_copy_paletted( tex_data *td, int width, int height ) {
	int i = 0;
	int j = 0;
	
	for( i = 0; i < width * height; i++ ) {
		td->data[j++] = pcx_palette[(pcx_buffer[i]*3)+0];
		td->data[j++] = pcx_palette[(pcx_buffer[i]*3)+1];
		td->data[j++] = pcx_palette[(pcx_buffer[i]*3)+2];
	}
}


void pcx_copy_RGB( tex_data *td, int width, int height ) {
	int src_row, src_col, pixel_component;
	int src_offset = 0, dest_offset;
	
	/*
	24-bit pcx files are stored:
	RRRR...
	GGGG...
	BBBB...
	RRRR... etc.
	*/
	
	for( src_row = 0; src_row < height; src_row++ ) {
		for( pixel_component = 0; pixel_component < 3; pixel_component++ ) {
			dest_offset = ( src_row * width * 3 ) + pixel_component;
			for( src_col = 0; src_col < width; src_col++ ) {
				td->data[dest_offset] = pcx_buffer[src_offset++];
				dest_offset += 3;
			}
		}
	}

}

