/* 
 * File:         tex_png.c
 * 
 * Description:  funcs for loading png files.  uses libpng.
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 *	Modified by Andrew Sampson to be a generalized png thing.
 *
 *	Based on example code from:
 *	Guillaume Cottenceau (gc at mandrakesoft.com)
 *
 *	...who didn't really do much... he just copied the example code 
 *	distributed with libpng.
 *
 */

#include "tex_png.h"

#ifdef ENABLE_PNG

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>

/*#define PNG_DEBUG 3*/
#include <png.h>


struct texture_file_fmt tex_fmt_png = {
	"Portable Network Graphics Format (png)",
	"png",
	png_load
};

void abort_(const char * s, ...)
{
	va_list args;
	va_start(args, s);
	vfprintf(stderr, s, args);
	fprintf(stderr, "\n");
	va_end(args);
	abort();
}


int width, height;
unsigned char *image_data;
png_byte color_type;
png_byte bit_depth;

png_structp png_ptr;
png_infop info_ptr;
int number_of_passes;
png_bytep * row_pointers = NULL;


int png_load( FILE *fp, Texture *tex ) {
	if( fp == NULL || tex == NULL )
		return TRUE;
	
	if( TRUE == read_png_file( fp ) )
		return TRUE;
	
	/* set up members of tex */
	
	
	return FALSE;
}


int read_png_file( FILE * fp )
{
	int x, y;
	char header[8];	// 8 is the maximum size that can be checked

	if (!fp) {
		printf("[read_png_file] File could not be opened for reading\n" );
		return TRUE;
	}

	fread(header, 1, 8, fp);
	if (png_sig_cmp(header, 0, 8)) {
		printf("[read_png_file] File is not recognized as a PNG file\n" );
		return TRUE;
	}


	/* initialize stuff */
	png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	
	if (!png_ptr) {
		printf("[read_png_file] png_create_read_struct failed\n");
		return TRUE;
	}

	info_ptr = png_create_info_struct(png_ptr);
	if (!info_ptr) {
		printf("[read_png_file] png_create_info_struct failed\n");
		return TRUE;
//FIXME - leak - png_ptr
	}

	if (setjmp(png_jmpbuf(png_ptr))) {
		printf("[read_png_file] Error during init_io\n");
		return TRUE;
//FIXME - leak - png_ptr, info_ptr
	}

	png_init_io(png_ptr, fp);
	png_set_sig_bytes(png_ptr, 8);

	png_read_info(png_ptr, info_ptr);

	width = info_ptr->width;
	height = info_ptr->height;
	color_type = info_ptr->color_type;
	bit_depth = info_ptr->bit_depth;

	number_of_passes = png_set_interlace_handling(png_ptr);
	png_read_update_info(png_ptr, info_ptr);


	/* read file */
	if (setjmp(png_jmpbuf(png_ptr)))
		abort_("[read_png_file] Error during read_image");

	row_pointers = (png_bytep*) malloc(sizeof(png_bytep) * height);
	for (y=0; y<height; y++)
		row_pointers[y] = (png_byte*) malloc(info_ptr->rowbytes);

	png_read_image(png_ptr, row_pointers);
}




/* DEAD CODE */

#if 0

void setup_png_params( int w, int h, int depth, int colors ) {
	width = w;
	height = h;
	bit_depth = depth;
	color_type = colors;
}


void make_test_image( void ) {
	int x, y, i=0;

	image_data = (unsigned char *)malloc( width * height * 4 );

	for( y = 0; y < height; y++ ) {
		for( x = 0; x < width; x++ ) {
			image_data[i++] = (int)(255.0 * ( (float)y / (float)height ));
			image_data[i++] = 0;
			image_data[i++] = (int)(255.0 * ( (float)x / (float)width ));
			image_data[i++] = 255;
		}
	}
}


void set_row_pointers( unsigned char *img ) {

	int i;
	if( row_pointers == NULL )
		row_pointers = (png_bytep*) malloc(sizeof(png_bytep) * height);
	
	for( i = 0; i < height; i++ ) {
		/* The height-1 bit is needed because ogl's screen origin is in 
		the lower left, but png's origin is in the upper left.  We're 
		flipping the image. */
		row_pointers[i] = image_data + ((height - 1 - i)*width*4);
	}
}


void write_png_file(char* file_name)
{
	/* create file */
	FILE *fp = fopen(file_name, "wb");
	if (!fp)
		abort_("[write_png_file] File %s could not be opened for writing", file_name);


	/* initialize stuff */
	png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	
	if (!png_ptr)
		abort_("[write_png_file] png_create_write_struct failed");

	info_ptr = png_create_info_struct(png_ptr);
	if (!info_ptr)
		abort_("[write_png_file] png_create_info_struct failed");

	if (setjmp(png_jmpbuf(png_ptr)))
		abort_("[write_png_file] Error during init_io");

	png_init_io(png_ptr, fp);

	/* set compression level to some value appropriate 
	   for writing pngs quickly, but with decent compression 
	   Compression levels range from 0 (no comp.) to Z_BEST_COMPRESSION (9)
	*/
	png_set_compression_level( png_ptr, 3 );
	
	/* write header */
	if (setjmp(png_jmpbuf(png_ptr)))
		abort_("[write_png_file] Error during writing header");

	png_set_IHDR(png_ptr, info_ptr, width, height,
		     bit_depth, color_type, PNG_INTERLACE_NONE,
		     PNG_COMPRESSION_TYPE_BASE, PNG_FILTER_TYPE_BASE);

	png_write_info(png_ptr, info_ptr);


	/* write bytes */
	if (setjmp(png_jmpbuf(png_ptr)))
		abort_("[write_png_file] Error during writing bytes");

	set_row_pointers( image_data );
	
	png_write_image(png_ptr, row_pointers);


	/* end write */
	if (setjmp(png_jmpbuf(png_ptr)))
		abort_("[write_png_file] Error during end of write");

	png_write_end(png_ptr, NULL);

}


void write_png( char *filename, int w, int h, unsigned char * img ) {

	if( filename == NULL || img == NULL ) return;

	setup_png_params( w, h, 8, PNG_COLOR_TYPE_RGBA );
	image_data = img;
	write_png_file( filename );
}


void process_file(void)
{
	int x, y;
	if (info_ptr->color_type != PNG_COLOR_TYPE_RGBA)
		abort_("[process_file] color_type of input file must be PNG_COLOR_TYPE_RGBA (is %d)", info_ptr->color_type);

	for (y=0; y<height; y++) {
		png_byte* row = row_pointers[y];
		for (x=0; x<width; x++) {
			png_byte* ptr = &(row[x*4]);
			printf("Pixel at position [ %d - %d ] has the following RGBA values: %d - %d - %d - %d\n",
			       x, y, ptr[0], ptr[1], ptr[2], ptr[3]);

			/* set red value to 0 and green value to the blue one */
			ptr[0] = 0;
			ptr[1] = ptr[2];
		}
	}

}


#endif

#endif

