/* 
 * File:         tex_sgi.c
 * 
 * Description:  loads sgi-format images (.rgb files, aka .sgi files)
 * 		 I don't know where this source came from.
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "tex_sgi.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


struct texture_file_fmt tex_fmt_sgi = {
	"SGI Image Format (sgi)",
	"sgi",
	sgi_load
};
struct texture_file_fmt tex_fmt_rgb = {
	"SGI Image Format (rgb)",
	"rgb",
	sgi_load
};


/**********************************************************************
* LOAD
**********************************************************************/
int sgi_load( FILE *fp, Texture *tex ) {
	
	unsigned char * image;

	g_return_val_if_fail( tex != NULL, TRUE );

	image = read_sgi( fp, &tex->original.width, 
				&tex->original.height, &tex->original.depth );

	if ( image == NULL ) return TRUE;

	tex->original.data = image;
	
	/* sgi's are upside down (I think) */
	tex->upside_down = TRUE;

	return FALSE;
}


/**********************************************************************
* FUNCS
**********************************************************************/

static void
bwtorgba( unsigned char *b, unsigned char *l, int n ) {
    while ( n-- ) {
        l[ 0 ] = *b;
        l[ 1 ] = *b;
        l[ 2 ] = *b;
        l[ 3 ] = 0xff;
        l += 4;
        b++;
    }
}

static void
latorgba( unsigned char *b, unsigned char *a, unsigned char *l, int n ) {
    while ( n-- ) {
        l[ 0 ] = *b;
        l[ 1 ] = *b;
        l[ 2 ] = *b;
        l[ 3 ] = *a;
        l += 4;
        b++;
        a++;
    }
}

static void
rgbtorgba( unsigned char *r, unsigned char *g, unsigned char *b, unsigned char *l, int n ) {
    while ( n-- ) {
        l[ 0 ] = r[ 0 ];
        l[ 1 ] = g[ 0 ];
        l[ 2 ] = b[ 0 ];
        l[ 3 ] = 0xff;
        l += 4;
        r++;
        g++;
        b++;
    }
}

static void
rgbatorgba( unsigned char *r, unsigned char *g, unsigned char *b, unsigned char *a, unsigned char *l, int n ) {
    while ( n-- ) {
        l[ 0 ] = r[ 0 ];
        l[ 1 ] = g[ 0 ];
        l[ 2 ] = b[ 0 ];
        l[ 3 ] = a[ 0 ];
        l += 4;
        r++;
        g++;
        b++;
        a++;
    }
}

static void
bwtobw( unsigned char *b, unsigned char *l, int n ) {
    while ( n-- ) {
        l[ 0 ] = *b;
        l += 1;
        b++;
    }
}

static void
latola( unsigned char *b, unsigned char *a, unsigned char *l, int n ) {
    while ( n-- ) {
        l[ 0 ] = *b;
        l[ 1 ] = *a;
        l += 2;
        b++;
        a++;
    }
}

static void
rgbtorgb( unsigned char *r, unsigned char *g, unsigned char *b, unsigned char *l, int n ) {
    while ( n-- ) {
        l[ 0 ] = r[ 0 ];
        l[ 1 ] = g[ 0 ];
        l[ 2 ] = b[ 0 ];
        l += 3;
        r++;
        g++;
        b++;
    }
}

typedef struct _ImageRec {
    unsigned short imagic;
    unsigned short type;
    unsigned short dim;
    unsigned short xsize, ysize, zsize;
    unsigned int min, max;
    unsigned int wasteBytes;
    char name[ 80 ];
    unsigned long colorMap;
    FILE *file;
    unsigned char *tmp, *tmpR, *tmpG, *tmpB;
    unsigned long rleEnd;
    unsigned int *rowStart;
    int *rowSize;
}
ImageRec;

static void
ConvertShort( unsigned short *array, long length ) {
    unsigned b1, b2;
    unsigned char *ptr;

    ptr = ( unsigned char * ) array;
    while ( length-- ) {
        b1 = *ptr++;
        b2 = *ptr++;
        *array++ = ( b1 << 8 ) | ( b2 );
    }
}

static void
ConvertLong( unsigned *array, long length ) {
    unsigned long b1, b2, b3, b4;
    unsigned char *ptr;

    ptr = ( unsigned char * ) array;
    while ( length-- ) {
        b1 = *ptr++;
        b2 = *ptr++;
        b3 = *ptr++;
        b4 = *ptr++;
        *array++ = ( b1 << 24 ) | ( b2 << 16 ) | ( b3 << 8 ) | ( b4 );
    }
}

static ImageRec *ImageOpen( FILE *fp ) {
    union {
        int testWord;
        char testByte[ 4 ];
    } endianTest;
    ImageRec *image;
    int swapFlag;
    int x;

    endianTest.testWord = 1;
    if ( endianTest.testByte[ 0 ] == 1 ) {
        swapFlag = 1;
    } else {
        swapFlag = 0;
    }

    image = ( ImageRec * ) malloc( sizeof( ImageRec ) );
    if ( image == NULL ) {
        fprintf( stderr, "Out of memory in sgi.c!\n" );
        exit( 1 );
    }
	memset( image, '\0', sizeof( ImageRec ) );

    if ( fp == NULL ) {
        fprintf( stderr, "fp is null in sgi.c!\n" );
        return NULL;
    }
	image->file = fp;

    fread( image, 1, 12, image->file );

    if ( swapFlag ) {
        ConvertShort( &image->imagic, 6 );
    }

    image->tmp = ( unsigned char * ) malloc( image->xsize * 256 );
    image->tmpR = ( unsigned char * ) malloc( image->xsize * 256 );
    image->tmpG = ( unsigned char * ) malloc( image->xsize * 256 );
    image->tmpB = ( unsigned char * ) malloc( image->xsize * 256 );
    if ( image->tmp == NULL || image->tmpR == NULL || image->tmpG == NULL ||
            image->tmpB == NULL ) {
        fprintf( stderr, "Out of memory!\n" );
        exit( 1 );
    }

    if ( ( image->type & 0xFF00 ) == 0x0100 ) {
        x = image->ysize * image->zsize * sizeof( unsigned );
        image->rowStart = ( unsigned * ) malloc( x );
        image->rowSize = ( int * ) malloc( x );
        if ( image->rowStart == NULL || image->rowSize == NULL ) {
            fprintf( stderr, "Out of memory!\n" );
            exit( 1 );
        }
        image->rleEnd = 512 + ( 2 * x );
        fseek( image->file, 512, SEEK_SET );
        fread( image->rowStart, 1, x, image->file );
        fread( image->rowSize, 1, x, image->file );
        if ( swapFlag ) {
            ConvertLong( image->rowStart, x / sizeof( unsigned ) );
            ConvertLong( ( unsigned * ) image->rowSize, x / sizeof( int ) );
        }
    }
    return image;
}

static void
ImageClose( ImageRec *image ) {
/* not needed here any more.  file will be closed in texture.c
    fclose( image->file );*/
    free( image->tmp );
    free( image->tmpR );
    free( image->tmpG );
    free( image->tmpB );

	if( image->rowStart ) free( image->rowStart );
	if( image->rowSize ) free( image->rowSize );

    free( image );
}

static void
ImageGetRow( ImageRec *image, unsigned char *buf, int y, int z ) {
    unsigned char * iPtr, *oPtr, pixel;
    int count;

    if ( ( image->type & 0xFF00 ) == 0x0100 ) {
        fseek( image->file, image->rowStart[ y + z * image->ysize ], SEEK_SET );
        fread( image->tmp, 1, ( unsigned int ) image->rowSize[ y + z * image->ysize ],
               image->file );

        iPtr = image->tmp;
        oPtr = buf;
        while ( 1 ) {
            pixel = *iPtr++;
            count = ( int ) ( pixel & 0x7F );
            if ( !count ) {
                return ;
            }
            if ( pixel & 0x80 ) {
                while ( count-- ) {
                    *oPtr++ = *iPtr++;
                }
            } else {
                pixel = *iPtr++;
                while ( count-- ) {
                    *oPtr++ = pixel;
                }
            }
        }
    } else {
        fseek( image->file, 512 + ( y * image->xsize ) + ( z * image->xsize * image->ysize ),
               SEEK_SET );
        fread( buf, 1, image->xsize, image->file );
    }
}

unsigned *
read_sgi_and_convert( FILE *fp, int *width, int *height, int *components ) {
    unsigned * base, *lptr;
    unsigned char *rbuf, *gbuf, *bbuf, *abuf;
    ImageRec *image;
    int y;

    image = ImageOpen( fp );

    if ( !image )
        return NULL;
    ( *width ) = image->xsize;
    ( *height ) = image->ysize;
    ( *components ) = image->zsize;
    base = ( unsigned * ) malloc( image->xsize * image->ysize * sizeof( unsigned ) );
    rbuf = ( unsigned char * ) malloc( image->xsize * sizeof( unsigned char ) );
    gbuf = ( unsigned char * ) malloc( image->xsize * sizeof( unsigned char ) );
    bbuf = ( unsigned char * ) malloc( image->xsize * sizeof( unsigned char ) );
    abuf = ( unsigned char * ) malloc( image->xsize * sizeof( unsigned char ) );
    if ( !base || !rbuf || !gbuf || !bbuf )
        return NULL;
    lptr = base;
    for ( y = 0; y < image->ysize; y++ ) {
        if ( image->zsize >= 4 ) {
            ImageGetRow( image, rbuf, y, 0 );
            ImageGetRow( image, gbuf, y, 1 );
            ImageGetRow( image, bbuf, y, 2 );
            ImageGetRow( image, abuf, y, 3 );
            rgbatorgba( rbuf, gbuf, bbuf, abuf, ( unsigned char * ) lptr, image->xsize );
            lptr += image->xsize;
        } else if ( image->zsize == 3 ) {
            ImageGetRow( image, rbuf, y, 0 );
            ImageGetRow( image, gbuf, y, 1 );
            ImageGetRow( image, bbuf, y, 2 );
            rgbtorgba( rbuf, gbuf, bbuf, ( unsigned char * ) lptr, image->xsize );
            lptr += image->xsize;
        } else if ( image->zsize == 2 ) {
            ImageGetRow( image, rbuf, y, 0 );
            ImageGetRow( image, abuf, y, 1 );
            latorgba( rbuf, abuf, ( unsigned char * ) lptr, image->xsize );
            lptr += image->xsize;
        } else {
            ImageGetRow( image, rbuf, y, 0 );
            bwtorgba( rbuf, ( unsigned char * ) lptr, image->xsize );
            lptr += image->xsize;
        }
    }
    ImageClose( image );
    free( rbuf );
    free( gbuf );
    free( bbuf );
    free( abuf );

    return ( unsigned * ) base;
}

unsigned char *
read_sgi( FILE *fp, int *width, int *height, int *components ) {
    unsigned char * base, *lptr;
    unsigned char *rbuf, *gbuf, *bbuf, *abuf;
    ImageRec *image;
    int y;

    image = ImageOpen( fp );

    if ( !image )
        return NULL;
    ( *width ) = image->xsize;
    ( *height ) = image->ysize;
    ( *components ) = image->zsize;
    base = ( unsigned char * ) malloc( image->xsize * image->ysize * image->zsize *
                                       sizeof( unsigned char ) );
    rbuf = ( unsigned char * ) malloc( image->xsize * sizeof( unsigned char ) );
    gbuf = ( unsigned char * ) malloc( image->xsize * sizeof( unsigned char ) );
    bbuf = ( unsigned char * ) malloc( image->xsize * sizeof( unsigned char ) );
    abuf = ( unsigned char * ) malloc( image->xsize * sizeof( unsigned char ) );
    if ( !base || !rbuf || !gbuf || !bbuf )
        return NULL;
    lptr = base;
    for ( y = 0; y < image->ysize; y++ ) {
        if ( image->zsize >= 4 ) {
            ImageGetRow( image, rbuf, y, 0 );
            ImageGetRow( image, gbuf, y, 1 );
            ImageGetRow( image, bbuf, y, 2 );
            ImageGetRow( image, abuf, y, 3 );
            rgbatorgba( rbuf, gbuf, bbuf, abuf, lptr, image->xsize );
            lptr += image->xsize * image->zsize;
        } else if ( image->zsize == 3 ) {
            ImageGetRow( image, rbuf, y, 0 );
            ImageGetRow( image, gbuf, y, 1 );
            ImageGetRow( image, bbuf, y, 2 );
            rgbtorgb( rbuf, gbuf, bbuf, lptr, image->xsize );
            lptr += image->xsize * image->zsize;
        } else if ( image->zsize == 2 ) {
            ImageGetRow( image, rbuf, y, 0 );
            ImageGetRow( image, abuf, y, 1 );
            latola( rbuf, abuf, lptr, image->xsize );
            lptr += image->xsize * image->zsize;
        } else {
            ImageGetRow( image, rbuf, y, 0 );
            bwtobw( rbuf, lptr, image->xsize );
            lptr += image->xsize;
        }
    }
    ImageClose( image );
    free( rbuf );
    free( gbuf );
    free( bbuf );
    free( abuf );

    return ( void * ) base;
}
