/* 
 * File:         texture.c
 * 
 * Description:  funcs for creating and adding textures.  
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>  /* for memset (wtf?) */

#include "texture.h"
#include "globals.h"
#include "prefs.h"
#include "tex_pcx.h"
#include "tex_sgi.h"
#include "tex_gdkpixbuf.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif


#define TEX_DEFAULT_SEARCH_PATH ":.:./textures:./images:..:../textures:../images"


/* PROTOTYPES ***********************************************************/

void tex_remove_alpha( Texture *ntex ) ;
void vertical_flip( tex_data * td );
void tex_make_mipmaps( Texture * t ) ;
void tex_flip( Texture * t ) ;
int tex_examine_extension( const gchar *filename ) ;
gchar *tex_resolve_filename( const gchar *filename ) ;
char *tex_find_relative_filename( char *texname, char *modelname ) ;
void tex_populate_pixbuf( Texture * t ) ;

/* STRUCTS **************************************************************/

struct texture_file_fmt *tex_fmt_list[] = {
	&tex_fmt_pcx,
	&tex_fmt_sgi,
	&tex_fmt_rgb,

	&tex_fmt_gdkpixbuf, /* this should probably be listed last */
	NULL
};


/* FUNCS ****************************************************************/

Texture *tex_new( void ) {
    Texture * newt;

    newt = ( Texture * ) malloc( sizeof( Texture ) );
    memset( newt, '\0', sizeof( Texture ) );
    
    return newt;
}

void tex_delete( Texture * tt ) {
    
    int i;
    
    if( tt == NULL ) return;

	if( tt->pixbuf )
		g_object_unref( tt->pixbuf );

    if ( tt->filename )
        free( tt->filename );

    if ( tt->original.data )
        free( tt->original.data );

    for( i = 0; i < TEXTURE_NUM_MIP_LEVELS; i++ ) {
        if ( tt->mip[i].data )
            free( tt->mip[i].data );
    }
    
	free( tt );
}



void vertical_flip( tex_data * td ) {
    unsigned char * tmp_buff;
    int c, d, off, off2;
    int width, height, depth;
    
    if( td == NULL ) return;
    
    width  = td->width;
    height = td->height;
    depth  = td->depth;
    
    d = width * depth;

    tmp_buff = ( unsigned char * ) malloc( width * height * depth );

    for ( c = height - 1, off = 0, off2 = width * ( height - 1 ) * depth;
            c >= 0; c--, off += d, off2 -= d )
        memcpy( tmp_buff + off, td->data + off2, d );

    free( td->data );
    td->data = tmp_buff;
}


void tex_scale( tex_data *td_src, tex_data *td_dest, int width, int height ) {
    
    // this takes the info in td_src, scales it, and puts it in td_dest.
    // I couldn't find any algos online for image scaling (granted, I didn't 
    // look very hard) so I wrote my own.  It's icky, but it seems to work.

    unsigned char * tmp_buff;
    int x, y;
    int depth;
    int src_current_row;
    int src_current_col;
    int dest_offset = 0;
    
    if( td_src == NULL || td_src->data == NULL || td_dest == NULL )
        return;
    
    depth = td_src->depth;
    
    tmp_buff = ( unsigned char * ) malloc( width * height * depth );

    for( y = 0; y < height; y++ ) {

        src_current_row = (int)( (float)td_src->height * ( (float)y / (float)height ) ) * ( td_src->width * depth );
        // array index of the first element of the current row

        for( x = 0; x < width; x++ ) {
            
            src_current_col = (int)( (float)td_src->width * ( (float)x / (float)width ) ) * depth;

            tmp_buff[ dest_offset    ] = td_src->data[ src_current_row + src_current_col    ];
            tmp_buff[ dest_offset +1 ] = td_src->data[ src_current_row + src_current_col +1 ];
            tmp_buff[ dest_offset +2 ] = td_src->data[ src_current_row + src_current_col +2 ];
            
            dest_offset += depth;
            
        }

    }
    
    if( td_dest->data != NULL )
        free( td_dest->data );
    
    td_dest->data = tmp_buff;
    td_dest->width = width;
    td_dest->height = height;
    td_dest->depth = td_src->depth;
}




void tex_make_mipmaps( Texture * t ) {

    // generate mipmaps for this texture!
    
    g_return_if_fail( t != NULL );
    g_return_if_fail( t->original.data != NULL );
    
    // fixme - minor kludge - i want to use a loop here, but i can't, 
    // because TEXTURE_MIP_LEVEL_*_SIZE are individual defs, not in 
    // a nice convenient array
    
    tex_scale( &(t->original), &(t->mip[TEXTURE_MIP_LEVEL_SMALLEST]), 
               TEXTURE_MIP_LEVEL_SMALLEST_SIZE, TEXTURE_MIP_LEVEL_SMALLEST_SIZE );
    
    tex_scale( &(t->original), &(t->mip[TEXTURE_MIP_LEVEL_MEDIUM]), 
               TEXTURE_MIP_LEVEL_MEDIUM_SIZE, TEXTURE_MIP_LEVEL_MEDIUM_SIZE );
    
    tex_scale( &(t->original), &(t->mip[TEXTURE_MIP_LEVEL_LARGE]), 
               TEXTURE_MIP_LEVEL_LARGE_SIZE, TEXTURE_MIP_LEVEL_LARGE_SIZE );
    
    tex_scale( &(t->original), &(t->mip[TEXTURE_MIP_LEVEL_OPENGL]), 
               TEXTURE_MIP_LEVEL_OPENGL_SIZE, TEXTURE_MIP_LEVEL_OPENGL_SIZE );
}


void tex_flip( Texture * t ) {

	/* flips texture as needed */

	/* opengl's textures use a different origin than that of 
	x11/gtk/etc.  If the gtk textures need to be flipped, the ogl one 
	will not.  If the gtk textures don't need to be flipped, the ogl one 
	will. */
	if( t->upside_down ) {
		vertical_flip( &(t->mip[TEXTURE_MIP_LEVEL_SMALLEST]) );
		vertical_flip( &(t->mip[TEXTURE_MIP_LEVEL_MEDIUM]) );
		vertical_flip( &(t->mip[TEXTURE_MIP_LEVEL_LARGE]) );
		/* don't flip the opengl one... */
	} else {
		/* don't flip the gtk ones... */
		vertical_flip( &(t->mip[TEXTURE_MIP_LEVEL_OPENGL]) );
	}
}


Texture * tex_load( const gchar *file_name, int fmt_idx, char *model_fname ) {

	Texture *tex = NULL;
	FILE *fp = NULL;
	int status;
	char *tex_name = NULL;
	char *old_pwd = NULL;
	
	g_return_val_if_fail(file_name != NULL, NULL);
	g_return_val_if_fail(fmt_idx >= TEXTURE_FORMAT_UNKNOWN, NULL);

	/* remember the current working directory; change directory to the 
	model's directory.  This is important, because the texture names are 
	relative to the model's path. */
	if( model_fname ) {
		char *model_dir;
		old_pwd = g_get_current_dir();
printf( "%s - pwd is %s\n", __FUNCTION__, old_pwd );
		model_dir = g_path_get_dirname( model_fname );
		chdir( model_dir );
		g_free( model_dir );
	}

	if( fmt_idx == TEXTURE_FORMAT_UNKNOWN )
		fmt_idx = tex_examine_extension( file_name );
	if( fmt_idx == TEXTURE_FORMAT_UNKNOWN )
		goto quit;

	if( g_path_is_absolute( file_name ) ) {
		tex_name = g_strdup( file_name );
	} else {
		tex_name = tex_resolve_filename( file_name );
	}
	
	if( tex_name == NULL )
		goto quit;
	
	fp = fopen( tex_name, "r" );
	if( fp == NULL ) 
		goto quit;

	tex = tex_new();
	
	if( g_path_is_absolute( file_name ) ) {
		char *rel_name = 
			tex_find_relative_filename( tex_name, model_fname );
		tex->filename = strdup( rel_name );
		free( rel_name );
	} else {
		tex->filename = strdup( tex_name );
	}
	
	status = (*(tex_fmt_list[fmt_idx]->load_tex))( fp, tex );
	
	fclose( fp );
	
	if( status == TRUE ) {
		tex_delete( tex );
		tex = NULL;
		goto quit;
	}
	
	tex_remove_alpha( tex );
	tex_make_mipmaps( tex );
	tex_flip( tex );

	if( tex->pixbuf == NULL ) {
		tex_populate_pixbuf( tex );
	}
	
quit:
	if( tex_name )
		g_free( tex_name );
	if( old_pwd ) {
		chdir( old_pwd );
		g_free( old_pwd );
	}
	
	return tex;
}



int tex_examine_extension( const gchar *filename ) {
	gint i, j, l, m;
	int result = TEXTURE_FORMAT_UNKNOWN;

	m = strlen(filename);
	for(i = 0; (tex_fmt_list[i] != NULL); i++)
	{
		if( tex_fmt_list[i]->load_tex == NULL )
			 continue;
	
		/* NULL means wildcard */
		if( tex_fmt_list[i]->extension == NULL ) {
			/* We've encountered an entry in the tex_fmt_list that claims 
			to be able to load multiple file formats.  We'll remember that,
			in case we don't/didn't find a more suitable loader. */
			result = i;
			continue;
		}

		l = strlen(tex_fmt_list[i]->extension);
		if(l >= m)
			continue;

		for(j = 0; j < l; j++) {
			if(tex_fmt_list[i]->extension[l - j] != filename[m - j])
				break;
		}

		if((j == l) && (filename[m-l - 1] == '.'))
			return i;
	}
	return result;
}


gchar *tex_resolve_filename( const gchar *filename ) {
	char *pathstring, *ptr, buf[1024] = {'\0'};
	
	if( g_file_test( filename, G_FILE_TEST_EXISTS ) ) {
		return( g_strdup( filename ) );
	}
	
	pathstring = strdup( pref_get_string( 
		"Textures::Texture Search Path (colon-separated)", 
		TEX_DEFAULT_SEARCH_PATH ) );
	ptr = strtok( pathstring, ":" );
	while( ptr ) {
		snprintf( buf, 1023, "%s/%s", ptr, filename );
		if( g_file_test( buf, G_FILE_TEST_EXISTS ) )
			break;
		ptr = strtok( NULL, ":" );
	}
	
	free( pathstring );
	return( ptr ? g_strdup( buf ) : NULL );
}


char *tex_find_relative_filename( char *texname, char *modelname ) {
	char *mstr, buf[1024] = {'\0'};
	int i, depth = 0;
	int len_mstr, len_texname, first_diff;

#if 0
	/* if modelname is not valid, use pwd (make sure pwd is '/'-terminated) */
	if( modelname == NULL ) {
		mstr = g_get_current_dir();
printf( "%s - pwd is %s\n", __FUNCTION__, mstr );
	} else {
		mstr = g_strdup( modelname );
	}
#else
	/* if modelname is not valid, use full (absolute) texname */
	if( modelname == NULL ) {
		return strdup( texname );
	} else {
		mstr = g_strdup( modelname );
	}
#endif
	
	/* examine both until a difference is found */
	len_mstr = strlen( mstr );
	len_texname = strlen( texname );
	for( i = 0; i < len_mstr && i < len_texname; i++ ) {
		if( mstr[i] != texname[i] ) {
			break;
		}
	}
	
	first_diff = i;
	
	/* at first different char in mstr, start counting '/'s; store as 'depth' */
	for( i = first_diff; i < len_mstr; i++ ) {
		if( mstr[i] == '/' )
			depth++;
	}
	
	/* rewind texname until we hit a '/', and (if we do hit a '/') take one 
	step forward */
	while( first_diff >= 0 ) {
		if( texname[first_diff] == '/' || first_diff == 0 )
			break;
		else
			first_diff--;
	}
	if( texname[first_diff] == '/' )
		first_diff++;
	
	/* relative file name for texname is: 
		('../' * depth) + ptr-to-first-diff-char-in-texname */
	for( i = 0; i < depth; i++ ) {
		strcat( buf, "../" );
	}
	strcat( buf, texname + first_diff );
	
	g_free( mstr );
	return strdup( buf );
}


void tex_remove_alpha( Texture *ntex ) {

    unsigned char * tmp_buff;
    char *off_src, *off_dest;
    int i;
    int width, height, depth3, depth4;
    
    if( ntex == NULL )
        return;
    
    width  = ntex->original.width;
    height = ntex->original.height;
    depth4 = ntex->original.depth;
    depth3 = 3;
    
    if( depth4 != 4 )
        return;

    tmp_buff = ( unsigned char * ) malloc( width * height * depth3 );
    
    off_src = ntex->original.data;
    off_dest = tmp_buff;
    
    for ( i = width * height; i >= 0; i-- ) {
        
        off_dest[0] = off_src[0];
        off_dest[1] = off_src[1];
        off_dest[2] = off_src[2];
        
        off_src += depth4;
        off_dest += depth3;
    }

    free( ntex->original.data );
    ntex->original.data = tmp_buff;
    ntex->original.depth = 3;
}


void tex_populate_pixbuf( Texture * t ) {
	
	/* if the texture doesn't have a pixbuf containing a copy of the 
	image, then create one */
	
	tex_data *orig;
	
	if( t == NULL || t->pixbuf ) return;
	
	orig = &(t->original);
	
	t->pixbuf = gdk_pixbuf_new_from_data( 
		orig->data, GDK_COLORSPACE_RGB, 
		(orig->depth == 4), 8, 
		orig->width, orig->height, 
		orig->width * orig->depth, /* rowstride - dist in bytes between rows */
		NULL, NULL /* no destructor func is specified for the data that 
					  we've passed in; we will take care of that in 
					  tex_delete */
		);
	
	if( t->upside_down ) {
		GdkPixbuf *copy;
		int pb_width, pb_height;
		
		copy = gdk_pixbuf_copy( t->pixbuf );
		pb_width = gdk_pixbuf_get_width( copy );
		pb_height = gdk_pixbuf_get_height( copy );
		gdk_pixbuf_scale( 
			copy, t->pixbuf, 
			0, 0, 
			pb_width,
			pb_height,
			0.0, 
			(double)pb_height, 
			1.0, -1.0, 
			GDK_INTERP_HYPER );
	
		g_object_unref( copy );
	}
}



