/* 
 * File:         toolbox.c
 * 
 * Description:  creates the toolbox notebook tab
 * 
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */
 



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <gtk/gtk.h>


#include "config.h"
#include "globals.h"
#include "model.h"
#include "mesh.h"
#include "polygon.h"
#include "view_config.h"
#include "tools.h"
#include "toolbox.h"
#include "undo.h"
#include "rollup.h"
#ifdef USE_PYTHON
#include "scriptbox.h"
#endif

#ifdef MEMWATCH
#include "memwatch.h"
#endif



#include "pixmaps/grid_snap.xpm"
#include "pixmaps/vertex_snap_together.xpm"
#include "pixmaps/vertex_rot_90.xpm"
#include "pixmaps/vertex_rot.xpm"
#include "pixmaps/vertex_flip.xpm"
#include "pixmaps/poly_reverse_winding.xpm"
#include "pixmaps/poly_extrude.xpm"
#include "pixmaps/vertex_remove_unused.xpm"
#include "pixmaps/vertex_weld.xpm"
#include "pixmaps/poly_turn_edge.xpm"
#include "pixmaps/poly_make_triangles.xpm"
#include "pixmaps/poly_subdivide_quads.xpm"
#include "pixmaps/poly_subdivide_tris.xpm"
#include "pixmaps/poly_subdivide_midpt.xpm"
#include "pixmaps/poly_smooth.xpm"

void tb_active_axis_x_cb( GtkWidget *widget, gpointer data ) ;
void tb_active_axis_y_cb( GtkWidget *widget, gpointer data ) ;
void tb_active_axis_z_cb( GtkWidget *widget, gpointer data ) ;

void tb_snap_to_grid( GtkWidget *widget, gpointer data );
void tb_snap_together( GtkWidget *widget, gpointer data );
void tb_snap_together_active_axis( GtkWidget *widget, gpointer data );
void tb_weld_together( GtkWidget *widget, gpointer data );
void tb_weld_modelwide( GtkWidget *widget, gpointer data );
void tb_extrude( GtkWidget *widget, gpointer data );
void tb_extrude_connected( GtkWidget *widget, gpointer data );
void tb_rotate_90( GtkWidget *widget, gpointer data ) ;
void tb_rotate_angle_cb( GtkWidget *widget, gpointer data ) ;
void tb_rotate( GtkWidget *widget, gpointer data ) ;
void tb_reverse_polygon_winding( GtkWidget *widget, gpointer data ) ;
void tb_flip_vertices( GtkWidget *widget, gpointer data ) ;
void tb_remove_unused_vertices( GtkWidget *widget, gpointer data ) ;
void tb_polygon_turn_edge( GtkWidget *widget, gpointer data ) ;
void tb_polygon_make_triangles( GtkWidget *widget, gpointer data ) ;
void tb_polygon_subdivide_quads( GtkWidget *widget, gpointer data ) ;
void tb_polygon_subdivide_tris( GtkWidget *widget, gpointer data ) ;
void tb_polygon_subdivide_midpt( GtkWidget *widget, gpointer data ) ;
void tb_polygon_smooth( GtkWidget *widget, gpointer data ) ;
void tb_polygon_holes( GtkWidget *widget, gpointer data ) ;


/* every time the number of buttons exceeds a multiple of this number, a new 
column will be created. */
#define TOOLBOX_BUTTON_COLUMN_THRESHOLD 12

struct toolbox_button_data {
    gchar **pix;
    GCallback callback;
    gchar *tooltip;
};


struct toolbox_button_data toolbox_bd[] = {
    { grid_snap_xpm,			G_CALLBACK(tb_snap_to_grid),			"Snap selected verts to grid" },
    { vertex_snap_together_xpm,	G_CALLBACK(tb_snap_together),			"Snap selected verts together" },
    { vertex_snap_together_xpm,	G_CALLBACK(tb_snap_together_active_axis),   "Snap sel. verts together, along active axis\n(verts will line up)" },
    { vertex_weld_xpm,			G_CALLBACK(tb_weld_together),			"Weld sel. verts together" },
    { vertex_weld_xpm,			G_CALLBACK(tb_weld_modelwide),			"Weld all close verts together, model-wide" },
    { vertex_rot_90_xpm,		G_CALLBACK(tb_rotate_90),				"Rot. sel. verts 90deg about active axis" },
    { vertex_rot_xpm,			G_CALLBACK(tb_rotate),					"Rot. sel. verts about active axis" },
    { vertex_flip_xpm,			G_CALLBACK(tb_flip_vertices),			"Flip sel. verts along active axis" },
    { vertex_remove_unused_xpm,	G_CALLBACK(tb_remove_unused_vertices),	"Remove unused verts from model" },
    { poly_extrude_xpm,			G_CALLBACK(tb_extrude),					"Extrude selected polys individually" },
    { poly_extrude_xpm,			G_CALLBACK(tb_extrude_connected),		"Extrude selected polys, keeping them connected" },
    { poly_reverse_winding_xpm,	G_CALLBACK(tb_reverse_polygon_winding),	"Reverse vertex winding on sel. polys" },
    { poly_turn_edge_xpm,		G_CALLBACK(tb_polygon_turn_edge),		"Turn edge on two adjacent sel. triangles" },
    { poly_make_triangles_xpm,	G_CALLBACK(tb_polygon_make_triangles),	"Convert any sel. poly with >3 sides into triangles" },
    { poly_subdivide_quads_xpm,	G_CALLBACK(tb_polygon_subdivide_quads),	"Subdivide selected polygons, forming quads" },
    { poly_subdivide_tris_xpm,	G_CALLBACK(tb_polygon_subdivide_tris),	"Subdivide selected polygons, forming triangles" },
    { poly_subdivide_midpt_xpm,	G_CALLBACK(tb_polygon_subdivide_midpt),	"Subdivide selected polygons at edge mid-points\n(similar to 'forming quads' but triangles are handled differently)" },
    { poly_smooth_xpm,			G_CALLBACK(tb_polygon_smooth),			"Smooth selected polygons" },
/*    { poly_subdivide_midpt_xpm,	G_CALLBACK(tb_polygon_holes),			"Find holes in selected polygons" },*/
    { NULL, NULL, NULL }  /* kinda important... */
};

int active_axis = 0;



void tb_active_axis_x_cb( GtkWidget *widget, gpointer data ) {
    active_axis = 0;
}

void tb_active_axis_y_cb( GtkWidget *widget, gpointer data ) {
    active_axis = 1;
}

void tb_active_axis_z_cb( GtkWidget *widget, gpointer data ) {
    active_axis = 2;
}

void tb_snap_to_grid( GtkWidget *widget, gpointer data ) {

    tools_vertices_snap_to_grid(the_model, the_model->selected_verts, view_config.grid_spacing );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}

void tb_snap_together( GtkWidget *widget, gpointer data ) {

    tools_vertices_snap_together(the_model, the_model->selected_verts );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}

void tb_snap_together_active_axis( GtkWidget *widget, gpointer data ) {

    tools_vertices_snap_together_along_axis(the_model, the_model->selected_verts, active_axis );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}

void tb_weld_together( GtkWidget *widget, gpointer data ) {

    tools_vertices_weld_list(the_model, the_model->selected_verts );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}

void tb_weld_modelwide( GtkWidget *widget, gpointer data ) {

    tools_vertices_weld_modelwide(the_model );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}

void tb_extrude( GtkWidget *widget, gpointer data ) {

    g_slist_free( 
		tools_polys_extrude(the_model, the_model->selected_polys ) );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}

void tb_extrude_connected( GtkWidget *widget, gpointer data ) {

    g_slist_free( 
		tools_polys_extrude_connected(the_model, the_model->selected_polys ) );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}

void tb_rotate_90( GtkWidget *widget, gpointer data ) {

    tools_vertices_rotate_90(the_model, the_model->selected_verts, active_axis );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}

void tb_rotate_angle_cb( GtkWidget *widget, gpointer data ) {

    const gchar *entry_text;
    float angle;

    entry_text = gtk_entry_get_text ( GTK_ENTRY (data) );
    sscanf( entry_text, "%f", &angle);
    angle = angle * M_PI / 180.0;

    tools_vertices_rotate(the_model, the_model->selected_verts, active_axis, angle );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}

void tb_rotate( GtkWidget *widget, gpointer data ) {

    GtkWidget *dialog, *button, *entry;

    dialog = gtk_dialog_new();

    entry = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY (entry), 10);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
                        entry, TRUE, TRUE, 0);
    g_signal_connect( G_OBJECT( entry ), "activate",
                        G_CALLBACK( tb_rotate_angle_cb ), entry );
    gtk_widget_show(entry);

    button = gtk_button_new_with_label( "Cancel" );
    gtk_box_pack_start( GTK_BOX( GTK_DIALOG( dialog ) ->action_area ), button,
                        FALSE, FALSE, 5 );
	g_signal_connect_swapped( G_OBJECT( button ), "clicked",
					  G_CALLBACK( gtk_widget_destroy ), dialog );
    gtk_widget_show( button );
    button = gtk_button_new_with_label( "Rotate" );
    gtk_box_pack_start( GTK_BOX( GTK_DIALOG( dialog ) ->action_area ), button,
                        FALSE, FALSE, 5 );
    g_signal_connect( G_OBJECT( button ), "clicked",
                        G_CALLBACK( tb_rotate_angle_cb ), entry );
	g_signal_connect_swapped( G_OBJECT( button ), "clicked",
					  G_CALLBACK( gtk_widget_destroy ), dialog );
    gtk_widget_show( button );

    gtk_window_set_title(GTK_WINDOW(dialog), "Enter Angle");

    gtk_widget_show(dialog);

}

void tb_reverse_polygon_winding( GtkWidget *widget, gpointer data ) {

    GSList *l;
    
    if( the_model->selected_polys == NULL )
        return;
    
    for( l = the_model->selected_polys; l; l = l->next ) {
        Poly *p = (Poly*)l->data;
        poly_reverse_winding( p );
    }
    
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}

void tb_flip_vertices( GtkWidget *widget, gpointer data ) {

    tools_vertices_flip(the_model, the_model->selected_verts, active_axis );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}


void tb_remove_unused_vertices( GtkWidget *widget, gpointer data ) {

	model_remove_unused_vertices( the_model );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}


void tb_polygon_turn_edge( GtkWidget *widget, gpointer data ) {
	
	tools_polys_turn_edge(the_model, the_model->selected_polys );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}


void tb_polygon_make_triangles( GtkWidget *widget, gpointer data ) {
	
	tools_polys_make_triangles(the_model, the_model->selected_polys );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}


void tb_polygon_subdivide_quads( GtkWidget *widget, gpointer data ) {
	
	tools_polys_subdivide_quads(the_model, the_model->selected_polys );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}


void tb_polygon_subdivide_tris( GtkWidget *widget, gpointer data ) {
	
	tools_polys_subdivide_tris(the_model, the_model->selected_polys );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}


void tb_polygon_subdivide_midpt( GtkWidget *widget, gpointer data ) {
	
	tools_polys_subdivide_midpoint(the_model, the_model->selected_polys );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}


void tb_polygon_smooth( GtkWidget *widget, gpointer data ) {
	
	tools_polys_smooth(the_model, the_model->selected_polys, 1.0 );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}


void tb_polygon_holes( GtkWidget *widget, gpointer data ) {
	
	tools_polys_find_holes(the_model, the_model->selected_polys );
	action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );
}



GtkWidget *create_toolbox( void ) {

    GtkWidget * toolbox;
    GtkWidget * button;
    GtkWidget * hbox;
    GtkTooltips *tips;
	GdkPixbuf *pix;
    GtkWidget * pixwidget;
	GtkWidget *frame;
    GtkWidget * button_table;
	GtkWidget *scrolled_win;
	GtkWidget *rollup;
	
    int i = 0;
	int num_button_columns, num_button_rows, num_buttons;
	int row, column;
	char tiptext[] = 
		"Some of the tools below act on a single axis, rather than all 3.  "
		"Use this to select the axis.";

    toolbox = gtk_vbox_new( FALSE, 5 );
    gtk_container_set_border_width( GTK_CONTAINER(toolbox), 5 );

	tips = gtk_tooltips_new();

	/* ACTIVE AXIS SELECTOR ***********************************************/

	frame = gtk_frame_new( "Active Axis" );

    hbox = gtk_hbox_new( FALSE, 5 );
    button = gtk_radio_button_new_with_label( NULL, "X" );
    g_signal_connect (G_OBJECT( button ), "clicked",
                        G_CALLBACK( tb_active_axis_x_cb ), NULL );
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( button ), TRUE );
    gtk_box_pack_start( GTK_BOX( hbox ), button, TRUE, TRUE, 0 );
	gtk_tooltips_set_tip( tips, button, tiptext, NULL );

    button = gtk_radio_button_new_with_label( 
                 gtk_radio_button_get_group( GTK_RADIO_BUTTON( button ) ), 
                 "Y" );
    g_signal_connect (G_OBJECT( button ), "clicked",
                        G_CALLBACK( tb_active_axis_y_cb ), NULL );
    gtk_box_pack_start( GTK_BOX( hbox ), button, TRUE, TRUE, 0 );
	gtk_tooltips_set_tip( tips, button, tiptext, NULL );

    button = gtk_radio_button_new_with_label( 
                 gtk_radio_button_get_group( GTK_RADIO_BUTTON( button ) ), 
                 "Z" );
    g_signal_connect (G_OBJECT( button ), "clicked",
                        G_CALLBACK( tb_active_axis_z_cb ), NULL );
    gtk_box_pack_start( GTK_BOX( hbox ), button, TRUE, TRUE, 0 );
	gtk_tooltips_set_tip( tips, button, tiptext, NULL );

	gtk_container_add( GTK_CONTAINER( frame ), hbox );
	gtk_box_pack_start( GTK_BOX( toolbox ), frame, FALSE, FALSE, 0 );

	/* ROLLUP AND SCROLLED WINDOW *******************************************/

	scrolled_win = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( scrolled_win ),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC );
	rollup = rollup_new();
	gtk_box_pack_start( GTK_BOX( toolbox ), scrolled_win, TRUE, TRUE, 0 );
	gtk_scrolled_window_add_with_viewport( 
		GTK_SCROLLED_WINDOW( scrolled_win ), rollup );
	
	/* TOOL BUTTONS (organized into columns) ********************************/

	num_buttons = (sizeof(toolbox_bd)/sizeof(struct toolbox_button_data))-1;
	num_button_columns = num_buttons / TOOLBOX_BUTTON_COLUMN_THRESHOLD;
	if( num_buttons % TOOLBOX_BUTTON_COLUMN_THRESHOLD != 0 )
		num_button_columns++;
	num_button_rows = TOOLBOX_BUTTON_COLUMN_THRESHOLD;
	
	button_table = gtk_table_new( 
		num_button_rows, num_button_columns, TRUE );
	rollup_append_page( rollup, button_table, "Built-in", TRUE );
	row = 0;
	column = 0;
	for( i = 0; toolbox_bd[i].pix != NULL; i++ ) {

		/* button itself */
		button = gtk_button_new();
		g_signal_connect( G_OBJECT( button ), "clicked",
							G_CALLBACK( toolbox_bd[i].callback ), NULL );

		/* tooltip */
		gtk_tooltips_set_tip( tips, button, toolbox_bd[ i ].tooltip, NULL );

		/* pixbuf */
		pix = gdk_pixbuf_new_from_xpm_data( ( const char ** ) toolbox_bd[ i ].pix );
		pixwidget = gtk_image_new_from_pixbuf( pix );
		g_object_unref( pix );
		gtk_container_add( GTK_CONTAINER( button ), pixwidget );
		
		/* add button to table */
		gtk_table_attach_defaults( GTK_TABLE(button_table), button,
			column, column+1,
			row, row+1 );

		row++;
		if( row >= num_button_rows ) {
			column++;
			row = 0;
		}
	}

	gtk_tooltips_enable( tips );

	/* done here, and not later, because we don't want the scriptbox to be 
	expanded (see below for reason)  */
	gtk_widget_show_all( toolbox );

#ifdef USE_PYTHON
	/* SCRIPTBOX BUTTONS ****************************************************/
	
	/* we don't want the Scripts page to be expanded initially, since 
	there is a bug in gtk regarding adding widgets (after program startup)
	to non-topmost notebook pages */
	rollup_append_page( rollup, create_scriptbox(), "Scripts", FALSE );
#endif

	return toolbox;
}




