/*
 * File:         undo.c
 * 
 * Description:  handles undo (action) stack, knows how to undo actions
 * 
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */

#include <string.h>  /* riddle me this: why is memcpy in string.h? */

#include "undo.h"
#include "vertex.h"
#include "polygon.h"
#include "tools.h"
#include "selection.h"
#include "primitive.h"
#include "documents.h"
#include "prefs.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif

/* FILE-SCOPE VARS ******************************************************/

int undo_limit_stack = FALSE;
int undo_max_stack_len = 200; /* only valid if undo_limit_stack is true */


/* PROTOTYPES ***********************************************************/

int undo_retrieve_preferences( void ) ;

void action_push( Model *model, Action *action ) ;
void action_remove_top( Model *model ) ;
void action_delete( Action *action ) ;
void action_unref_objs( Action *action ) ;
void action_trim_stack( Model* m ) ;

void action_print_stack( Model *model );
void action_print( Action *action ) ;
char * action_type_string_lookup( ActionType type ) ;


/* FUNCS ****************************************************************/

void undo_init( void ) {
	/* listen for the preferences-changed signal */
	g_signal_connect( 
		notificationObj, "notify::preferences-changed", 
		G_CALLBACK(undo_retrieve_preferences), NULL );
	undo_retrieve_preferences();
}


int undo_retrieve_preferences( void ) {
	GSList *l;
	
	undo_limit_stack = pref_get_bool( 
		"Undo::Limit the Undo Stack Length", undo_limit_stack );
	undo_max_stack_len = pref_get_int( 
		"Undo::Maximum Undo Stack Length", undo_max_stack_len );

	/* undo_max_stack_len must be >= 0 */
	if( undo_max_stack_len < 0 ) {
		undo_max_stack_len = 0;
	}
	
	if( undo_limit_stack ) {
		for( l = models; l; l = l->next ) {
			action_trim_stack( (Model*)l->data );
		}
	}

	return FALSE;
}


void undo_enable(Model *model) { 
	model->undo_enabled = TRUE;
/*	printf( "undo is now %s\n", model->undo_enabled ? "on" : "off" );*/
}

void undo_disable(Model *model) { 
	model->undo_enabled = FALSE;
/*	printf( "undo is now %s\n", model->undo_enabled ? "on" : "off" );*/
}


/* Pushes an Action onto the action stack
 */
void action_push( Model *model, Action *action ) {

	model->action_stack = g_list_prepend( model->action_stack, action );	

	model->action_stack_length++;
}


/* Pops an Action off of the action stack, and frees it
 */
void action_remove_top( Model *model ) {

    Action *action;
    GList *first = model->action_stack;

	if( first == NULL ) return;

    action = (Action*)first->data;

    model->action_stack = g_list_remove_link( model->action_stack, first );
    g_list_free_1( first );
	model->action_stack_length--;
	
	if( model->action_stack == NULL )
		model->action_stack_tail = NULL;
	
	action_delete( action );
}


void action_delete( Action *action ) {
    if( action == NULL )
		return;
	
	action_unref_objs( action );
	if( action->objects != NULL )
        g_slist_free( action->objects );
    if( action->fargs != NULL )
        free( action->fargs );
    if( action->iargs != NULL )
        free( action->iargs );
    free( action );
}


void action_unref_objs( Action *action ) {
	switch( action->type ) {
		case ACTION_VERTEX_ADD_TO_MODEL:
		case ACTION_VERTEX_REMOVE_MODEL:
		case ACTION_POLY_ADD_TO_MESH:
		case ACTION_POLY_REMOVE_MESH:
			primitive_unref( (Primitive*) action->object );
			break;
		default:
			break;
	}
}


/* Clears the undo stack
 */
void action_clear_stack( Model *model ) {
    while( model->action_stack != NULL )
        action_remove_top( model );
}


void action_trim_stack( Model* m ) {
	/* if the model's action_stack has grown beyond the bounds set by the 
	user, remove actions from the bottom of the stack until it is 
	within the size restriction */
	int orig_len;

	if( m == NULL ) return;
	if( !undo_limit_stack ) return;

	if( !(m->action_stack_length > undo_max_stack_len) ) return;
	
	if( m->action_stack_tail == NULL ) {
		m->action_stack_tail = g_list_last( m->action_stack );
	}

	orig_len = m->action_stack_length;

	while( m->action_stack_tail != NULL &&
		   ( m->action_stack_length > undo_max_stack_len || 
		   ((Action *)m->action_stack_tail->data)->type != ACTION_MARKER ) ) 
	{
		GList *last = m->action_stack_tail;
		m->action_stack_tail = m->action_stack_tail->prev;
		if( m->action_stack_tail )
			m->action_stack_tail->next = NULL;
		last->prev = NULL;
		
		action_delete( (Action *)last->data );
		g_list_free_1( last );
		
		m->action_stack_length--;
	}
	
	if( m->action_stack_length == 0 ) {
		/* if we've trimmed away the entire stack, 
		set action_stack accordingly */
		m->action_stack = NULL;
		m->action_stack_tail = NULL;
	}
	
	printf( "Note: %i actions were discarded\n", orig_len - m->action_stack_length );
}


#define UNDO_MEMCPY_F(dest, size) \
            dest = malloc( sizeof( float ) * size ); \
            memcpy( dest, fargs, sizeof( float ) * size );

#define UNDO_MEMCPY_I(dest, size) \
            dest = malloc( sizeof( int ) * size ); \
            memcpy( dest, iargs, sizeof( int ) * size );


/* Registers a new Action with the system
 *
 */
void action_do( Model *model, ActionType type, GSList *objects, void *object, 
				Mesh *mesh, float *fargs, int *iargs, void *misc )
{
    Action *action;

	if( !model ) {
		printf( "Error in %s : model is null.  \n"
				"\tThe undo stack has probably been corrupted.\n", 
				__FUNCTION__ );
		return;
	}
	
	if( !model->undo_enabled ) {
//        printf( "Undo is disabled\n" );
        return;
    }

    action = malloc( sizeof( Action ) );
    memset( action, 0, sizeof( Action ) );

    action->type = type;
    action->objects = g_slist_copy( objects );
	action->object = object;
    action->mesh = mesh;
    action->fargs = NULL;
    action->iargs = NULL;
    action->misc = misc;

    switch( action->type ) {
        case ACTION_MARKER:
			/* Only trim the stack when pushing MARKER onto stack, not every 
			action.  The MARKER action not only indicates the completion of a 
			"user action", it also indicates that whatever tool was making 
			changes to the model is now finished.  This is important, as the 
			reference-counting code might get confused if the stack is trimmed 
			in the middle of a tool's manipulations, and primitives might 
			inadvertently be deleted.  Example: some tools, like the 
			polygon-subdivide tools, create vertices that are "in limbo" 
			temporarily.  If an action, one holding a reference to one of 
			those verts, were trimmed from the stack before the tool was 
			finished, the vert would be deleted. */
			action_push( model, action );
			action_trim_stack( model );
            break;
            
        case ACTION_VERTEX_SEL:
			action_push( model, action );
            break;
            
        case ACTION_VERTEX_UNSEL:
			action_push( model, action );
            break;
            
        case ACTION_VERTEX_MOVE:
            UNDO_MEMCPY_F( action->fargs, 3 )
			action_push( model, action );
            break;
            
        case ACTION_VERTEX_ROT:
            UNDO_MEMCPY_F( action->fargs, 4 )
            UNDO_MEMCPY_I( action->iargs, 1 )
			action_push( model, action );
            break;
            
		case ACTION_VERTEX_SCALE:
			UNDO_MEMCPY_F( action->fargs, 6 )
			action_push( model, action );
			break;
			
		case ACTION_VERTEX_ADD_TO_MODEL:
			primitive_ref( (Primitive*)action->object );
			action_push( model, action );
            break;
            
		case ACTION_VERTEX_REMOVE_MODEL:
			primitive_ref( (Primitive*)action->object );
			action_push( model, action );
            break;
            
		case ACTION_VERTEX_ADD_TO_MESH:
			action_push( model, action );
            break;
            
		case ACTION_VERTEX_REMOVE_MESH:
			action_push( model, action );
            break;
            
		case ACTION_VERTEX_ADD_TO_POLY:
			action_push( model, action );
            break;
            
		case ACTION_VERTEX_REMOVE_POLY:
			UNDO_MEMCPY_I( action->iargs, 1 )
			action_push( model, action );
            break;

		case ACTION_VERTEX_SWITCH_POLY:
			UNDO_MEMCPY_I( action->iargs, 1 )
			action_push( model, action );
            break;

        case ACTION_POLY_SEL:
			action_push( model, action );
            break;
            
        case ACTION_POLY_UNSEL:
			action_push( model, action );
            break;
            
		case ACTION_POLY_ADD_TO_MESH:
			primitive_ref( (Primitive*)action->object );
			action_push( model, action );
            break;

		case ACTION_POLY_REMOVE_MESH:
			primitive_ref( (Primitive*)action->object );
			action_push( model, action );
            break;

		case ACTION_POLY_REVERSE_WINDING:
			action_push( model, action );
            break;

        case ACTION_UNKNOWN:
        default:
            /* if the action is not one of the above, or is UNKNOWN, then 
               this is an action that can not be undone, and we need to 
               clear the undo stack (rather than risk corrupting the model)
               */
/*            printf( "This action is not undo-able.  The undo stack has been cleared\n" );*/
/* FIXME - post message to error queue */
			action_delete( action );
			action = NULL;
            action_clear_stack( model );
            break;
    }
}


/* Pops an Action off of the action stack, and un-does the action.
 * This one's a doozie...
 */
void action_undo( Model *model ) {
    
    Action *action;
    
	if( !model ) return;

    if( !model->undo_enabled ) {
/* FIXME - post message to error queue */
        printf( "Undo is disabled\n" );
        return;
    }
    
    if( model->action_stack == NULL ) {
/* FIXME - post message to error queue */
        printf( "The undo stack is empty.\n" );
        return;
    }

    undo_disable( model );
    
    /* if this is a MARKER action, pop it off */
    action = (Action*)model->action_stack->data;
    if( action->type == ACTION_MARKER ) {
        action_remove_top( model );
        if( model->action_stack != NULL )
            action = (Action*)model->action_stack->data;
    }
    
    /* undo actions until we hit the next marker, or the stack is empty */
    while( model->action_stack != NULL && action->type != ACTION_MARKER ) {
    
        switch( action->type ) {
    
            case ACTION_VERTEX_SEL:
                sel_vert_unselect_list( action->objects );
                break;
        
            case ACTION_VERTEX_UNSEL:
                sel_vert_select_list( action->objects );
                break;
        
            case ACTION_VERTEX_MOVE:
                if( action->objects ) {
                    tools_vertices_move( model, action->objects, 
                                         -1.0f * action->fargs[0], 
                                         -1.0f * action->fargs[1], 
                                         -1.0f * action->fargs[2] );
                } else {
                    tools_vertices_move( model, model->selected_verts, 
                                         -1.0f * action->fargs[0], 
                                         -1.0f * action->fargs[1], 
                                         -1.0f * action->fargs[2] );
                }
                break;
        
            case ACTION_VERTEX_ROT:
                if( action->objects ) {
                    tools_vertices_rotate_about_point( model, 
						action->objects, 
						action->iargs[0],
						action->fargs, 
						-1.0f * action->fargs[3],
						FALSE );
                } else {
                    tools_vertices_rotate_about_point( model, 
						model->selected_verts, 
						action->iargs[0],
						action->fargs, 
						-1.0f * action->fargs[3],
						FALSE );
                }
                break;
        
			case ACTION_VERTEX_SCALE:
				{
					float temp[3];
					temp[0] = 1. / action->fargs[0];
					temp[1] = 1. / action->fargs[1];
					temp[2] = 1. / action->fargs[2];
					tools_vertices_scale_about_point( model, 
						( action->objects ? 
							action->objects : model->selected_verts ),
						temp, action->fargs + 3, FALSE );
				}
				break;

			case ACTION_VERTEX_ADD_TO_MODEL:
				model_vertex_remove( model, (Vertex*)action->object );
	            break;
	            
			case ACTION_VERTEX_REMOVE_MODEL:
				model_vertex_add( model, (Vertex*)action->object );
	            break;
	            
			case ACTION_VERTEX_ADD_TO_MESH:
				mesh_vertex_remove( action->mesh, (Vertex*)action->object );
	            break;
	            
			case ACTION_VERTEX_REMOVE_MESH:
				mesh_vertex_add( action->mesh, (Vertex*)action->object );
	            break;
	            
			case ACTION_VERTEX_ADD_TO_POLY:
				poly_remove_vertex( (Poly*)action->misc, (Vertex*)action->object );
	            break;
	            
			case ACTION_VERTEX_REMOVE_POLY:
				poly_add_vertex_nth( (Poly*)action->misc, (Vertex*)action->object, action->iargs[0] );
	            break;
			
			case ACTION_VERTEX_SWITCH_POLY:
				poly_replace_vertex( (Poly*)action->misc, (Vertex*)action->object, action->iargs[0] );
				break;

            case ACTION_POLY_SEL:
                sel_poly_unselect_list( action->objects );
                break;
        
            case ACTION_POLY_UNSEL:
                sel_poly_select_list( action->objects );
                break;
        
			case ACTION_POLY_ADD_TO_MESH:
				mesh_polygon_remove( action->mesh, (Poly*)action->object );
            	break;

			case ACTION_POLY_REMOVE_MESH:
				mesh_polygon_add( action->mesh, (Poly*)action->object );
            	break;

			case ACTION_POLY_REVERSE_WINDING:
				poly_reverse_winding( (Poly*)action->object );
            	break;

            case ACTION_UNKNOWN:
            default:
                printf( "The undo stack contains weird stuff.  This is bad.\n" );
                printf( "\taction->type is %i\n", action->type );
                break;
        }

        action_remove_top( model );
        if( model->action_stack != NULL )
            action = (Action*)model->action_stack->data;
    }
    
    undo_enable( model );
}


void action_print_stack( Model *model ) {
	GList *l;
	fprintf( stderr, "Model %x has %i items in action stack:\n", 
			(unsigned int)model, g_list_length( model->action_stack ) );
	for( l = model->action_stack; l; l = l->next ) {
		fprintf( stderr, "\t" );
		action_print( (Action*)l->data );
	}
}


void action_print( Action *action ) {
	fprintf( stderr, 
		"Action: addr %x; type %s; objs %x (%i items); obj %x \n",
		(unsigned int)action,
		action_type_string_lookup( action->type ),
		(unsigned int)action->objects,
		g_slist_length( action->objects ),
		(unsigned int)action->object 
		);
}


char * action_type_string_lookup( ActionType type ) {
	
	switch( type ) {
	case ACTION_UNKNOWN:
		return "***ACTION_UNKNOWN***";
		break;
	case ACTION_VERTEX_SEL:
		return "ACTION_VERTEX_SEL";
		break;
	case ACTION_VERTEX_UNSEL:
		return "ACTION_VERTEX_UNSEL";
		break;
	case ACTION_VERTEX_MOVE:
		return "ACTION_VERTEX_MOVE";
		break;
	case ACTION_VERTEX_ROT:
		return "ACTION_VERTEX_ROT";
		break;
	case ACTION_VERTEX_SCALE:
		return "ACTION_VERTEX_SCALE";
		break;
	case ACTION_VERTEX_ADD_TO_MODEL:
		return "ACTION_VERTEX_ADD_TO_MODEL";
		break;
	case ACTION_VERTEX_REMOVE_MODEL:
		return "ACTION_VERTEX_REMOVE_MODEL";
		break;
	case ACTION_VERTEX_ADD_TO_MESH:
		return "ACTION_VERTEX_ADD_TO_MESH";
		break;
	case ACTION_VERTEX_REMOVE_MESH:
		return "ACTION_VERTEX_REMOVE_MESH";
		break;
	case ACTION_VERTEX_ADD_TO_POLY:
		return "ACTION_VERTEX_ADD_TO_POLY";
		break;
	case ACTION_VERTEX_REMOVE_POLY:
		return "ACTION_VERTEX_REMOVE_POLY";
		break;
	case ACTION_VERTEX_SWITCH_POLY:
		return "ACTION_VERTEX_SWITCH_POLY";
		break;
	case ACTION_POLY_SEL:
		return "ACTION_POLY_SEL";
		break;
	case ACTION_POLY_UNSEL:
		return "ACTION_POLY_UNSEL";
		break;
	case ACTION_POLY_ADD_TO_MESH:
		return "ACTION_POLY_ADD_TO_MESH";
		break;
	case ACTION_POLY_REMOVE_MESH:
		return "ACTION_POLY_REMOVE_MESH";
		break;
	case ACTION_POLY_REVERSE_WINDING:
		return "ACTION_POLY_REVERSE_WINDING";
		break;
	case ACTION_MESH_ADD:
		return "ACTION_MESH_ADD";
		break;
	case ACTION_MESH_DEL:
		return "ACTION_MESH_DEL";
		break;
	case ACTION_MESH_EDIT:
		return "ACTION_MESH_EDIT";
		break;
	case ACTION_MAX:
		return "ACTION_MAX";
		break;
	case ACTION_MARKER:
		return "***ACTION_MARKER***";
		break;
	default:
		return "***Invalid***";
		break;
	}

	return "***Invalid***";
}


