/* 
 * File:         vertex.c
 * 
 * Description:  funcs for creating, adding, and moving vertices.
 * 		 Used to also do drawing, but not any more!
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <glib.h>
#include <string.h>
#include "win32stuff.h"

#include "vertex.h"
#include "polygon.h"
#include "model.h"
#include "vector.h"
#include "undo.h"
#include "selection.h"
#include "primitive.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif




Vertex* vertex_new( void ) {
    Vertex* v = (Vertex*)malloc( sizeof( Vertex ) );
    
    if( v == NULL ) return NULL;
    
    memset( v, 0, sizeof( Vertex ) );
	
	primitive_set_delete_cb( (Primitive*)v, vertex_delete );
	((Primitive*)v)->type = PRIM_TYPE_VERTEX;
	
    v->v[0] = v->v[1] = v->v[2] = 0.0f;
    v->selected = FALSE;
    v->hidden = FALSE;
    
    return v;
}


void vertex_delete( Vertex *v ) {
	if( !v ) return;
	vertex_remove( v );
	free( v );
}


void vertex_remove( Vertex *v ) {
	if( !v ) return;

	sel_vert_unselect( v, TRUE );
	sel_vert_show( v );
	
	while( v->num_polys > 0 ) {
		vertex_remove_poly( v, v->polys[0] );
	}
}


Vertex* vertex_dup( Vertex* p ) {
    
    Vertex* np = NULL;
    
    g_return_val_if_fail( p != NULL, NULL );
    
    np = vertex_new();
    
	vector_copy( np->v, p->v );
    
    np->selected = FALSE;
    np->hidden = FALSE;
	
	np->polys = NULL;
	np->num_polys = 0;
	
	np->model = NULL;
    
    return np;
}


/* called only by poly.c */
void vertex_add_poly( Vertex *v, Poly *p ) {
    Poly ** tempP;
    
    if( p == NULL || v == NULL ) return;
    if( vertex_has_poly( v, p ) ) return;
    
    tempP = malloc( sizeof( Poly * ) * ( v->num_polys + 1 ) );

    /* copy the list of pointers */
    if( v->polys ) {
        memcpy( tempP, v->polys, sizeof( Poly * ) * v->num_polys );
        free( v->polys );
    }

    v->polys = tempP;
    v->polys[v->num_polys] = p;
    v->num_polys++;
	
	primitive_ref( (Primitive*)p );
}


/* called only by poly.c */
void vertex_remove_poly( Vertex *v, Poly *p ) {
    Poly ** tempP;
	int i, j;
    
    if( p == NULL || v == NULL ) return;
	if( v->num_polys <= 0 ) return;
    if( !vertex_has_poly( v, p ) ) return;
    
	if( v->num_polys == 1 ) {
		/* special case: one poly left */
		tempP = NULL;
	} else {
	    tempP = malloc( sizeof( Poly * ) * ( v->num_polys - 1 ) );
	
	    /* copy the list of pointers */
		for( i = 0, j = 0; i < v->num_polys; i++ ) {
			if( p != v->polys[i] ) {
				tempP[j] = v->polys[i];
				j++;
			}
		}
	}

	free( v->polys );
	v->polys = tempP;
	v->num_polys--;

	primitive_unref( (Primitive*)p );
}


int vertex_has_poly( Vertex *v, Poly *p ) {
    int i;
    for( i = 0; i < v->num_polys; i++ ) {
        if( v->polys[i] == p ) {
            return TRUE;
        }
    }
    return FALSE;
}


int vertex_is_in_use( Vertex *v ) {
	/* returns FALSE if v is not being used by any polys, TRUE otherwise */
	return( v->num_polys > 0 );
}


// fixme - should use the vector_add func
void vertex_move( Vertex *v, float deltaX, float deltaY, float deltaZ ) {

    v->v[0] += deltaX;
    v->v[1] += deltaY;
    v->v[2] += deltaZ;
}


void vertices_find_mean( GSList * plist, Vertex* average ) {
    
    /*
        takes a list of vertices and places the mean value in average
    */
    
    Vertex * p = NULL;
    int i, list_size = 0;
    
    if( plist == NULL || average == NULL )
        return;

    average->v[0] = average->v[1] = average->v[2] = 0.0f;
    
    while( plist != NULL ) {
        p = (Vertex*)plist->data;

        for( i = 0; i < 3; i++ ) {
            average->v[i] += p->v[i];
        }
        
        list_size++;
        plist = plist->next;
    }

    if( list_size > 0 ) {
        for( i = 0; i < 3; i++ ) {
            average->v[i] /= list_size;
        }
    }

}


/* Finds the angle between three points in space, by way of the dot product.
 * Vertex "c" is the middle vert, the point of rotation.
 * The result is in radians.
 */
float vertices_find_angle( Vertex *c, Vertex *v1, Vertex *v2 ) {
    float angle = 0.0f;
    float cosine;
    
    float vect1[3], vect2[3];
    
    vector_sub( vect1, v1->v, c->v );
    vector_sub( vect2, v2->v, c->v );
    
printf("dp: %f, vl1: %f, vl2: %f\n", vector_dot_prod( vect1, vect2 ),
             vector_length( vect1 ), vector_length( vect2 ) );

    cosine = vector_dot_prod( vect1, vect2 ) / 
             ( vector_length( vect1 ) * vector_length( vect2 ) );
    
    if( cosine < -1.0 ) cosine = -1.0;
    if( cosine > 1.0 ) cosine = 1.0;

    angle = acos( cosine );
    
    return angle;
}


gint vertices_isIdentical( Vertex* p1, Vertex* p2 ) {

    if( p1 == NULL || p2 == NULL )
        return FALSE;
    
    if( p1 == p2 ) 
        return TRUE;
    
    return FALSE;
}


gint vertices_isEqual( Vertex* p1, Vertex* p2 ) {

    float deltas[3];
    int i;
    
    if( p1 == NULL || p2 == NULL )
        return FALSE;
    
    if( p1 == p2 ) 
        return TRUE;
    
    for( i = 0; i < 3; i++ ) {
        deltas[i] = fabs( p1->v[i] - p2->v[i] );
        if( deltas[i] > POINTS_FUDGE_FACTOR )
            return FALSE;
    }
    
    
    return TRUE;
}


gint vertices_isNotEqual( Vertex* p1, Vertex* p2 ) {

    return( !vertices_isEqual( p1, p2 ) );

}



GSList * verts_get_polys_unique( GSList *verts ) {

	GSList *polys = NULL;
	int i;
	Vertex *v;
	
	for( ; verts != NULL; verts = verts->next ) {
		v = (Vertex *)verts->data;
		for( i = 0; i < v->num_polys; i++ ) {
			if( g_slist_index( polys, v->polys[i] ) == -1 )
				polys = g_slist_append( polys, v->polys[i] );
		}
	}
	
	return polys;
}


GSList * verts_get_polys( GSList *verts ) {

	GSList *polys = NULL;
	int i;
	Vertex *v;
	
	for( ; verts != NULL; verts = verts->next ) {
		v = (Vertex *)verts->data;
		for( i = 0; i < v->num_polys; i++ ) {
			polys = g_slist_append( polys, v->polys[i] );
		}
	}
	
	return polys;
}




///////////////////////////////////////////////////////
//
// dead code...
//
///////////////////////////////////////////////////////



/*
void vertex_move_selected_snap( float deltaX, float deltaY, float deltaZ, 
                               float spacing ) {

    GSList *cl;
    Vertex * p;
    float x, y, z, mult;
    
    if( grid_snap && the_model->selected_verts ) {
        // vertex snap stuff; this probably shouldn't be here...
        cl = the_model->selected_verts;
        p = ( Vertex * ) cl->data;
        
        x = p->v[0] + deltaX;
        y = p->v[1] + deltaY;
        z = p->v[2] + deltaZ;
        
        mult = x / spacing;
        x = spacing * rint( mult );
        mult = y / spacing;
        y = spacing * rint( mult );
        mult = z / spacing;
        z = spacing * rint( mult );
        
        deltaX = x - p->v[0];
        deltaY = y - p->v[1];
        deltaZ = z - p->v[2];
    }
    
    
    
    for ( cl = the_model->selected_verts; cl != NULL; cl = cl->next ) {
        // go through all the selected vetices and move them

        p = ( Vertex * ) cl->data;

        p->v[0] += deltaX;
        p->v[1] += deltaY;
        p->v[2] += deltaZ;
    }


}
*/

