/*
 * File:         view.c
 * 
 * Description:  handles 3d views, vertex&triangle selection, etc
 * 
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */


#include <stdio.h>
#include <math.h>
#include <string.h>
#include "win32stuff.h"

#include "vector.h"
#include "vertex.h"
#include "polygon.h"
#include "toolbar.h"    /* for click_mode and NUM_MOUSE_BUTTONS */
#include "notebook.h"
#include "globals.h"
#include "tex_app.h"
#include "glpreview.h"
#include "view.h"
#include "view_config.h"
#include "model.h"
#include "tools.h"
#include "bottombar.h"
#include "undo.h"
#include "selection.h"
#include "misc.h"

#ifdef MEMWATCH
#include "memwatch.h"
#endif

// DEFINES /////////////////////////////////////////////////////////////

#define RCV_VIEWPORT    0x0001
#define RCV_PROPERTIES  0x0002

struct _mouse_state {
    int button_pressed;   /* is this button down? */
    int has_moved;        /* has the mouse been moved while this button was down? */
    int startx, starty;   /* starting location for a drag */
    int endx, endy;
    int view_index;       /* one of the view types, ie VIEW_XY, etc */
};


// PROTOTYPES ///////////////////////////////////////////////////////////

model_view* view_new( int type );
gboolean view_delete_all( gpointer data ) ;
void view_init( model_view * view );
void initialize_mouse_state( void ) ;
static GdkGLConfig *view_create_glconfig( void );


gboolean view_mouse_down( GtkWidget *widget, GdkEventButton *event,
                          model_view *view );
gboolean view_mouse_motion( GtkWidget *widget, GdkEventMotion *event,
                            model_view *view );
gboolean view_mouse_up( GtkWidget *widget, GdkEventButton *event,
                        model_view *view );
gboolean view_mouse_leave( GtkWidget *widget, GdkEventCrossing *event,
                           model_view *view ) ;


void view_pan( model_view* view, float deltaX, float deltaY, float deltaZ );
void view_zoom( model_view* view, float deltaX, float deltaY, float deltaZ );
void view_object_select( model_view *view, struct _mouse_state* ms, guint event_state );

void view_rot_3d( model_view* view, int dx, int dy ) ;
void view_pan_3d( model_view* view, int dx, int dy ) ;

int view_rebuild_display_lists( void ) ;

void view_glarea_destroy( GtkWidget *widget, gpointer user_data ) ;
void view_glarea_realize( GtkWidget *widget, gpointer user_data );
gboolean view_glarea_configure( GtkWidget *widget, GdkEventConfigure *event,
                                gpointer user_data );
gboolean view_glarea_expose( GtkWidget *widget, GdkEventExpose *event,
                             gpointer user_data );

void view_reconfigure( model_view *view, gint flags );

Vertex *view_unproject( model_view *view, guint x, guint y );

void view_draw_selection_box( model_view *view, struct _mouse_state* ms );

int view_get_axis( model_view *view );


// GLOBALS AND FILE-SCOPE VARS ////////////////////////////////////////////

model_view *views[4] = { NULL, NULL, NULL, NULL };

int changed_views = FALSE;

struct _mouse_state mouse_state[NUM_MOUSE_BUTTONS];  /* one for each of the mouse buttons */

float cumulative_delta[4];  /* used to keep track of cumulative deltas, for 
                               translations and rotations */

GdkGLConfig *glconfig = NULL;  /* the glconfig used by all 4 views */
GdkGLContext *shared_glcontext = NULL;
GtkWidget *dummy_drawable = NULL;


///////////////////////////////////////////////////////////////////////////
//
//
//  initialization stuff
//
//
///////////////////////////////////////////////////////////////////////////



model_view* view_new( int type ) {

    model_view* v = (model_view*)malloc( sizeof(model_view) );

    memset( v, 0, sizeof( model_view ) );

    v->type = type;
    view_init( v );

    return v;
}


gboolean view_delete_all( gpointer data ) {

	int i;

printf( "%s : freeing views\n", __FUNCTION__ );

    for( i = 0; i < VIEW_3D + 1; i++ ) {
		if( views[i] )
			free( views[i] );
	}
	return FALSE;
}


void view_init( model_view * view ) {
	/* set the view parameters to some reasonable defaults */

    if( view == NULL ) return;

    view->dirty = TRUE;

    switch( view->type ) {

    case VIEW_XY:
        view->heading = 0.0;
        view->pitch = 0.0;
        view->dist = 10.0;

        view->grid_mins[ 0 ] = -20.0f;
        view->grid_mins[ 1 ] = -20.0f;
        view->grid_mins[ 2 ] = 0.0f;

        view->grid_maxes[ 0 ] = 20.0f;
        view->grid_maxes[ 1 ] = 20.0f;
        view->grid_maxes[ 2 ] = 0.0f;

        break;
    case VIEW_XZ:
        view->heading = 0.0;
        view->pitch = 90.0;
        view->dist = 10.0;

        view->grid_mins[ 0 ] = -20.0f;
        view->grid_mins[ 1 ] = 0.0f;
        view->grid_mins[ 2 ] = -20.0f;

        view->grid_maxes[ 0 ] = 20.0f;
        view->grid_maxes[ 1 ] = 0.0f;
        view->grid_maxes[ 2 ] = 20.0f;

        break;
    case VIEW_YZ:
        view->heading = 270.0;
        view->pitch = 90.0;
        view->dist = 10.0;

        view->grid_mins[ 0 ] = 0.0f;
        view->grid_mins[ 1 ] = -20.0f;
        view->grid_mins[ 2 ] = -20.0f;

        view->grid_maxes[ 0 ] = 0.0f;
        view->grid_maxes[ 1 ] = 20.0f;
        view->grid_maxes[ 2 ] = 20.0f;

        break;
    case VIEW_3D:
        view->heading = 0.0;
        view->pitch = 0.0;
        view->dist = 10.0;

        view->grid_mins[ 0 ] = -20.0f;
        view->grid_mins[ 1 ] = -20.0f;
        view->grid_mins[ 2 ] = 0.0f;

        view->grid_maxes[ 0 ] = 20.0f;
        view->grid_maxes[ 1 ] = 20.0f;
        view->grid_maxes[ 2 ] = 0.0f;

        break;
    default:
        break;
    }

    view->cursX = 0.0;
    view->cursY = 0.0;
    view->cursZ = 0.0;
    view->cursA = 0.0;
}


GtkWidget *create_view( int type ) {

    GtkWidget *frame;
	model_view *view;
	static int firstTime = TRUE;

	view = view_new( type );
	views[type] = view;

	if( firstTime ) {
		view_config_init();
		initialize_mouse_state();

		g_signal_connect( 
			notificationObj, "notify::exit", 
			G_CALLBACK(view_delete_all), NULL );
		g_signal_connect( 
			notificationObj, "notify::model-appearance-changed", 
			G_CALLBACK(view_redraw_all), NULL );

		gtk_widget_realize( dummy_drawable );
		shared_glcontext = gtk_widget_get_gl_context( dummy_drawable );

		if( glconfig == NULL ) {
			glconfig = view_create_glconfig();
		}

		firstTime = FALSE;
	}

	view->glarea = gtk_drawing_area_new();
	gtk_widget_set_gl_capability( 
			view->glarea,
			glconfig,
			shared_glcontext,
			TRUE,
			GDK_GL_RGBA_TYPE );

    gtk_widget_set_events( GTK_WIDGET( view->glarea ),
//                           gtk_widget_get_events( GTK_WIDGET( view->glarea ) ) |
                           GDK_EXPOSURE_MASK|
                           GDK_BUTTON_PRESS_MASK|
                           GDK_BUTTON_RELEASE_MASK|
                           GDK_POINTER_MOTION_MASK|
                           GDK_POINTER_MOTION_HINT_MASK|
                           GDK_ENTER_NOTIFY_MASK|
                           GDK_LEAVE_NOTIFY_MASK );

g_signal_connect( G_OBJECT( view->glarea ), "destroy",
                  G_CALLBACK( view_glarea_destroy ), view );

    g_signal_connect( G_OBJECT( view->glarea ), "realize",
                        G_CALLBACK( view_glarea_realize ), view );
    g_signal_connect( G_OBJECT( view->glarea ), "configure_event",
                        G_CALLBACK( view_glarea_configure ), view );
    g_signal_connect( G_OBJECT( view->glarea ), "expose_event",
                        G_CALLBACK( view_glarea_expose ), view );
    g_signal_connect( G_OBJECT( view->glarea ), "button_press_event",
                        G_CALLBACK( view_mouse_down ), view );
    g_signal_connect( G_OBJECT( view->glarea ), "motion_notify_event",
                        G_CALLBACK( view_mouse_motion ), view );
    g_signal_connect( G_OBJECT( view->glarea ), "button_release_event",
                        G_CALLBACK( view_mouse_up ), view );
    g_signal_connect( G_OBJECT( view->glarea ), "leave_notify_event",
                        G_CALLBACK( view_mouse_leave ), view );

    /*
      gint (* enter_notify_event)      (GtkWidget          *widget,
                                        GdkEventCrossing   *event);
      gint (* leave_notify_event)      (GtkWidget          *widget,
                                        GdkEventCrossing   *event);
    */

    frame = gtk_frame_new( NULL );
    gtk_frame_set_shadow_type( GTK_FRAME( frame ), GTK_SHADOW_OUT );
    gtk_container_add( GTK_CONTAINER( frame ), view->glarea );
    gtk_widget_show( frame );

    view->gltable = gtk_table_new( 2, 2, FALSE );

    gtk_table_attach( GTK_TABLE( view->gltable ), frame, 0, 1, 0, 1,
                      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0 );

    // the table used to contain more stuff than just the glarea... namely,
    // the 3d view used to have scroll bars for rotating the camera.  I'll
    // leave the tables in place, since they might come in handy at some point

    gtk_widget_show( view->gltable );
    gtk_widget_show( view->glarea );

	/* using the gtk_timeout feature ensures that the views will be redrawn 
	at least once following the first time the gtk_main func is entered */
/* not needed	gtk_timeout_add( 50, view_redraw_all, NULL );*/

    return view->gltable;
}


GtkWidget *view_create_dummy_drawable( void ) {
	/* 
	In order to share glcontexts amongst multiple widgets using gtkglext, a 
	very specific (and convoluted, and inconvenient, ...) sequence of events 
	must occur:
		1) a drawing area (or some other widget) must be created
		2) gtk_widget_set_gl_capability must be called on that DA, 
			with NULL for the 'shared' arg
		3) gtk_widget_realize must be called on that widget, which in turn 
			means that THAT WIDGET MUST HAVE A VALID PARENT, AND *THAT* PARENT 
			MUST HAVE A VALID PARENT, and so on... all the way up to the 
			top-level window.
		4) gtk_widget_get_gl_context can then be called to retrieve a valid 
			glcontext, which can then be shared with other drawables via the 
			gtk_widget_set_gl_capability func
	The architecture of kludge3d prevents us from doing these steps all at 
	once.  The kludge3d UI is assembled piecemeal, hence step 3 (above) 
	presents a very real problem.
	
	As a work-around, we will create a dummy drawable here, which will be 
	passed back to gui.c, where it will be added to the top-level window.
	Later, when create_view is called and runs the first-time-through stuff, 
	the dummy drawable will be realized and its glcontext retrieved.  This 
	glcontext will then be shared by all of the views.
	
	BTW - if you're asking "what's so important about sharing GL contexts?", 
	here's why:
		When gtk_widget_set_gl_capability is called with a 'shared' context, 
		it allows multiple drawables to share display lists and texture 
		objects.  This saves memory and makes the use of display lists much 
		more practical (and display lists can be used to improve rendering 
		speed).
	*/
	
	if( glconfig == NULL ) {
		glconfig = view_create_glconfig();
	}

	dummy_drawable = gtk_drawing_area_new();
	gtk_widget_set_gl_capability( 
			dummy_drawable,
			glconfig,
			NULL,
			TRUE,
			GDK_GL_RGBA_TYPE );
	return dummy_drawable;
}


void initialize_mouse_state( void ) {

    int btn;

    for( btn = 0; btn < 3; btn++ ) {
        mouse_state[btn].button_pressed = FALSE;
        mouse_state[btn].has_moved = FALSE;
        mouse_state[btn].startx = 0;
        mouse_state[btn].starty = 0;
        mouse_state[btn].endx = 0;
        mouse_state[btn].endy = 0;
        mouse_state[btn].view_index = -1;
    }
}


/***
 *** Configure the OpenGL framebuffer.  (nabbed from a gtkglext example)
 ***/
static GdkGLConfig *view_create_glconfig( void ) {

	GdkGLConfig *glconf;

	/* Try double-buffered visual */
	glconf = gdk_gl_config_new_by_mode( GDK_GL_MODE_RGB	 |
										GDK_GL_MODE_DEPTH  |
										GDK_GL_MODE_DOUBLE );
	if( glconf == NULL ) {
		fprintf( stderr, "\n*** Cannot find the double-buffered visual.\n");
		fprintf( stderr, "\n*** Trying single-buffered visual.\n");
	
		/* Try single-buffered visual */
		glconf = gdk_gl_config_new_by_mode( GDK_GL_MODE_RGB	|
											GDK_GL_MODE_DEPTH );
		if( glconf == NULL ) {
			fprintf( stderr, "*** No appropriate OpenGL-capable visual found.\n");
			exit( 1 );
		}
	}

	return glconf;
}



//////////////////////////////////////////////////////////////////////////
//
//
//  callbacks, of all kinds
//
//
///////////////////////////////////////////////////////////////////////////


void view_glarea_destroy( GtkWidget *widget, gpointer user_data ) {
#ifdef VERBOSE
	model_view * view = ( model_view * ) user_data;
	
	printf( "widget for view %i was destroyed\n", view->type );
#endif
}


void view_glarea_realize( GtkWidget *widget, gpointer user_data ) {

    model_view * view = ( model_view * ) user_data;

	/* gtk_widget_get_gl_context and gtk_widget_get_gl_drawable won't return 
	proper values until the glarea has been realized.  That is why these 
	assignments must be made here, and not earlier. */
	view->glcontext = gtk_widget_get_gl_context( view->glarea );
	view->gldrawable = gtk_widget_get_gl_drawable( view->glarea );

    view_reconfigure( view, RCV_PROPERTIES | RCV_VIEWPORT );

    // for some reason, it seems that X wants to call the "expose" routine
    // on our widget before the "realize" routine.  This results in a black
    // background in the ogl windows, because glp_render is being called
    // *before* glp_glconfigure.  Hmmph.  Riddle me this: why does X try to
    // draw a window before it's been created?

    // anyway, this call to glp_render replaces that icky black blankness
    // with what *should* have been drawn
	gtk_widget_queue_draw( view->glarea );
}


gboolean view_glarea_configure( 
	GtkWidget *widget, GdkEventConfigure *event, gpointer user_data ) 
{
    model_view * view = ( model_view * ) user_data;

    view_reconfigure( view, RCV_VIEWPORT );
	gtk_widget_queue_draw( view->glarea );
    return TRUE;
}


gboolean view_glarea_expose( 
	GtkWidget *widget, GdkEventExpose *event, gpointer user_data ) 
{
    model_view * view = ( model_view * ) user_data;

/* 
FIXME FIXME FIXME 

If the view doesn't need to be re-rendered (ie not dirty), then 
it is more efficient to just paste the contents of the double buffer 
onto the screen.

HOWEVER,

it seems that this little trick doesn't work any longer.  (I don't 
think it has *ever* worked perfectly for non-mesa opengl implementations.  
They've always experienced problems similar to those described below.) 

For the most recent version of mesa (at this time, 4.0.4): 
A opengl viewport that is exposed will (sometimes!) display 
a black screen instead of the expected image.  My guess is that the double 
buffer is (sometimes!) zeroed when an ogl context/screen/etc is obscured.
Ugh.

FIXME FIXME FIXME 
*/

	/* if the glarea has not been realized yet, don't continue.  Damn X11 
	and its bizarre policies.  Why is an expose event generated before a 
	widget has been realized? */
	if( view->glcontext == NULL || view->gldrawable == NULL )
		return TRUE;

//    if( view->dirty )
        glp_render( view );
//    else
//        glp_expose( view );
    return TRUE;
}


gboolean view_redraw_all( void ) {
    // forces a redraw of all views

    int i;

    for( i = 0; i < 4; i++ ) {
        views[i]->dirty = TRUE;
        gtk_widget_queue_draw( views[i]->glarea );
    }

	return FALSE;
}


void view_redraw_n( int frame_num ) {
    // forces a redraw of view number n

    if( frame_num >= VIEW_XY && frame_num <= VIEW_3D ) {
        views[frame_num]->dirty = TRUE;
        gtk_widget_queue_draw( views[frame_num]->glarea );
    }
}


gboolean view_mouse_leave( GtkWidget *widget, GdkEventCrossing *event,
                           model_view *view ) {

    /* if the cursor left the glarea, with a mouse button held down,
       then set changed_views */
    if( mouse_state[1].button_pressed ||
        mouse_state[2].button_pressed ||
        mouse_state[3].button_pressed )
        changed_views = TRUE;
    else
        changed_views = FALSE;    // courtesy of dp


    //printf( "exited view %i, changed_views is %i\n", view->type, changed_views );
    return TRUE;
}



gboolean view_mouse_down( GtkWidget *widget, GdkEventButton *event,
                          model_view *view ) {

    // mouse button has been pressed

    int rerender = FALSE;
    Vertex * v = NULL;
    GLfloat z;
	int i;
	
	/* check to see if any of the buttons are being pressed */
	for( i = 1; i < 4; i++ ) {
		if( mouse_state[i].button_pressed ) {
			/* we should bail out if the user tries something like this */
			return TRUE;
		}
	}
	
        cumulative_delta[0] = 0.0f;
        cumulative_delta[1] = 0.0f;
        cumulative_delta[2] = 0.0f;
        cumulative_delta[3] = 0.0f;
		if( click_mode[ event->button ] == CM_OBJECT_SCALE ) {
			cumulative_delta[3] = 1.0f;
		} 
        
        switch( global_scope ) {

        case SCOPE_VERTEX:
            switch( click_mode[ event->button ] ) {

            case CM_OBJECT_ADD:
                v = view_unproject( view, event->x, event->y );

                if( v != NULL ) {
                    if( the_model->current_node->parent != NULL ) {
                        // IMPORTANT - we don't want to add vertices to the root mesh
                        //FIXME - this still needs work (users can paste vertices into root mesh, for example)
                        model_vertex_add( the_model, v );
                        mesh_vertex_add( the_model->current_mesh, v );
						action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
                    } else {
                        bb_push_message( 3.0, "You can't add vertices to the root group" );
                        free( v );
                        v = NULL;
                    }
                } 
else { printf("v was null\n"); }
                rerender = TRUE;

                break;

            case CM_OBJECT_DEL:

                v = glp_test_click_vertex( view, &z, event->x, event->y );

                if( v != NULL ) {
                    model_vertex_remove( the_model, v );
					action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
                    v = NULL;
                    rerender = TRUE;
                }

                break;
            default:
                break;
            }
            break;

        case SCOPE_POLYGON:

            switch( click_mode[ event->button ] ) {

            case CM_OBJECT_ADD:

                if( the_model->partially_completed_poly == NULL ) {
                    /* create a new polygon */
                    the_model->partially_completed_poly = poly_new();
                    bb_push_message( 1.5, "Click on at least 2 vertices" );
                }

                v = glp_test_click_vertex( view, &z, event->x, event->y );

                if( v != NULL ) {
/*printf( "vert has %i references\n", ((Primitive*)v)->ref_count );*/
                    poly_add_vertex( the_model->partially_completed_poly, v );
					if( the_model->partially_completed_poly->num_verts == 2 ) {
						bb_push_message( 1.5, "... the polygon, or you can continue to add vertices" );
						bb_push_message( 1.5, "You can click on an empty area to complete ..." );
					}
/*printf( "partially-completed-tri now has %i verts\n", the_model->partially_completed_poly->num_verts );*/
                } else {
					/* user has clicked on an empty space, indicating that 
					   the polygon is complete */
                    if( the_model->partially_completed_poly->num_verts >= 2 && 
                        the_model->current_mesh != NULL ) 
                    {
                        /* partially_completed_poly has at least two 
						   vertices... time to add it to the model */
                        mesh_polygon_add( the_model->current_mesh, the_model->partially_completed_poly );
						action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
                        bb_push_message( 1.5, "Completed a polygon" );
/*{
int j;
printf( "Completed a polygon: " );
for(j=0; j < the_model->partially_completed_poly->num_verts; j++ )
    printf( "%x ", (unsigned int)the_model->partially_completed_poly->verts[j] );
printf( "\n" );
}*/
                    } else {
                        poly_delete( the_model->partially_completed_poly );
                        bb_push_message( 1.5, "Aborted a polygon" );
                    }
                    the_model->partially_completed_poly = NULL;
                }
                rerender = TRUE;

                break;

            case CM_OBJECT_DEL:
                {
                    Poly * p = glp_test_click_polygon( view, &z, event->x, event->y );
                    if( p ) {
                        model_polygon_remove( the_model, p );
						action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
                        p = NULL;
                        rerender = TRUE;
                    }
                }
                break;

            case CM_OBJECT_EXTRUDE:
				g_slist_free( 
					tools_polys_extrude_connected(the_model, the_model->selected_polys ) );
            case CM_OBJECT_ROT:
			case CM_OBJECT_SCALE:
            case CM_OBJECT_MOVE:
                if( view->type != VIEW_3D ) {
                    GSList *verts = polys_get_verts( the_model->selected_polys );
                    sel_vert_unselect_all( the_model );
                    sel_vert_select_list( verts );
                    g_slist_free( verts );
                }
                break;


            default:
                break;
            }
        default:
            break;
        }


    mouse_state[event->button].button_pressed = TRUE;
    mouse_state[event->button].has_moved = FALSE;
    mouse_state[event->button].startx = event->x;
    mouse_state[event->button].starty = event->y;
    mouse_state[event->button].endx = 0;
    mouse_state[event->button].endy = 0;
    mouse_state[event->button].view_index = view->type;

    if ( rerender ) {
		g_signal_emit_by_name( notificationObj, 
			"notify::model-appearance-changed", NULL );
    }

    return TRUE;
}




gboolean view_mouse_motion( GtkWidget *widget, GdkEventMotion *event,
                            model_view *view ) {

    // mouse has been moved (this gets ugly...)

    static int prev_mouse_x = -1;
    static int prev_mouse_y = -1;

    gchar *bb_coords;

    //    static model_view* prev_view = NULL;

    int x, y;
    int btn = 0;
    Vertex * current_coords = NULL;
    Vertex * prev_coords = NULL;
    Vertex * orig_coords = NULL;
    GdkModifierType state;
    int redraw_current = FALSE;
    int redraw_all = FALSE;
    float deltaX, deltaY, deltaZ;
    float delta[3];

    g_return_val_if_fail( view != NULL, TRUE );

    //    if( prev_view == NULL )
    //        prev_view = view;

    //printf( "prev_view is %i\n", prev_view->type );
    //printf( "     view is %i\n", view->type );


    if ( event->is_hint )
        gdk_window_get_pointer ( event->window, &x, &y, &state );
    else {
        x = event->x;
        y = event->y;
        state = event->state;
    }


    if( x < 0 || x > view->glarea->allocation.width ||
        y < 0 || y > view->glarea->allocation.height )
        return TRUE;

    if( prev_mouse_x < 0 || prev_mouse_y < 0 ) {
        // first time we've been called.
        prev_mouse_x = x;
        prev_mouse_y = y;
    }

	{
		int num_buttons_held_down = 0;
		if( state & GDK_BUTTON1_MASK ) {
			btn = 1;
			num_buttons_held_down++;
		}
		if( state & GDK_BUTTON2_MASK ) {
			btn = 2;
			num_buttons_held_down++;
		}
		if( state & GDK_BUTTON3_MASK ) {
			btn = 3;
			num_buttons_held_down++;
		}
		
		if( num_buttons_held_down > 1 ) {
			int i;
			/* If more than one button is being held, we need to determine 
			which one should be "in control".  To do this, we'll check the 
			mouse_states to see if any has_moved */
			for( i = 1; i < 4; i++ ) {
				if( mouse_state[i].has_moved ) {
					btn = i;
					break;
				}
			}
		}
	}

    current_coords = view_unproject( view, x, y );
    if (current_coords != NULL) {
	bb_coords = g_strdup_printf("[ %f %f %f ]", 
				    current_coords->v[0],
				    current_coords->v[1],
				    current_coords->v[2]);
	bb_update_coords( bb_coords );
	g_free( bb_coords );
    }

    if( !changed_views && btn != 0 ) {

        prev_coords = view_unproject( view, prev_mouse_x, prev_mouse_y );

        if( current_coords == NULL || prev_coords == NULL )
            return TRUE;


#if 0
        if( prev_view != view ) {
            prev_coords->v[0] = current_coords->v[0];
            prev_coords->v[1] = current_coords->v[1];
            prev_coords->v[2] = current_coords->v[2];
        }
#endif

        deltaX = current_coords->v[0] - prev_coords->v[0];
        deltaY = current_coords->v[1] - prev_coords->v[1];
        deltaZ = current_coords->v[2] - prev_coords->v[2];
        vector_sub( delta, current_coords->v, prev_coords->v );

        switch( click_mode[ btn ] ) {

        case CM_OBJECT_MOVE:

            switch( global_scope ) {

            case SCOPE_POLYGON:
            case SCOPE_VERTEX:
                if( view->type != VIEW_3D ) {
                    vector_add( cumulative_delta, cumulative_delta, delta );
                    tools_vertices_move(the_model, the_model->selected_verts, deltaX, deltaY, deltaZ );
                    redraw_all = TRUE;
                }
                break;
            default:
                break;
            }
            break;

        case CM_OBJECT_ROT:

            orig_coords = view_unproject( view, mouse_state[btn].startx, mouse_state[btn].starty );
            
            switch( global_scope ) {

            case SCOPE_POLYGON:
            case SCOPE_VERTEX:
                if( view->type != VIEW_3D ) {
                    if( y - prev_mouse_y != 0 ) {
                    float *pt = orig_coords->v;
                    float angle = (float)( y - prev_mouse_y ) / (float) view->glarea->allocation.height;
                    angle *= 3.0 * M_PI;

                    cumulative_delta[0] = pt[0];
                    cumulative_delta[1] = pt[1];
                    cumulative_delta[2] = pt[2];
                    cumulative_delta[3] += angle;
                    tools_vertices_rotate_about_point(the_model, the_model->selected_verts, 
                    	view_get_axis( view ), pt, 
                        angle, FALSE );
                    }
                    redraw_all = TRUE;
                }
                break;
            default:
                break;
            }
            
			if( orig_coords ) free( orig_coords );
			
			break;

        case CM_OBJECT_SCALE:
			
			orig_coords = view_unproject( view, mouse_state[btn].startx, mouse_state[btn].starty );
			
			switch( global_scope ) {

			case SCOPE_POLYGON:
			case SCOPE_VERTEX:
				if( view->type != VIEW_3D ) {
					if( y - prev_mouse_y != 0 ) {
					float *pt = orig_coords->v;
					float factor = (float)( y - prev_mouse_y ) 
									/ (float) view->glarea->allocation.height;
					factor += 1.;

					cumulative_delta[0] = pt[0];
					cumulative_delta[1] = pt[1];
					cumulative_delta[2] = pt[2];
					cumulative_delta[3] *= factor;
					tools_vertices_scale_symmetrically_about_center(the_model,
						the_model->selected_verts, factor, FALSE );
					}
					redraw_all = TRUE;
				}
				break;
			default:
				break;
			}
			
			if( orig_coords ) free( orig_coords );
			
			break;

        case CM_OBJECT_EXTRUDE:

            switch( global_scope ) {

            case SCOPE_POLYGON:
                if( view->type != VIEW_3D ) {
                    vector_add( cumulative_delta, cumulative_delta, delta );
                    tools_vertices_move(the_model, the_model->selected_verts, deltaX, deltaY, deltaZ );
                    redraw_all = TRUE;
                }
                break;
            default:
                break;
            }
            break;

		case CM_OBJECT_THIN:

			if( view->type != VIEW_3D ) {
				vector_add( cumulative_delta, cumulative_delta, delta );
				switch( global_scope ) {

					case SCOPE_POLYGON:
						tools_scale_along_normals(the_model, the_model->selected_polys, NULL, deltaX + deltaY + deltaZ );
						break;
					case SCOPE_VERTEX:
						tools_scale_along_normals(the_model, NULL, the_model->selected_verts, deltaX + deltaY + deltaZ );
						break;
					default:
						break;
				}
				redraw_all = TRUE;
			}
			break;

        case CM_ZOOM:
            view_zoom( view, deltaX, deltaY, deltaZ );
            redraw_current = TRUE;
            break;

        case CM_PAN:
            if( view->type == VIEW_3D && state & GDK_SHIFT_MASK )
                view_pan_3d( view, x - prev_mouse_x, y - prev_mouse_y );
            else if( view->type == VIEW_3D )
                view_rot_3d( view, x - prev_mouse_x, y - prev_mouse_y );
            else
                view_pan( view, deltaX, deltaY, deltaZ );

            redraw_current = TRUE;
            break;

        default:
            break;
        }
        mouse_state[btn].has_moved = TRUE;

    } else if( btn == 0 ) {
        // if none of the buttons are being held down...
        changed_views = FALSE;
    }


    prev_mouse_x = x;
    prev_mouse_y = y;

    if( current_coords ) free( current_coords );
    if( prev_coords ) free( prev_coords );


    if( redraw_all ) {
		g_signal_emit_by_name( notificationObj, 
			"notify::model-appearance-changed", NULL );
    } else if( redraw_current ) {
        view_redraw_n( view->type );
    }

    // if we're in object sel. mode:
    // now that rendering has been completed, draw a selection box
    if( click_mode[ btn ] == CM_OBJECT_SEL ) {
        mouse_state[btn].endx = event->x;
        mouse_state[btn].endy = event->y;
        view_draw_selection_box( view, &(mouse_state[btn]) );
    }

    return TRUE;
}





gboolean view_mouse_up( GtkWidget *widget, GdkEventButton *event,
                        model_view *view ) {

    // mouse button has been released

    int btn;
    int rerender = FALSE;
    int axis = view_get_axis( view );


    btn = event->button;

    mouse_state[btn].endx = event->x;
    mouse_state[btn].endy = event->y;

    if( mouse_state[btn].button_pressed ) {
        switch( click_mode[ btn ] ) {

        case CM_OBJECT_SEL:

            //FIXME - check that starting view is same as present view

            view_object_select( view, mouse_state + btn, event->state );
            /*         pointer math -----------^   */
            
            action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
            rerender = TRUE;

            break;

        case CM_OBJECT_MOVE:

            action_do( the_model, ACTION_VERTEX_MOVE, NULL, NULL, NULL, cumulative_delta, NULL, NULL );

            action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
            rerender = TRUE;

            break;

		case CM_OBJECT_SCALE:

			{
			float temp[6];
			Vertex avg;
			
			temp[0] = cumulative_delta[3];
			temp[1] = cumulative_delta[3];
			temp[2] = cumulative_delta[3];
			vertices_find_mean( the_model->selected_verts, &avg );
			vector_copy( temp + 3, avg.v );

			action_do( the_model, ACTION_VERTEX_SCALE, NULL, NULL, NULL, temp, NULL, NULL );
			action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
			
			}
			rerender = TRUE;

			break;

		case CM_OBJECT_EXTRUDE:

			switch( global_scope ) {

			case SCOPE_POLYGON:
				action_do( the_model, ACTION_VERTEX_MOVE, NULL, NULL, NULL, cumulative_delta, NULL, NULL );
				action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
				break;
			default:
				break;
			}
			rerender = TRUE;

			break;

		case CM_OBJECT_THIN:

			if( view->type != VIEW_3D ) {
				action_do( the_model, ACTION_UNKNOWN, NULL, NULL, NULL, NULL, NULL, NULL );
				action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
			}
			rerender = TRUE;

			break;

        case CM_OBJECT_ROT:

            action_do( the_model, ACTION_VERTEX_ROT, NULL, NULL, NULL, cumulative_delta, &axis, NULL );

            action_do( the_model, ACTION_MARKER, NULL, NULL, NULL, NULL, NULL, NULL );
            rerender = TRUE;

            break;


        default:
            break;

        }
    }

    mouse_state[btn].button_pressed = FALSE;
    mouse_state[btn].has_moved = FALSE;
    mouse_state[btn].startx = 0;
    mouse_state[btn].starty = 0;
    mouse_state[btn].endx = 0;
    mouse_state[btn].endy = 0;
    mouse_state[btn].view_index = -1;


    if ( rerender ) {
		g_signal_emit_by_name( notificationObj, 
			"notify::model-appearance-changed", NULL );
    }

    return TRUE;
}







void view_reconfigure( model_view *view, gint flags ) {
    g_return_if_fail( view != NULL );
	view->dirty = TRUE;

	if( view->glcontext == NULL || view->gldrawable == NULL )
		return;

    if ( flags & RCV_PROPERTIES )
        glp_glconfigure( view );
    if ( flags & RCV_VIEWPORT )
        glp_glviewport( view );
}


int view_rebuild_display_lists( void ) {
	
	/* because this code is called at a point when the views are still being 
	initialized, we have to take extra steps to ensure that, when we call 
	glp_rebuild_display_lists, the view->glcontext and gldrawable are valid.
	We'll do this through the use of a gtk timeout... this ensures that, if 
	the context and drawable aren't ready yet (such as during startup), the 
	function will be called again later (once we've hit a gtk_main, at which 
	point the context and drawable should be ready) */
	
	model_view * view = views[0];
	
	if( view == NULL || view->glcontext == NULL || view->gldrawable == NULL ) {
		/* try again later */
		gtk_timeout_add( 50, (GtkFunction)view_rebuild_display_lists, NULL );
	} else {
		glp_rebuild_display_lists( view );
	}
	
	return FALSE;
}


///////////////////////////////////////////////////////////////////////
//
// other stuff
//
///////////////////////////////////////////////////////////////////////


void view_pan( model_view* view, float deltaX, float deltaY, float deltaZ ) {

    if( view == NULL )
        return;

    if( view->type == VIEW_3D )
        return;

    view->camX += deltaX;
    view->camY += deltaY;
    view->camZ += deltaZ;

    view->grid_mins[0] = -20.0f - view->camX;
    view->grid_mins[1] = -20.0f - view->camY;

    view->grid_maxes[0] = 20.0f - view->camX;
    view->grid_maxes[1] = 20.0f - view->camY;

    view->grid_mins[2] = -20.0f - view->camZ;
    view->grid_maxes[2] = 20.0f - view->camZ;


}

void view_rot_3d( model_view* view, int dx, int dy ) {

    if( view == NULL )
        return;

    if( view->type != VIEW_3D )
        return;

    view->heading += dx;
    view->pitch   -= dy;  // I don't like inverted mouselook. So there.

}


void view_pan_3d( model_view* view, int dx, int dy ) {

    float deltaX, deltaY, h;

    if( view == NULL )
        return;

    if( view->type != VIEW_3D )
        return;


    // up/down motion is backwards... this is to counter for it
    dy = -dy;


    // take care of up/down mouse movement
    h = view->heading * DEGREES_TO_RADIANS;

    deltaX = sin( h ) * (float)dy;
    deltaY = cos( h ) * (float)dy;

    view->camX += deltaX;
    view->camY += deltaY;


    // take care of left/right mouse movement
    h = (view->heading + 90.0) * DEGREES_TO_RADIANS;

    deltaX = sin( h ) * (float)dx;
    deltaY = cos( h ) * (float)dx;

    view->camX += deltaX;
    view->camY += deltaY;

    // update the grid position
    view->grid_mins[0] = -20.0f - view->camX;
    view->grid_mins[1] = -20.0f - view->camY;

    view->grid_maxes[0] = 20.0f - view->camX;
    view->grid_maxes[1] = 20.0f - view->camY;
}


void view_zoom( model_view* view, float deltaX, float deltaY, float deltaZ ) {

    if( view == NULL )
        return;

    view->dist += (deltaX + deltaY + deltaZ) / ( 1.0 / view->dist );  // a hack...

    if( view->dist < 1.0f )
        view->dist = 1.0f;

    //fixme - is the present impl. of the ortho view zoom acceptable?  will this be needed again?
    //        if( view->type != VIEW_3D )
    //            glp_ortho_viewport( view );

}


void view_set_height( float height ) {
	views[VIEW_3D]->camZ = height;
	view_redraw_n( VIEW_3D );
}


void view_fov_changed( void ) {
	/* VIEW_3D is the only view that uses the fov setting... */
	view_reconfigure( views[VIEW_3D], RCV_VIEWPORT );
}


void view_object_select( model_view *view, struct _mouse_state* ms, guint event_state ) {
    /* called after buttondown, drag, buttonup */

    GSList* l = NULL;

    float z;

    int xavg, yavg, xsize, ysize;

    int keystate = 0;  // 0 = none, GDK_SHIFT_MASK = shift, etc...


    if( event_state & GDK_SHIFT_MASK )
        keystate = GDK_SHIFT_MASK;
    else if( event_state & GDK_CONTROL_MASK )
        keystate = GDK_CONTROL_MASK;



    if( ms->has_moved ) {

        xavg = ( ms->startx + ms->endx ) / 2;
        yavg = ( ms->starty + ms->endy ) / 2;

        xsize = abs( ms->startx - ms->endx );
        ysize = abs( ms->starty - ms->endy );

        if( global_scope == SCOPE_VERTEX ) {
            l = glp_test_click_vertices( view, &z, xavg, yavg, xsize, ysize );
        } else if( global_scope == SCOPE_POLYGON ) {
            l = glp_test_click_polygons( view, &z, xavg, yavg, xsize, ysize );
        } else if( global_scope == SCOPE_MESH ) {
        }


    } else {

        if( global_scope == SCOPE_VERTEX ) {
            l = g_slist_append( l, glp_test_click_vertex( view, &z,
                                ms->endx, ms->endy ) );
        } else if( global_scope == SCOPE_POLYGON ) {
            l = g_slist_append( l, glp_test_click_polygon( view, &z,
                                ms->endx, ms->endy ) );
        } else if( global_scope == SCOPE_MESH ) {
        }

    }



    switch( keystate ) {

    case 0:
        // user isn't holding down the shift of control keys,
        // so unselect everything before continuing


        if( global_scope == SCOPE_VERTEX ) {
            sel_vert_unselect_all( the_model );
        } else if( global_scope == SCOPE_POLYGON ) {
            sel_poly_unselect_all( the_model );
        } else if( global_scope == SCOPE_MESH ) {
        }

        // fall through

    case GDK_SHIFT_MASK:
        if( global_scope == SCOPE_VERTEX ) {
            sel_vert_select_list( l );
        } else if( global_scope == SCOPE_POLYGON ) {
            sel_poly_select_list( l );
        }
        break;

    case GDK_CONTROL_MASK:
        if( global_scope == SCOPE_VERTEX ) {
            sel_vert_unselect_list( l );
        } else if( global_scope == SCOPE_POLYGON ) {
            sel_poly_unselect_list( l );
        }
        break;

    default:
        break;

    }

    g_slist_free( l );

}


Vertex *view_unproject( model_view *view, guint x, guint y ) {

    // this is used *only* for zooming, panning, adding vertices.
    // presently, it only returns relative values (ie if you pan a view,
    // and then click in the middle of the view, the value returned by this func
    // will be <0, 0, 0> not <camX, camY, camZ> )

    // this func only exists because i can't figure out how to do
    // ray-plane intersections in opengl.  anybody care to help?

    //UPDATE - now calls glp_unproject, which returns the _actual_
    // world-space coordinate under the cursor.  Note that it's probably
    // not going to handle the 3d-projection very well... this func only
    // handles the orthographic views correctly

    Vertex* p = NULL;

    GLfloat ox, oy, oz;

    g_return_val_if_fail ( view != NULL, NULL );
    g_return_val_if_fail ( view->glarea != NULL, NULL );

    y = view->glarea->allocation.height - y;  // flip y axis

    glp_unproject( view, x, y, &ox, &oy, &oz );

//printf( "unproject test   x: %f    y: %f    z: %f\n",  ox, oy, oz  );

    p = vertex_new();
    p->v[0] = ox;
    p->v[1] = oy;
    p->v[2] = oz;

    // now let's zero-out the non-planar axis...
    // depends on whis view we're dealing w/
    
    if( view->type == VIEW_3D )
        /* used only for zoom & pan stuff... 
        doesn't actually return sensible values */
        p->v[2] = 0.0;
    else 
        p->v[ view_get_axis( view ) ] = 0.0;

    return p;
}



/* rotates a view around so that it's pointing in the other direction.
only used for orthographic views. */
gint view_flip_cb( GtkWidget *widget, gpointer data ) {
	int type;
	model_view *view;
	
	type = (int) data;
	if( type == VIEW_3D )
		return FALSE;
	
	view = views[type];
	
	switch( type ) {
	case VIEW_XY:
		view->pitch += 180.0;
		break;
	case VIEW_XZ:
		view->heading += 180.0;
		break;
	case VIEW_YZ:
		view->heading += 180.0;
		break;
	default:
		break;
	}
	
	view->flipped = !view->flipped;
	
	view_redraw_n( type );

	return FALSE;
}


void view_draw_selection_box( model_view *view, struct _mouse_state* ms ) {

    float startx, starty, endx, endy;
    float width, height;

    g_return_if_fail ( view != NULL );
    g_return_if_fail ( ms != NULL );

    width = view->glarea->allocation.width;
    height = view->glarea->allocation.height;

    startx = ms->startx;
    endx = ms->endx;

    // ogl origin is in lower left, X11 origin is in upper left...
    starty = height - ms->starty;
    endy = height - ms->endy;

    // glp needs coords in range 0.0 - 1.0
    startx = startx / width;
    starty = starty / height;
    endx = endx / width;
    endy = endy / height;

    glp_draw_selection_box( view, startx, starty, endx, endy );


}


/* Returns the number of the axis that this view is projected along
 * X = 0, Y = 1, Z = 2
 */
int view_get_axis( model_view *view ) {
    
    switch( view->type ) {
        case VIEW_XY:
            return 2;
        case VIEW_YZ:
            return 0;
        case VIEW_XZ:
            return 1;
        case VIEW_3D:
            return -1;
        default:
            break;
    }
    return -1;
}


