/*
 * File:         view_config.c
 * 
 * Description:  handles initializing, storing, etc some values related to 
 *               the views' appearance and how they behave
 * 
 * 
 * This source code is part of kludge3d, and is released under the 
 * GNU General Public License.
 * 
 * 
 */


#include <stdio.h>
#include <math.h>
#include <string.h>
#include <gtk/gtk.h>

#include "view_config.h"
#include "editor_models.h"
#include "view.h"
#include "prefs.h"
#include "globals.h"
#include "glpreview.h"
#include "vector.h"


#ifdef MEMWATCH
#include "memwatch.h"
#endif

/* GLOBALS **************************************************************/

model_view_config view_config;

GLfloat *cursor_glarray = NULL;		/* GL arrays describing */
GLfloat *vertex_glarray = NULL;		/* helper objects */
GLfloat *girder_glarray = NULL;


/* PROTOTYPES ***********************************************************/

gboolean view_delete_glarrays( gpointer data ) ;


/* VIEW-CONFIG STUFF ******************************************************/

/* initializes the view_config's members to reasonable defaults, and does 
some other one-time-only initializations */
void view_config_init( void ) {

	/* register the signals */
	g_signal_connect( 
		notificationObj, "notify::exit", 
		G_CALLBACK(view_delete_glarrays), NULL );
	g_signal_connect( 
		notificationObj, "notify::preferences-changed", 
		G_CALLBACK(view_retrieve_preferences), NULL );

	view_retrieve_preferences();

	view_config.fov = 90.0f;
	view_config.near_plane = 1.0f;
	view_config.far_plane = 200.0f;

	view_config.grid_spacing = 1.0f;

	view_config.vertices_visible = TRUE;
	view_config.grid_visible = TRUE;
	view_config.normals_visible = FALSE;

	view_config.cursor_visible = FALSE;

	view_config.polygon_mode = MP_SHOW_WIREFRAME;
	view_config.render_mode = MP_MODE_RENDER;
	view_config.triangle_order = MP_TRI_ORDER_CCW;

	view_config.cursor_rad = 0.2;
	view_config.vertex_rad = 0.1;
	view_config.girder_rad = 0.07;

	view_rebuild_glarrays( );

}


int view_retrieve_preferences( void ) {
	float *temp;
	float default_bg[] =		{ 0.35, 0.35, 0.35 };
	float default_grid[] =		{ 0.25, 0.25, 0.25 };
	float default_nmlvert[] =	{ 1.00, 1.00, 1.00 };
	float default_selvert[] =	{ 1.00, 0.00, 0.00 };
	float default_nmlpoly[] =	{ 0.80, 0.80, 0.80 };
	float default_selpoly[] =	{ 1.00, 0.00, 0.00 };
	
	pref_get_string( "View::Note", "Changes to the background color will take effect next time the program is started." );
	temp = pref_get_vector( "View::Background Color", default_bg );
	vector_copy( view_config.bkgnd_color, temp );
	temp = pref_get_vector( "View::Grid Color", default_grid );
	vector_copy( view_config.grid_color, temp );
	temp = pref_get_vector( "View::Vertex Color", default_nmlvert );
	vector_copy( view_config.vertex_unsel_color, temp );
	temp = pref_get_vector( "View::Selected Vertex Color", default_selvert );
	vector_copy( view_config.vertex_sel_color, temp );
	temp = pref_get_vector( "View::Polygon Color", default_nmlpoly );
	vector_copy( view_config.poly_unsel_color, temp );
	temp = pref_get_vector( "View::Selected Polygon Color", default_selpoly );
	vector_copy( view_config.poly_sel_color, temp );

	view_config.draw_verts_as_cubes = 
		pref_get_bool( "View::Draw Vertices Using Cubes (slower)", FALSE );

	g_signal_emit_by_name( notificationObj, 
		"notify::model-appearance-changed", NULL );

	return FALSE;
}


/* FUNCS RELATED TO THE HELPER OBJS *************************************/

gboolean view_delete_glarrays( gpointer data ) {

printf( "%s : freeing cursor objects\n", __FUNCTION__ );
	if( cursor_glarray ) {
		free( cursor_glarray );
		cursor_glarray = NULL;
	}
	if( vertex_glarray ) {
		free( vertex_glarray );
		vertex_glarray = NULL;
	}
	if( girder_glarray ) {
		free( girder_glarray );
		girder_glarray = NULL;
	}

	return FALSE;
}


void view_rebuild_glarrays( void ) {
	/* rebuilds the glarrays for the vertices (little cubes) the
	"girders" (little rectangular prisms) and the cursor (diamond-shaped) */

	int i;

	if ( cursor_glarray != NULL )
		free( cursor_glarray );
	if ( vertex_glarray != NULL )
		free( vertex_glarray );
	if ( girder_glarray != NULL )
		free( girder_glarray );

	if ( view_config.triangle_order == MP_TRI_ORDER_CW ) {
		cursor_glarray = ( GLfloat * ) malloc( sizeof( view_cursor_cw ) );
		memcpy( cursor_glarray, view_cursor_cw, sizeof( view_cursor_cw ) );
		vertex_glarray = ( GLfloat * ) malloc( sizeof( view_vertex_cw ) );
		memcpy( vertex_glarray, view_vertex_cw, sizeof( view_vertex_cw ) );
		girder_glarray = ( GLfloat * ) malloc( sizeof( view_girder_cw ) );
		memcpy( girder_glarray, view_girder_cw, sizeof( view_girder_cw ) );
	} else {
		cursor_glarray = ( GLfloat * ) malloc( sizeof( view_cursor_ccw ) );
		memcpy( cursor_glarray, view_cursor_ccw, sizeof( view_cursor_ccw ) );
		vertex_glarray = ( GLfloat * ) malloc( sizeof( view_vertex_ccw ) );
		memcpy( vertex_glarray, view_vertex_ccw, sizeof( view_vertex_ccw ) );
		girder_glarray = ( GLfloat * ) malloc( sizeof( view_girder_ccw ) );
		memcpy( girder_glarray, view_girder_ccw, sizeof( view_girder_ccw ) );
	}


	/* size the arrays to fit the view_config.* parameters */
	for ( i = 0; i < ( sizeof( view_cursor_ccw ) / sizeof( GLfloat ) ); i += 10 ) {
		cursor_glarray[ i + 7 ] *= view_config.cursor_rad;
		cursor_glarray[ i + 8 ] *= view_config.cursor_rad;
		cursor_glarray[ i + 9 ] *= view_config.cursor_rad;
	}
	for ( i = 0; i < ( sizeof( view_vertex_ccw ) / sizeof( GLfloat ) ); i += 6 ) {
		vertex_glarray[ i + 3 ] *= view_config.vertex_rad;
		vertex_glarray[ i + 4 ] *= view_config.vertex_rad;
		vertex_glarray[ i + 5 ] *= view_config.vertex_rad;
	}
	for ( i = 0; i < ( sizeof( view_girder_ccw ) / sizeof( GLfloat ) ); i += 6 ) {
		girder_glarray[ i + 3 ] *= view_config.girder_rad;
		girder_glarray[ i + 4 ] *= view_config.girder_rad;
	}
	
}



/* SETS *****************************************************************/

void view_set_rendermode( int mode ) {

	view_config.polygon_mode = mode;
}


void view_set_triangleorder( int order ) {

	int i;
	view_config.triangle_order = order;
	for( i = 0; i < VIEW_3D + 1; i++ ) {
		glp_set_triangleorder( views[i] );
	}
	view_rebuild_glarrays(  );
}


void view_set_vertexsize( float size ) {

	view_config.vertex_rad = size;
	view_config.cursor_rad = size * 2.0;
	view_config.girder_rad = size * 0.7;

	view_rebuild_glarrays(  );
}


void view_set_fov( float fov ) {

	view_config.fov = fov;
	view_fov_changed();
}

