/***************************************************************************
 *   Copyright (C) 2004 by Norbert Andres                                  *
 *   nandres@web.de                                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "kpicframer.h"
#include "kpicframerview.h"
#include "prefs.h"
#include "scores.h"

#include <qmessagebox.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>

#include <kglobal.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kdeversion.h>
#include <kedittoolbar.h>
#include <kmenubar.h>
#include <kstatusbar.h>
#include <kkeydialog.h>
#include <kaccel.h>
#include <kconfig.h>

#include <kedittoolbar.h>

#include <kstdaccel.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kstdgameaction.h>

KPicFramer::KPicFramer()
 : KMainWindow(0, "KPicFramer"),
   m_options(new Options(kapp->config())),
   m_view(new KPicFramerView(this))
{
  // tell the KMainWindow that this is indeed the main widget
  setCentralWidget(m_view);
  setupActions();
  statusBar()->show();

  // apply the saved mainwindow settings, if any, and ask the mainwindow
  // to automatically save settings if changed: window size, toolbar
  // position, icon size, etc.
  setAutoSaveSettings();
  enableUndo(false);

  // allow the view to change the statusbar and caption
  connect(m_view, SIGNAL(signalChangeStatusbar(const QString&)),
          this,   SLOT(changeStatusbar(const QString&)));
  connect(m_view, SIGNAL(signalChangeCaption(const QString&)),
          this,   SLOT(changeCaption(const QString&)));
  connect(m_view, SIGNAL(signalChangeScore(int)),
          this,   SLOT(changeScore(int)));
  connect(m_view, SIGNAL(signalIncreaseStoppedGames(int)),
          this,   SLOT(changeStoppedGames(int)));
  connect(m_view, SIGNAL(signalIncreaseWonGames(int)),
          this,   SLOT(changeWonGames(int )));
  connect(m_view, SIGNAL(signalIncreaseLostGames(int, int)),
          this,   SLOT(changeLostGames(int, int)));
  connect(m_view, SIGNAL(signalGameOver(const QString &)),
          this,   SLOT(showGameOver(const QString &)));
  connect(m_view, SIGNAL(signalEnableUndo(bool)),
          this,   SLOT(enableUndo(bool)));
  connect(m_view, SIGNAL(signalPauseEnd()),
          this,   SLOT(pauseGame()));

  changeStatusbar(i18n("Press %1 to start a game!")
                  .arg(m_newAction->shortcut().toString()));
}

KPicFramer::~KPicFramer()
{
  m_options->writeConfig(kapp->config());
  delete m_options;
}

void KPicFramer::setupActions()
{
  m_newAction   = KStdGameAction::gameNew( this, SLOT(newGame()), actionCollection() );
  m_pauseAction = KStdGameAction::pause(this, SLOT(pauseGame()), actionCollection());

  KStdGameAction::quit(this, SLOT(close()), actionCollection() );
  KStdGameAction::highscores(this, SLOT(showHighscore()), actionCollection() );

  m_toolbarAction
    = KStdAction::showToolbar(this, SLOT(optionsShowToolbar()), actionCollection());
  m_statusbarAction
    = KStdAction::showStatusbar(this, SLOT(optionsShowStatusbar()), actionCollection());

  KStdAction::keyBindings(this, SLOT(optionsConfigureKeys()), actionCollection());
  KStdAction::configureToolbars(this, SLOT(optionsConfigureToolbars()), actionCollection());
  KStdAction::preferences(this, SLOT(optionsPreferences()), actionCollection());

  m_restartAction = new KAction(i18n("Restart &Game"), QString::fromLatin1("reload"), 0,
                                this, SLOT(restartGame()),
                                actionCollection(), "restart_game");
  m_undoAction = new KAction(i18n("&Undo"), QString::fromLatin1("undo"),
                             KStdAccel::shortcut(KStdAccel::Undo),
                             this, SLOT(undo()), actionCollection(), "undo");

  m_restartAction->setEnabled(false);
  KStatusBar * sBar = statusBar();

  sBar->insertItem("", 0, 0, true);
  sBar->setItemAlignment(0, Qt::AlignVCenter | Qt::AlignRight);

  createGUI();
  optionsShowToolbar();
  optionsShowStatusbar();
}

void KPicFramer::enableUndo(bool enable)
{
  m_undoAction->setEnabled(enable);
}

void KPicFramer::saveProperties(KConfig * config)
{
  m_options->writeConfig(config);
}

void KPicFramer::readProperties(KConfig * config)
{
  m_options->readConfig(config);
}

void KPicFramer::newGame()
{
  m_view->newGame();
  m_restartAction->setEnabled(true);
}

void KPicFramer::pauseGame()
{
  if (m_view->pause())
    m_pauseAction->setChecked(true);
  else
    m_pauseAction->setChecked(false);
}

void KPicFramer::restartGame()
{
  m_view->restart();
}

void KPicFramer::undo()
{
  m_view->undo();
}

void KPicFramer::optionsShowToolbar()
{
  if (m_toolbarAction->isChecked())
    toolBar()->show();
  else
    toolBar()->hide();
}

void KPicFramer::optionsShowStatusbar()
{
  if (m_statusbarAction->isChecked())
    statusBar()->show();
  else
    statusBar()->hide();
}

void KPicFramer::optionsConfigureKeys()
{
  KKeyDialog::configureKeys(actionCollection(), "kpicframerui.rc");
}

void KPicFramer::optionsConfigureToolbars()
{
  KEditToolbar dlg(actionCollection());
  connect(&dlg, SIGNAL(newToolbarConfig()), this, SLOT(newToolbarConfig()));
  dlg.exec();

#if defined(KDE_MAKE_VERSION)
# if KDE_VERSION >= KDE_MAKE_VERSION(3,1,0)
  saveMainWindowSettings(KGlobal::config(), autoSaveGroup());
# else
  saveMainWindowSettings(KGlobal::config());
# endif
#else
  saveMainWindowSettings(KGlobal::config());
#endif
}

void KPicFramer::newToolbarConfig()
{
  // this slot is called when user clicks "Ok" or "Apply" in the toolbar editor.
  // recreate our GUI, and re-apply the settings (e.g. "text under icons", etc.)
  createGUI();

#if defined(KDE_MAKE_VERSION)
# if KDE_VERSION >= KDE_MAKE_VERSION(3,1,0)
  applyMainWindowSettings(KGlobal::config(), autoSaveGroup());
# else
  applyMainWindowSettings(KGlobal::config());
# endif
#else
  applyMainWindowSettings(KGlobal::config());
#endif
}

void KPicFramer::optionsPreferences()
{
  KPicPreferences dlg(m_options);

  connect(&dlg, SIGNAL(saveConfig()),
          this, SLOT(optionsSave()));

  dlg.exec();
}

void KPicFramer::optionsSave()
{
  m_options->writeConfig(kapp->config());
  m_view->reinit();
}

void KPicFramer::showHighscore()
{
  ScoresDialog dlg;
  dlg.m_lastScore->setNum(m_options->getLastScore());
  dlg.m_highScore->setNum(m_options->getHightScore());
  dlg.m_wonGames->setNum(m_options->getWonGames());
  dlg.m_lostGames->setNum(m_options->getLostGames());
  dlg.m_stoppedGames->setNum(m_options->getStoppedGames());

  if (!dlg.exec())
  {
    m_options->setHighscore(0);
    m_options->setStoppedGames(0);
    m_options->setLostGames(0);
    m_options->setWonGames(0);
    m_options->setLastScore(0);

    m_options->writeConfig(kapp->config());
  }
}

void KPicFramer::changeStatusbar(const QString& text)
{
  statusBar()->message(text);
}

void KPicFramer::changeScore(int score)
{
  // display the text on the statusbar
  statusBar()->changeItem(i18n("Score: %1").arg(score), 0);
}

void KPicFramer::changeStoppedGames(int score)
{
  int old = m_options->getStoppedGames();
  m_options->setStoppedGames(++old);

  m_options->setLastScore(score);
  m_options->writeConfig(kapp->config());
}

void KPicFramer::changeWonGames(int score)
{
  int old = m_options->getWonGames();
  m_options->setWonGames(++old);

  checkHighscore(score);
}

void KPicFramer::changeLostGames(int score, int num)
{
  int old = m_options->getLostGames();
  m_options->setLostGames(old + num);

  checkHighscore(score);
}

void KPicFramer::checkHighscore(int score)
{
  int old = m_options->getHightScore();
  if (score > old)
  {
    m_options->setHighscore(score);
    QMessageBox::information(this, i18n("New Highscore"),
                             i18n("You've reached a new highscore: %1").arg(score), i18n("Ok"));
  }

  m_options->setLastScore(score);
  m_options->writeConfig(kapp->config());
}

bool KPicFramer::queryClose()
{
  if (!m_view->close())
    return false;

  return true;
}

void KPicFramer::showGameOver(const QString & title)
{
  QMessageBox mb(title, i18n("Do you want to play again?"), QMessageBox::Question,
                 QMessageBox::Yes | QMessageBox::Default, QMessageBox::No, QMessageBox::Cancel, this);
  mb.setButtonText( QMessageBox::Yes, i18n("Yes") );
  mb.setButtonText( QMessageBox::No,  i18n("Quit"));
  mb.setButtonText( QMessageBox::Cancel,  i18n("Cancel"));

  switch( mb.exec() )
  {
    case QMessageBox::Yes:
    {
      newGame();
      return;
    }
    case QMessageBox::No:
    {
      close();
      return;
    }
  }
}

void KPicFramer::changeCaption(const QString& text)
{
  setCaption(text);
}
#include "kpicframer.moc"
