/***************************************************************************
 *   Copyright (C) 2004 by Norbert Andres                                  *
 *   nandres@web.de                                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "card.h"
#include "kpicframer.h"
#include "kpicframerview.h"
#include "options.h"

#include <klocale.h>
#include <krandomsequence.h>

#include <stdio.h>
#include <math.h>

#include <qframe.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qmessagebox.h>
#include <qpainter.h>
#include <qtimer.h>
#include <qwhatsthis.h>

KPicFramerView::KPicFramerView(QWidget * parent)
 : QWidget(parent),
   m_phase(gameOver),
   m_backupPhase(gameOver),
   m_stack(0),
   m_activeCard(0),
   m_aCard(0),
   m_scoreTimer(new QTimer(this)),
   m_wonAniTimer(new QTimer(this)),
   m_cardAniTimer(new QTimer(this))
{
  initCards();
  initPlayField();
  reinit();

  connect( m_scoreTimer, SIGNAL(timeout()), this, SLOT(scoreTimerEvent()) );
  connect( m_wonAniTimer, SIGNAL(timeout()), this, SLOT(wonAniTimerEvent()) );
  connect( m_cardAniTimer, SIGNAL(timeout()), this, SLOT(cardAniTimerEvent()) );
}

KPicFramerView::~KPicFramerView()
{
  cleanup();
}

void KPicFramerView::undo()
{
  for (int i = 0; i < 16; ++i)
  {
    m_cards[i]->setCardNumber(m_backupField[i]);
  }

  m_cardIndex = m_backupIndex;
  m_activeCard->setCardNumber(m_backupACard);

  if (m_phase != pickUp)
    m_lCounter->setNum(m_cardIndex + 1);

  if (m_phase == layDown)
  {
    if ((m_activeCard->getCardNumber() >= spadesJack)
        && (m_activeCard->getCardNumber() <= clubKing))
      changeScore(-10);
    else
      changeScore(-5);
  }

  if (m_phase == gameOver)
  {
    setPhase(layDown);
    emit signalIncreaseLostGames(0, -1);
  }

  setPhase(m_backupPhase);
  emit signalEnableUndo(false);
  m_lastField = -1;
}

void KPicFramerView::cleanup()
{
  for (int i = 0; i < 16; ++i)
  {
    delete m_cards[i];
  }

  delete m_stack;
  delete m_activeCard;
}

void KPicFramerView::reinit()
{
  Options * options = ((KPicFramer *) parent())->options();
  if (options->setBackground())
  {
    setBackgroundColor(options->getBgColor());
    m_frameLeft->setBackgroundColor(options->getBgColor());
    m_frameRight->setBackgroundColor(options->getBgColor());
  }
  else
  {
    m_frameLeft->setBackgroundMode(Qt::PaletteBackground);
    m_frameRight->setBackgroundMode(Qt::PaletteBackground);
    setBackgroundMode(Qt::PaletteBackground);
  }

  m_lCounter->setBackgroundColor(options->getCounterBgColor());
  m_lCounter->setPaletteForegroundColor(options->getCounterFgColor());

  for (int i = 0; i < 16; ++i)
  {
    m_cards[i]->setMColor(options->getHlColor());
    m_cards[i]->setMBColor(options->getBlinkColor());
  }
}

void KPicFramerView::initCards()
{
  for (int i = 0; i < 16; ++i)
  {
    m_cards[i] = 0;
  }
}

void KPicFramerView::initPlayField()
{
  Options * options = ((KPicFramer *) parent())->options();

  QGridLayout * mainLayout = new QGridLayout(this, 1, 1, 11, 6, "mainLayout");

  m_frameLeft = new QFrame(this, "m_frameLeft");
  m_frameLeft->setGeometry(0, 0, 110, ((QWidget *)parent())->height());
  QGridLayout * frameLeftLayout = new QGridLayout( m_frameLeft, 1, 1, 11, 6, "layout");
  m_frameLeft->setSizePolicy(QSizePolicy( (QSizePolicy::SizeType)0,
                             (QSizePolicy::SizeType)5, 0, 0,
                              m_frameLeft->sizePolicy().hasHeightForWidth()));

  m_stack = new Card(m_frameLeft, 0);
  m_stack->enableHighlightning(false);
  m_stack->setCardNumber(-1);
  QWhatsThis::add(m_stack, i18n("Pile of cards left to be played"));

  frameLeftLayout->addMultiCellWidget(m_stack, 1, 1, 0, 1);


  m_lCounter = new QLabel("52", m_frameLeft, 0, 0);
  m_lCounter->setFrameStyle( QFrame::WinPanel | QFrame::Raised );
  m_lCounter->setLineWidth(1);
  m_lCounter->setMidLineWidth(3);
  m_lCounter->setAutoResize(false);
  m_lCounter->setAlignment(AlignCenter);
  m_lCounter->setFixedWidth(75);
  m_lCounter->setBackgroundColor(options->getCounterBgColor());
  QWhatsThis::add(m_lCounter, i18n("Number of cards left (including the active card below)"));

  frameLeftLayout->addMultiCellWidget(m_lCounter, 2, 2, 0, 1);


  m_activeCard = new Card(m_frameLeft, 0);
  m_activeCard->enableMarking(false);
  m_activeCard->setCardNumber(-1);
  QWhatsThis::add(m_activeCard, i18n("Card to be played next"));

  frameLeftLayout->addMultiCellWidget(m_activeCard, 4, 4, 0, 1);


  QSpacerItem * spacer1 = new QSpacerItem(20, 21, QSizePolicy::Minimum, QSizePolicy::Expanding);
  frameLeftLayout->addItem(spacer1, 0, 0);
  QSpacerItem * spacer2 = new QSpacerItem(20, 21, QSizePolicy::Minimum, QSizePolicy::Expanding);
  frameLeftLayout->addItem(spacer2, 3, 0);
  QSpacerItem * spacer3 = new QSpacerItem(20, 21, QSizePolicy::Minimum, QSizePolicy::Expanding);
  frameLeftLayout->addItem(spacer3, 5, 0);

  m_frameRight = new QFrame(this, "frameRight");
  QGridLayout * frameRightLayout = new QGridLayout(m_frameRight, 1, 1, 11, 6, "frl");

  int i = 0;

  for (int y = 0; y < 4; ++y)
  {
    for (int x = 0; x < 4; ++x)
    {
      m_cards[i] = new Card(m_frameRight, 0);
      m_cards[i]->setMColor(options->getHlColor());
      m_cards[i]->setMBColor(options->getBlinkColor());
      connect(m_cards[i], SIGNAL(clicked(Card *)), this, SLOT(cardClicked(Card* )));
      frameRightLayout->addWidget(m_cards[i], x, y);
      if ((i == 0) || (i == 3) || (i == 12) || (i == 15))
        QWhatsThis::add(m_cards[i], i18n("Place for putting cards. Kings can only be placed here."));
      else
      if ((i == 1) || (i == 2) || (i == 13) || (i == 14))
        QWhatsThis::add(m_cards[i], i18n("Place for putting cards. Queens can only be placed here."));
      else
      if ((i == 4) || (i == 8) || (i == 7) || (i == 11))
        QWhatsThis::add(m_cards[i], i18n("Place for putting cards. Jacks can only be placed here."));
      else
        QWhatsThis::add(m_cards[i], i18n("Place for putting cards."));

      ++i;
    }
  }

  mainLayout->addWidget(m_frameLeft, 0, 0);
  mainLayout->addWidget(m_frameRight, 0, 1);

  connect(m_stack, SIGNAL(clicked(Card *)),
          this, SLOT(nextCard(Card *)));
  connect(m_activeCard, SIGNAL(clicked(Card *)),
          this, SLOT(nextCard(Card *)));
  connect(m_activeCard, SIGNAL(released(Card *)),
          this, SLOT(endDrag(Card *)));
  connect(m_activeCard, SIGNAL(pressed(Card *)),
          this, SLOT(startDrag(Card *)));
  clearWState( WState_Polished );
}

void KPicFramerView::newGame()
{
  // stop timer of animation
  if (m_wonAniTimer->isActive())
    m_wonAniTimer->stop();

  for (int j = 0; j < 16; ++j)
  {
    m_backupField[j] = empty;
  }
  m_backupIndex = 51;
  m_backupACard = empty;

        // if the game is still running it should be stopped
  if (m_phase != gameOver)
  {
    if (!stopCurrentGame())
      return;
  }

  // the cards:
  m_lastField = -1;
  m_cStack[0] = -1;
  for (int n = 0; n < 52; ++n)
    m_cStack[n] = n + 1;

  // mix up the stack of the cards:
  KRandomSequence R(0);

  int t;
  long z;
  for (int m = 1; m < 3; ++m)
  {
    for (int j = 51; j >= 0; --j)
    {
      z = R.getLong(52);
      t = m_cStack[z];
      m_cStack[z] = m_cStack[j];
      m_cStack[j] = t;
      // lCounter->setNum((int)(m * j / 3));
    }
  }

  restart();
}

bool KPicFramerView::stopCurrentGame()
{
  if (emptyFieldsCount() == 16)
    return true;

  int result = QMessageBox::question(this, "KPicFramer",
                                     i18n("Do you want to stop the current game?"), i18n("Yes"),
                                     i18n("No"), QString::null, 0, 1);
  if (result == 0)
  {
    emit signalIncreaseStoppedGames(m_score);

    if (m_scoreTimer->isActive())
      m_scoreTimer->stop();

    return true;
  }

  return false;
}

void KPicFramerView::setPhase(playPhase p)
{
  m_phase = p;
  QColor c;
  if (m_phase == layDown)
  {
    c = QColor("lightgrey");
  }
  else
  if (m_phase == pickUp)
  {
    c = QColor("white");
  }

  for (int i = 0; i < 16; ++i)
    m_cards[i]->setMBColor(c);
}

void KPicFramerView::cardClicked(Card * card)
{
  // there is no use of clicking on a card if the game is over:-)
  if (m_phase == gameOver)
    return;

  if ((m_phase == pickUp) && (card->getCardNumber() == empty))
    return;

  for (int i = 0; i < 16; ++i)
  {
     m_backupField[i] = m_cards[i]->getCardNumber();
  }

  m_backupIndex = m_cardIndex;
  m_backupACard = m_activeCard->getCardNumber();
  m_backupPhase = m_phase;

  if ((m_phase != pickUp) || (m_mindex1 != -1))
    emit signalEnableUndo(true);
  else
    emit signalEnableUndo(false);

  // stop the animation timer if running
  if (m_wonAniTimer->isActive())
    m_wonAniTimer->stop();

  // if the score timer isn't active start it:
  if (!m_scoreTimer->isActive())
  {
    emit signalPauseEnd();
    m_scoreTimer->start(4000);
  }

  // if the picking up is stopped before there is nothing more to pick up
  // the next card of the stack should be turned over
  if ((card->getCardNumber() == -1) && (m_phase == pickUp))
  {
    nextCard(card);
    return;
  }

  // if a card is blinking it has to be stopped
  for (int i = 0; i < 16; ++i)
    m_cards[i]->stopBlink();

  // if an empty field is clicked the state of the game is "lay_down"
  int value = card->getCardNumber();
  if (value == -1)
    setPhase(layDown);

  // if there are no more empty fields cards have to be picked up
  if (emptyFieldsCount() == 0)
    setPhase(pickUp);

  switch(m_phase)
  {
    case layDown:
      layDownCard(card);
      break;
    case pickUp:
      pickUpCard(card);
      break;
    default:
      break;
  }
}

bool KPicFramerView::layDownCard(Card * card)
{
  // is the field empty?
  if (card->getCardNumber() != -1)
    return false;

  int actCardNumber = m_activeCard->getCardNumber();

  // may the card be placed here?
  if (!cardTest(card, actCardNumber))
    return false;

  m_aCard = card;

  // lay down the card perhaps with an animation
  Options * options = ((KPicFramer *) parent())->options();
  if (options->showAnimations())
  {
    finishLayDown();
    // TODO: cardAnimation(card->x(), card->y(), options->getAnimationSpeed());
  }
  else
  {
    finishLayDown();
  }

  return true;
}

bool KPicFramerView::finishLayDown()
{
  int actCardNumber = m_activeCard->getCardNumber();
  m_aCard->setCardNumber(actCardNumber);

  //m_activeCard->setPos(20,320);
  m_activeCard->setCardNumber(empty);
  changeScore(actCardNumber);
  // is the picture frame now ready?
  if (picFrameReady())
  {
    won();
    return true;
  }
  m_lastField = getFieldIndex(m_aCard);

  newCard();
  return true;
}

bool KPicFramerView::pickUpCard(Card * card)
{
  int value = card->getCardNumber();
  int index = getFieldIndex(card);
  m_lastField = -1;

  if (!card->isMarked())
  {
    if (m_mcount > 1)
      return false;

    // if card is a special one it must not be marked
    if ( (value >= spadesJack) && (value <= clubKing) )
      return false;

    // mark the card
    card->markField(true);
    ++m_mcount;
    // save the index of the new marked card
    switch(m_mcount)
    {
      case 1: m_mindex1 = index;
        break;
      case 2: m_mindex2 = index;
        break;
    }
  }
  else
  {
    // the card should not be marked anymore
    card->markField(false);
    --m_mcount;
    // clear the saved index
    switch(m_mcount)
    {
      case 0: m_mindex1 = -1;
        break;
      case 1: m_mindex2 = -1;
        break;
    }
  }
  if (m_mcount > 0)
  {
    switch(m_mcount)
    {
      // if there is only one card marked the value has to be ten
      case 1:
      if ((m_cards[m_mindex1]->getCardNumber() >= spades10)
          && (m_cards[m_mindex1]->getCardNumber() <= club10))
      {
        m_cards[m_mindex1]->setCardNumber(-1);
        m_cards[m_mindex1]->markField(false);
        --m_mcount;
        m_mindex1 = -1;
      }
      break;
          // if there is more than one marked card: add the values
    case 2:
      if ((getSum() == 10) && (m_mindex1 != m_mindex2))
      {
        m_cards[m_mindex1]->setCardNumber(-1);
        m_cards[m_mindex1]->markField(false);
        --m_mcount;
        m_cards[m_mindex2]->setCardNumber(-1);
        m_cards[m_mindex2]->markField(false);
        --m_mcount;
        m_mindex1 = -1;
        m_mindex2 = -1;
      }
      else
      {  // shouldn't be marked anymore
        m_cards[m_mindex1]->markField(false);
        --m_mcount;
        m_cards[m_mindex2]->markField(false);
        --m_mcount;
        m_mindex1 = -1; m_mindex2 = -1;
      }
    }
  }

  // if there are no more cards that can be picked up the next card of the
  // stack should be turned over
  if (!pickupPossible())
  {
    nextCard(card);
    if (m_phase != gameOver)
      setPhase(layDown);
  }

  return true;
}

void KPicFramerView::nextCard(Card *)
{
  if (m_phase != pickUp)
    return;

  setPhase(layDown);

  if (m_cardIndex >= 0)
  {
    m_activeCard->setCardNumber(m_cStack[m_cardIndex]);
    m_lCounter->setNum(m_cardIndex);
    --m_cardIndex;
  }
  if (m_cardIndex == -1)
  {
    m_stack->setEmpty(true);
    m_stack->update();
  }
  if (gameLost())
  {
    lost();
    return;
  }
}

bool KPicFramerView::cardTest(Card * card, int actCardNumber)
{
  // gets the index of the card
  int index = getFieldIndex(card);

  if ( (actCardNumber >= spadesJack) && (actCardNumber <= clubJack) )
  {
    if ( (index != 1) && (index != 2) && (index != 13) && (index != 14) )
    // wrong field
    {
      // if the field is empty it should blink
      if (m_cards[1]->getCardNumber() == empty)
        m_cards[1]->startBlink(3);
      if (m_cards[2]->getCardNumber() == empty)
        m_cards[2]->startBlink(3);
      if (m_cards[13]->getCardNumber() == empty)
        m_cards[13]->startBlink(3);
      if (m_cards[14]->getCardNumber() == empty)
        m_cards[14]->startBlink(3);

      emit signalChangeStatusbar(i18n("Illegal move - Place the card on one of the blinking fields!"));
      return false;
    }
  }

  if ( (actCardNumber >= spadesQueen) && (actCardNumber <= clubQueen) ) // restricted cards (2)
  {
    if ( (index != 4) && (index != 7) && (index != 8) && (index != 11) )
    {
      if (m_cards[4]->getCardNumber() == empty)
        m_cards[4]->startBlink(3);
      if (m_cards[7]->getCardNumber() == empty)
        m_cards[7]->startBlink(3);
      if (m_cards[8]->getCardNumber() == empty)
        m_cards[8]->startBlink(3);
      if (m_cards[11]->getCardNumber() == empty)
        m_cards[11]->startBlink(3);

      emit signalChangeStatusbar(i18n("Illegal move - Place the card on one of the blinking fields!"));
      return false;
    }
  }

  if ( (actCardNumber >= spadesKing) && (actCardNumber <= clubKing) ) // restricted cards (3)
  {
    if ( (index != 0) && (index != 3) && (index != 12) && (index != 15) )
    {
      if (m_cards[0]->getCardNumber() == empty)
        m_cards[0]->startBlink(3);
      if (m_cards[3]->getCardNumber() == empty)
        m_cards[3]->startBlink(3);
      if (m_cards[12]->getCardNumber() == empty)
        m_cards[12]->startBlink(3);
      if (m_cards[15]->getCardNumber() == empty)
        m_cards[15]->startBlink(3);

      emit signalChangeStatusbar(i18n("Illegal move - Place the card on one of the blinking fields!"));
      return false;
    }
  }

  return true;
}

void KPicFramerView::changeScore(int number)
{
  int diff;
  // for every card the score is increased by 5
  // for every special card the score is increased by 10
  if (number > 0)
  {
    diff = 5;
    if ((number >= spadesJack) && (number <= clubKing))
      diff = 10;
  }
  else
  {       // negative numbers: score = score - number
    diff = number;
  }

  m_score += diff;
  emit signalChangeScore(m_score);
}

void KPicFramerView::cardAnimation(int, int, int)
{
/*  m_startx  = m_activeCard->x();
  m_starty  = m_activeCard->y();
  m_targetx = x;
  m_targety = y;
  m_actx    = m_startx;
  m_acty    = m_starty;
  m_ccount  = 0;

  if (m_cardAniTimer->isActive())
    m_cardAniTimer->stop();

  cardAniTimerEvent();
  m_cardAniTimer->start(speed);*/
}

void KPicFramerView::startDrag(Card *)
{
}

void KPicFramerView::endDrag(Card *)
{
}

void KPicFramerView::scoreTimerEvent()
{
  emit signalChangeScore(--m_score);
}

void KPicFramerView::wonAniTimerEvent()
{
  int val = m_cards[0]->getCardNumber();

  m_cards[0]->setCardNumber(m_cards[4]->getCardNumber());
  m_cards[4]->setCardNumber(m_cards[8]->getCardNumber());
  m_cards[8]->setCardNumber(m_cards[12]->getCardNumber());
  m_cards[12]->setCardNumber(m_cards[13]->getCardNumber());
  m_cards[13]->setCardNumber(m_cards[14]->getCardNumber());
  m_cards[14]->setCardNumber(m_cards[15]->getCardNumber());
  m_cards[15]->setCardNumber(m_cards[11]->getCardNumber());
  m_cards[11]->setCardNumber(m_cards[7]->getCardNumber());
  m_cards[7]->setCardNumber(m_cards[3]->getCardNumber());
  m_cards[3]->setCardNumber(m_cards[2]->getCardNumber());
  m_cards[2]->setCardNumber(m_cards[1]->getCardNumber());
  m_cards[1]->setCardNumber(val);

  ++m_aniCount;
  if (m_aniCount > 95)
  {
    if (m_wonAniTimer->isActive())
      m_wonAniTimer->stop();
  }
}

void KPicFramerView::cardAniTimerEvent()
{
/*  ++m_ccount;
  m_actx = (m_targetx - m_startx) / 9 * m_ccount + m_startx;
  m_activeCard->highlightField(true);

  if (m_actx >= m_targetx)
  {
    m_actx = m_targetx;
    m_cardAniTimer->stop();
    m_acty = m_targety;
    //m_activeCard->setPos(m_actx, m_acty);
    m_activeCard->highlightField(false);

    finishLayDown();
    return;
  }

  m_acty = (int)(m_starty + (m_targety - m_starty) * sin(M_PI / 2
            * (m_actx - m_startx) / (m_targetx - m_startx)));
  m_activeCard->setPos(m_actx, m_acty);*/
}

void KPicFramerView::newCard()
{
  emit signalChangeStatusbar(i18n("Good luck!"));

  if (picFrameReady())
  {
    won();
    return;
  }

  int ec = emptyFieldsCount();
  if (ec == 0)
  {
    if (m_phase != gameOver)
      setPhase(pickUp);

    m_activeCard->setCardNumber(empty);

    if (!pickupPossible())
    {
      lost();
      return;
    }

    return;
  }

  if (m_cardIndex >= 0)
  {
    m_activeCard->setCardNumber(m_cStack[m_cardIndex]);
    m_lCounter->setNum(m_cardIndex);
    --m_cardIndex;
  }
  if (m_cardIndex == -1)
  {
    m_stack->setEmpty(true);
    m_stack->update();
  }
  if (gameLost())
  {
    lost();
    return;
  }
}

void KPicFramerView::won()
{
  m_score += m_cardIndex * 5;
  emit signalChangeStatusbar(i18n("You've won! Your score: %1").arg(m_score));
  emit signalEnableUndo(false);
  Options * options = ((KPicFramer *) parent())->options();
  if (options->showAnimations())
    wonAnimation(options->getAnimationSpeed());

  if (m_scoreTimer->isActive())
    m_scoreTimer->stop();

  setPhase(gameOver);

  m_activeCard->setCardNumber(empty);
  m_stack->setEmpty(true);
  m_stack->update();
  m_lCounter->setText("0");
  m_cards[5]->setCardNumber(empty);
  m_cards[6]->setCardNumber(empty);
  m_cards[9]->setCardNumber(empty);
  m_cards[10]->setCardNumber(empty);

  emit signalIncreaseWonGames(m_score);
  emit signalChangeScore(m_score);
  emit signalGameOver(i18n("You've won!"));
}

void KPicFramerView::lost()
{
  setPhase(gameOver);

  if (m_scoreTimer->isActive())
    m_scoreTimer->stop();

  emit signalChangeStatusbar(i18n("You've lost. Your score: %1").arg(m_score));
  emit signalIncreaseLostGames(m_score, 1);
  emit signalGameOver(i18n("You've lost"));
}

bool KPicFramerView::pause()
{
  if (m_scoreTimer->isActive())
  {
    m_scoreTimer->stop();
    return true;
  }

  m_scoreTimer->start(4000);
  return false;
}

bool KPicFramerView::close()
{
  if (m_wonAniTimer->isActive())
    m_wonAniTimer->stop();

  if ((m_phase != gameOver)
    && (m_cardIndex != 51))
  {
    if (!stopCurrentGame())
      return false;
  }

  return true;
}

void KPicFramerView::restart()
{
  // empty all fields
  m_mcount = 0;
  m_mindex1 = m_mindex2 = -1;
  m_cardIndex = 51;
  m_score = 0;
  setPhase(layDown);
  for (int i = 0; i < 16; ++i)
    m_cards[i]->setCardNumber(empty);

  emit signalChangeStatusbar(i18n("Good luck!"));
  emit signalChangeScore(0);

  // turn over first card:
  newCard();
  m_stack->setCardNumber(back);

  // the timer starts when the first card is layed down
  if (m_scoreTimer->isActive())
    m_scoreTimer->stop();
  emit signalEnableUndo(false);
}

bool KPicFramerView::gameLost()
{
  // if there are no more fields it is not lost because you can pick up cards
  if (emptyFieldsCount() == 0)
    return false;

  int actCardNumber = m_activeCard->getCardNumber();
  if ((actCardNumber >= spadesJack) && (actCardNumber <= clubJack) ) // restricted cards (1)
  {
    // if the active card is a jack and no "special" field is empty the game is lost
    if ((m_cards[1]->getCardNumber() != -1) && (m_cards[2]->getCardNumber() != -1)
        && (m_cards[13]->getCardNumber() != -1) && (m_cards[14]->getCardNumber() != -1))
      return true;
  }

  if ( (actCardNumber >= spadesQueen) && (actCardNumber <= clubQueen) ) // restricted cards (2)
  {
    if ( (m_cards[4]->getCardNumber() != -1) && (m_cards[7]->getCardNumber() != -1)
        && (m_cards[8]->getCardNumber() != -1) && (m_cards[11]->getCardNumber() != -1) )
      return true;
  }

  if ( (actCardNumber >= spadesKing) && (actCardNumber <= clubKing) ) // restricted cards (3)
  {
    if ( (m_cards[0]->getCardNumber() != -1) && (m_cards[3]->getCardNumber() != -1) &&
           (m_cards[12]->getCardNumber() != -1) && (m_cards[15]->getCardNumber() != -1) )
      return true;
  }

  return false;
}

void KPicFramerView::wonAnimation(int speed)
{
  m_aniCount = 0;
  wonAniTimerEvent();
  m_wonAniTimer->start(20 * speed);
}

int KPicFramerView::getSum()
{
  // adds the values of two marked cards
  int val1 = m_cards[m_mindex1]->getValue();
  int val2 = m_cards[m_mindex2]->getValue();
  return val1 + val2;
}

bool KPicFramerView::picFrameReady()
{
  bool result = true;
  int n = 1;
  // if one of the fields is not filled with one of the e.g. jacks the picture frame is not completed yet
  if ((m_cards[n]->getCardNumber() < spadesJack) || (m_cards[n]->getCardNumber() > clubJack))
    result = false;
  n = 2;
  if ((m_cards[n]->getCardNumber() < spadesJack) || (m_cards[n]->getCardNumber() > clubJack))
    result = false;
  n = 13;
  if ((m_cards[n]->getCardNumber() < spadesJack) || (m_cards[n]->getCardNumber() > clubJack))
    result = false;
  n = 14;
  if ((m_cards[n]->getCardNumber() < spadesJack) || (m_cards[n]->getCardNumber() > clubJack))
    result = false;

  n = 0;
  if ((m_cards[n]->getCardNumber() < spadesKing) || (m_cards[n]->getCardNumber() > clubKing))
    result = false;
  n = 3;
  if ((m_cards[n]->getCardNumber() < spadesKing) || (m_cards[n]->getCardNumber() > clubKing))
    result = false;
  n = 12;
  if ((m_cards[n]->getCardNumber() < spadesKing) || (m_cards[n]->getCardNumber() > clubKing))
    result = false;
  n = 15;
  if ((m_cards[n]->getCardNumber() < spadesKing) || (m_cards[n]->getCardNumber() > clubKing))
    result = false;

  n = 4;
  if ((m_cards[n]->getCardNumber() < spadesQueen) || (m_cards[n]->getCardNumber() > clubQueen))
    result = false;
  n = 7;
  if ((m_cards[n]->getCardNumber() < spadesQueen) || (m_cards[n]->getCardNumber() > clubQueen))
    result = false;
  n = 8;
  if ((m_cards[n]->getCardNumber() < spadesQueen) || (m_cards[n]->getCardNumber() > clubQueen))
    result = false;
  n = 11;
  if ((m_cards[n]->getCardNumber() < spadesQueen) || (m_cards[n]->getCardNumber() > clubQueen))
    result = false;

  return result;
}

bool KPicFramerView::pickupPossible()
{
  // is it possible to pick up cards (sum: 10)
  // First: is there a card with the value 10?
  for (int i = 0; i < 16; ++i)
  {
    if ((m_cards[i]->getCardNumber() >= spades10) && (m_cards[i]->getCardNumber() <= club10))
      return true;
  }

  // Second: are there cards that sum up to 10
   for (int j = 0; j < 16; ++j)
  {
    if ((m_cards[j]->getCardNumber() > club9) && (m_cards[j]->getCardNumber() < spadesAce))
      continue;

    int v1 = m_cards[j]->getValue();
    for (int n = j + 1; n < 16; ++n)
    {
       if ((m_cards[n]->getCardNumber() > club9) && (m_cards[n]->getCardNumber() < spadesAce))
        continue;
      int v2 = m_cards[n]->getValue();
      if (v1 + v2 == 10)
        return true;
    }
  }

  return false;
}

int KPicFramerView::emptyFieldsCount()
{
  int count = 0;

  for (int i = 0; i < 16; ++i)
  {
    if (m_cards[i]->getCardNumber() == -1)
      ++count;
  }

  return count;
}

int KPicFramerView::getFieldIndex(Card * card)
{
  for (int i = 0; i < 16; ++i)
  {
    if (m_cards[i] == card)
      return i;
  }

  return -1;
}

#include "kpicframerview.moc"
