/***************************************************************************
                          filterhandler.cpp  -  description
                             -------------------
    begin                : Mon Oct 15 2001
    copyright            : (C) 2001 by Chris Wallace
    email                : ranchhard@netflash.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "filterhandler.h"

#include <qdatetime.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qdir.h>
#include <qdom.h>

#include <klistview.h>
#include <klocale.h>
#include <kmdcodec.h>

#include <iostream>

FilterHandler::FilterHandler(FilterRule::Action defaultAction)
{
	mFilters.setAutoDelete(true);
	clearRules();
	setDefaultAction(defaultAction);
}

/*
FilterHandler::FilterHandler(const FilterHandler & f)
{
	operator = (f);
}
*/

FilterHandler::~FilterHandler()
{
	clearRules();
}

void FilterHandler::operator = (const FilterHandler & f)
{
	mDefaultAction = f.mDefaultAction;
	mFilters = f.mFilters;
}

void FilterHandler::clearRules(void)
{
	mFilters.clear();
}

void FilterHandler::addFilterRule(FilterRule::Location location, FilterRule::Test test, const QString & re, FilterRule::Action action)
{
	FilterRule *fr = new FilterRule(location, test, re, action);
	addFilterRule(fr);
}

void FilterHandler::addFilterRule(FilterRule *fr)
{
	if(!fr)
		return;
		
	KMD5 md5(QString::number(fr->location()) + QString::number(fr->test()) + fr->regularExpression() + QString::number(fr->action()) + QDateTime::currentDateTime().toString());
	fr->setId(QString(md5.hexDigest()));
	mFilters.append(fr);
}

FilterRule::Action FilterHandler::filter(const QString & title, const QString & description)
{
	FilterRule::Action action = defaultAction();
	
	FilterRule *fr = mFilters.first();
	while(fr) {
		if(fr->doesTrigger(title, description)) {
			action = fr->action();
			break;
		}
		fr = mFilters.next();
	}
	
	return action;
}

void FilterHandler::list(KListView * mFilterList)
{
	mFilterList->clear();
	FilterRule *fr = mFilters.first();
	KListViewItem *item = 0;
	while(fr) {
		item = new KListViewItem(mFilterList, locationToString(fr->location()), testToString(fr->test()), fr->regularExpression(), actionToString(fr->action()), fr->id());
		//mFilterList->insertItem(item);
		fr = mFilters.next();
	}
}

FilterRule * FilterHandler::find(const QString & id)
{
	FilterRule *ret = 0;
	FilterRule *fr = mFilters.first();
	while(fr) {
		if(fr->id().compare(id) == 0) {
			ret = fr;
			break;
		}
		fr = mFilters.next();
	}
	return ret;
}

void FilterHandler::remove(const QString & id)
{
	FilterRule *fr = find(id);
	if(!fr)
		return;
		
	mFilters.remove(fr);
}

const QString & FilterHandler::locationToString(FilterRule::Location location)
{
	switch(location) {
		case FilterRule::NewsTitle: mTempStr = i18n("News Title"); break;
		case FilterRule::NewsDescription: mTempStr = i18n("News Description"); break;
		case FilterRule::Both: mTempStr = i18n("Both"); break;
		case FilterRule::Either: mTempStr = i18n("Either"); break;
	}
	return mTempStr;
}

const QString & FilterHandler::testToString(FilterRule::Test test)
{
	switch(test) {
		case FilterRule::Contains: mTempStr = i18n("Matches"); break;
		case FilterRule::DoesNotContain: mTempStr = i18n("Does Not Match"); break;
	}
	return mTempStr;
}

const QString & FilterHandler::actionToString(FilterRule::Action action)
{
	switch(action) {
		case FilterRule::Allow: mTempStr = i18n("Allow"); break;
		case FilterRule::Deny: mTempStr = i18n("Deny"); break;
	}
	return mTempStr;
}

bool FilterHandler::startDocument(void)
{
	mReadingFilters = mReadingDefaultAction = mReadingRule = mReadingLocation = false;
	mReadingTest = mReadingRegExp = mReadingAction = false;
	mFilters.clear();
	cerr << "Starting Document" << endl;
	return true;
}

bool FilterHandler::endDocument(void)
{
	cerr << "Ending Document" << endl;
	return true;
}

bool FilterHandler::startElement(const QString &, const QString &, const QString & qName, const QXmlAttributes &)
{
	if(qName.compare(QString("filters")) == 0)
		mReadingFilters = true;
	else if(qName.compare(QString("defaultaction")) == 0)
		mReadingDefaultAction = true;
	else if(qName.compare(QString("rule")) == 0) {
		mReadingRule = true;
		mTempRule = new FilterRule();
	}
	else if(qName.compare(QString("location")) == 0)
		mReadingLocation = true;
	else if(qName.compare(QString("test")) == 0)
		mReadingTest = true;
	else if(qName.compare(QString("regexp")) == 0)
		mReadingRegExp = true;
	else if(qName.compare(QString("action")) == 0)
		mReadingAction = true;
	else
		cerr << "Unknown Element: " << qName << endl;
		
	return true;
}

bool FilterHandler::endElement(const QString &, const QString &, const QString & qName)
{
	if(qName.compare(QString("filters")) == 0)
		mReadingFilters = false;
	else if(qName.compare(QString("defaultaction")) == 0)
		mReadingDefaultAction = false;
	else if(qName.compare(QString("rule")) == 0) {
		mReadingRule = false;
		addFilterRule(mTempRule);
	}
	else if(qName.compare(QString("location")) == 0)
		mReadingLocation = false;
	else if(qName.compare(QString("test")) == 0)
		mReadingTest = false;
	else if(qName.compare(QString("regexp")) == 0)
		mReadingRegExp = false;
	else if(qName.compare(QString("action")) == 0)
		mReadingAction = false;
	else
		cerr << "Unknown Element: " << qName << endl;
		
	return true;
}

bool FilterHandler::warning(const QXmlParseException & e)
{
	cerr << "WARNING: at " << e.lineNumber() << "," << e.columnNumber() << endl;
	cerr << "        " << e.message();
	return true;
}

bool FilterHandler::error(const QXmlParseException & e)
{
	cerr << "ERROR: at " << e.lineNumber() << "," << e.columnNumber() << endl;
	cerr << "     " << e.message();
	return false;
}

bool FilterHandler::fatalError(const QXmlParseException & e)
{
	cerr << "FATAL ERROR: at " << e.lineNumber() << "," << e.columnNumber() << endl;
	cerr << "           " << e.message();
	return false;
}

bool FilterHandler::characters(const QString & ch)
{
	if(mReadingFilters) {
		if(mReadingDefaultAction) {
			mDefaultAction = (FilterRule::Action)ch.toInt();
		}
		else if(mReadingRule) {
			if(mReadingLocation) {
				mTempRule->setLocation((FilterRule::Location)ch.toInt());
			}
			else if(mReadingTest) {
				mTempRule->setTest((FilterRule::Test)ch.toInt());
			}
			else if(mReadingRegExp) {
				mTempRule->setRegularExpression(ch);
			}
			else if(mReadingAction) {
				mTempRule->setAction((FilterRule::Action)ch.toInt());
			}
		}
	}
	return true;
}

void FilterHandler::writeFilters(void)
{
	QDomDocument doc("KrssFilters");
		QDomElement felm = doc.createElement("filters");
			QDomElement daelm = doc.createElement("defaultaction");
				QDomText datxt = doc.createTextNode(QString::number(mDefaultAction));
			daelm.appendChild(datxt);
		felm.appendChild(daelm);
		
		FilterRule * fr = mFilters.first();
		while(fr) {
			fr->write(doc, felm);
			fr = mFilters.next();
		}
		
	doc.appendChild(felm);
	
	QFile f(QDir::homeDirPath() + QString("/.krss/filters.xml"));
	if(!f.open(IO_WriteOnly))
		return;
	
	QTextStream t(&f);
	t << doc.toString();
	
	f.close();
}

void FilterHandler::readFilters(void)
{
	QFile xmlFile(QDir::homeDirPath() + QString("/.krss/filters.xml"));
	QXmlInputSource source(xmlFile);
	QXmlSimpleReader reader;
	reader.setContentHandler(this);
	reader.setErrorHandler(this);
	reader.parse(source);
}

void FilterHandler::moveUp(const QString & id)
{
	cerr << "Moving filter rule up" << endl;
	
	FilterRule * cfr = mFilters.first();
	
	while(cfr) {
		if(cfr->id().compare(id) == 0) {
			unsigned int pos = mFilters.at();
			cerr << "Current Position: " << pos << endl;
			if(pos < mFilters.count()) {
				FilterRule *nfr = mFilters.take(pos+1);
				if(nfr)
					mFilters.insert(pos, nfr);
			}
			cfr = mFilters.next();
		}
		cfr = mFilters.next();
	}
}

void FilterHandler::moveDown(const QString & id)
{
	cerr << "Moving filter rule down" << endl;
	
	FilterRule * cfr = mFilters.first();
	while(cfr) {
		if(cfr->id().compare(id) == 0) {
			unsigned int pos = mFilters.at();
			cerr << "Current Position: " << pos << endl;
			if(pos > 0) {
				FilterRule * pfv = mFilters.take(pos-1);
				if(pfv)
					mFilters.insert(pos, pfv);
			}
		}
		cfr = mFilters.next();
	}
}
