/***************************************************************************
                          mychannelbar.cpp  -  description
                             -------------------
    begin                : Thu Oct 11 2001
    copyright            : (C) 2001 by Chris Wallace
    email                : ranchhard@netflash.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "mychannelbar.h"

#include <qdir.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qxml.h>
#include <qtooltip.h>
#include <qfileinfo.h>

#include <kurldrag.h>
#include <kurl.h>
#include <kio/job.h>
#include <kconfig.h>
#include <kglobal.h>
#include <klocale.h>

#include <iostream>

#include "rssparser.h"
#include "rdftree.h"
#include "sitehandler.h"
#include "mychannelbaritem.h"

MyChannelBar::MyChannelBar(KConfig * config, SiteHandler *sitehandler, QWidget *parent, const char *name )
: KIconView(parent,name)
{
	mConfig = config;
	mSiteHandler = sitehandler;
	mHaveToolTip = false;
	
	connect(horizontalScrollBar(), SIGNAL(valueChanged(int)), this, SLOT(slotScrollBy(int)));
	connect(verticalScrollBar(), SIGNAL(valueChanged(int)), this, SLOT(slotScrollBy(int)));
	connect(this, SIGNAL(onItem(QIconViewItem *)), this, SLOT(slotOnItem(QIconViewItem *)));
	connect(this, SIGNAL(onViewport()), this, SLOT(slotOnViewport()));
	connect(this, SIGNAL(rightButtonPressed(QIconViewItem *, const QPoint & )), this, SLOT(slotRightButtonPressed(QIconViewItem *, const QPoint & )));
	connect(&mExTimer, SIGNAL(timeout()), this, SLOT(slotExpirationTimeOut()));
	
	expirationChanged();
	
}

MyChannelBar::~MyChannelBar()
{
	if(mExTimer.isActive())
		mExTimer.stop();
}

void MyChannelBar::drawBackground(QPainter *p, const QRect & )
{
	p->eraseRect(rect());
	p->fillRect(rect(), viewport()->palette().brush(QPalette::Normal, QColorGroup::Background));
}


void MyChannelBar::slotScrollBy(int)
{
	repaint(0, 0, -1, -1, true);
}

void MyChannelBar::contentsDragEnterEvent(QDragEnterEvent* event)
{
	event->accept(KURLDrag::canDecode(event));
}

void MyChannelBar::contentsDropEvent(QDropEvent* event)
{
	KURL::List urls;
	if(KURLDrag::decode(event, urls)) {
		KURL dest;
		dest.setPath(QDir::homeDirPath() + QString("/.krss/tmp/"));
		KIO::CopyJob * cpjob = KIO::copy(urls, dest, true);
		connect(cpjob, SIGNAL(result(KIO::Job *)), this, SLOT(slotResult(KIO::Job *)));
		connect(cpjob, SIGNAL(copyingDone(KIO::Job *, const KURL &, const KURL &, bool, bool)), this, SLOT(slotCopyingDone(KIO::Job *, const KURL &, const KURL &, bool, bool)));
	}
}

void MyChannelBar::slotResult(KIO::Job *)
{
	emit signalRefresh();
}

void MyChannelBar::slotCopyingDone(KIO::Job *, const KURL & from, const KURL & to, bool, bool)
{	
	QFile f(to.path());
	if(!f.exists()) return;
	
	RSSParser parser;
	QFile xmlFile(to.path());
	QXmlInputSource source(xmlFile);
	QXmlSimpleReader reader;
	reader.setContentHandler(&parser);
	reader.setErrorHandler(&parser);
	reader.parse(source);
	
	Site * s = 0;
	
	s = new Site();
	
	s->setName(parser.rdfTree().channelTitle());
	s->setURL(from);
	s->setDescription(parser.rdfTree().channelDescription());
	s->setUsed(true);
	s->makeFileName();
	
	if(mSiteHandler) {
		mSiteHandler->addSite(s);
		mSiteHandler->writeSites();	
	}
	
	QFile fw(s->fileNameParsed());
	
	if(!f.open(IO_ReadOnly)) return;
	if(!fw.open(IO_WriteOnly)) return;
	
	QTextStream tr(&f);
	QTextStream tw(&fw);
	
	tw << tr.read();
	
	f.close();
	fw.close();
	
	QDir().remove(to.path(), true);
}

void MyChannelBar::slotOnItem(QIconViewItem *item)
{
	mConfig->setGroup(QString("MyChannelBar"));
	if(!mConfig->readBoolEntry(QString("Display Channel Information"), true))
		return;
		
	if(!item)
		return;
		
	MyChannelBarItem *mcitem = (MyChannelBarItem*)item;
	
	if(!mcitem)
		return;
		
	if(mHaveToolTip)
		return;
				
	bool shortf = mConfig->readBoolEntry(QString("Use Short Format"), true);
	bool isecs = mConfig->readBoolEntry(QString("Include Seconds"), false);
		
	QString mToolTip = mcitem->text() + QString("\n\n");
	mToolTip.append(i18n("Downloaded On: ") + KGlobal::locale()->formatDateTime(mcitem->downloadDateTime(), shortf, isecs) + QChar('\n'));
	mToolTip.append(i18n("Displayed On: ") + KGlobal::locale()->formatDateTime(mcitem->displayDateTime(), shortf, isecs) + QChar('\n'));
 	QDateTime edt = mcitem->expireDateTime();
  if(!edt.isValid())
  	mToolTip.append(i18n("Never Expires"));
  else if(edt <= QDateTime::currentDateTime())
		mToolTip.append(i18n("Expired On: ") + KGlobal::locale()->formatDateTime(edt, shortf, isecs));
  else
  	mToolTip.append(i18n("Expires On: ") +KGlobal::locale()->formatDateTime(edt, shortf, isecs));

	QToolTip::add(viewport(), mToolTip);
	mHaveToolTip = true;
}

void MyChannelBar::slotOnViewport(void)
{	
	mConfig->setGroup(QString("MyChannelBar"));
	if(!mConfig->readBoolEntry(QString("Display Channel Information"), true))
		return;
		
	if(!mHaveToolTip)
		return;
		
	//cerr << "removing tooltip" << endl;
	QToolTip::remove(viewport());
	mHaveToolTip = false;
}

void MyChannelBar::expirationChanged(void)
{
	mConfig->setGroup(QString("MyChannelBar"));
	if(mConfig->readBoolEntry(QString("No Expiration"), false)) {
		mExpirationDays = mExpirationHours = mExpirationMinutes = -1;
		if(mExTimer.isActive())
			mExTimer.stop();
	}
	else {
		mExpirationDays = mConfig->readNumEntry(QString("Expiration Days"), 1);
		mExpirationHours = mConfig->readNumEntry(QString("Expiration Hours"), 0);
		mExpirationMinutes = mConfig->readNumEntry(QString("Expiration Minutes"), 0);
		if(mExTimer.isActive())
			mExTimer.stop();
		mExTimer.start(15*1000);
	}
}

void MyChannelBar::slotExpirationTimeOut(void)
{
	if(mExpirationDays < 0 && mExpirationHours < 0 && mExpirationMinutes < 0)
		return;
		
  setUpdatesEnabled(false);
	QIconViewItem *item = firstItem();
	while(item) {
		MyChannelBarItem * mcitem = (MyChannelBarItem*)item;
		if(mcitem) {
			if(mcitem->isNew()) {
				QFileInfo fi(mcitem->key() + QString(".md5"));
				QDateTime idt = fi.lastModified();
				idt = idt.addSecs(mExpirationDays * 24*60*60);
				idt = idt.addSecs(mExpirationHours * 60*60);
				idt = idt.addSecs(mExpirationMinutes * 60);
				//cout << idt.toString() << " >= " << QDateTime::currentDateTime().toString() << " = ";
				if(idt <= QDateTime::currentDateTime()) {
					//cout << "true" << endl;
					mcitem->setNew(false);
					mcitem->setColour(mOldColour);
				}
				else {
					//cout << "false" << endl;
					mcitem->setNew(true);
					mcitem->setColour(mNewColour);
				}
			}
			else {
				mcitem->setColour(mOldColour);
			}
		}
  	item->repaint();
		item = item->nextItem();
	}
	setUpdatesEnabled(true);
 	erase();
 	update();
	repaint();
}

void MyChannelBar::setItemColours(const QColor & newcolour, const QColor & oldcolour)
{
	mNewColour = newcolour;
	mOldColour = oldcolour;
}

void MyChannelBar::slotRightButtonPressed(QIconViewItem * item, const QPoint & p)
{
	if(!item)
		emit signalMyChannelBarPopupMenu(QString::null, p);
	else
		emit signalMyChannelBarPopupMenu(item->key(), p);
}



