/***************************************************************************
                          rssparser.cpp  -  description
                             -------------------
    begin                : Sun Aug 5 2001
    copyright            : (C) 2001 by Chris Wallace
    email                : ranchhard@netflash.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "rssparser.h"

#include <qdir.h>
#include <qfileinfo.h>
#include <qdatetime.h>
#include <qtextstream.h>

#include <kmessagebox.h>
#include <kglobal.h>
#include <klocale.h>
#include <kconfig.h>
#include <kmdcodec.h>

#include <iostream>

#include "schemetree.h"
#include "schemeparser.h"
#include "mychannelbaritem.h"
#include "filterhandler.h"

RSSParser::RSSParser()
: QXmlDefaultHandler()
{
	rdftag = QString("rdf:RDF");
	channeltag = QString("channel");
	titletag = QString("title");
	linktag = QString("link");
	desctag = QString("description");
	imagetag = QString("image");
	itemstag = QString("items");
	rdfseqtag = QString("rdf:Seq");
	rdflitag = QString("rdf:li");
	textinputtag = QString("textinput");
	urltag = QString("url");
	nametag = QString("name");
	itemtag = QString("item");
	rsstag = QString("rss");
	htmltag = QString("html");
	bodytag = QString("body");
	clear();
}

RSSParser::RSSParser(const RSSParser & p)
: QXmlDefaultHandler()
{
	operator = (p);
}

RSSParser::~RSSParser()
{
}

void RSSParser::operator = (const RSSParser & p)
{
	mRDFTree = p.mRDFTree;
}

void RSSParser::clear(void)
{
	mRDFTree.clear();
	mReadingFile = mReadingRDF = mReadingChannel = mReadingImage = mReadingItem = false;
	mReadingTextInput = mReadingItems = mReadingItemSeq = mReadingItemSeqList = false;
	mReadingDescription = mReadingTitle = mReadingLink = mReadingUrl = mReadingName = false;
	mHadRDFTag = mPrevVersion = false;
	mIsHTML = mReadingBody = false;
	mIsEmpty = true;
	mHaveComment = false;
	mComment = QString::null;
	
}

bool RSSParser::startDocument(void)
{
	clear();
	mReadingFile = true;
	return true;
}

bool RSSParser::endDocument(void)
{
	mReadingFile = false;
	if(mIsEmpty) {
		mRDFTree.setChannelTitle(i18n("Feed Error"));
		if(mHaveComment)
			mRDFTree.setChannelDescription(mComment);
		else
			mRDFTree.setChannelDescription(i18n("This feed appears to be empty."));
	}
	/*
	if(!mHadRDFTag) {
		mRDFTree.setChannelTitle(i18n("Feed Error"));
		mRDFTree.setChannelDescription(i18n("This file does not appear to be in a RDF Site Summary (RSS) version 1.0 format."));
	}
	*/
	return true;
}

bool RSSParser::startElement(const QString & , const QString & , const QString & qName, const QXmlAttributes & atts)
{
	if(qName.compare(rdftag) == 0) {
		mReadingRDF = true;
		mHadRDFTag = true;
		for(int i=0; i<atts.length(); i++) {
			if(atts.qName(i).compare(QString("xmlns:rdf")) == 0) {
				mRDFTree.setXmlNSRDF(atts.value(i));
			}
			else if(atts.qName(i).compare(QString("xmlns")) == 0) {
				mRDFTree.setXmlNS(atts.value(i));
			}
		}
	}
	else if(qName.compare(channeltag) == 0) {
		mIsEmpty = false;
		mReadingChannel = true;
		if(!mPrevVersion) {
			for(int i=0; i<atts.length(); i++) {
				if(atts.qName(i).compare(QString("rdf:about")) == 0) {
					mRDFTree.setChannelRDFAbout(atts.value(i));
				}
			}
		}
	}
	else if(qName.lower().compare(titletag) == 0) {
		mReadingTitle = true;
		mIsEmpty = false;
	}
	else if(qName.compare(linktag) == 0) {
		mReadingLink = true;
		mIsEmpty = false;
	}
	else if(qName.compare(desctag) == 0) {
		mReadingDescription = true;
		mIsEmpty = false;
	}
	else if(qName.compare(imagetag) == 0) {
		mReadingImage = true;
		mIsEmpty = false;
		if(!mPrevVersion) {
			if(mReadingChannel) {
				for(int i=0; i<atts.length(); i++) {
					if(atts.qName(i).compare(QString("rdf:resource")) == 0) {
						mRDFTree.setChannelImageUri(atts.value(i));
					}
				}
			}
			else {
				for(int i=0; i<atts.length(); i++) {
					if(atts.qName(i).compare(QString("rdf:about")) == 0) {
						mRDFTree.setImageUri(atts.value(i));
					}
				}
			}
		}
		else {
			mRDFTree.setImageUri(QString("image"));
			mRDFTree.setChannelImageUri(QString("image"));
		}
	}
	else if(qName.compare(textinputtag) == 0) {
		mReadingTextInput = true;
		mIsEmpty = false;
		if(!mPrevVersion) {
			if(mReadingChannel) {
				for(int i=0; i<atts.length(); i++) {
					if(atts.qName(i).compare(QString("rdf:resource")) == 0) {
						mRDFTree.setChannelTextInputUri(atts.value(i));
					}
				}
			}
			else {
				for(int i=0; i<atts.length(); i++) {
					if(atts.qName(i).compare(QString("rdf:about")) == 0) {
						mRDFTree.setTextInputUri(atts.value(i));
					}
				}
			}
		}
		else {
			mRDFTree.setChannelTextInputUri(QString("textinput"));
			mRDFTree.setTextInputUri(QString("textinput"));
		}
	}
	else if(qName.compare(itemstag) == 0) {
		mReadingItems = true;
		mIsEmpty = false;
	}
	else if(qName.compare(rdfseqtag) == 0) {
		mReadingItemSeq = true;
		mIsEmpty = false;
	}
	else if(qName.compare(rdflitag) == 0) {
		mReadingItemSeqList = true;
		mIsEmpty = false;
		if(!mPrevVersion) {
			for(int i=0; i<atts.length(); i++) {
				if(atts.qName(i).compare(QString("resource")) == 0) {
					mRDFTree.addChannelItemUri(atts.value(i));
				}
			}
		}
		else {
			mRDFTree.addChannelItemUri(QString("item"));
		}
	}
	else if(qName.compare(itemtag) == 0) {
		mReadingItem = true;
		mIsEmpty = false;
		if(!mPrevVersion) {
			for(int i=0; i<atts.length(); i++) {
				if(atts.qName(i).compare(QString("rdf:about")) == 0) {
					mRDFTree.setItemUri(atts.value(i));
				}
			}
		}
		else {
			mRDFTree.setItemUri(QString("item"));
		}
	}
	else if(qName.compare(nametag) == 0) {
		mReadingName = true;
		mIsEmpty = false;
	}
	else if(qName.compare(urltag) == 0) {
		mReadingUrl = true;
		mIsEmpty = false;
	}
	else if(qName.compare(rsstag) == 0) {
		mReadingRDF = true;
		mHadRDFTag = true;
		mPrevVersion = true;
	}
	else if(qName.lower().compare(htmltag) == 0) {
		mIsHTML = true;
		mHadRDFTag = true;
		mPrevVersion = true;
	}
	else if(qName.lower().compare(bodytag) == 0) {
		mReadingBody = true;
	}
	return true;
}

bool RSSParser::endElement(const QString & , const QString & , const QString & qName)
{
	if(qName.compare(rdftag) == 0) {
		mReadingRDF = false;
	}
	else if(qName.compare(channeltag) == 0) {
		mReadingChannel = false;
	}
	else if(qName.lower().compare(titletag) == 0) {
		mReadingTitle = false;
	}
	else if(qName.compare(linktag) == 0) {
		mReadingLink = false;
	}
	else if(qName.compare(desctag) == 0) {
		mReadingDescription = false;
	}
	else if(qName.compare(imagetag) == 0) {
		mReadingImage = false;
	}
	else if(qName.compare(textinputtag) == 0) {
		mReadingTextInput = false;
	}
	else if(qName.compare(itemstag) == 0) {
		mReadingItems = false;
	}
	else if(qName.compare(rdfseqtag) == 0) {
		mReadingItemSeq = false;
	}
	else if(qName.compare(rdflitag) == 0) {
		mReadingItemSeqList = false;
	}
	else if(qName.compare(itemtag) == 0) {
		mReadingItem = false;
		mRDFTree.addItem();
	}
	else if(qName.compare(nametag) == 0) {
		mReadingName = false;
	}
	else if(qName.compare(urltag) == 0) {
		mReadingUrl = false;
	}
	else if(qName.compare(rsstag) == 0) {
		mReadingRDF = false;
	}
	else if(qName.lower().compare(bodytag) == 0) {
		mReadingBody = false;
	}
	return true;
}

bool RSSParser::characters(const QString & ch)
{
	if(mReadingFile) {
		if(mIsHTML) {
			if(mReadingTitle) {
				mRDFTree.setChannelTitle(ch);
			}
			else if(mReadingBody) {
				mRDFTree.setChannelDescription(ch);
			}
		}
		else if(!mPrevVersion) {
			if(mReadingChannel) {
				if(mReadingTitle) {
					mRDFTree.setChannelTitle(ch);
				}
				else if(mReadingLink) {
					mRDFTree.setChannelLink(KURL(ch.stripWhiteSpace()));
				}
				else if(mReadingDescription) {
					mRDFTree.setChannelDescription(ch);
				}
			}
			else if(mReadingItem) {
				if(mReadingTitle) {
					mRDFTree.setItemTitle(ch);
				}
				else if(mReadingLink) {
					mRDFTree.setItemLink(KURL(ch.stripWhiteSpace()));
				}
				else if(mReadingDescription) {
					mRDFTree.setItemDescription(ch);
				}
			}
			else if(mReadingImage) {
				if(mReadingTitle) {
					mRDFTree.setImageTitle(ch);
				}
				else if(mReadingUrl) {
					mRDFTree.setImageUrl(KURL(ch.stripWhiteSpace()));
				}
				else if(mReadingLink) {
					mRDFTree.setImageLink(KURL(ch));
				}
			}
			else if(mReadingTextInput) {
				if(mReadingTitle) {
					mRDFTree.setTextInputTitle(ch);
				}
				else if(mReadingDescription) {
					mRDFTree.setTextInputDescription(ch);
				}
				else if(mReadingName) {
					mRDFTree.setTextInputName(ch);
				}
				else if(mReadingLink) {
					mRDFTree.setTextInputLink(KURL(ch.stripWhiteSpace()));
				}
			}
		}
		else {
			if(mReadingChannel) {
				if(mReadingTitle) {
					if(mReadingItem) {
						mRDFTree.setItemTitle(ch);
					}
					else if(mReadingImage) {
						mRDFTree.setImageTitle(ch);
					}
					else if(mReadingTextInput) {
						mRDFTree.setTextInputTitle(ch);
					}
					else {
						mRDFTree.setChannelTitle(ch);
					}
				}
				else if(mReadingLink) {
					if(mReadingItem) {
						mRDFTree.setItemLink(KURL(ch.stripWhiteSpace()));
					}
					else if(mReadingImage) {
						mRDFTree.setImageLink(KURL(ch.stripWhiteSpace()));
					}
					else if(mReadingTextInput) {
						mRDFTree.setTextInputLink(KURL(ch.stripWhiteSpace()));
					}
					else {
						mRDFTree.setChannelLink(KURL(ch.stripWhiteSpace()));
					}
				}
				else if(mReadingDescription) {
					if(mReadingItem) {
						mRDFTree.setItemDescription(ch);
					}
					else if(mReadingTextInput) {
						mRDFTree.setTextInputDescription(ch);
					}
					else {
						mRDFTree.setChannelDescription(ch);
					}
				}
				else if(mReadingUrl) {
					if(mReadingImage) {
						mRDFTree.setImageUrl(KURL(ch));
					}
				}
				else if(mReadingName) {
					if(mReadingTextInput) {
						mRDFTree.setTextInputName(ch.stripWhiteSpace());
					}
				}
			}
		}			
	}
	return true;
}

bool RSSParser::comment(const QString & ch)
{
	cerr << "Comment: " << ch << endl;
	mComment.append(QChar(' ') + ch);
	mHaveComment = true;
	return true;
}

bool RSSParser::warning(const QXmlParseException & exception)
{
	cerr << endl;
	cerr << "RSSParser reported: " << endl;
	cerr << "Waring: (" << exception.lineNumber() << "," << exception.columnNumber() << ")" << endl;
	cerr << "      " << exception.message() << endl;
	return true;
}

bool RSSParser::error(const QXmlParseException & exception)
{
	cerr << endl;
	cerr << "RSSParser reported: " << endl;
	cerr << "Error: (" << exception.lineNumber() << "," << exception.columnNumber() << ")" << endl;
	cerr << "     " << exception.message() << endl;
	return true;
}

bool RSSParser::fatalError(const QXmlParseException & exception)
{
	cerr << endl;
	cerr << "RSSParser reported: " << endl;
	cerr << "Fatal Error: (" << exception.lineNumber() << "," << exception.columnNumber() << ")" << endl;
	cerr << "           " << exception.message() << endl;
	return true;
}

bool RSSParser::parseFile(const QString & fileName, bool fixImage, KURL & oldUrl, KURL & newUrl, bool updateMD5)
{
	RSSParser parser;
	QFile xmlFile(fileName);
	if(!xmlFile.exists()) {
		RDFTree ttree;
		ttree.setChannelTitle(QString("Error"));
		ttree.setChannelDescription(QString("The file: ") + fileName + QString(" does not exist.  Please update your feeds (File->Get Feeds...) to retreive the file."));
		ttree.write(fileName);
	}
	QXmlInputSource source(xmlFile);
	QXmlSimpleReader reader;
	reader.setContentHandler(&parser);
	reader.setErrorHandler(&parser);
	reader.setLexicalHandler(&parser);
	bool ret = reader.parse(source);
	
	if(ret == false) return ret;
	
	if(fixImage) {
		ret = parser.fixImage(oldUrl, newUrl);
	}

	parser.write(fileName);
	
	if(updateMD5)
		isNew(fileName, true);
	
	return ret;
}

void RSSParser::parseFile(const QString & fileName, RDFTree & tree)
{
	cerr << "Parsing File: " << fileName << endl;
	
	RSSParser parser;
	QFile xmlFile(fileName);
	if(!xmlFile.exists()) {
		RDFTree ttree;
		ttree.setChannelTitle(QString("Error"));
		ttree.setChannelDescription(QString("The file: ") + fileName + QString(" does not exist.  Please update your feeds (File->Get Feeds...) to retreive the file."));
		ttree.write(fileName);
	}
	QXmlInputSource source(xmlFile);
	QXmlSimpleReader reader;
	reader.setContentHandler(&parser);
	reader.setErrorHandler(&parser);
	reader.setLexicalHandler(&parser);
	reader.parse(source);
	
	cerr << "Done Parsing" << endl;
	
	tree = parser.rdfTree();
}

void RSSParser::makeNews(const QString & siteTitle, const QString & fileName, QString & mStr, SchemeTree & tree, FilterHandler & filters, const QString & buttonText, KIconView *mIcons, int imagex, int imagey)
{
	RSSParser parser;
	QFile xmlFile(fileName);
	if(!xmlFile.exists()) {
		RDFTree ttree;
		ttree.setChannelTitle(QString("Error"));
		ttree.setChannelDescription(QString("The file: ") + fileName + QString(" does not exist.  Please update your feeds (File->Get Feeds...) to retreive the file."));
		ttree.write(fileName);
	}
	QXmlInputSource source(xmlFile);
	QXmlSimpleReader reader;
	reader.setContentHandler(&parser);
	reader.setErrorHandler(&parser);
	reader.parse(source);
	
	MyChannelBarItem *mIcon = 0;
	
	QString ofileName;
	
	if(mIcons) {
		mIcon = new MyChannelBarItem(mIcons, tree.myChannelBarTextColour());
		mIcon->setDisplayDateTime(QDateTime::currentDateTime());
		QFileInfo nfi(fileName);
		ofileName = QDir::homeDirPath() + QString("/.krss/feeds/unparsed/") + nfi.fileName();
		mIcon->setKey(ofileName);
		QFileInfo ofi(ofileName);
		mIcon->setDownloadDateTime(ofi.lastModified());
  	mIcon->calculateExpireDateTime();
		
		if(isNew(ofileName)) {
			mIcon->setColour(tree.myChannelBarNewFeedColour());
			mIcon->setNew(true);
		}
		else {
			mIcon->setColour(tree.myChannelBarTextColour());
			mIcon->setNew(false);
		}
	}
	
	mStr.append(QString("<a name=\""));
	mStr.append(ofileName);
	mStr.append(QString("\"></a>"));
	
	parser.displayNews(siteTitle, mStr, tree, filters, buttonText, mIcon, imagex, imagey);
		
}

bool RSSParser::isNew(const QString & filename, bool updateMD5)
{
	QString md5filename = filename + QString(".md5");
	
	QFile md5f(md5filename);
	QFile f(filename);
	
	if(!f.open(IO_ReadOnly)) return true;
	
	QString md5f_str = QString::null;
	if(md5f.open(IO_ReadOnly)) {
		QTextStream md5ft(&md5f);
		md5f_str = md5ft.read();
		md5f.close();
	}
		
	QTextStream ft(&f);
	QString f_str = ft.read();
	
	KMD5 md5(f_str);
	QString md5s(md5.hexDigest());
	
	bool ret = true;
	
	if(md5s.compare(md5f_str) == 0) {
		KConfig * config = KGlobal::config();
		config->setGroup(QString("MyChannelBar"));
		if(!config->readBoolEntry(QString("No Expiration"), false)) {
			QFileInfo fi(md5f);
			QDateTime fidt = fi.lastModified();
			fidt = fidt.addSecs(config->readNumEntry(QString("Expiration Days"), 1) * 24*60*60);
			fidt = fidt.addSecs(config->readNumEntry(QString("Expiration Hours"), 0) * 60*60);
			fidt = fidt.addSecs(config->readNumEntry(QString("Expiration Minutes"), 0) * 60);
			if(fidt >= QDateTime::currentDateTime())
				ret = true;
			else
				ret = false;
		}
		else		
			ret = false;
	}
		
	f.close();
	
	if(ret && updateMD5) {
		if(!md5f.open(IO_WriteOnly))
			return ret;
			
		QTextStream md5wf(&md5f);
		md5wf << md5s;
		
		md5f.close();
	}
	
	return ret;
}

