/***************************************************************************
                          schemeeditor.cpp  -  description
                             -------------------
    begin                : Fri Oct 12 2001
    copyright            : (C) 2001 by Chris Wallace
    email                : ranchhard@netflash.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "schemeeditor.h"

#include <qlayout.h>
#include <qdir.h>
#include <qlabel.h>
#include <qspinbox.h>
#include <qcheckbox.h>
#include <qtoolbutton.h>
#include <qslider.h>

#include <kcombobox.h>
#include <klistview.h>
#include <khtml_part.h>
#include <khtmlview.h>
#include <kfontdialog.h>
#include <kcolorbtn.h>
#include <klocale.h>
#include <klineeditdlg.h>
#include <klineedit.h>
#include <kfiledialog.h>
#include <kurl.h>
#include <kstddirs.h>

#include <iostream>

#include "schemeparser.h"
#include "colourconverter.h"

SchemeEditor::SchemeEditor(QWidget *parent, const char *name )
: QWidget(parent,name)
{
	mLastType = Information;
	mLastLocate = Nowhere;
	
	QVBoxLayout * lay = new QVBoxLayout(this, 6, 3);
	
	QWidget * top = new QWidget(this);
	QHBoxLayout * topLay = new QHBoxLayout(top, 6, 3);
	mSchemes = new KComboBox(top, "mSchemes");
	mSchemes->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred));
	topLay->addWidget(mSchemes);
	mNewScheme = new QPushButton(i18n("New..."), top, "mNewScheme");
	mNewScheme->setSizePolicy(QSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred));
	topLay->addWidget(mNewScheme);
	mSaveScheme = new QPushButton(i18n("Save"), top, "mSaveScheme");
	mSaveScheme->setSizePolicy(QSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred));
	topLay->addWidget(mSaveScheme);
	mSaveSchemeAs = new QPushButton(i18n("Save As..."), top, "mSaveSchemeAs");
	mSaveSchemeAs->setSizePolicy(QSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred));
	topLay->addWidget(mSaveSchemeAs);
	mRemoveScheme = new QPushButton(i18n("Remove"), top, "mRemoveScheme");
	mRemoveScheme->setSizePolicy(QSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred));
	topLay->addWidget(mRemoveScheme);
	
	lay->addWidget(top);
		
	QFrame *prevFrame = new QFrame(this, "prevFrame");
	prevFrame->setCaption(i18n("Preview"));
	QVBoxLayout * preLay = new QVBoxLayout(prevFrame, 6, 3);
	mPreview = new KHTMLPart(prevFrame, "mPreview");
	preLay->addWidget(mPreview->view());
	prevFrame->setMaximumHeight(200);
	
	lay->addWidget(prevFrame);
	
	QWidget *bot = new QWidget(this);
	QHBoxLayout *botLay = new QHBoxLayout(bot, 6, 3);
	
	QFrame * bleFrame = new QFrame(bot, "bleFrame");
	bleFrame->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Preferred));
	bleFrame->setCaption(i18n("Options"));
	QHBoxLayout * bleLay = new QHBoxLayout(bleFrame, 6, 3);
	
	botLay->addWidget(bleFrame);
	
	mOptions = new KListView(bleFrame, "mOptions");
	mOptions->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Preferred));
	mOptions->addColumn(i18n("Options"));
	
	bleLay->addWidget(mOptions);
	
	QFrame * briFrame = new QFrame(bot, "briFrame");
	briFrame->setCaption(i18n("Settings"));
	QVBoxLayout *briLay = new QVBoxLayout(briFrame, 6, 3);
	
	mFont = new KFontChooser(briFrame, "mFont");
	
	briLay->addWidget(mFont);
	
	mColour = new KColorButton(briFrame, "mColour");
	
	briLay->addWidget(mColour);
	
	mInformation = new QLabel(briFrame, "mInformation");
	
	briLay->addWidget(mInformation);
	
	mMargin = new QWidget(briFrame, "mMargins");
	QGridLayout * marLay = new QGridLayout(mMargin, 6, 3);
	mTopMarginText = new QLabel(i18n("Top Margin: "), mMargin, "mTopMarginText");
	marLay->addWidget(mTopMarginText, 0, 0);
	mTopMargin = new QSpinBox(0, 100, 1, mMargin, "mTopMargin");
	marLay->addWidget(mTopMargin, 0, 1);
	mLeftMarginText = new QLabel(i18n("Left Margin: "), mMargin, "mLeftMarginText");
	marLay->addWidget(mLeftMarginText, 1, 0);
	mLeftMargin = new QSpinBox(0, 100, 1, mMargin, "mleftMargin");
	marLay->addWidget(mLeftMargin, 1, 1);
	mBottomMarginText = new QLabel(i18n("Bottom Margin: "), mMargin, "mBottomMargin");
	marLay->addWidget(mBottomMarginText, 0, 2);
	mBottomMargin = new QSpinBox(0, 100, 1, mMargin, "mBottomMargin");
	marLay->addWidget(mBottomMargin, 0, 3);
	mRightMarginText = new QLabel(i18n("Right Margin: "), mMargin, "mRightMarginText");
	marLay->addWidget(mRightMarginText, 1, 2);
	mRightMargin = new QSpinBox(0, 100, 1, mMargin, "mRightMargin");
	marLay->addWidget(mRightMargin, 1, 3);
	
	briLay->addWidget(mMargin);
	
	mImage = new QWidget(briFrame, "mImage");
	QVBoxLayout *imgLay = new QVBoxLayout(mImage, 6, 3);
	mUseImage = new QCheckBox(mImage, "mUseImage");
	mUseImage->setText(i18n("Use an image"));
	imgLay->addWidget(mUseImage);
	mImageFrame = new QFrame(mImage, "mImageFrame");
	mImageFrame->setCaption(i18n("Image Options"));
	QVBoxLayout * ifrLay = new QVBoxLayout(mImageFrame, 6, 3);
	QWidget *imgWidget = new QWidget(mImageFrame);
	QHBoxLayout *img2Lay = new QHBoxLayout(imgWidget, 6, 3);
	mImageURLText = new QLabel(i18n("URL: "), imgWidget, "mImageText");
	img2Lay->addWidget(mImageURLText);
	mImageURL = new KLineEdit(imgWidget, "mImageURL");
	img2Lay->addWidget(mImageURL);
	mImageBrowser = new QToolButton(imgWidget, "mImageBrowser");
	mImageBrowser->setMaximumSize(21, 21);
	mImageBrowser->setText(QString("..."));
	img2Lay->addWidget(mImageBrowser);
	ifrLay->addWidget(imgWidget);
	mFixedImage = new QCheckBox(mImageFrame, "mImageFixed");
	mFixedImage->setText(i18n("Do not Scroll Image"));
	ifrLay->addWidget(mFixedImage);
	imgLay->addWidget(mImageFrame);
	
	briLay->addWidget(mImage);
	
	mAlign = new QWidget(briFrame, "mAlign");
	QHBoxLayout *algLay = new QHBoxLayout(mAlign, 6, 3);
	mAlignmentText = new QLabel(i18n("Alignment: "), mAlign, "mAlignmentText");
	algLay->addWidget(mAlignmentText);
	mAlignment = new KComboBox(mAlign, "mAlignment");
	mAlignment->insertItem(i18n("Default"));
	mAlignment->insertItem(i18n("Left"));
	mAlignment->insertItem(i18n("Right"));
	mAlignment->insertItem(i18n("Center"));
	mAlignment->insertItem(i18n("Justified"));
	algLay->addWidget(mAlignment);
	
	briLay->addWidget(mAlign);
	
	mHeader = new QWidget(briFrame, "mHeader");
	QHBoxLayout *hdrLay = new QHBoxLayout(mHeader, 6, 3);
	mHeaderText = new QLabel(i18n("Header Level: "), mHeader, "mHeaderText");
	hdrLay->addWidget(mHeaderText);
	mHeaders = new KComboBox(mHeader, "mHeaders");
	mHeaders->insertItem(i18n("None"));
	mHeaders->insertItem(i18n("Level 1"));
	mHeaders->insertItem(i18n("Level 2"));
	mHeaders->insertItem(i18n("Level 3"));
	mHeaders->insertItem(i18n("Level 4"));
	mHeaders->insertItem(i18n("Level 5"));
	mHeaders->insertItem(i18n("Level 6"));
	hdrLay->addWidget(mHeaders);
	
	briLay->addWidget(mHeader);
	
	mBorder = new QWidget(briFrame, "mBorder");
	QHBoxLayout *bdrLay = new QHBoxLayout(mBorder, 6, 3);
	mBorderText = new QLabel(i18n("Image Border: "), mBorder, "mBorderText");
	bdrLay->addWidget(mBorderText);
	mBorderValue = new QSpinBox(0, 100, 1, mBorder, "mBorderValue");
	bdrLay->addWidget(mBorderValue);
	
	briLay->addWidget(mBorder);
	
	mSize = new QWidget(briFrame, "mSize");
	QHBoxLayout *sizeLay = new QHBoxLayout(mSize, 6, 3);
	mSizeText = new QLabel(i18n("Size: "), mSize, "mSizeText");
	sizeLay->addWidget(mSizeText);
	mSizeValue = new QSpinBox(0, 100, 1, mSize, "mSizeValue");
	sizeLay->addWidget(mSizeValue);
	
	briLay->addWidget(mSize);
	
	mWidth = new QWidget(briFrame, "mWidth");
	QHBoxLayout *widLay = new QHBoxLayout(mWidth, 6, 3);
	mWidthText = new QLabel(i18n("Width (in percentage): "), mWidth, "mWidthText");
	widLay->addWidget(mWidthText);
	mWidthValue = new QSlider(0, 100, 10, 0, Qt::Horizontal, mWidth, "mWidthValue");
	mWidthValue->setTickmarks(QSlider::Below);
	widLay->addWidget(mWidthValue);
	
	briLay->addWidget(mWidth);
	
	mSeparator = new QWidget(briFrame, "mSeparator");
	QHBoxLayout *sepLay = new QHBoxLayout(mSeparator, 6, 3);
	mUseSeparator = new QCheckBox(mSeparator, "mUseSeparator");
	mUseSeparator->setText(i18n("Use Separator"));
	sepLay->addWidget(mUseSeparator);
	
	briLay->addWidget(mSeparator);
	
	botLay->addWidget(briFrame);
	
	
	lay->addWidget(bot);
	
	loadOptions();
	loadSchemes();
	
	connect(mSchemes, SIGNAL(activated(const QString &)), this, SLOT(slotSchemeChanged(const QString &)));
	connect(mOptions, SIGNAL(executed(QListViewItem *)), this, SLOT(slotOptionChanged(QListViewItem *)));
	connect(mNewScheme, SIGNAL(clicked()), this, SLOT(slotNewScheme()));
	connect(mSaveScheme, SIGNAL(clicked()), this, SLOT(slotSaveScheme()));
	connect(mSaveSchemeAs, SIGNAL(clicked()), this, SLOT(slotSaveSchemeAs()));
	connect(mRemoveScheme, SIGNAL(clicked()), this, SLOT(slotRemoveScheme()));
	connect(mUseImage, SIGNAL(toggled(bool)), mImageFrame, SLOT(setEnabled(bool)));
	connect(mImageBrowser, SIGNAL(clicked()), this, SLOT(slotImageBrowse()));
}

SchemeEditor::~SchemeEditor()
{
}

void SchemeEditor::loadOptions(void)
{
	KListViewItem * item = new KListViewItem(mOptions, i18n("Background Colour"), QString::number(Colour), QString::number(Background));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Active Link Colour"), QString::number(Colour), QString::number(ActiveLink));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Visited Link Colour"), QString::number(Colour), QString::number(VisitedLink));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel Title Colour"), QString::number(Colour), QString::number(ChannelTitle));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel Description Colour"), QString::number(Colour), QString::number(ChannelDescription));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("News Item Title Colour"), QString::number(Colour), QString::number(NewsTitle));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("News Item Description Colour"), QString::number(Colour), QString::number(NewsDescription));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel Separator Colour"), QString::number(Colour), QString::number(ChannelSeparator));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("News Item Separator Colour"), QString::number(Colour), QString::number(NewsSeparator));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel/News Separator Colour"), QString::number(Colour), QString::number(ChannelNewsSeparator));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Background Image"), QString::number(Image), QString::number(Background));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Margins"), QString::number(Margins), QString::number(Margin));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel Title Font"), QString::number(Font), QString::number(ChannelTitle));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel Description Font"), QString::number(Font), QString::number(ChannelDescription));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel Title Alignment"), QString::number(Align), QString::number(ChannelTitle));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel Image Alignment"), QString::number(Align), QString::number(ChannelImage));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel Description Alignment"), QString::number(Align), QString::number(ChannelDescription));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel Title Header Level"), QString::number(Header), QString::number(ChannelTitle));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel Image Border"), QString::number(Border), QString::number(ChannelImage));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("News Item Title Font"), QString::number(Font), QString::number(NewsTitle));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("News Item Description Font"), QString::number(Font), QString::number(NewsDescription));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("News Item Title Alignment"), QString::number(Align), QString::number(NewsTitle));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("News Item Title Header Level"), QString::number(Header), QString::number(NewsTitle));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("News Item Description Alignment"), QString::number(Align), QString::number(NewsDescription));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel Separator"), QString::number(Separator), QString::number(ChannelSeparator));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel Separator Alignment"), QString::number(Align), QString::number(ChannelSeparator));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel Separator Size"), QString::number(Size), QString::number(ChannelSeparator));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel Separator Width"), QString::number(Width), QString::number(ChannelSeparator));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel/News Separator"), QString::number(Separator), QString::number(ChannelNewsSeparator));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel/News Items Separator Alignment"), QString::number(Align), QString::number(ChannelNewsSeparator));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel/News Items Separator Size"), QString::number(Size), QString::number(ChannelNewsSeparator));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("Channel/News Items Separator Width"), QString::number(Width), QString::number(ChannelNewsSeparator));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("News Item Separator"), QString::number(Separator), QString::number(NewsSeparator));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("News Item Separator Alignment"), QString::number(Align), QString::number(NewsSeparator));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("News Item Separator Size"), QString::number(Size), QString::number(NewsSeparator));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("News Item Separator Width"), QString::number(Width), QString::number(NewsSeparator));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("MyChannelBar Text Colour"), QString::number(Colour), QString::number(MyChannelBarText));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("MyChannelBar Background Colour"), QString::number(Colour), QString::number(MyChannelBarBackground));
	mOptions->insertItem(item);
	item = new KListViewItem(mOptions, i18n("MyChannelBar New Feed Text Colour"), QString::number(Colour), QString::number(MyChannelBarNewFeed));
	//To fix mychannelbar, for background images
	//item = new KListViewItem(mOptions, i18n("MyChannelBar Background Image"), QString::number(Image), QString::number(MyChannelBarImage));
	//mOptions->insertItem(item);
}

void SchemeEditor::loadSchemes(void)
{
	QDir d(QDir::homeDirPath() + QString("/.krss/schemes/"));
	d.setFilter(QDir::Files | QDir::NoSymLinks);
	d.setSorting(QDir::Name);
	
	const QFileInfoList *list = d.entryInfoList();
	QFileInfoListIterator it(*list);
	QFileInfo *fi;
	
	mSchemes->clear();
	
	mSchemes->insertItem(i18n("Current Scheme"));
	
	while((fi = it.current())) {
		if(fi->extension().compare(QString("scheme")) == 0) {
			mSchemes->insertItem(fi->baseName());
		}
		++it;
	}
	slotSchemeChanged(i18n("Current Scheme"));
}

void SchemeEditor::slotSchemeChanged(const QString & schemename)
{
	QString filename = QDir::homeDirPath() + QString("/.krss/schemes/") + schemename + QString(".scheme");
	QFile f(filename);
	if(!f.exists())
		filename = QDir::homeDirPath() + QString("/.krss/schemes/currentscheme");
	
	cerr << "Loading Scheme: " << filename << endl;
		
	mTree.clear();
	SchemeParser::parseFile(filename, mTree);
	
	mLastType = Information;
	mLastLocate = Nowhere;
	slotOptionChanged(mOptions->selectedItem());
}

void SchemeEditor::slotNewScheme(void)
{
	bool ok = false;
	QString schemename = KLineEditDlg::getText(i18n("Please enter a name for the new scheme"), i18n("New Scheme"), &ok, 0);
	if(!ok)
		return;
		
	QString filename = QDir::homeDirPath() + QString("/.krss/schemes/") + schemename + QString(".scheme");
	QFile f(filename);
	if(f.exists()) //We should warn the user here!
		return;	
		
	//Create an empty (default) scheme
	SchemeTree tree;
	
	//Save it
	tree.write(filename);
	
	//Reload
	loadSchemes();
}

void SchemeEditor::slotSaveScheme(void)
{
	QString filename = QDir::homeDirPath() + QString("/.krss/schemes/") + mSchemes->currentText() + QString(".scheme");
	QFile f(filename);
	if(!f.exists())
		filename = QDir::homeDirPath() + QString("/.krss/schemes/currentscheme");
		
	cerr <<	"Saving Scheme: " << filename << endl;
	
	saveOption(mLastType, mLastLocate);
	
	mTree.write(filename);
}

void SchemeEditor::saveCurrentScheme(void)
{
	QString filename = QDir::homeDirPath() + QString("/.krss/schemes/currentscheme");
	
	saveOption(mLastType, mLastLocate);
	
	mTree.write(filename);
}

void SchemeEditor::slotSaveSchemeAs(void)
{
	bool ok = false;
	QString schemename = KLineEditDlg::getText(i18n("Please enter a name for the new scheme"), i18n("New Scheme"), &ok, 0);
	if(!ok)
		return;
		
	QString filename = QDir::homeDirPath() + QString("/.krss/schemes/") + schemename + QString(".scheme");
	QFile f(filename);
	if(f.exists()) //We should warn the user here!
		return;	
	
	saveOption(mLastType, mLastLocate);
		
	mTree.write(filename);
	
	loadSchemes();
}

void SchemeEditor::slotRemoveScheme(void)
{
	QString filename = QDir::homeDirPath() + QString("/.krss/schemes/") + mSchemes->currentText() + QString(".scheme");
	
	//Test if the file exits
	QFile f(filename);
	if(!f.exists())
		return;
	
	//Remove it
	QDir().remove(filename);
	
	//Reload
	loadSchemes();
}

void SchemeEditor::slotOptionChanged(QListViewItem *lvi)
{
	//Hide all of the settings widgets
	mFont->hide();
	mColour->hide();
	mInformation->hide();
	mMargin->hide();
	mImage->hide();
	mSeparator->hide();
	mAlign->hide();
	mHeader->hide();
	mBorder->hide();
	mSize->hide();
	mWidth->hide();
	
	Type t;
	Locate l;
	if(!lvi) {
		t = Information;
		l = Nowhere;
	}
	else {
		t = (Type)lvi->text(1).toInt();
		l = (Locate)lvi->text(2).toInt();
	}
	
	saveOption(mLastType, mLastLocate);
	loadOption(t, l);
	
	
	switch(t) {
		case Font: mFont->show();	break;
		case Colour: mColour->show();	break;
		case Margins:	mMargin->show(); break;
		case Image:	mImage->show();	break;
		case Separator: mSeparator->show(); break;
		case Align:	mAlign->show();	break;
		case Header: mHeader->show();	break;
		case Border: mBorder->show();	break;
		case Size: mSize->show(); break;
		case Width: mWidth->show(); break;
			
		case Information:
		case Text:
		case OneLine:
		default:
			//if(l == Information) {
				mInformation->setText(i18n("Please select an option on the left."));
				mInformation->show();
			//}
			break;
	}
	mLastType = t;
	mLastLocate = l;
	displayHTML();
}

void SchemeEditor::saveOption(SchemeEditor::Type t, SchemeEditor::Locate l)
{
	switch (t) {
		case Font:
			switch(l) {
				case ChannelTitle: mTree.setChannelTitleFont(mFont->font()); break;
				case ChannelDescription: mTree.setChannelDescriptionFont(mFont->font()); break;
				case NewsTitle: mTree.setNewsTitleFont(mFont->font()); break;
				case NewsDescription: mTree.setNewsDescriptionFont(mFont->font()); break;
				case Nowhere:
				default: break;
			}
			break;
		
		case Colour:
			switch(l) {
				case Background: mTree.setBackgroundColour(mColour->color()); break;
				case ActiveLink: mTree.setActiveLinkColour(mColour->color()); break;
				case VisitedLink: mTree.setVisitedLinkColour(mColour->color()); break;
				case ChannelTitle: mTree.setChannelTitleColour(mColour->color()); break;
				case ChannelDescription: mTree.setChannelDescriptionColour(mColour->color()); break;
				case ChannelSeparator: mTree.setChannelSeparatorColour(mColour->color()); break;
				case NewsTitle: mTree.setNewsTitleColour(mColour->color()); break;
				case NewsDescription: mTree.setNewsDescriptionColour(mColour->color()); break;
				case NewsSeparator: mTree.setNewsSeparatorColour(mColour->color()); break;
				case ChannelNewsSeparator: mTree.setChannelNewsSeparatorColour(mColour->color()); break;
				case MyChannelBarText: mTree.setMyChannelBarTextColour(mColour->color()); break;
				case MyChannelBarBackground: mTree.setMyChannelBarBackgroundColour(mColour->color()); break;
				case MyChannelBarNewFeed: mTree.setMyChannelBarNewFeedColour(mColour->color()); break;
				case Nowhere:
				default : break;
			}
			break;
			
		case Margins:
			mTree.setTopMargin(mTopMargin->value());
			mTree.setLeftMargin(mLeftMargin->value());
			mTree.setRightMargin(mRightMargin->value());
			mTree.setBottomMargin(mBottomMargin->value());
			break;
		
		case Image:
			switch(l) {
				case Background:
					mTree.setUseBackgroundImage(mUseImage->isChecked());
					mTree.setBackgroundImageUrl(KURL(mImageURL->text()));
					mTree.setBackgroundImageFixed(mFixedImage->isChecked());
					break;
				case MyChannelBarImage:
					mTree.setMyChannelBarUseBackgroundImage(mUseImage->isChecked());
					mTree.setMyChannelBarBackgroundImageUrl(KURL(mImageURL->text()));
					mTree.setMyChannelBarBackgroundImageFixed(mFixedImage->isChecked());
					break;
			}
			break;
		
		case Separator:
			switch(l) {
				case ChannelSeparator: mTree.setSeparateBetweenChannels(mUseSeparator->isChecked()); break;
				case NewsSeparator: mTree.setSeparateBetweenNews(mUseSeparator->isChecked()); break;
				case ChannelNewsSeparator: mTree.setSeparateBetweenChannelsAndNews(mUseSeparator->isChecked()); break;
			}
			break;
			
		case Align:
			switch(l) {
				case ChannelTitle: mTree.setChannelTitleAlign(mAlignment->currentItem()); break;
				case ChannelImage: mTree.setChannelImageAlign(mAlignment->currentItem()); break;
				case ChannelDescription: mTree.setChannelDescriptionAlign(mAlignment->currentItem()); break;
				case NewsTitle: mTree.setNewsTitleAlign(mAlignment->currentItem()); break;
				case NewsDescription: mTree.setNewsDescriptionAlign(mAlignment->currentItem()); break;
				case ChannelSeparator: mTree.setSeparateBetweenChannelsAlign(mAlignment->currentItem()); break;
				case NewsSeparator: mTree.setSeparateBetweenNewsAlign(mAlignment->currentItem()); break;
				case ChannelNewsSeparator: mTree.setSeparateBetweenChannelsAndNewsAlign(mAlignment->currentItem()); break;
			}
			break;
			
		case Header:
			switch(l) {
				case ChannelTitle: mTree.setChannelTitleHeaderLevel(mHeaders->currentItem()); break;
				case NewsTitle: mTree.setNewsTitleHeaderLevel(mHeaders->currentItem()); break;
			}
			break;
		
		case Border:
			switch(l) {
				case ChannelImage: mTree.setChannelImageBorder(mBorderValue->value()); break;
			}
			break;
			
		case Size:
			switch(l) {
				case ChannelSeparator: mTree.setSeparateBetweenChannelsSize(mSizeValue->value()); break;
				case NewsSeparator: mTree.setSeparateBetweenNewsSize(mSizeValue->value()); break;
				case ChannelNewsSeparator: mTree.setSeparateBetweenChannelsAndNewsSize(mSizeValue->value()); break;
			}
			break;
			
		case Width:
			switch(l) {
				case ChannelSeparator: mTree.setSeparateBetweenChannelsWidth(mWidthValue->value()); break;
				case NewsSeparator: mTree.setSeparateBetweenNewsWidth(mWidthValue->value()); break;
				case ChannelNewsSeparator: mTree.setSeparateBetweenChannelsAndNewsWidth(mWidthValue->value()); break;
			}
			break;
			
		case Information:
		case Text:
		case OneLine:
		default:
			break;
	}
	
}

void SchemeEditor::loadOption(SchemeEditor::Type t, SchemeEditor::Locate l)
{
	switch (t) {
		case Font:
			switch(l) {
				case ChannelTitle: mFont->setFont(mTree.channelTitleFont()); break;
				case ChannelDescription: mFont->setFont(mTree.channelDescriptionFont()); break;
				case NewsTitle: mFont->setFont(mTree.newsTitleFont()); break;
				case NewsDescription: mFont->setFont(mTree.newsDescriptionFont()); break;
				case Nowhere:
				default: break;
			}
			break;
		
		case Colour:
			switch(l) {
				case Background: mColour->setColor(mTree.backgroundColour()); break;
				case ActiveLink: mColour->setColor(mTree.activeLinkColour()); break;
				case VisitedLink: mColour->setColor(mTree.visitedLinkColour()); break;
				case ChannelTitle: mColour->setColor(mTree.channelTitleColour()); break;
				case ChannelDescription: mColour->setColor(mTree.channelDescriptionColour()); break;
				case ChannelSeparator: mColour->setColor(mTree.channelSeparatorColour()); break;
				case NewsTitle: mColour->setColor(mTree.newsTitleColour()); break;
				case NewsDescription: mColour->setColor(mTree.newsDescriptionColour()); break;
				case NewsSeparator: mColour->setColor(mTree.newsSeparatorColour()); break;
				case ChannelNewsSeparator: mColour->setColor(mTree.channelNewsSeparatorColour()); break;
				case MyChannelBarText: mColour->setColor(mTree.myChannelBarTextColour()); break;
				case MyChannelBarBackground: mColour->setColor(mTree.myChannelBarBackgroundColour()); break;
				case MyChannelBarNewFeed: mColour->setColor(mTree.myChannelBarNewFeedColour()); break;
				case Nowhere:
				default : break;
			}
			break;
			
		case Margins:
			mLeftMargin->setValue(mTree.leftMargin());
			mTopMargin->setValue(mTree.topMargin());
			mBottomMargin->setValue(mTree.bottomMargin());
			mRightMargin->setValue(mTree.rightMargin());
			break;
		
		case Image:
			switch(l) {
				case Background:
					mUseImage->setChecked(mTree.useBackgroundImage());
					mImageFrame->setEnabled(mTree.useBackgroundImage());
					mImageURL->setText(mTree.backgroundImageUrl().url());
					mFixedImage->setChecked(mTree.backgroundImageFixed());
					break;
				case MyChannelBarImage:
					mUseImage->setChecked(mTree.myChannelBarUseBackgroundImage());
					mImageFrame->setEnabled(mTree.myChannelBarUseBackgroundImage());
					mImageURL->setText(mTree.myChannelBarBackgroundImageUrl().url());
					mFixedImage->setChecked(mTree.myChannelBarBackgroundImageFixed());
					break;
			}
			break;
		
		case Separator:
			switch(l) {
				case ChannelSeparator: mUseSeparator->setChecked(mTree.separateBetweenChannels()); break;
				case NewsSeparator: mUseSeparator->setChecked(mTree.separateBetweenNews()); break;
				case ChannelNewsSeparator: mUseSeparator->setChecked(mTree.separateBetweenChannelsAndNews()); break;
			}
			break;
			
		case Align:
			switch(l) {
				case ChannelTitle: mAlignment->setCurrentItem(mTree.channelTitleAlign()); break;
				case ChannelImage: mAlignment->setCurrentItem(mTree.channelImageAlign()); break;
				case ChannelDescription: mAlignment->setCurrentItem(mTree.channelDescriptionAlign()); break;
				case NewsTitle: mAlignment->setCurrentItem(mTree.newsTitleAlign()); break;
				case NewsDescription: mAlignment->setCurrentItem(mTree.newsDescriptionAlign()); break;
				case ChannelSeparator: mAlignment->setCurrentItem(mTree.separateBetweenChannelsAlign()); break;
				case NewsSeparator: mAlignment->setCurrentItem(mTree.separateBetweenNewsAlign()); break;
				case ChannelNewsSeparator: mAlignment->setCurrentItem(mTree.separateBetweenChannelsAndNewsAlign()); break;
			}
			break;
			
		case Header:
			switch(l) {
				case ChannelTitle: mHeaders->setCurrentItem(mTree.channelTitleHeaderLevel()); break;
				case NewsTitle: mHeaders->setCurrentItem(mTree.newsTitleHeaderLevel()); break;
			}
			break;
		
		case Border:
			switch(l) {
				case ChannelImage: mBorderValue->setValue(mTree.channelImageBorder()); break;
			}
			break;
			
		case Size:
			switch(l) {
				case ChannelSeparator: mSizeValue->setValue(mTree.separateBetweenChannelsSize()); break;
				case NewsSeparator: mSizeValue->setValue(mTree.separateBetweenNewsSize()); break;
				case ChannelNewsSeparator: mSizeValue->setValue(mTree.separateBetweenChannelsAndNewsSize()); break;
			}
			break;
			
		case Width:
			switch(l) {
				case ChannelSeparator: mWidthValue->setValue(mTree.separateBetweenChannelsWidth()); break;
				case NewsSeparator: mWidthValue->setValue(mTree.separateBetweenNewsWidth()); break;
				case ChannelNewsSeparator: mWidthValue->setValue(mTree.separateBetweenChannelsAndNewsWidth()); break;
			}
			break;
			
		case Information:
		case Text:
		case OneLine:
		default:
			break;
	}
}

void SchemeEditor::displayHTML(void)
{
	mHTMLText = QString::null;
	mHTMLText = QString("<HTML><HEAD><TITLE>Colour Scheme Preview</TITLE>");
	mHTMLText.append(QString("<STYLE TYPE=\"text/css\" MEDIA=\"screen, print\">"));
	mTree.makeCSS(mHTMLText);
	mHTMLText.append(QString("</STYLE></HEAD>\n"));
	mHTMLText.append(QString("<BODY class=\"BODY_KRSS\">"));
	mHTMLText.append(QString("<a class=\"A_KRSS\" href=\"\"><img"));
	switch(mTree.channelImageAlign()) {
		case 1: mHTMLText.append(QString(" align=\"top\"")); break;
		case 2:	mHTMLText.append(QString(" align=\"texttop\"")); break;
		case 3:	mHTMLText.append(QString(" align=\"middle\""));	break;
		case 4:	mHTMLText.append(QString(" align=\"absmiddle\"")); break;
		case 5:	mHTMLText.append(QString(" align=\"bottom\""));	break;
		case 6:	mHTMLText.append(QString(" align=\"absbottom\"")); break;
		case 7:	mHTMLText.append(QString(" align=\"baseline\""));	break;
		case 8:	mHTMLText.append(QString(" align=\"left\""));	break;
		case 9:	mHTMLText.append(QString(" align=\"right\"")); break;
		case 10: mHTMLText.append(QString(" align=\"center\"")); break;
	}
	mHTMLText.append(QString(" border=") + QString::number(mTree.channelImageBorder()));
	cerr << "file:/usr/" << KStandardDirs::kde_default("data") << "krss/tempimage.png" << endl;
	mHTMLText.append(QString(" src=\"file:/usr/") + KStandardDirs::kde_default("data") + QString("krss/tempimage.png\" alt=\"Test Image\"></a>"));
	
	switch(mTree.channelTitleAlign()) {
		case 1:	mHTMLText.append(QString("<div align=\"left\">")); break;
		case 2:	mHTMLText.append(QString("<div align=\"right\">"));	break;
		case 3:	mHTMLText.append(QString("<div align=\"center\">")); break;
		case 4:	mHTMLText.append(QString("<div align=\"justify\">"));	break;
	}
	switch(mTree.channelTitleHeaderLevel()) {
		case 1:	mHTMLText.append(QString("<h1>")); break;
		case 2:	mHTMLText.append(QString("<h2>")); break;
		case 3:	mHTMLText.append(QString("<h3>")); break;
		case 4:	mHTMLText.append(QString("<h4>")); break;
		case 5:	mHTMLText.append(QString("<h5>")); break;
		case 6:	mHTMLText.append(QString("<h6>")); break;
	}
	
	mHTMLText.append(QString("<P class=\"P_KRSS_CH_TITLE\">A Channel Title</P>"));	
	
	switch(mTree.channelTitleHeaderLevel()) {
		case 1:	mHTMLText.append(QString("</h1>"));	break;
		case 2:	mHTMLText.append(QString("</h2>"));	break;
		case 3:	mHTMLText.append(QString("</h3>"));	break;
		case 4:	mHTMLText.append(QString("</h4>"));	break;
		case 5:	mHTMLText.append(QString("</h5>"));	break;
		case 6:	mHTMLText.append(QString("</h6>"));	break;
	}
	if(mTree.channelTitleAlign() > 0 && mTree.channelTitleAlign() < 5)
		mHTMLText.append(QString("</div>"));
		
	mHTMLText.append(QString("<P>A <A class=\"A_KRSS\" href=\"\">Link</A></P>"));
		
	switch(mTree.channelDescriptionAlign()) {
		case 1:	mHTMLText.append(QString("<div align=\"left\">")); break;
		case 2:	mHTMLText.append(QString("<div align=\"right\">"));	break;
		case 3:	mHTMLText.append(QString("<div align=\"center\">")); break;
		case 4:	mHTMLText.append(QString("<div align=\"justify\">"));	break;
	}	
	
	mHTMLText.append(QString("<P class=\"P_KRSS_CH_DESC\">A Channel Description</P>"));
	
	if(mTree.channelDescriptionAlign() > 0 && mTree.channelDescriptionAlign() < 5)
		mHTMLText.append(QString("</div>"));
		
	if(mTree.separateBetweenChannelsAndNews()) {
		mHTMLText.append(QString("<hr class=\"HR_KRSS_CN\""));
		switch(mTree.separateBetweenChannelsAndNewsAlign()) {
			case 1: mHTMLText.append(QString(" align=\"left\"")); break;
			case 2:	mHTMLText.append(QString(" align=\"right\""));	break;
			case 3:	mHTMLText.append(QString(" align=\"center\"")); break;
		}
		mHTMLText.append(QString(" size=") + QString::number(mTree.separateBetweenChannelsAndNewsSize()));
		mHTMLText.append(QString(" width=\"") + QString::number(mTree.separateBetweenChannelsAndNewsWidth()) + QString("%\""));
		mHTMLText.append(QString(">"));
	}
	
	switch(mTree.newsTitleAlign()) {
		case 1:	mHTMLText.append(QString("<div align=\"left\">"));	break;
		case 2:	mHTMLText.append(QString("<div align=\"right\">")); break;
		case 3:	mHTMLText.append(QString("<div align=\"center\">"));	break;
		case 4:	mHTMLText.append(QString("<div align=\"justify\">")); break;
	}
	
	switch(mTree.newsTitleHeaderLevel()) {
		case 1: mHTMLText.append(QString("<h1>")); break;
		case 2:	mHTMLText.append(QString("<h2>"));	break;
		case 3:	mHTMLText.append(QString("<h3>"));	break;
		case 4:	mHTMLText.append(QString("<h4>"));	break;
		case 5:	mHTMLText.append(QString("<h5>"));	break;
		case 6:	mHTMLText.append(QString("<h6>"));	break;
	}
	
	mHTMLText.append(QString("<P class=\"P_KRSS_NW_TITLE\">A News Item Title</P>"));
	
	switch(mTree.newsTitleHeaderLevel()) {
		case 1:	mHTMLText.append(QString("</h1>"));	break;
		case 2:	mHTMLText.append(QString("</h2>"));	break;
		case 3:	mHTMLText.append(QString("</h3>"));	break;
		case 4:	mHTMLText.append(QString("</h4>"));	break;
		case 5:	mHTMLText.append(QString("</h5>"));	break;
		case 6:	mHTMLText.append(QString("</h6>"));	break;
	}
	
	if(mTree.newsTitleAlign() > 0 && mTree.newsTitleAlign() < 5)
		mHTMLText.append(QString("</div>"));
		
	switch(mTree.newsDescriptionAlign()) {
		case 1:	mHTMLText.append(QString("<div align=\"left\">"));	break;
		case 2:	mHTMLText.append(QString("<div align=\"right\">")); break;
		case 3:	mHTMLText.append(QString("<div align=\"center\">"));	break;
		case 4:	mHTMLText.append(QString("<div align=\"justify\">")); break;
	}
		
	mHTMLText.append(QString("<P class=\"P_KRSS_NW_DESC\">A News Item Description</P>"));
	
	if(mTree.newsDescriptionAlign() > 0 && mTree.newsDescriptionAlign() < 5)
		mHTMLText.append(QString("</div>"));
		
	if(mTree.separateBetweenNews()) {
		mHTMLText.append(QString("<hr class=\"HR_KRSS_NW\""));
		switch(mTree.separateBetweenNewsAlign()) {
			case 1: mHTMLText.append(QString(" align=\"left\"")); break;
			case 2: mHTMLText.append(QString(" align=\"right\""));	break;
			case 3:	mHTMLText.append(QString(" align=\"center\"")); break;
		}
		mHTMLText.append(QString(" size=") + QString::number(mTree.separateBetweenNewsSize()));
		mHTMLText.append(QString(" width=\"") + QString::number(mTree.separateBetweenNewsWidth()) + QString("%\""));
		mHTMLText.append(QString(" color=\"#"));
		ColourConverter::makeHTMLColour(mHTMLText, mTree.newsSeparatorColour());
		mHTMLText.append(QString("\">"));
	}
	
	switch(mTree.newsTitleAlign()) {
		case 1:	mHTMLText.append(QString("<div align=\"left\">"));	break;
		case 2:	mHTMLText.append(QString("<div align=\"right\">")); break;
		case 3:	mHTMLText.append(QString("<div align=\"center\">"));	break;
		case 4:	mHTMLText.append(QString("<div align=\"justify\">")); break;
	}
	
	switch(mTree.newsTitleHeaderLevel()) {
		case 1: mHTMLText.append(QString("<h1>")); break;
		case 2:	mHTMLText.append(QString("<h2>"));	break;
		case 3:	mHTMLText.append(QString("<h3>"));	break;
		case 4:	mHTMLText.append(QString("<h4>"));	break;
		case 5:	mHTMLText.append(QString("<h5>"));	break;
		case 6:	mHTMLText.append(QString("<h6>"));	break;
	}
	
	mHTMLText.append(QString("<P class=\"P_KRSS_NW_TITLE\">Another News Item Title</P>"));
	
	switch(mTree.newsTitleHeaderLevel()) {
		case 1:	mHTMLText.append(QString("</h1>"));	break;
		case 2:	mHTMLText.append(QString("</h2>"));	break;
		case 3:	mHTMLText.append(QString("</h3>"));	break;
		case 4:	mHTMLText.append(QString("</h4>"));	break;
		case 5:	mHTMLText.append(QString("</h5>"));	break;
		case 6:	mHTMLText.append(QString("</h6>"));	break;
	}
	
	if(mTree.newsTitleAlign() > 0 && mTree.newsTitleAlign() < 5)
		mHTMLText.append(QString("</div>"));
		
	switch(mTree.newsDescriptionAlign()) {
		case 1:	mHTMLText.append(QString("<div align=\"left\">"));	break;
		case 2:	mHTMLText.append(QString("<div align=\"right\">")); break;
		case 3:	mHTMLText.append(QString("<div align=\"center\">"));	break;
		case 4:	mHTMLText.append(QString("<div align=\"justify\">")); break;
	}
		
	mHTMLText.append(QString("<P class=\"P_KRSS_NW_DESC\">Another News Item Description</P>"));
	
	if(mTree.newsDescriptionAlign() > 0 && mTree.newsDescriptionAlign() < 5)
		mHTMLText.append(QString("</div>"));
	
	if(mTree.separateBetweenChannels()) {
		mHTMLText.append(QString("<hr class=\"HR_KRSS_CH\""));
		switch(mTree.separateBetweenChannelsAlign()) {
			case 1:	mHTMLText.append(QString(" align=\"left\"")); break;
			case 2: mHTMLText.append(QString(" align=\"right\"")); break;
			case 3:	mHTMLText.append(QString(" align=\"center\"")); break;
		}
		mHTMLText.append(QString(" size=") + QString::number(mTree.separateBetweenChannelsSize()));
		mHTMLText.append(QString(" width=\"") + QString::number(mTree.separateBetweenChannelsWidth()) + QString("%\""));
		mHTMLText.append(QString(" color=\"#"));
		ColourConverter::makeHTMLColour(mHTMLText, mTree.channelSeparatorColour());
		mHTMLText.append(QString("\""));
		mHTMLText.append(QString(">"));
	}
	
	mHTMLText.append(QString("<BR><BR><BR><BR><BR><BR></BODY>\n</HTML>"));
	
	//cerr << mHTMLText << endl;
	
	mPreview->begin();
	mPreview->write(mHTMLText);
	mPreview->end();
}

void SchemeEditor::slotImageBrowse(void)
{
	KURL url = KFileDialog::getOpenURL();
	if(url.url().isEmpty() || url.url().isNull())
		return;
		
	mImageURL->setText(url.url());
}
		
