/***************************************************************************
                          sitehandler.cpp  -  description
                             -------------------
    begin                : Sat Aug 4 2001
    copyright            : (C) 2001 by Chris Wallace
    email                : ranchhard@netflash.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "sitehandler.h"
#include <qfile.h>
#include <qdir.h>
#include <qtextstream.h>
#include <qlistview.h>
#include <qdatetime.h>

#include <klocale.h>
#include <klistview.h>
//#include <qpalette.h>

#include <iostream>

#include "schemetree.h"
#include "schemeparser.h"
#include "colourconverter.h"
#include "filterhandler.h"
#include "mychannelbar.h"
#include "editfeedlistviewitem.h"

SiteHandler::SiteHandler()
{
	mReadingSites = mReadingSite = mReadingName = mReadingUsed = false;
	mReadingURL = mReadingDescription = mReadingFileNameParsed = false;
	mReadingFileNameUnParsed = false;
	//mSites.setAutoDelete(true);
	mDontClear = false;
}
SiteHandler::~SiteHandler()
{
}

void SiteHandler::readSites()
{
	readSites(QDir::homeDirPath() + QString("/.krss/sites.xml"));
}

void SiteHandler::readSites(const QString & filename, bool clear)
{
	mDontClear = !clear;
	QFile xmlFile(filename);
	QXmlInputSource source(xmlFile);
	QXmlSimpleReader reader;
	reader.setContentHandler(this);
	reader.setErrorHandler(this);
	reader.parse(source);
}

void SiteHandler::writeSites()
{
	QDomDocument doc("RSS_Sites");
	QDomElement elm = doc.createElement("sites");
	Site *s = mSites.first();
	while(s) {
		s->write(doc, elm);
		s = mSites.next();
	}
	
	doc.appendChild(elm);
	
	QFile xmlFile(QDir::homeDirPath() + QString("/.krss/sites.xml"));
	if(!xmlFile.open(IO_WriteOnly))
		return;
	QTextStream t(&xmlFile);
	t << doc.toString();
	xmlFile.close();
}

//This fixes the site file for any changes in versions
void SiteHandler::fixSiteFile_2_2(void)
{
	QFile f(QDir::homeDirPath() + QString("/.krss/sites.xml"));
	if(!f.exists()) return; //we do not have a file yet, no need to so anything
	
	readSites();
	
	Site *s = mSites.first();
	while(s) {
		s->makeFileName();
		s = mSites.next();
	}
	
	writeSites();
	
	clearSites();
}

void SiteHandler::displaySites(KListView *lview)
{
	if(!lview)
		return;
		
	for(Site *s = mSites.first(); s != 0; s = mSites.next()) {
		EditFeedListViewItem *cli = new EditFeedListViewItem(lview, "");
		cli->setText(1, s->name());
		cli->setText(2, s->URL().url());
		cli->setText(3, s->description());
		cli->setText(4, s->fileNameUnParsed());
		cli->setText(5, s->fileNameParsed());
		cli->setOn(s->used());
		lview->insertItem(cli);
	}
}

void SiteHandler::getSites(KListView *lview)
{
	if(!lview)
		return;
		
	clearSites();
	
	for(QListViewItem *lvi = lview->firstChild(); lvi != 0; lvi = lvi->nextSibling()) {
		QCheckListItem *cli = (QCheckListItem *) lvi;
		if(cli) {
			Site *s = new Site();
			s->setName(cli->text(1));
			s->setURL(KURL(cli->text(2)));
			s->setDescription(cli->text(3));
			s->setFileNameUnParsed(cli->text(4));
			s->setFileNameParsed(cli->text(5));
			s->setUsed(cli->isOn());
			mSites.append(s);
		}
	}
}

bool SiteHandler::startDocument(void)
{
	if(!mDontClear)
		clearSites();
		
	mReadingSites = mReadingSite = mReadingName = mReadingUsed = false;
	mReadingURL = mReadingDescription = mReadingFileNameParsed = false;
	mReadingFileNameUnParsed = false;
	cerr << "Starting Document" << endl;
	return true;
}

bool SiteHandler::endDocument(void)
{
	mReadingSites = mReadingSite = mReadingName = mReadingUsed = false;
	mReadingURL = mReadingDescription = mReadingFileNameParsed = false;
	mReadingFileNameUnParsed = false;
	cerr << "Ending Document" << endl;
	return true;
}

bool SiteHandler::warning(const QXmlParseException & e)
{
	cerr << "WARNING: at " << e.lineNumber() << "," << e.columnNumber() << endl;
	cerr << "        " << e.message();
	return true;
}

bool SiteHandler::error(const QXmlParseException & e)
{
	cerr << "ERROR: at " << e.lineNumber() << "," << e.columnNumber() << endl;
	cerr << "     " << e.message();
	return false;
}

bool SiteHandler::fatalError(const QXmlParseException & e)
{
	cerr << "FATAL ERROR: at " << e.lineNumber() << "," << e.columnNumber() << endl;
	cerr << "           " << e.message();
	return false;
}

bool SiteHandler::characters(const QString & ch)
{
	if(mReadingSites) {
		if(mReadingSite) {
			if(mReadingUsed) {
				if(ch.compare(QString("yes")) == 0) {
					mSite->setUsed(true);
				}
				else {
					mSite->setUsed(false);
				}
			}
			else if(mReadingName) {
				mSite->setName(ch);
			}
			else if(mReadingURL) {
				mSite->setURL(KURL(ch.stripWhiteSpace()));
			}
			else if(mReadingDescription) {
				mSite->setDescription(ch);
			}
			else if(mReadingFileNameUnParsed) {
				mSite->setFileNameUnParsed(ch);
			}
			else if(mReadingFileNameParsed) {
				mSite->setFileNameParsed(ch);
			}
		}
	}
	
	return true;
}

bool SiteHandler::startElement(const QString &, const QString &, const QString & qName, const QXmlAttributes &atts)
{
	if(qName.compare(QString("sites")) == 0) {
		mReadingSites = true;
	}
	else if(qName.compare(QString("site")) == 0) {
		QDate d;
		QTime t;
		int year=0, month=0, day=0,hour=0,minute=0,second=0;
		for(int i=0; i<atts.length(); i++) {
			if(atts.qName(i).compare(QString("year")) == 0) {
				year = atts.value(i).toInt();
			}
			else if(atts.qName(i).compare(QString("month")) == 0) {
				month = atts.value(i).toInt();
			}
			else if(atts.qName(i).compare(QString("day")) == 0) {
				day = atts.value(i).toInt();
			}
			else if(atts.qName(i).compare(QString("hour")) == 0) {
				hour = atts.value(i).toInt();
			}
			else if(atts.qName(i).compare(QString("minute")) == 0) {
				minute = atts.value(i).toInt();
			}
			else if(atts.qName(i).compare(QString("second")) == 0) {
				second = atts.value(i).toInt();
			}
		}
		t.setHMS(hour,minute,second);
		d.setYMD(year,month,day);
		mReadingSite = true;
		mSite = new Site();
		mSite->setAddedDate(d);
		mSite->setAddedTime(t);
	}
	else if(qName.compare(QString("used")) == 0) {
		mReadingUsed = true;
	}
	else if(qName.compare(QString("name")) == 0) {
		mReadingName = true;
	}
	else if(qName.compare(QString("url")) == 0) {
		mReadingURL = true;
	}
	else if(qName.compare(QString("description")) == 0) {
		mReadingDescription = true;
	}
	else if(qName.compare(QString("filename")) == 0) {
		mReadingFileNameParsed = true;
	}
	else if(qName.compare(QString("unparsed_filename")) == 0) {
		mReadingFileNameUnParsed = true;
	}
	return true;
}

bool SiteHandler::endElement(const QString &, const QString &, const QString & qName)
{
	if(qName.compare(QString("sites")) == 0) {
		mReadingSites = false;
	}
	else if(qName.compare(QString("site")) == 0) {
		//if(!isDuplicate(mSite))
		mSites.append(mSite);
		mReadingSite = false;
	}
	else if(qName.compare(QString("used")) == 0) {
		mReadingUsed = false;
	}
	else if(qName.compare(QString("name")) == 0) {
		mReadingName = false;
	}
	else if(qName.compare(QString("url")) == 0) {
		mReadingURL = false;
	}
	else if(qName.compare(QString("description")) == 0) {
		mReadingDescription = false;
	}
	else if(qName.compare(QString("filename")) == 0) {
		mReadingFileNameParsed = false;
	}
	else if(qName.compare(QString("unparsed_filename")) == 0) {
		mReadingFileNameUnParsed = false;
	}
	return true;
}

void SiteHandler::makeQueue(QQueue<FileTransfer> & files)
{
	files.clear();
	Site *s = mSites.first();
	while(s) {
		if(s->used()) {
			FileTransfer * ft = new FileTransfer();
			ft->setDestination(s->fileNameUnParsed());
			ft->setFinalDestination(s->fileNameParsed());
			ft->setURL(s->URL());
			ft->setFeedName(s->name());
			ft->setIsFeed(true);
			files.enqueue(ft);
		}
		s = mSites.next();
	}
}

void SiteHandler::makeQueue(QQueue<FileTransfer> & files, const QString & unparsedfilename)
{
	if(unparsedfilename.isEmpty() || unparsedfilename.isNull()) {
		makeQueue(files);
		return;
	}
	files.clear();
	Site *s = mSites.first();
	while(s) {
		if(s->used() && s->fileNameUnParsed().compare(unparsedfilename) == 0) {
			FileTransfer * ft = new FileTransfer();
			ft->setDestination(s->fileNameUnParsed());
			ft->setFinalDestination(s->fileNameParsed());
			ft->setURL(s->URL());
			ft->setFeedName(s->name());
			ft->setIsFeed(true);
			files.enqueue(ft);
		}
		s = mSites.next();
	}
}

void SiteHandler::makeSiteList(QList<Site> & sites)
{
	sites.clear();
	Site *s = mSites.first();
	while(s) {
		Site *ns = new Site();
		ns->setName(s->name());
		ns->setURL(s->URL());
		ns->setDescription(s->description());
		ns->setAddedDateTime(s->addedDateTime());
		sites.append(ns);
		s = mSites.next();
	}
}

bool SiteHandler::displayNews(KConfig *mConfig, QString & mStr, QProgressBar * mProgress, KApplication * mKApp, MyChannelBar *mIcons)
{
	SchemeTree tree;
	SchemeParser::parseFile(QDir::homeDirPath() + QString("/.krss/schemes/currentscheme"), tree);
	
	mIcons->setItemColours(tree.myChannelBarNewFeedColour(), tree.myChannelBarTextColour());
	
	FilterHandler filters;
	filters.readFilters();
	
	mConfig->setGroup(QString("General"));
	QString buttonText = mConfig->readEntry(QString("Button Text"), i18n("Go"));
	
	// Load the scheme stuff for the MyChannelBar
	QPalette pal = mIcons->viewport()->palette();
	//mIcons->viewport()->setBackgroundMode(QWidget::PaletteBackground);
		
	if(tree.myChannelBarUseBackgroundImage()) {
		QBrush brush;
		brush.setPixmap(QPixmap(tree.myChannelBarBackgroundImageUrl().path()));
		//pal.setBrush(QColorGroup::Background, QBrush(QColor(0, 0, 0), QPixmap(tree.myChannelBarBackgroundImageUrl().path())));
		pal.setBrush(QColorGroup::Background, brush);
		mIcons->viewport()->setBackgroundMode(QWidget::NoBackground);
	}
	else {
		pal.setBrush(QColorGroup::Background, tree.myChannelBarBackgroundColour());
		mIcons->viewport()->setBackgroundMode(QWidget::PaletteBackground);
	}
	mIcons->viewport()->setPalette(pal);
	
	// Start the actual loading of the sites
		
	mStr = QString("<html><head><title>Krss Feed Display</title>");
	mStr.append(QString("<style type=\"text/css\" media=\"screen, print\">"));
	
	tree.makeCSS(mStr);
	
	mConfig->setGroup(QString("Style Sheet"));
	if(mConfig->readBoolEntry(QString("Use Style Sheet"), true)) {
		mStr.append(mConfig->readEntry(QString("Style Sheet"), QString("body { font-family: \"Times New Roman\", Times, serif;\n       background-color: white; color: black;\n       margin: 1em 1em 1em 1em; }\np, .version {\n    margin-left: 2.5em; }\n\ndl, blockquote {\n     margin-left: 3.2em; }\n\nul, ol {\n     margin-left: 4.5em; }\n\npre { font-family: \"Courier New\", monospace;\n      padding: 1em;\n      margin: 0.2em 2.5em 0.2em 3em;\n      background-color: #efeff5;\n      border: 1px solid #cfcfcf;\n      white-space: pre;\n }\n\nh1, h2, h3, h4 {font-family: Arial, Helvetica, sans-serif;}\nh1 { font-size: 2.2em; }\n\nh2 { font-size: 1.6em; }\n\nh2, h3, em { font-weight: bold;\n             margin-left: 0.3em; }\n\ndd.indent { padding-left: 1em; }\n\n.example-dc { color: blue; }\n.example-taxo { color: green; }\n.example-syn, .example-other { color: red; }\n.example-co { color: purple; }\n.example-ti { color: #ff9900; /* orange */ }\n.example-rdf { color: #996633; /* brown */}\n\npre {\n  white-space: pre;\n}")));
	}
	
	mStr.append(QString("</style></head>"));
	
	mStr.append(QString("<BODY class=\"BODY_KRSS\">"));
	
	mConfig->setGroup(QString("MyChannelBar"));
	int imagex = mConfig->readNumEntry(QString("Image X"), 16);
	int imagey = mConfig->readNumEntry(QString("Image Y"), 16);
	
	unsigned int total = mSites.count();
	if(total == 0) total = 1;
	unsigned int count = 0;
	
	if(mProgress) mProgress->reset();
	
	Site *s = mSites.first();
	bool mHaveSites = false;
	while(s) {
		if(s->used()) {
			s->displayNews(mStr, tree, filters, buttonText, mIcons, imagex, imagey);
			mHaveSites = true;
		}
		s = mSites.next();
		count++;
		if(mProgress)	mProgress->setProgress((count * 100) / total);
		if(mKApp)	mKApp->processEvents(100);
	}
	mStr.append(QString("<br><br><br><br><br><br><br><br>End of the news<br>"));
	mStr.append("</BODY></html>");
	return mHaveSites;
}

bool SiteHandler::isDuplicate(Site *s)
{
	if(s->URL().url().isEmpty() || s->URL().url().isNull()) {
		return true;
	}
	
	for(Site *si = mSites.first(); si != 0; si = mSites.next()) {
		if(si->URL().url().compare(s->URL().url()) == 0) {
			return true;
		}
	}
	
	return false;
}

bool SiteHandler::isDuplicate(const KURL & url)
{
	for(Site *si = mSites.first(); si != 0; si = mSites.next()) {
		if(si->URL().url().compare(url.url()) == 0) {
			return true;
		}
	}
	
	return false;
}

void SiteHandler::addSite(Site *site)
{
	if(!site) return;
	
	if(site->fileNameUnParsed().isEmpty() || site->fileNameUnParsed().isNull())
		site->makeFileName();
		
	if(site->fileNameParsed().isEmpty() || site->fileNameParsed().isNull())
		site->makeFileName();
	
	mSites.append(site);
}
