// -*- C++ -*-
/*****************************************************************************
 * KWATCH --- KDE Log File Viewer
 *
 * $Id: klogwidget.cpp,v 0.25 2004/06/22 20:26:27 mj Exp $
 *
 * QMultiLineEdit based widget for viewing log files
 *
 *****************************************************************************
 * Copyright (C) 2000-2003
 *  _____ _____
 * |     |___  |   Martin Junius             Internet:  mailto:mj@m-j-s.net
 * | | | |   | |   Radiumstr. 18                        http://www.m-j-s.net/
 * |_|_|_|@home|   D-51069 Koeln, Germany
 *
 * Based on kwatch 0.3, with the following copyright notice:
 *
 * Kwatch is Copyright (C) 1997,1998 Ralph C. Weichert and can be
 * distributed freely under the terms of the GNU General Public License
 * (see file COPYING)
 *****************************************************************************
 * This file is part of KWATCH.
 *
 * KWATCH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * KWATCH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KWATCH; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *****************************************************************************/


#include <string.h>

#include <qcursor.h>
#include <qdragobject.h>
#include <qregexp.h>

#include <kurl.h>
#include <kapp.h>
#include <klocale.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <kaudioplayer.h>

#include "klogwidget.h"
#include "prefs.h"
#include "kwatchmsg.h"
#include "iptables.h"



/*
 * KLogFile
 */
KLogFile::KLogFile(const QString &name, QObject *parent, kwatchConf &cf)
    : QObject(parent), m_cf(cf)
{
    m_buf  = new char[m_cf.bufsize];
    m_file = new QFile(name);
    activate();
}


KLogFile::~KLogFile()
{
    delete m_buf;
    delete m_file;
}



void KLogFile::activate()
{
    if(m_file->isOpen())
	return;
//    m_file->open( IO_ReadOnly | IO_Raw);
    m_file->open( IO_ReadOnly );
    if(m_file->isOpen())
        // Go to EOF
        m_file->at(m_file->size());
    else {
        // Try again in a minute
        QTimer::singleShot( 60000, this, SLOT(activate()) );
    }
}



bool KLogFile::readMore()
{
    if( !m_file->isOpen() )
	return FALSE;
    
    QString line, text;
    int len;

    // read lines, check them individually, but pass as much as can
    // be read to the log viewer to avoid slow scrolling
    do 
    {
	len = m_file->readLine(line, m_cf.bufsize);
	if(len>0 && !line.isEmpty() && line!="\n" && line!="\r\n") 
	{
	    emit checkLine(this, line);
	    text += line;
	}
	
    }
    while(len > 0);
    
    if(!text.isEmpty())
    {
        emit moreLines(this, text);
        return TRUE;
    }

    return FALSE;
}




/*
 * KLogWidget
 */
KLogWidget::KLogWidget(QWidget *parent, kwatchConf &cf)
    : QTextEdit(parent),
      m_cf(cf), m_popup(0),
      m_timer(0), m_files(0), m_current(0),
      m_r(0), m_c(0)
{
    // set QMultiLineEdit parameters
    setReadOnly(TRUE);

    // set configuration parameters
    updateConf();

    // accept drops
    setAcceptDrops(true);

    setTextFormat(PlainText);
    setWordWrap(NoWrap);
    
    // start timer
    m_timer = new QTimer(this);
    m_timer->start(0);
    connect(m_timer, SIGNAL(timeout()), SLOT(timeout()));

    m_idle  = new QTimer(this);
    m_idle->start(m_cf.idleTimeout * 1000);
    connect(m_idle, SIGNAL(timeout()), SLOT(timeout_idle()));
    
    for(int i=0; i<MAXKWATCHMSG; i++)
	m_msgbox[i] = 0;
    m_msgi = 0;
}



KLogWidget::~KLogWidget()
{
    if(m_timer)
	delete m_timer;
    if(m_idle)
	delete m_idle;
    if(m_files)
	delete m_files;

    for(int i=0; i<MAXKWATCHMSG; i++)
    {
	delete m_msgbox[i];
	m_msgbox[i] = 0;
    }
}



void KLogWidget::setPopupMenu(QPopupMenu *popup)
{
    m_popup = popup;
}



void KLogWidget::clear()
{
    QTextEdit::clear();
    m_r = m_c = 0;
}



void KLogWidget::mousePressEvent(QMouseEvent *e)
{
    if(m_popup && e->button() == RightButton)
    {
	m_popup->popup(QCursor::pos());
    }
    else 
    {
	QTextEdit::mousePressEvent(e);
    }
}



QPopupMenu *KLogWidget::createPopupMenu(const QPoint &pos)
{
//    QPopupMenu *menu = QTextEdit::createPopupMenu(pos);

    if(m_popup)
//	m_popup->popup(pos);
	m_popup->popup(QCursor::pos());

    return 0;
}



QPopupMenu *KLogWidget::createPopupMenu()
{
    return QTextEdit::createPopupMenu();
}



void KLogWidget::dropEvent(QDropEvent *ev)
{
    if(QUriDrag::canDecode(ev))
    {
	QStringList files;
 
	if(QUriDrag::decodeLocalFiles(ev, files))
	{
	    QStringList::Iterator it;
	    for (it=files.begin(); it!=files.end(); it++)
	    {
		QString f = (*it);
		// add to list and open
		if (m_cf.logfiles.find(f) == m_cf.logfiles.end()) {
		    m_cf.logfiles.append(f);
		    addFile(f);
		}
	    }
	}
    }
}

void KLogWidget::dragEnterEvent(QDragEnterEvent *ev)
{
    ev->accept(QUriDrag::canDecode(ev));
}
    


void KLogWidget::timeout()
{
    bool text_added = FALSE;
    QListIterator<KLogFile> it(*m_files);

    for(; it.current(); ++it)
        if(it.current()->readMore())
            text_added = TRUE;
    if(text_added)
    {
        m_timer->changeInterval(0);
	m_idle ->changeInterval(m_cf.idleTimeout * 1000);
    }
    else
        m_timer->changeInterval(1000);
}



void KLogWidget::timeout_idle()
{
    printf("KLogWidget::timeout_idle() - reopen files\n");
    // reopen files after idle timer timeout
    reopen();
}



void KLogWidget::moreText(KLogFile *f, QString s)
{
    if(s.isEmpty())
	return;

    QString more = s;

    if(f && f != m_current)
    {
        m_current = f;
        more.prepend(formatName(f->m_file->name()));
    }

    // insert new text
    append(more);
    moveCursor(MoveEnd, FALSE);
    ensureCursorVisible();

    // remove lines at beginning if necessary
    int l = paragraphs() - m_cf.maxlines;
    if(l > 0) 
    {
	// disable updates while removing to make this a bit faster
	// for several lines/paragraphs at once
	setUpdatesEnabled(FALSE);
	while(l-- > 0)
	    removeParagraph(0);
	moveCursor(MoveEnd, FALSE);
	ensureCursorVisible();
	setUpdatesEnabled(TRUE);
	update();
    }
}



QString KLogWidget::formatName(const QString &name)
{
    QString std = "---------------[" + name + "]---------------\n";

#ifndef KDE2
    return std;
#else
    QString s = textLine(0);
    if(s.isEmpty() || s.isNull())
	return std;
    
    int l    = textWidth(0);
    int cols = width() / (l / s.length());
    int n    = cols - name.length() - 15 - 1 - 1 - 4;
    if(n < 1)
	return std;
    
    QString f;
    f.fill('-', n);
    
    return "---------------[" + name + "]" + f + "\n";
#endif
}



void KLogWidget::updateConf()
{
    // list of files
    if(m_files)
	delete m_files;
    m_files = new QList<KLogFile>();
    m_files->setAutoDelete(TRUE);
    m_current = NULL;

    QStringList::Iterator it;
    for (it=m_cf.logfiles.begin(); it!=m_cf.logfiles.end(); it++)
        addFile((*it));

    // set font, fg, bg colors
    kwatchPrefs::setFontFgBg(this, m_cf.font, m_cf.fg, m_cf.bg);
}



void KLogWidget::reopen()
{
    QListIterator<KLogFile> it(*m_files);
    KLogFile *f;

    for(; (f = it.current()); ++it)
    {
        if(f->m_file->isOpen())
	    f->m_file->close();
        f->activate();
    }
}



void KLogWidget::addFile(const QString &name)
{
    KLogFile *f = new KLogFile(name, this, m_cf);
    if(! f->m_file->isOpen())
	moreText(f, "ERROR: cannot open file\n");
    connect(f, 
	    SIGNAL(checkLine(KLogFile *, QString)),
	    SLOT(logcheck(KLogFile *, QString))    );
    connect(f, 
	    SIGNAL(moreLines(KLogFile *, QString)),
	    SLOT(moreText(KLogFile *, QString))    );
    m_files->append(f);
}



void KLogWidget::logcheck(KLogFile *f, QString line)
{
    QString name = f->m_file->name();
    int len;
    int index;

    static IPTablesLog last;
    IPTablesLog log;

    QString ts = line.mid(0, 15);
    
    
    /***** ISDN incomming call **********************************************/
    QRegExp isdn("isdn_tty: call from [0-9]* ->", false);
    if(m_cf.enableCall && (index = isdn.match(line, 0, &len)) != -1) 
    {
	// +/- 20 = "isdn_tty: call from ", -3 = " ->"
	QString num = "0" + line.mid(index+20, len-20-3);
	if(num == "00")
	    num = i18n("unknown number");
//	kdDebug() << "call from ----- " << num << " -----" << endl;

	// called party number
	QRegExp r2(" -> [0-9]*", false);
	QString num2;
	if( (index = r2.match(line, 0, &len)) != -1 )
	    num2 = i18n("to phone ") + line.mid(index+4, len-4);

	// play sound if configured
	if(!m_cf.soundCall.isNull() && !m_cf.soundCall.isEmpty())
	    KAudioPlayer::play(m_cf.soundCall);

	// show dialog
	kwatchMsg *msgbox = nextMsg();
	msgbox->setCaption("Incoming call");
	msgbox->set1(ts + i18n("\nIncoming call from"));
	msgbox->set2(num);
	msgbox->set3(num2);
	msgbox->show();

	return;
    }
    
    /***** IP packets logged by firewall iptables rules *********************/
    // kernel 2.4.x, iptables
    QRegExp fw("kernel: IN=.* OUT=.* MAC=.*", false);
    if(m_cf.enableLog && (index = fw.match(line, 0, &len)) != -1) 
    {
	log.parseLog(line.mid(index+8 /**"kernel: "**/));
	if(log.eq(last))
	    return;
	
	// play sound if configured
	if(!m_cf.soundLog.isNull() && !m_cf.soundLog.isEmpty())
	    KAudioPlayer::play(m_cf.soundLog);

	// show dialog
	kwatchMsg *msgbox = nextMsg();
	msgbox->setCaption("Connection attempt");
	msgbox->set1(ts + i18n("\nConnection attempt from"));
	msgbox->set2(log.srcAddr());
	msgbox->set3(i18n("to port ") + log.dstPort());
	msgbox->show();

	last = log;
	return;
    }
    
}



kwatchMsg *KLogWidget::nextMsg()
{
    m_msgi = (++m_msgi % MAXKWATCHMSG);
    delete m_msgbox[m_msgi];
    m_msgbox[m_msgi] = new kwatchMsg(this);

    return m_msgbox[m_msgi];
}



#include "klogwidget.moc"
