/*--License:
	Kyra Sprite Engine
	Copyright Lee Thomason (Grinning Lizard Software) 2001-2002
	www.grinninglizard.com/kyra
	www.sourceforge.net/projects/kyra

	Kyra is provided under 2 licenses:

	- The GPL, with no additional restrictions.
	- The LGPL, provided you display the Kyra splash screen, described below.


--- GPL License --
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	The full text of the license can be found in license.txt


--- LGPL License --
  **Provided you kindly display the Kyra splash screen (details below), 
	you	may use the LGPL license:**

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	The full text of the license can be found in lgpl.txt


--- Kyra Splash Screen.

	It would be appreciate if you display the Kyra splash screen when using
	either license, however it is only required for the LGPL. All the
	resources for the splash are compiled into the library, and it can be
	accessed through the following API:

		KrEngine::StartSplash
		KrEngine::UpdateSplash
		KrEngine::EndSplash

	Full documentation is provided with the KrEngine class. The splash screen
	should be displayed for 2 seconds.

	Thank you.
*/


#include "engine.h"
#include "../util/glperformance.h"
#ifdef KYRA_SUPPORT_OPENGL
	#include "SDL_opengl.h"
	#include "ogltexture.h"
#endif
#include "splash.h"
#include "canvasresource.h"
#include "canvas.h"


// Yes, we use 'this' in a member initialization list. Quote from
// Microsoft help: "This is a level-1 warning when Microsoft extensions 
// are enabled (/Ze) and a level-4 warning otherwise." I'm sure that
// made sense to someone. -lee
#pragma warning ( disable : 4355 )

// Check some SDL stuff here:
#if !( SDL_VERSION_ATLEAST(1, 2, 0) )
	#error Kyra requires SDL 1.2.0 or greater.
#endif


/*static*/ int KrEngine::maxOglTextureSize = 0;

KrEngine::KrEngine( SDL_Surface* _screen )
	: paintInfo( _screen )
{
	KrRect bounds;
	bounds.Set( 0, 0, _screen->w - 1, _screen->h - 1 );

	Init( _screen, 1, &bounds, 0 );
}


KrEngine::KrEngine( SDL_Surface* _screen, const KrRect& bounds, const KrRGBA* extra )
	: paintInfo( _screen )
{
	Init( _screen, 1, &bounds, extra );
}


KrEngine::KrEngine( SDL_Surface* _screen, int _nWindows, const KrRect* bounds, const KrRGBA* extra )
	  : paintInfo( _screen )
{
	Init( _screen, _nWindows, bounds, extra );
}


void KrEngine::Init(	SDL_Surface* _screen,
						int _nWindows, 
						const KrRect* bounds, 
						const KrRGBA* extra )
{
	screen = _screen;
	nWindows = _nWindows;

	splashStart = 0;
	splashVault = 0;
	splash = splashText = 0;

	windowBounds.Set( 0, 0, screen->w-1, screen->h-1 );
	extraBackground.Set( 0, 0, 0, 255 );

	// If this assert is thrown, increase KR_MAX_WINDOWS to an
	// appropriate value and re-compile.
	GLASSERT( nWindows <= KR_MAX_WINDOWS );

	// fullScreenUpdate draws *outside* of the windows.
	needFullScreenUpdate = ( extra != 0 );

	if ( extra )
	{
		extraBackground = *extra;

		if ( !paintInfo.OpenGL() )
		{
			U32 color = SDL_MapRGB( screen->format, extra->c.red, extra->c.green, extra->c.blue );
			SDL_FillRect( screen, 0, color ); 
		}	
	}

	int i;
	for( i=0; i<nWindows; ++i )
	{
		// Default to filling the background to black.
		fillBackground[i] = true;
		backgroundColor[i].Set( 0, 0, 0, 255 );

		// Set the screenbounds to a window.
		screenBounds[i] = bounds[i];

		GLASSERT( bounds[i].xmin >= 0 );
		GLASSERT( bounds[i].ymin >= 0 );
		GLASSERT( bounds[i].xmax < screen->w );
		GLASSERT( bounds[i].ymax < screen->h );

		// Really aweful bugs result if the indivual windows
		// aren't clipped to the screen.
		screenBounds[i].DoClip( windowBounds );

		// Initialize the DR to repaint everything and clip to the screen
		// IMPORTANT: Set clip before adding any rectangles.
		dirtyRectangle[i].SetClipping( screenBounds[i] );
		dirtyRectangle[i].AddRectangle( screenBounds[i] );
	}

	// Check that none overlap.
	#ifdef DEBUG
		int j;
		for( i=0; i<nWindows; ++i )
		{
			for( j=i+1; j<nWindows; ++j )
			{
				GLASSERT( !bounds[i].Intersect( bounds[j] ) );
			}
		}
	#endif

	if ( paintInfo.openGL )
		InitOpenGL();

	// Initialization of stuff that has "this" usage.
	vault = new KrResourceVault;
	tree = new KrImageTree( this );
}


void KrEngine::InitOpenGL()
{
	#ifdef KYRA_SUPPORT_OPENGL
	GLASSERT( KrTextureManager::TextureIndex() == 0 );

	int w = windowBounds.Width();
	int h =  windowBounds.Height();
	glViewport( 0, 0, w, h );
	glClearColor( extraBackground.Redf(),
				  extraBackground.Greenf(),
				  extraBackground.Bluef(),
				  255.0f );

	glClearDepth(1.0);

	// The depth buffer isn't actually used. This only occured to me at the end:
	// Kyra sorts everything and draws in order...so you don't need the depth testing.
	// Oddly...if you turn this on, it creates a bug in 16bit mode. I'de really
	// like to understand that.  -- lee
	//	glDepthFunc(GL_LESS);
	//	glEnable(GL_DEPTH_TEST);	// disabled by default.

	// TODO: after reading the above comment...we don't need the openGLZ parameter...
	//		 maybe. I suspect you could implement openGL without the depth buffer
	//		 in which case you need the z depths.
	
	glShadeModel(GL_FLAT);		// Don't need smooth for 2D.

	// Create and reset the projection matrix
	glMatrixMode(GL_PROJECTION);
	
	// This matrix is set up to map pixels to screen the same way the bitbuffer does.
	// Trick from the OpenGL docs.

	glLoadIdentity();
	glOrtho( 0, w,
			 h, 0, 
			 -kKrOglDistance, kKrOglDistance );

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	glTranslatef(0.375, 0.375, 0.0);

	// A more reasonable mode to start in.
	glMatrixMode(GL_MODELVIEW);

	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

	// Enable the texturing and the blending mode needed.
	glEnable( GL_TEXTURE_2D );
	glEnable( GL_BLEND );
	glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);

	GLASSERT( glGetError() == GL_NO_ERROR );
	#endif
}


void KrEngine::FillBackground( const KrRGBA* fillColor )
{
	for( int i=0; i<nWindows; ++i )
	{
		FillBackgroundWindow( i, fillColor );
	}
}


void KrEngine::FillBackgroundWindow( int i, const KrRGBA* fillColor )
{
	if ( fillColor )
	{
		fillBackground[i]   = true;
		backgroundColor[i]	= *fillColor;
		//SDL_MapRGB( screen->format, fillColor->c.red, fillColor->c.green, fillColor->c.blue );
	}
	else
	{
		fillBackground[i] = false;
		backgroundColor[i].Set( 0, 0, 0 );
	}
	dirtyRectangle[i].AddRectangle( screenBounds[i] );
}


//void KrEngine::ClearScreen( int red, int green, int blue )
//{
//	U32 color = SDL_MapRGB( screen->format, red, green, blue );
//	SDL_FillRect( screen, 0, color ); 
//
//	SDL_UpdateRect( screen, 0, 0, 0, 0 );
//	Tree()->Root()->Invalidate( KR_ALL_WINDOWS );
//}



#if defined( DRAWDEBUG_RLE ) || defined( DRAWDEBUG_BLTRECTS )
	extern int debugFrameCounter;
#endif


void KrEngine::UpdateScreen( GlDynArray< KrRect >* rectArray )
{
	if ( paintInfo.openGL )
	{
		SDL_GL_SwapBuffers();
	}
	else
	{
		if ( rectArray->Count() == 0 )
			return;

		SDL_Rect* sdlrect;
		sdlrect = new SDL_Rect[ rectArray->Count() ];

		for( unsigned i=0; i<rectArray->Count(); i++ )
		{
			sdlrect[i].x = rectArray->Item( i ).xmin;
			sdlrect[i].y = rectArray->Item( i ).ymin;
			sdlrect[i].w = rectArray->Item( i ).Width();
			sdlrect[i].h = rectArray->Item( i ).Height();
			
			GLASSERT( sdlrect[i].x >= 0 );
			GLASSERT( sdlrect[i].y >= 0 );
			GLASSERT( sdlrect[i].w > 0 && sdlrect[i].w <= screen->w );
			GLASSERT( sdlrect[i].h > 0 && sdlrect[i].h <= screen->h );
			GLASSERT( sdlrect[i].x + sdlrect[i].w <= screen->w );
			GLASSERT( sdlrect[i].y + sdlrect[i].h <= screen->h );
		}

//		GLOUTPUT( "Updating %d rects\n", rectArray->Count() );
		SDL_UpdateRects( screen, rectArray->Count(), sdlrect );
		delete [] sdlrect;
	}
}


void KrEngine::Draw( bool updateRect, GlDynArray< KrRect >* _rectangles )
{
	GlDynArray< KrRect > rectArrayOnStack;

	#if defined( DRAWDEBUG_RLE ) || defined( DRAWDEBUG_BLTRECTS )
		debugFrameCounter++;
	#endif

//	GLOUTPUT( "Engine::Draw Walk\n" );
	tree->Walk();

	GlDynArray< KrRect >* rectArray = 0;
	if ( !paintInfo.openGL )
	{
		// Queue up the rectangles that will be used to blit to screen:

		// We either use the passed in rectangles,
		// or the one here on the stack. Set the pointer
		// rectArray to the right thing.
		if ( _rectangles )
			rectArray = _rectangles;
		else
			rectArray = &rectArrayOnStack;
	
		rectArray->Clear();

		if ( needFullScreenUpdate )
		{
			needFullScreenUpdate = false;			
			KrRect rect;
			rect.Set( 0, 0, screen->w-1, screen->h-1 );
			rectArray->PushBack( rect );
		}
		else
		{
			for ( int i=0; i<nWindows; ++i )
			{
				for ( int j=0; j<dirtyRectangle[i].NumRect(); ++j )
				{
					rectArray->PushBack( dirtyRectangle[i].Rect(j) );
				}
			}
		}	
	}
	
	if ( paintInfo.openGL )
	{
		#ifdef KYRA_SUPPORT_OPENGL

		// OpenGL drawing
		glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

		for( int j=0; j<nWindows; ++j )
		{
			if ( fillBackground[j] )
			{
				glBindTexture( GL_TEXTURE_2D, 0 );
				glColor4f( backgroundColor[j].Redf(), backgroundColor[j].Greenf(), backgroundColor[j].Bluef(), 1.0f );
				glBegin( GL_QUADS );
				{
					glVertex3i( screenBounds[j].xmin, screenBounds[j].ymin, 0 );
					glVertex3i( screenBounds[j].xmin + screenBounds[j].Width(), screenBounds[j].ymin, 0 );
					glVertex3i( screenBounds[j].xmin + screenBounds[j].Width(), screenBounds[j].ymin + screenBounds[j].Height(), 0 );
					glVertex3i( screenBounds[j].xmin, screenBounds[j].ymin + screenBounds[j].Height(), 0 );
				}
				glEnd();
			}

			bool clipping = ( screenBounds[j] != windowBounds );

			if ( clipping )
			{
				glEnable(GL_CLIP_PLANE0);
				glEnable(GL_CLIP_PLANE1);
				glEnable(GL_CLIP_PLANE2);
				glEnable(GL_CLIP_PLANE3);

				double plane0[4] = { 1.0, 0.0, 0.0, -screenBounds[j].xmin };
				double plane1[4] = { -1.0, 0.0, 0.0, (screenBounds[j].xmin + screenBounds[j].Width() ) };
				double plane2[4] = { 0.0, 1.0, 0.0, -screenBounds[j].ymin };
				double plane3[4] = { 0.0, -1.0, 0.0, (screenBounds[j].ymin + screenBounds[j].Height() ) };

				glClipPlane( GL_CLIP_PLANE0, plane0 );
				glClipPlane( GL_CLIP_PLANE1, plane1 );
				glClipPlane( GL_CLIP_PLANE2, plane2 );
				glClipPlane( GL_CLIP_PLANE3, plane3 );
			}

			tree->DrawWalk( screenBounds[j], &paintInfo, j );

			if ( clipping )
			{
				glDisable(GL_CLIP_PLANE0);
				glDisable(GL_CLIP_PLANE1);
				glDisable(GL_CLIP_PLANE2);
				glDisable(GL_CLIP_PLANE3);
			}
			dirtyRectangle[j].Clear();
		}
		UpdateScreen( 0 );
		#else
			// No openGl support, but openGl surface used
			GLASSERT( 0 );
		#endif
	}
	else
	{
		// Bitmap drawing.

		// Draw the background, if necessary. Then
		// do a draw walk for every DR.
		for( int win=0; win<nWindows; ++win  )
		{
			for( int i=0; i<dirtyRectangle[win].NumRect(); ++i )
			{
				const KrRect& rect = dirtyRectangle[win].Rect( i );

				// Draw the background.
				//GLASSERT( fillBackground[j] );
				if ( fillBackground[win] )
				{
					SDL_Rect sdlrect = { rect.xmin, rect.ymin, rect.Width(), rect.Height() };
					U32 sdlColor = SDL_MapRGB( screen->format,	backgroundColor[win].c.red,
																backgroundColor[win].c.green,
																backgroundColor[win].c.blue );
					//GLASSERT( sdlColor == 0 );
					SDL_FillRect( screen, &sdlrect, sdlColor );
				}
				tree->DrawWalk( rect, &paintInfo, win );

				/*
				#ifdef DRAWDEBUG_BLTRECTS
					KrPainter painter( &paintInfo );
					painter.DrawBox( rect.xmin, rect.ymin, rect.Width(), rect.Height(), 200, 0, 0 );
				#endif
				*/
			}

			#ifdef DRAWDEBUG_BLTRECTS
				dirtyRectangle[win].DrawRects( screen );
			#endif
			dirtyRectangle[win].Clear();
		}

		// The windows and DRs have been walked. Now transfer to physical screen.
		if ( updateRect )
		{
			// Use the composite list of rectangles.
			UpdateScreen( rectArray );
		}
	}
}


void KrEngine::QueryRenderDesc( std::string* desc )
{
	QueryRenderDesc( screen, desc );
}


/* static */ void KrEngine::QueryRenderDesc( SDL_Surface* screen, std::string* desc )
{
	char buf[ 256 ];
	sprintf( buf, "v%d.%d.%d %dbbp Fullscreen=%d %s ",
					KyraVersionMajor, KyraVersionMinor, KyraVersionBuild,
					screen->format->BitsPerPixel,
					( screen->flags & SDL_FULLSCREEN ) ? 1 : 0,
					#ifdef WIN32
						"Win32"
					#elif defined ( linux )
						"Linux"
					#else
						"UnknownPlatform"
					#endif
		);

	char render[256];
	#ifdef KYRA_SUPPORT_OPENGL
	if ( screen->flags & SDL_OPENGL )
	{
		const unsigned char* vendor   = glGetString( GL_VENDOR );
		const unsigned char* renderer = glGetString( GL_RENDERER );
		const unsigned char* version  = glGetString( GL_VERSION );
		sprintf( render, "OpenGL render: Vendor: '%s'  Renderer: '%s'  Version: '%s'",
					vendor, renderer, version );
	} else
	#endif
	{
		sprintf( render, "Software render" );
	}
	*desc = buf;
	desc->append( render );
}


int KrEngine::GetWindowFromPoint( int x, int y )
{
	for( int i=0; i<nWindows; ++i )
	{
		if ( screenBounds[i].Intersect( x, y ) )
			return i;
	}
	return -1;
}


void KrEngine::StartSplash( U32 msec )
{
	splashStart = msec;
	GLASSERT( splashVault == 0 );

	splashVault = new KrResourceVault();
	splashVault->LoadDatFileFromMemory( splash_DAT, splash_SIZE );

	KrSpriteResource* splashRes     = splashVault->GetSpriteResource( "splash" );
	KrSpriteResource* splashTextRes = splashVault->GetSpriteResource( "splashText" );

	splash = new KrSprite( splashRes );
	splashText = new KrSprite( splashTextRes );

	KrRect bounds, boundsText;
	splash->QueryBoundingBox( &bounds, 0 );
	splashText->QueryBoundingBox( &boundsText, 0 );

	tree->AddNode( 0, splash );
	tree->AddNode( 0, splashText );

	splash->SetPos( screenBounds[0].Width() / 2  - bounds.Width() / 2,
					screenBounds[0].Height() / 4 - bounds.Height() / 2 );
	splash->SetZDepth( 5000 );

	splashText->SetPos( screenBounds[0].Width() / 2  - boundsText.Width() / 2,
						splash->Y() + bounds.Height() + 20 );

	splashText->SetZDepth( 5000 );
}


bool KrEngine::UpdateSplash( U32 msec )
{
	U32 delta = msec - splashStart;

	KrColorTransform xcolor;

	if ( delta < 1000 )
	{
		xcolor.SetAlpha( 255 * delta / 1000 );
	}
	splash->SetColor( xcolor );
	splashText->SetColor( xcolor );

	return ( delta >= 2000 );
}


void KrEngine::EndSplash()
{
	tree->DeleteNode( splash );
	tree->DeleteNode( splashText );
	delete splashVault;

	splash = 0;
	splashText = 0;
	splashVault = 0;
}


