/*--License:
	Kyra Sprite Engine
	Copyright Lee Thomason (Grinning Lizard Software) 2001-2002
	www.grinninglizard.com/kyra
	www.sourceforge.net/projects/kyra

	Kyra is provided under 2 licenses:

	- The GPL, with no additional restrictions.
	- The LGPL, provided you display the Kyra splash screen, described below.


--- GPL License --
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	The full text of the license can be found in license.txt


--- LGPL License --
  **Provided you kindly display the Kyra splash screen (details below), 
	you	may use the LGPL license:**

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	The full text of the license can be found in lgpl.txt


--- Kyra Splash Screen.

	It would be appreciate if you display the Kyra splash screen when using
	either license, however it is only required for the LGPL. All the
	resources for the splash are compiled into the library, and it can be
	accessed through the following API:

		KrEngine::StartSplash
		KrEngine::UpdateSplash
		KrEngine::EndSplash

	Full documentation is provided with the KrEngine class. The splash screen
	should be displayed for 2 seconds.

	Thank you.
*/

#include "SDL.h"
#include "SDL_endian.h"
#include "fontresource.h"
#include "engine.h"
#include "encoder.h"
#include "sdlutil.h"


const std::string KrFontResource::fontName = "Font";

KrFontResource::KrFontResource( U32 size, 
								SDL_RWops* data )

	: KrSpriteResource( size, data )
{
	// After the sprite has initialized, read our startIndex and type.
	startIndex = SDL_ReadLE32( data );
	fontType   = SDL_ReadLE16( data );
	GLASSERT( fontType == FIXED || fontType == SFONT );
	space      = SDL_ReadLE32( data );

	#ifdef DEBUG
		GLOUTPUT( "startindex=%d type=%d\n", startIndex, fontType );
	#endif
	
//	actionArr = GetActionByIndex( 0 );
//	GLASSERT( action );

	CalcSpaceWidth();
}


KrFontResource::KrFontResource( const std::string& name,
								KrPaintInfo* info,
								int startingGlyph,
								int addSpaceGlyph,
								int type,
								int length )
	: KrSpriteResource( name )
{
	startIndex = startingGlyph;
	fontType   = type;
	space      = addSpaceGlyph;

	KrAction* action = new KrAction( "NONE" );

	if ( fontType == FIXED )
	{
		int width  = info->width / length;
		int height = info->height;

		for( int i = 0; i < length; i++ )
		{
			action->AddFrame();
			KrRle* rle = action->GetFrame( i );

			rle->Create(	info,
							i * width, 0, width, height,
							i * width, 0,	// the hotspots are absolute coordinates!
							width, height );
		}
	}
	else
	{
		KrPainter painter( info );
		int height = info->height - 1;
		int x = 0;
		int transparent = 0;

		while ( x < info->width )
		{
			x += painter.CalcNotTransparentRun( x, info->width - 1, 0 );
			if ( x < info->width )
				transparent = painter.CalcTransparentRun( x, info->width - 1, 0 );
			else
				transparent = 0;

			if ( x < info->width && transparent > 0 )
			{
				action->AddFrame();
				KrRle* rle = action->GetFrame( action->NumFrames() - 1 );

 				rle->Create(	info,
								x, 1, transparent, height,
								x, 1,	// the hotspots are absolute coordinates!
								transparent, height );
			}
			x += transparent;
		}
	}

	AddAction( action );

	CalcSpaceWidth();
}


KrFontResource::~KrFontResource()
{
}


void KrFontResource::CalcSpaceWidth()
{
	int total = 0;
	for( int i=0; i<actionArr[0]->NumFrames(); ++i )
		total += actionArr[0]->Frame( i ).Delta().x;
	
	spaceWidth = total / actionArr[0]->NumFrames();
	spaceWidth = GlMax( 1, spaceWidth );
}


int KrFontResource::FontWidth( const U16* str )
{
	// Walk the string, check to make sure each character is in
	// the font, and add up the widths.
	const U16* p;
	int width = 0;
	int glyph;

	for( p=str; p && *p; ++p )
	{
		if ( *p == space )
		{
			width += spaceWidth;
		}
		else
		{
			glyph = (*p) - startIndex;
			if ( glyph >= 0 && glyph < actionArr[0]->NumFrames() )
			{
				width += actionArr[0]->Frame( glyph ).Delta().x;
			}
		}
	}
	return width;
}


int KrFontResource::FontWidthN( const U16* str, int n )
{
	// Walk the string, check to make sure each character is in
	// the font, and add up the widths.
	const U16* p;
	int width = 0;
	int glyph;
	int i;

	for( p=str, i=0; p && *p && i < n; ++p, ++i )
	{
		if ( *p == space )
		{
			width += spaceWidth;
		}
		else
		{
			glyph = (*p) - startIndex;
			if ( glyph >= 0 && glyph < actionArr[0]->NumFrames() )
			{
				width += actionArr[0]->Frame( glyph ).Delta().x;
			}
		}
	}
	return width;
}



int KrFontResource::FontWidth1( U16 glyphCode )
{
	if ( glyphCode == space )
	{
		return spaceWidth;
	}

	int glyph = glyphCode - startIndex;
	if ( glyph >= 0 && glyph < actionArr[0]->NumFrames() )
	{
		return actionArr[0]->Frame( glyph ).Delta().x;
	}
	return 0;
}


bool KrFontResource::GlyphInFont( U16 glyphCode )
{
	int glyph = glyphCode - startIndex;
	if ( glyph >= 0 && glyph < actionArr[0]->NumFrames() )
	{
		return true;
	}
	return false;	
}


int  KrFontResource::GlyphToFrame(  U16 glyphCode )
{
	int glyph = glyphCode - startIndex;
	GLASSERT( glyph >= 0 && glyph < actionArr[0]->NumFrames() );
	return glyph;
}


void KrFontResource::Draw( KrPaintInfo* paintInfo,
						   U16 rawGlyphCode,
						   const KrMatrix2& matrix,
						   const KrColorTransform& cForm,
						   const KrRect& clipping,
						   int openGLZ )
{
	#ifdef ASSERT_IF_NOT_CACHED
		GLASSERT( !matrix.IsScaled() );
	#endif

	if ( rawGlyphCode == space )
		return;

	int glyph = rawGlyphCode - startIndex;

	if ( glyph >=0 && glyph < actionArr[0]->NumFrames() )
	{
		actionArr[0]->Draw(	paintInfo,
							glyph,
							matrix,
							cForm,
							clipping,
							openGLZ );
		}
}


void KrFontResource::Save( KrEncoder* encoder )
{
	encoder->StartTag( KYRATAG_FONT );
	
	WriteString( encoder->Stream(), ResourceName() );
	encoder->WriteCached( ResourceName() );

	SDL_WriteLE32( encoder->Stream(), NumActions() );

	for( int i=0; i<NumActions(); ++i )
	{
		actionArr[i]->Save( encoder );
	}
	SDL_WriteLE32( encoder->Stream(), startIndex );
	SDL_WriteLE16( encoder->Stream(), fontType );
	SDL_WriteLE32( encoder->Stream(), space );
	encoder->EndTag();
}

