/*--License:
	Kyra Sprite Engine
	Copyright Lee Thomason (Grinning Lizard Software) 2001-2002
	www.grinninglizard.com/kyra
	www.sourceforge.net/projects/kyra

	Kyra is provided under 2 licenses:

	- The GPL, with no additional restrictions.
	- The LGPL, provided you display the Kyra splash screen, described below.


--- GPL License --
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	The full text of the license can be found in license.txt


--- LGPL License --
  **Provided you kindly display the Kyra splash screen (details below), 
	you	may use the LGPL license:**

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	The full text of the license can be found in lgpl.txt


--- Kyra Splash Screen.

	It would be appreciate if you display the Kyra splash screen when using
	either license, however it is only required for the LGPL. All the
	resources for the splash are compiled into the library, and it can be
	accessed through the following API:

		KrEngine::StartSplash
		KrEngine::UpdateSplash
		KrEngine::EndSplash

	Full documentation is provided with the KrEngine class. The splash screen
	should be displayed for 2 seconds.

	Thank you.
*/


#ifndef IO_COMPRESSOR_INCLUDED
#define IO_COMPRESSOR_INCLUDED

#define RLE_WRITE

#include "SDL.h"
#include "SDL_rwops.h"

#include "../util/glutil.h"
#include "../util/gllist.h"
#include "../util/gldynarray.h"
#include "../util/glstring.h"

#include "rle.h"
#include "kyraresource.h"

class KrResource;
class KrSpriteResource;
class KrTileResource;
class KrFontResource;

/**	The vault is a container for resources. The sequence of
	steps generally followed is:
	
	- A vault is created and resources are read from a stream. 
	- Nhe engine is created, used, and destroyed.
	- The vault is deleted.

	It would be bad to delete resources the engine is using.

	Note that the engine has a "default vault" Engine()->Vault() 
	that it manages for you. It is usually simpler to use that
	Vault rather than manage your own.
*/
class KrResourceVault
{
  public:
	KrResourceVault()			{ memoryPoolRGBA = 0; memoryPoolLine = 0; memoryPoolSegment = 0; }
	virtual ~KrResourceVault();

	/**	Load in a .dat file from disk. Only one dat files can be
		loaded into a vault; but the engine can use multiple vaults.
	*/
	bool LoadDatFile( const char* fileName );

	/**	Load in a .dat file from disk. Only one dat files can be
		loaded into a vault; but the engine can use multiple vaults.
	*/
	bool LoadDatFileFromMemory( void* memory, int size );

	/// How many resources are in the vault?
	int ResourceCount();
	/** Get an interator that can walk all the resources in this vault.
		Useful if you want to list resources, for instance if making
		an editor or allowing the user to choose between available fonts.
	*/
	GlSListIterator< KrResource* > GetResourceIterator();

	/// Add user created resources - generally canvases. Will be owned and deleted by the vault.
	void AddResource( KrResource* resource );

	/// Return a pointer to a resource. Null if the resource isn't found.
	KrResource* GetResource( const std::string& type, const std::string& name );
	/// Return a pointer to a resource. Null if the resource isn't found.
	KrResource* GetResource( U32 type, const std::string& name );
	/// Return a pointer to a resource. Null if the resource isn't found.
	KrResource* GetResource( U32 type, U32 resourceId );

	/// Return a pointer to a resource. Null if the resource isn't found.
	KrSpriteResource*	GetSpriteResource( const std::string& name );
	/// Return a pointer to a resource. Null if the resource isn't found.
	KrSpriteResource*	GetSpriteResource( U32 resourceId );

	/// Return a pointer to a resource. Null if the resource isn't found.
	KrTileResource*		GetTileResource( const std::string& name );
	/// Return a pointer to a resource. Null if the resource isn't found.
	KrTileResource*		GetTileResource( U32 resourceId );

	/// Return a pointer to a resource. Null if the resource isn't found.
	KrFontResource*		GetFontResource( const std::string& name );
	/// Return a pointer to a resource. Null if the resource isn't found.
	KrFontResource*		GetFontResource( U32 resourceId );

	/// Return a pointer to a resource. Null if the resource isn't found.
	KrTextDataResource*		GetTextDataResource( const std::string& name );
	/// Return a pointer to a resource. Null if the resource isn't found.
	KrTextDataResource*		GetTextDataResource( U32 resourceId );

	/// Return a pointer to a resource. Null if the resource isn't found.
	KrBinaryDataResource*	GetBinaryDataResource( const std::string& name );
	/// Return a pointer to a resource. Null if the resource isn't found.
	KrBinaryDataResource*	GetBinaryDataResource( U32 resourceId );

	/// For cached scaling, cache all the resources in this vault.
	virtual void CacheScale( GlFixed xScale, GlFixed yScale );
	/// For cached scaling, free all the scale caches in this vault.
	virtual void FreeScaleCache();

  private:
	bool LoadDatFile( SDL_RWops* stream );

	GlSList< KrResource* > resList;

	// The vault is the owner of these memory allocators:
	GlLinearMemoryPool*	memoryPoolRGBA;
	GlLinearMemoryPool*	memoryPoolLine;
	GlLinearMemoryPool*	memoryPoolSegment;
};
	
	

#endif
