/*--License:
	Kyra Sprite Engine
	Copyright Lee Thomason (Grinning Lizard Software) 2001-2002
	www.grinninglizard.com/kyra
	www.sourceforge.net/projects/kyra

	Kyra is provided under 2 licenses:

	- The GPL, with no additional restrictions.
	- The LGPL, provided you display the Kyra splash screen, described below.


--- GPL License --
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	The full text of the license can be found in license.txt


--- LGPL License --
  **Provided you kindly display the Kyra splash screen (details below), 
	you	may use the LGPL license:**

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	The full text of the license can be found in lgpl.txt


--- Kyra Splash Screen.

	It would be appreciate if you display the Kyra splash screen when using
	either license, however it is only required for the LGPL. All the
	resources for the splash are compiled into the library, and it can be
	accessed through the following API:

		KrEngine::StartSplash
		KrEngine::UpdateSplash
		KrEngine::EndSplash

	Full documentation is provided with the KrEngine class. The splash screen
	should be displayed for 2 seconds.

	Thank you.
*/

#include "eventmanager.h"
#include "widget.h"

/*static*/ KrEventManager* KrEventManager::instance = 0;


KrEventManager::KrEventManager()
{
	keyFocus = -1;
	mouseFocus = 0;
	mouseDown  = false;
	SDL_EnableUNICODE( true );	// Force this! Too many bugs that it's not on.
}

KrEventManager::~KrEventManager()
{
	instance = 0;	// Very important, so we don't screw up if we're re-created.
}


void KrEventManager::AddListener( KrWidget* widget )
{
	if ( widget->IsKeyListener() )
	{
		keyListeners.PushBack( widget );
		if ( keyFocus < 0 )
		{
			ChangeKeyFocus( 0 );
		}
	}
	if ( widget->IsMouseListener() & KrWidget::LEFT_MOUSE )
	{
		mouseListeners.PushBack( widget );
	}
	if ( widget->IsSelectable() )
	{
		selectListeners.PushBack( widget );
	}
}


void KrEventManager::RemoveListener( KrWidget* widget )
{
	keyListeners.FindAndDelete( widget );
	mouseListeners.FindAndDelete( widget );
	selectListeners.FindAndDelete( widget );

	for( unsigned i=0; i<accelListeners.Count(); ++i )
	{
		if ( accelListeners[i].target == widget )
		{
			accelListeners.Remove( i );
			break;
		}
	}
	if ( mouseFocus == widget )
	{
		mouseFocus = 0;
	}
}


void KrEventManager::ChangeKeyFocus( int newFocus )
{
	if ( newFocus >= (int) keyListeners.Count() )
		newFocus -= (int) keyListeners.Count();

	if ( keyListeners.Count() > 0 )
	{
		if ( newFocus != keyFocus )
		{
			if ( GlInRange( keyFocus, (int) 0, int( keyListeners.Count() - 1 ) ) )
				keyListeners[ keyFocus ]->KeyFocus( false );
			if ( GlInRange( newFocus, (int) 0, int( keyListeners.Count() - 1 ) ) )
				keyListeners[ newFocus ]->KeyFocus( true );
			keyFocus = newFocus;
		}
	}
	else
	{
		keyFocus = -1;
	}
}


void KrEventManager::GrabKeyFocus( KrWidget* w )
{
	int i = keyListeners.Find( w );
	if ( i >= 0 )
		ChangeKeyFocus( i );
}


void KrEventManager::SetAccelerator( int keymod, int keysym, KrWidget* target )
{
	unsigned i;
	for ( i=0; i<accelListeners.Count(); ++i )
	{
		if ( accelListeners[i].target == target )
		{
			accelListeners[i].keymod = keymod;
			accelListeners[i].keysym = keysym;
			break;
		}
	}
	if ( i == accelListeners.Count() )
	{
		Accel a = { keymod, keysym, target };
		accelListeners.PushBack( a );
	}
}


void KrEventManager::SelectMe( KrWidget* w )
{
	if ( w->IsSelectable() )
	{
		int group = w->Group();

		for( unsigned i=0; i < selectListeners.Count(); ++i )
		{
			if ( selectListeners[i]->Group() == group )
			{
				if ( selectListeners[i] == w )
					selectListeners[i]->Selected( true );
				else
					selectListeners[i]->Selected( false );
			}
		}
	}
}


void KrEventManager::HandleEvent( const SDL_Event& event, KrEngine* engine )
{
	if ( event.type == SDL_KEYDOWN )
	{
		//	- the tab key changes key focus.
		//	- accelerators are checked
		//	- keys passed through to the handler.

//		#ifdef DEBUG
//		GLOUTPUT( "KeyDown mod=%d sym=%d\n", event.key.keysym.mod, event.key.keysym.sym );
//		#endif

		if (    event.key.keysym.sym == SDLK_TAB 
				 && keyListeners.Count() > 1 )
		{
			if ( event.key.keysym.mod & KMOD_SHIFT )
				ChangeKeyFocus( keyFocus + keyListeners.Count() - 1 );
			else	
				ChangeKeyFocus( keyFocus + 1 );
			return;
		}

		for( unsigned i=0; i<accelListeners.Count(); ++i )
		{
			int sym = accelListeners[i].keysym;
			int mod = accelListeners[i].keymod;

			if (    event.key.keysym.sym == sym
				 && event.key.keysym.mod & mod )
			{
				accelListeners[i].target->Accelerate( true );
				return;
			}
		}

		if ( keyListeners.Count() > 0 )
		{
			keyFocus = GlClamp( keyFocus, 0, int( keyListeners.Count()-1 ) );
			KrWidget* widget = keyListeners[ keyFocus ];

			// Go up the chain until handled.
			while( widget && !widget->KeyEvent( event ) )
			{
				widget = widget->ParentWidget();
			}
		}
	}
	else if ( event.type == SDL_KEYUP )
	{
		// - only accelerates key up
		for( unsigned i=0; i<accelListeners.Count(); ++i )
		{
			if (    event.key.keysym.sym == accelListeners[i].keysym
				 && event.key.keysym.mod & accelListeners[i].keymod )
			{
				accelListeners[i].target->Accelerate( false );
				return;
			}
		}
	}
	else if ( event.type == SDL_MOUSEMOTION )
	{
		GlDynArray<KrImage*> hitArray;
		KrWidget* hit = 0;
		int window = engine->GetWindowFromPoint( event.motion.x, event.motion.y );

		engine->Tree()->HitTest( event.motion.x, event.motion.y,
								 KrImageTree::ALWAYS_INSIDE_BOX, //| GET_ALL_HITS,
								 &hitArray,
								 &window );

		KrVector2T< GlFixed > object;
		for( unsigned i=0; i<hitArray.Count(); ++i )
		{
			KrImNode* parent = hitArray[i]->Parent();
			while( parent )
			{
				if ( parent->ToWidget() )
				{
					hit = parent->ToWidget();
					hit->ScreenToObject( event.motion.x, event.motion.y, &object, window );
					break;
				}
				parent = parent->Parent();
			}
		}

		// 1) Something has the focus. Nothing had it before.
		// 2) Something has the focus, something else had it before.
		// 3) Something loses the focus.
		// 5) The thing with focus gets a move.
		if ( hit && !mouseFocus )
		{
			mouseFocus = hit;
			mouseFocus->MouseIn( mouseDown, true );
			mouseFocus->MouseMove( mouseDown, object.x.ToIntRound(), object.y.ToIntRound() );
		}
		else if ( hit && mouseFocus && mouseFocus != hit )
		{
			mouseFocus->MouseIn( mouseDown, false );
			mouseFocus = hit;
			mouseFocus->MouseIn( mouseDown, true );
			mouseFocus->MouseMove( mouseDown, object.x.ToIntRound(), object.y.ToIntRound() );
		}
		else if ( !hit && mouseFocus )
		{
			mouseFocus->MouseIn( mouseDown, false );
			mouseFocus = hit;
		}
		else if ( hit && hit == mouseFocus )
		{
			GLASSERT( hit == mouseFocus );
			mouseFocus->MouseMove( mouseDown, object.x.ToIntRound(), object.y.ToIntRound() );
		}	
		else if ( !hit && !mouseFocus )
		{
			// nothing to do
		}
		else
		{
			GLASSERT( 0 );
		}					
	}
	else if ( event.type == SDL_MOUSEBUTTONDOWN || event.type == SDL_MOUSEBUTTONUP )
	{
		if ( event.button.button == SDL_BUTTON_LEFT )
		{
			bool down = event.button.state != 0;	// & SDL_BUTTON_LMASK;
			if ( down != mouseDown )
			{
				mouseDown = down;
				if ( mouseFocus )
				{
					int window = engine->GetWindowFromPoint( event.motion.x, event.motion.y );
					KrVector2T< GlFixed > object;
					mouseFocus->ScreenToObject( event.motion.x, event.motion.y, &object, window );

					mouseFocus->MouseClick( mouseDown ? KrWidget::LEFT_DOWN : KrWidget::LEFT_UP, 
											object.x.ToIntRound(), 
											object.y.ToIntRound() );
				}
			}
		}
	}
}
