/*--License:
	Kyra Sprite Engine
	Copyright Lee Thomason (Grinning Lizard Software) 2001-2002
	www.grinninglizard.com/kyra
	www.sourceforge.net/projects/kyra

	Kyra is provided under 2 licenses:

	- The GPL, with no additional restrictions.
	- The LGPL, provided you display the Kyra splash screen, described below.


--- GPL License --
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	The full text of the license can be found in license.txt


--- LGPL License --
  **Provided you kindly display the Kyra splash screen (details below), 
	you	may use the LGPL license:**

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	The full text of the license can be found in lgpl.txt


--- Kyra Splash Screen.

	It would be appreciate if you display the Kyra splash screen when using
	either license, however it is only required for the LGPL. All the
	resources for the splash are compiled into the library, and it can be
	accessed through the following API:

		KrEngine::StartSplash
		KrEngine::UpdateSplash
		KrEngine::EndSplash

	Full documentation is provided with the KrEngine class. The splash screen
	should be displayed for 2 seconds.

	Thank you.
*/

#include "textwidget.h"
#include "eventmanager.h"
#include "../engine/boxresource.h"
#include "../engine/box.h"
#include "../engine/textbox.h"
#include "../engine/fontresource.h"


KrTextWidget::KrTextWidget(	int _width, int _height, 
							bool _drawBorder,
							bool _drawBackground,
							bool _editable,
							const KrScheme& _scheme ) : KrWidget( _scheme )
{
	width = _width;
	height = _height;
	drawBorder = _drawBorder;
	drawBackground = _drawBackground;
	editable = _editable;
	//KrScheme scheme = _scheme;

	plateRes = 0;
	plate = 0;
	textBox = 0;
	cursorPos = 0;
	cursor = 0;
	bevel = 0;
}

KrTextWidget::~KrTextWidget()
{
	// Children are deleted first, else this explodes.
	delete plateRes;
	delete cursorRes;
	delete bevel;
}


void KrTextWidget::AddedtoTree()
{
	KrWidget::AddedtoTree();
	
	holder = new KrImNode();
	Engine()->Tree()->AddNode( this, holder );

	// Add a background plate
	if ( drawBackground )
	{
		plateRes = new KrBoxResource( "TextWidget Plate",
									  width, height,
									  &scheme.primary, 1,
									  KrBoxResource::FILL );

		plate = new KrBox( plateRes );
		Engine()->Tree()->AddNode( holder, plate );
	}

	// Add a text box
	textBox = new KrTextBox( scheme.font, 
							 ( drawBorder ) ? width -2 : width,
							 scheme.font->FontHeight(), 
							 0 );

	textBox->SetPos(	( drawBorder ) ? 1 : 0,
						( height - scheme.font->FontHeight() ) / 2  );
	
	Engine()->Tree()->AddNode( holder, textBox );

	// The cursor
	cursorRes = new KrBoxResource(	"KrTextWidget cursor",
									CURSOR_WIDTH, scheme.font->FontHeight(),
									&scheme.cursor, 1,
									KrBoxResource::FILL );

	cursor = new KrBox( cursorRes );
	cursor->SetVisible( false );
//	cursor->SetColor( scheme.secondary );
	Engine()->Tree()->AddNode( textBox, cursor );

	// Borders
	if ( drawBorder )
	{
		bevel = new KrBevelElement( width, height, scheme );
		bevel->AddToTree( Engine(), holder );
		bevel->DrawIn();
	}
	KrEventManager::Instance()->AddListener( this );
}


bool KrTextWidget::KeyEvent( const SDL_Event &event )
{
	if (	Engine() 
		 && event.type == SDL_KEYDOWN )
	{
		int length = textBox->GetLineLength( 0 );
		GlDynArray<U16> text = textBox->GetText16Array( 0 );

		// Make sure the cursor position is in range.
		int cursorPosTemp = GlClamp( cursorPos, 0, length );
		GLASSERT( cursorPosTemp == cursorPos );
		cursorPos = cursorPosTemp;

		// Line processing keys:
		if ( event.key.keysym.sym == SDLK_BACKSPACE )
		{
			if ( cursorPos > 0 )
			{
				text.Remove( cursorPos-1 );
				textBox->SetText16( text.Memory(), 0 );
				--cursorPos;
			}
		} 
		else if ( event.key.keysym.sym == SDLK_DELETE )
		{
			if ( cursorPos < length )
			{
				text.Remove( cursorPos );
				textBox->SetText16( text.Memory(), 0 );
			}
		} 
		else if ( event.key.keysym.sym == SDLK_LEFT )
		{
			if ( cursorPos > 0 )
				--cursorPos;
		}
		else if ( event.key.keysym.sym == SDLK_RIGHT )
		{
			if ( cursorPos <= textBox->GetLineLength( 0 ) )
				++cursorPos;
		}
		else if ( event.key.keysym.sym == SDLK_HOME )
		{
			cursorPos = 0;
		}
		else if ( event.key.keysym.sym == SDLK_END )
		{
			cursorPos = length;
		} 
		else if ( event.key.keysym.sym == SDLK_RETURN )
		{
			PublishEvent( ACTIVATED, 0, &event, 0, 0 );
		}
		else if ( event.key.keysym.unicode >= ' ' )
		{
			text.Insert( event.key.keysym.unicode, cursorPos );
			textBox->SetText16( text.Memory(), 0 );
			++cursorPos;
		}
		else
		{
//			PublishEvent( KEY, 0, &event, 0, 0 );
			return false;	// we didn't handle.
		}

		PositionCursor();
		return true;			// Almost all keys are handled.
	}
	return false;
}


void KrTextWidget::PositionCursor()
{
	if ( Engine() )
	{
		// Get the position of the bottom line, and 
		// put the cursor there.
		int length = textBox->GetLineLength( 0 );
		int y      = textBox->GetLineY( 0 );

		// Make sure the cursor position is in range.
		cursorPos = GlClamp( cursorPos, 
							 0, length );

		int x = textBox->FontResource()->FontWidthN( textBox->GetText16( 0 ), cursorPos );

		//GLOUTPUT( "Cursor (%d,%d)\n", x, y );
		cursor->SetPos( x, y );
	}
}


void KrTextWidget::KeyFocus( bool receivingFocus )
{
	cursor->SetVisible( receivingFocus );
	if ( receivingFocus )
		PublishEvent( SELECTION, 0, 0, 0, 0 );
}


void KrTextWidget::SetTextChar( const std::string& t )
{
	if ( textBox )
	{
		textBox->SetTextChar( t, 0 );
		cursorPos = textBox->GetLineLength( 0 );
		PositionCursor();
	}
}


void KrTextWidget::GetTextChar( std::string* buffer )
{
	if ( textBox )
		textBox->GetTextChar( buffer, 0 );
}


void KrTextWidget::MouseIn( bool down, bool in )
{
	if ( in )
	{
		holder->SetColor( scheme.CalcHiSec() );
	}
	else
	{
		KrColorTransform none;
		holder->SetColor( none );
	}
}


void KrTextWidget::MouseClick( int down, int x, int y )
{
	bool selectionSent = false;
	if ( down && editable )
	{
		cursorPos = textBox->GetLineLength();
		for( int i=0; i<textBox->GetLineLength(); ++i )
		{
			int lineX = textBox->FontResource()->FontWidthN( textBox->GetText16( 0 ), i );
			if ( lineX >= x )
			{
				cursorPos = i;
				break;		
			}
		}
		PositionCursor();

		KrEventManager::Instance()->GrabKeyFocus( this );
		selectionSent = true;
	}
	if (    down
		 && !selectionSent )		// don't double send selection if we sent one for grabbing focus
	{
		PublishEvent( SELECTION, 0, 0, 0, 0 );
	}
}


