/*
Copyright (c) 2000-2003 Lee Thomason (www.grinninglizard.com)

Grinning Lizard Utilities. Note that software that uses the 
utility package (including Lilith3D and Kyra) have more restrictive
licences which applies to code outside of the utility package.


This software is provided 'as-is', without any express or implied 
warranty. In no event will the authors be held liable for any 
damages arising from the use of this software.

Permission is granted to anyone to use this software for any 
purpose, including commercial applications, and to alter it and 
redistribute it freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must 
not claim that you wrote the original software. If you use this 
software in a product, an acknowledgment in the product documentation 
would be appreciated but is not required.

2. Altered source versions must be plainly marked as such, and 
must not be misrepresented as being the original software.

3. This notice may not be removed or altered from any source 
distribution.
*/

#ifndef KYRA_DYNARRAY_INCLUDED
#define KYRA_DYNARRAY_INCLUDED

#include <limits.h>
#include "glutil.h"
#include "gldebug.h"

/**	A dynamic array class.
	Has a notion of both Size (the allocated
	memory) and Count (the number of items.) Generally, as a user
	of the array you want to be sure you are using Count.
	
	Left to its own devices, GlDynArray will use power of 2 
	memory allocation when it expands.
*/
template <class T>
class GlDynArray
{
  public:
	GlDynArray( int _size )	{ size = 0; count = 0; data = 0; Resize( _size ); }
	GlDynArray()			{ size = 0; count = 0; data = 0; }
	~GlDynArray()			{ delete [] data; }

	GlDynArray( const GlDynArray<T>& copy )			{ size = 0; count = 0; data = 0; this->SetEqual( copy ); }
	void operator=( const GlDynArray<T>& copy )		{ this->SetEqual( copy ); }

	unsigned NotFound()	const { return UINT_MAX; }		///< Returned by failed search.
	bool     Empty()    const { return count == 0; }	///< Returns true if this contains nothing.

	void Clear()	{ delete [] data; size  = 0; count = 0; data = 0; } ///< delete the contents of this array

	/// Array access.
	T& operator[] ( unsigned i ) const	{	GLASSERT( i<count );
											GLASSERT( i>=0 );
											return data[i];		}
 	/// Access to the base of array memory.
	T* Memory()		{ return data; }

	void SetEqual( const GlDynArray<T>& rhs )
	{
		delete [] data;
		data = new T[ rhs.size ];
		for ( unsigned i=0; i<rhs.count; i++ )
		{
			data[i] = rhs.data[i];
		}
		size = rhs.size;
		count = rhs.count;
	}

	/// Fetch an item, by reference, at the index given. 
	T Item( unsigned i ) const	{ GLASSERT( i<count );
								  return data[i]; }
	/// Fetch an item, by pointer, at the index given.
	T* ItemPointer( unsigned i ) const	{ GLASSERT( i<count );
										  return &data[i]; }
	/// Set the item at the given index.
	void SetItem( unsigned i, const T& t )
											{ GLASSERT( i<count );
											  data[i] = t; }

	/** The number of items in the array. (Managed by the user.)
	*/
	unsigned Count() const 	{ return count; }

	/** The allocated memory, in total number of items.
	*/
	unsigned AllocatedSize()	const	{ return size; }

	/** Set the number of items in the array. This will also
		re-allocate memory.
	*/
	void SetCount( unsigned _count )	{	if ( count != _count )
											{
												if ( _count != size )
													ResizePower2( _count );
												count = _count;
											}
										}

	/** Push T to the last element of the array. Will never
		reduce the size, only increase it. (Useful for setting the
		size to something reasonable, and then pushing items
		to the array.

		Will increment count.
	*/				
	unsigned PushBack( const T& t)			
	{
		if ( count+1 > size )
			ResizePower2( count+1 );
		data[count] = t;
		count++;
		return count -1;
	}

	/** Get the last element. */
	T Back()
	{
		GLASSERT( count > 0 );
		return data[ count - 1 ];
	}

	/** Pop the last element of the array. Will never
		change the size of allocated memory, but will
		update the count.
	*/				
	T PopBack()			
	{
		GLASSERT( count > 0 );
		count--;
		return data[count];
	}

	/** Insert an element into this array at index 'n'.
		Will resource and adjust count as necessary.
	*/
	void Insert( T t, unsigned n )
	{
		if ( n > count ) 
			n = count;
		if ( count + 1 > size )
			ResizePower2( count + 1 );
		if ( count > 0 )
		{
			for ( unsigned i = count; i > n; i-- )
				data[i] = data[i-1];
		}
		data[n] = t;
		++count;
	}

	/** Remove the element at index 'n'.
		Adjusts count as necessary. Won't reduce memory.
	*/
	void Remove( unsigned n )
	{
		if ( n >= count ) 
			return;

		GLASSERT( count > 0 );
		if ( count > 0 )
		{
			for ( unsigned i = n; i < count-1; i++ )
				data[i] = data[i+1];
			--count;
		}
	}

	/** Trys to find the selected item (criteria being t == data[i])
		returns an index if successful, NOTFOUND if not.
	*/
	unsigned Find( const T& t ) const
	{	
		for ( unsigned i=0; i<count; i++ )
			if ( data[i] == t )
				return i;
		return NotFound();
	}

	/** Trys to find the selected item (criteria being t == data[i])
		and delete it.
	*/
	bool FindAndDelete( const T& t )
	{	
		for ( unsigned i=0; i<count; i++ )
		{
			if ( data[i] == t )
			{
				Remove( i );
				return true;
			}
		}
		return false;
	}

	/** Resize the array. Will take no action of the array
		size does not change. Will not increase count.

	*/
	void Resize( unsigned _size )	
	{
		if ( _size != size ) {
			T* newData = new T[_size];
			unsigned copy = GlMin( _size, count );

			for( unsigned i=0; i<copy; i++ )
				newData[i] = data[i];

			delete [] data;
			data = newData;

			size = _size;
			count = GlMin( count, _size );
		}
	}

	/** A more useful and efficient resize. Note resize normally
		does not need to be called, since it will be called by other functions.
	*/
	void ResizePower2( unsigned _size )
	{
		unsigned newSize = 1;
		while ( newSize < _size )
			newSize <<= 1;
		Resize( newSize );
	}

	/** Appends another DynArray (of the same type) on to this
		one. The appended array is unchanged. Uses PowerOf2 allocation
		in hopes of optimizing multiple appends.
	*/
	void Append( const GlDynArray<T>& rhs )
	{
		int addAt = size;
		ResizePower2( size + rhs.size );

		for ( int i=0; i<rhs.size; i++ )
			data[i+addAt] = rhs.data[addAt];
	}

	/** Perform a shellsort on the array. Note that "Find" is not
		cordinated with the sort, so Find will still be in linear 
		time.
	*/
	void Sort()
	{
		// Shellsort. Sedgewick, Algorithms in C++. p109
		// Note N = count-1
		int i, j, h;
		T value;
		int N = count-1;

		for( h = 1; h <= N/9; h = 3*h+1 )	
		{}

		for( ; h > 0; h /= 3 )
		{
			for( i = h+1; i <= N; ++i )
			{
				value = data[i];
				j = i;
				while( j>h && data[j-h]>value )
				{
					data[j] = data[j-h];
					j -= h;
				}
				data[j] = value;
			}
		}
	}

  private:
	unsigned count;
	unsigned size;
	T*		 data;

};


#endif
