/*
Copyright (c) 2000-2003 Lee Thomason (www.grinninglizard.com)

Grinning Lizard Utilities. Note that software that uses the 
utility package (including Lilith3D and Kyra) have more restrictive
licences which applies to code outside of the utility package.


This software is provided 'as-is', without any express or implied 
warranty. In no event will the authors be held liable for any 
damages arising from the use of this software.

Permission is granted to anyone to use this software for any 
purpose, including commercial applications, and to alter it and 
redistribute it freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must 
not claim that you wrote the original software. If you use this 
software in a product, an acknowledgment in the product documentation 
would be appreciated but is not required.

2. Altered source versions must be plainly marked as such, and 
must not be misrepresented as being the original software.

3. This notice may not be removed or altered from any source 
distribution.
*/

#include <stdio.h>
#include "glgraph.h"

GlGraph::GlGraph( int _numVertex )
{
	numVertex = _numVertex;
	dataPoint = 0;
	vertex = new Vertex[ numVertex ];
}


GlGraph::~GlGraph()
{
	delete [] dataPoint;
	delete [] vertex;
}


void GlGraph::LowerAddEdge( int v0, int v1, int d01, int d10 )
{
	//int v[2] = { v0, v1 };

	GLASSERT( v0 < numVertex );
	GLASSERT( v1 < numVertex );
	GLASSERT( !dataPoint );

	Edge edge;
	edge.lengthTo = d01;
	edge.lengthFrom = d10;
	edge.toVertex = v1;
	vertex[v0].edgeList.PushFront( edge );
}


void GlGraph::AddBiEdge( int v0, int v1, int distance )
{
	LowerAddEdge( v0, v1, distance, distance );	
	LowerAddEdge( v1, v0, distance, distance );	
}


void GlGraph::AddBiEdge( int v0, int v1, int d01, int d10 )
{
	LowerAddEdge( v0, v1, d01, d10 );	
	LowerAddEdge( v1, v0, d10, d01 );	
}


void GlGraph::AddEdge( int v0, int v1, int distance )
{
	LowerAddEdge( v0, v1, distance, GL_INFINITE );
	LowerAddEdge( v1, v0, GL_INFINITE, distance );
}


void GlGraph::GetPath(	int source, int dest, 
						int *nextVertex, 
 						int *length,
						int *distance )
{
	// Check if we need to calc:
	if ( !vertex[dest].destinationCalculated )
	{
		ShortestPathCalc( dest );
	}
	GLASSERT( dataPoint );
	DataPoint* dp = GetDataPoint( source, dest );

	if ( dp->distance >= GL_INFINITE )
	{
		// There is no connection.
		*nextVertex = -1;
		if ( length )	*length = GL_INFINITE;
		if ( distance )	*distance = GL_INFINITE;
	}
	else
	{
		*nextVertex = dp->vertex;
		if ( distance )
		{
			*distance   = dp->distance;
		}
		if ( length )
		{
			*length = GL_INFINITE;
			GlSListIterator< Edge > it( vertex[source].edgeList );
			for( it.Begin(); !it.Done(); it.Next() )
			{
				if (	it.Current().toVertex == dp->vertex 
					 && it.Current().lengthTo < GL_INFINITE )
				{
					*length = it.Current().lengthTo;
					break;
				}
			}
			GLASSERT( *length != GL_INFINITE );
		}
	}
}


int GlGraph::FindCheapest( GlCircleList<int>* set, int dest )
{
	int ret = -1;
	int distance = GL_INFINITE;

	GlCircleListIterator< int > it( *set );
	for( it.Begin(); !it.Done(); it.Next() )
	{
		DataPoint *dp = GetDataPoint( it.Current(), dest );
		if ( dp->distance < distance )
		{
			distance = dp->distance;
			ret = it.Current();
		}
	}
	return ret;
}


void GlGraph::ShortestPathCalc( int dest )
{
	int i;
	#ifdef DEBUG
		GLOUTPUT( "Shortest path to %d\n", dest );
	#endif

	// The 'calculated' flags and the setQ are redundant to each other. But it is slow to
	// iterate through 'calculated' and slow to fnd in 'setQ' so both are here.
	bool *calculated = new bool[ numVertex ];	// Marks whether the source vertex has been calcalated
	GlCircleList<int>	setQ;					// The set of all the vertices not yet processed

	for( i=0; i<numVertex; ++i )
	{
		calculated[i] = false;
		setQ.PushFront( i );
	}	

	if ( !dataPoint )
	{
		int count = numVertex * numVertex;
		dataPoint = new DataPoint[ count ];
		for ( int source=0; source<numVertex; ++source )
		{
			for( int dest=0; dest<numVertex; ++dest )
			{
				DataPoint* dp = GetDataPoint( source, dest );
				dp->vertex = dest;

				if ( source == dest )
					dp->distance = 0;
				else
					dp->distance = GL_INFINITE;
			}
		}
	}
	vertex[dest].destinationCalculated = true;

	
	// Now, move from set Q to S...processing as we go.
	while ( !setQ.Empty() )
	{
		int cheapest = FindCheapest( &setQ, dest );
		if ( cheapest < 0 )
		{
			// disconnected set.
			break;
		}

		// Get the closest node in Q
		GlCircleNode<int>* node = setQ.Find( cheapest );
		GLASSERT( node );
		setQ.Delete( node );

		DataPoint* cheapDp = GetDataPoint( cheapest, dest );
		calculated[cheapest] = true;

		// For each vertex adjacent to the cheapest, that is still in Q,
		// relax it. Note that edges can have an GL_INFINITE distance, if
		// they have a real distance in one direction, it can be GL_INFINITE 
		// in the other.
		// Also, we are interested in the distance from the 'nextDp' back
		// to the 'cheapest'. It's easy to get confused.

// 		GLOUTPUT( "Relaxing %d\n", cheapest );
		GlSListIterator< Edge > it( vertex[cheapest].edgeList );
		for( it.Begin(); !it.Done(); it.Next() )
		{
			int toVertex = it.Current().toVertex;
			DataPoint* nextDp = GetDataPoint( toVertex, dest );
			if ( !calculated[toVertex] )
			{
				// note we use the 'lengthFrom', which is the
				// disance from 'it.current()' back to 'cheap'
				int distance = cheapDp->distance + it.Current().lengthFrom;
				if ( distance < nextDp->distance )
				{
					nextDp->distance = distance;
					nextDp->vertex = cheapest;
// 					GLOUTPUT( "  to %d, dist=%d\n", toVertex, distance );
				}
			}
		}
	}
	delete [] calculated;
}



#ifdef DEBUG
void GlGraph::DumpVertex()
{
	int i;
	for( i=0; i<numVertex; ++i )
	{
		printf( "vertex %c\n", i + 'a' );
		GlSListIterator< Edge > it( vertex[i].edgeList );
		for( it.Begin(); !it.Done(); it.Next() )
		{
			printf( "  edge to: %c  dist: %2d\n", 
					it.Current().toVertex + 'a',
					it.Current().lengthTo );
		}
	}
}

#endif
