/*
Copyright (c) 2000-2003 Lee Thomason (www.grinninglizard.com)

Grinning Lizard Utilities. Note that software that uses the 
utility package (including Lilith3D and Kyra) have more restrictive
licences which applies to code outside of the utility package.


This software is provided 'as-is', without any express or implied 
warranty. In no event will the authors be held liable for any 
damages arising from the use of this software.

Permission is granted to anyone to use this software for any 
purpose, including commercial applications, and to alter it and 
redistribute it freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must 
not claim that you wrote the original software. If you use this 
software in a product, an acknowledgment in the product documentation 
would be appreciated but is not required.

2. Altered source versions must be plainly marked as such, and 
must not be misrepresented as being the original software.

3. This notice may not be removed or altered from any source 
distribution.
*/

#ifndef GLMAP_INCLUDED
#define GLMAP_INCLUDED

#pragma warning( disable : 4530 )
#pragma warning( disable : 4786 )
#include <string>
#include <ctype.h>
#include "glutil.h"
#include "gldebug.h"
#include "glprime.h"
#include "gltypes.h"

// Standard Hash classes.
class GlHash
{
  public:
	U32 HashValue()	{ return val; }
  protected:
	U32 val;	
};

class GlStringHash : public GlHash
{
  public:
	GlStringHash( const std::string& str )
	{	
		val = 0;
		for ( unsigned i=0; i<str.length() && i < 32; ++i )
		{
			val <<= 1;
			val |= str[i];
		}
	}
};

template < class T >
class GlNumberHash : public GlHash
{
  public:
	GlNumberHash( T num )	{ val = (T) num; }
};


// A map template class. It associates a KEY with a VALUE.
// It uses a HASH class to get a hash value from the KEY. 2 Hash classes are
// provided: GlStringHash and GlNumberHash.

template < class KEY, class VALUE, class HASH >
class GlMap
{
  public:
	GlMap( U32 startSize = 64, U32 grow = 60 );
	~GlMap();

	// Adds a key value pair. Will fail if the key is not unique.
	bool Add( const KEY& key, const VALUE& value );
	
	// Remove a key value pair. Will fail if the key is not found.
	bool Remove( const KEY& key );
	
	// Returns true if the key is found, and writes the result to value.
	// Value will not be set if the key is not found.
	bool Find( const KEY& key, VALUE* value );

	// PRIVATE! But silly template friends too hard.
	enum {
		EXPAND = 4
	};

	struct Item
	{
		KEY		key;
		VALUE	value;
		Item*	next;
	};

	// Grow the hash table. Dicey work, and slow.
	void Grow( unsigned newsize );

	U32 numBuckets;
	Item** buckets;
	U32 grow;
	U32 numItems;
};




template < class KEY, class VALUE, class HASH >
inline GlMap< KEY, VALUE, HASH>::GlMap( U32 startSize, U32 _grow )
{
	numBuckets = GlPrime( startSize, 1 );
	buckets = new Item*[ numBuckets ];
	memset( buckets, 0, sizeof( Item* ) * numBuckets );
	grow = _grow;
	numItems = 0;

	#ifdef DEBUG
		//GLOUTPUT( "Created Map: %d buckets\n", numBuckets );
	#endif
};


template < class KEY, class VALUE, class HASH >
inline GlMap< KEY, VALUE, HASH>::~GlMap()
{
	for( U32 i=0; i<numBuckets; ++i )
	{	
		while ( buckets[i] )
		{
			Item* next = buckets[i]->next;
			delete buckets[i];
			buckets[i] = next;
		}
	}
		
	// Everything is in the new list, create new buckets and put it back in.
	delete [] buckets;
};


template < class KEY, class VALUE, class HASH >
inline void GlMap< KEY, VALUE, HASH>::Grow( unsigned newsize )
{
	#ifdef DEBUG
	int itemcount = 0;
	#endif

	// Yep. Unlink everything, put in a linked list, and re-insert.
	Item *root = 0;
	for( U32 i=0; i<numBuckets; ++i )
	{	
		while ( buckets[i] )
		{
			Item* next = buckets[i]->next;

			buckets[i]->next = root;
			root = buckets[i];

			buckets[i] = next;

			#ifdef DEBUG
			++itemcount;
			#endif
		}
	}
	#ifdef DEBUG
	int comparecount = 0;
	for( Item* it = root; it; it = it->next )
		++comparecount;
	GLASSERT( comparecount == itemcount );
	#endif
		
	// Everything is in the new list, create new buckets and put it back in.
	delete [] buckets;
	buckets = 0;

	#ifdef DEBUG
		//GLOUTPUT( "Rebuilding map (from %d)...", numBuckets );
	#endif

	numBuckets = GlPrime( newsize, 1 );
	#ifdef DEBUG
		//GLOUTPUT( "%d buckets\n", numBuckets );
	#endif

	buckets = new Item*[ numBuckets ];
	memset( buckets, 0, sizeof( Item* ) * numBuckets );

	while ( root )
	{
		Item* next = root->next;

		HASH hash( root->key );
		U32 which = hash.HashValue() % numBuckets;

		root->next = buckets[ which ];
		buckets[ which ] = root;

		root = next;
	}
}


template < class KEY, class VALUE, class HASH >
inline bool GlMap< KEY, VALUE, HASH>::Add( const KEY& key, const VALUE& value )
{
	VALUE dummy;
	if ( Find( key, &dummy ) ) 
		return false;

	// Do we need to get bigger?
	if ( ( ( numItems + 1 ) * 100 / numBuckets ) > grow )
	{
		Grow( GlMax( numItems * EXPAND, numBuckets * EXPAND ) );
	}

	// Add the key in, if it is unique.
	HASH hash( key );
	U32 which = hash.HashValue() % numBuckets;

	Item* item	= new Item;
	item->key	= key;
	item->value = value;
	item->next	= buckets[ which ];

	buckets[ which ] = item;
	++numItems;
	return true;
}


template < class KEY, class VALUE, class HASH >
inline bool GlMap< KEY, VALUE, HASH>::Find( const KEY& key, VALUE* value )
{
	HASH hash( key );
	U32 which = hash.HashValue() % numBuckets;

	Item* item = buckets[which];
	
	while( item )
	{
		if ( item->key == key )
		{
			*value = item->value;
			return true;
		}
		item = item->next;
	}
	return false;	
}


template < class KEY, class VALUE, class HASH >
inline bool GlMap< KEY, VALUE, HASH>::Remove( const KEY& key )
{
	HASH hash( key );
	U32 which = hash.HashValue() % numBuckets;

	Item* item = buckets[which];
	Item* prev = 0;
	
	while( item )
	{
		if ( item->key == key )
		{
			if ( prev )
				prev->next = item->next;
			else
				buckets[ which ] = item->next;

			delete item;
			--numItems;
			return true;
		}
		prev = item;
		item = item->next;
	}
	return false;	
}


template < class KEY, class VALUE, class HASH >
class GlMapIterator
{
  public:
	GlMapIterator( GlMap< KEY, VALUE, HASH >& _map )	{ map = &_map; bucket=0; item=0; }

	void Begin()		{ bucket=0; item=map->buckets[0]; FindValid(); }
	void Next()			{ item=item->next; FindValid(); }
	bool Done()			{ return ( bucket < 0 ); }

	void Current( KEY* key, VALUE* value )
	{
		if ( item )
		{
			*key = item->key;
			*value = item->value;
		}
	}

  private:
	GlMap< KEY, VALUE, HASH >* map;
	int bucket;
	typename GlMap< KEY, VALUE, HASH >::Item* item;

	void FindValid()
	{
		if (	bucket < 0				// we are done
			 || ( item && bucket >=0 ))	// we have a current value
		{
			return;
		}
		GLASSERT( item == 0 );			// else we should have current value
		for ( ++bucket; bucket < map->numBuckets; ++bucket )
		{
			if ( map->buckets[bucket] )
			{
				item = map->buckets[ bucket ];
				return;
			}
		}
		bucket = -1;
		item = 0;
	}
};


#endif

